/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;

import javax.swing.JComponent;
import javax.swing.JSeparator;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.basic.BasicPopupMenuSeparatorUI;

import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.api.ComponentState;
import org.jvnet.substance.api.SubstanceColorScheme;
import org.jvnet.substance.api.SubstanceConstants.MenuGutterFillKind;
import org.jvnet.substance.painter.utils.SeparatorPainterUtils;
import org.jvnet.substance.painter.utils.SubstanceFillBackgroundDelegate;
import org.jvnet.substance.utils.*;
import org.jvnet.substance.utils.menu.MenuUtilities;

/**
 * UI for popup menu separators in <b>Substance</b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstancePopupMenuSeparatorUI extends BasicPopupMenuSeparatorUI {
	/**
	 * Delegate for painting fill background.
	 */
	private SubstanceFillBackgroundDelegate fillBackgroundDelegate;

	public static ComponentUI createUI(JComponent comp) {
		SubstanceCoreUtilities.testComponentCreationThreadingViolation(comp);
		return new SubstancePopupMenuSeparatorUI();
	}

	/**
	 * Creates a new UI delegate.
	 */
	public SubstancePopupMenuSeparatorUI() {
		super();
		this.fillBackgroundDelegate = new SubstanceFillBackgroundDelegate();
	}

	@Override
	public void paint(Graphics g, JComponent c) {
		Graphics2D graphics = (Graphics2D) g.create();

		try {
			JSeparator sep = (JSeparator) c;

			this.fillBackgroundDelegate.updateIfOpaque(graphics, c);
			int xOffset = MenuUtilities.getTextOffset(sep, sep.getParent());

			Dimension s = c.getSize();
			int startX = 0;
			int width = s.width;
			if (c.getComponentOrientation().isLeftToRight()) {
				MenuGutterFillKind fillKind = SubstanceCoreUtilities
						.getMenuGutterFillKind();
				if (fillKind != MenuGutterFillKind.NONE) {
					SubstanceColorScheme scheme = SubstanceColorSchemeUtilities
							.getColorScheme(sep, ComponentState.DEFAULT);
					Color leftColor = ((fillKind == MenuGutterFillKind.SOFT_FILL) || (fillKind == MenuGutterFillKind.HARD)) ? scheme
							.getUltraLightColor()
							: scheme.getLightColor();
					Color rightColor = ((fillKind == MenuGutterFillKind.SOFT_FILL) || (fillKind == MenuGutterFillKind.SOFT)) ? scheme
							.getUltraLightColor()
							: scheme.getLightColor();
					GradientPaint gp = new GradientPaint(0, 0, leftColor,
							xOffset, 0, rightColor);

					graphics.setComposite(TransitionLayout.getAlphaComposite(
							sep, 0.7f, g));
					graphics.setPaint(gp);
					graphics.fillRect(0, 0, xOffset - 2, c.getHeight());
				}

				//graphics.setComposite(TransitionLayout.getAlphaComposite(sep))
				// ;
				// graphics.setColor(c.getForeground());
				// graphics.drawLine(xOffset, 1, s.width, 1);
				// graphics.setColor(c.getBackground());
				// graphics.drawLine(xOffset, 2, s.width, 2);
				// graphics.drawLine(xOffset, 0, xOffset, 0);
				// graphics.drawLine(xOffset, 3, xOffset, 3);
				startX = xOffset - 2;
				width = s.width - startX;
			} else {
				// fix for defect 125 - support of RTL menus
				MenuGutterFillKind fillKind = SubstanceCoreUtilities
						.getMenuGutterFillKind();
				if (fillKind != MenuGutterFillKind.NONE) {
					SubstanceColorScheme scheme = SubstanceColorSchemeUtilities
							.getColorScheme(sep, ComponentState.DEFAULT);
					Color leftColor = ((fillKind == MenuGutterFillKind.HARD_FILL) || (fillKind == MenuGutterFillKind.HARD)) ? scheme
							.getLightColor()
							: scheme.getUltraLightColor();
					Color rightColor = ((fillKind == MenuGutterFillKind.HARD_FILL) || (fillKind == MenuGutterFillKind.SOFT)) ? scheme
							.getLightColor()
							: scheme.getUltraLightColor();
					GradientPaint gp = new GradientPaint(xOffset, 0, leftColor,
							s.width, 0, rightColor);

					graphics.setComposite(TransitionLayout.getAlphaComposite(
							sep, 0.7f, g));
					graphics.setPaint(gp);
					graphics.fillRect(xOffset - 2, 0, s.width, c.getHeight());
				}
				startX = 0;
				width = xOffset - 4;
			}
			graphics.translate(startX, 0);
			graphics.setComposite(TransitionLayout.getAlphaComposite(sep));
			SeparatorPainterUtils.paintSeparator(sep, graphics,
					SubstanceColorSchemeUtilities.getColorScheme(c,
							ComponentState.DEFAULT), width, s.height, sep
							.getOrientation(), true, 2);
		} finally {
			graphics.dispose();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * javax.swing.plaf.basic.BasicPopupMenuSeparatorUI#getPreferredSize(javax
	 * .swing.JComponent)
	 */
	@Override
	public Dimension getPreferredSize(JComponent c) {
		float borderStrokeWidth = SubstanceSizeUtils
				.getBorderStrokeWidth(SubstanceSizeUtils
						.getComponentFontSize(c));
		int prefSize = (int) (Math.ceil(2.0 * borderStrokeWidth));
		return new Dimension(0, prefSize);
	}
}
