/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.skin;

import java.awt.Color;

import org.jvnet.substance.api.*;
import org.jvnet.substance.colorscheme.BaseDarkColorScheme;
import org.jvnet.substance.colorscheme.BaseLightColorScheme;
import org.jvnet.substance.painter.border.FlatBorderPainter;
import org.jvnet.substance.painter.decoration.*;
import org.jvnet.substance.painter.gradient.SubduedGradientPainter;
import org.jvnet.substance.painter.highlight.ClassicHighlightPainter;
import org.jvnet.substance.shaper.ClassicButtonShaper;

/**
 * <code>Nebula</code> skin. This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 * @since version 4.0
 */
public class NebulaSkin extends SubstanceSkin {
	/**
	 * Display name for <code>this</code> skin.
	 */
	public static String NAME = "Nebula";

	/**
	 * Color scheme for active visual state.
	 * 
	 * @author Kirill Grouchnikov
	 */
	protected static class ActiveScheme extends BaseLightColorScheme {
		/**
		 * The main ultra-light color.
		 */
		private static final Color mainUltraLightColor = Color
				.decode("#F6F8FA");

		/**
		 * The main extra-light color.
		 */
		private static final Color mainExtraLightColor = Color
				.decode("#DFE6ED");

		/**
		 * The main light color.
		 */
		private static final Color mainLightColor = Color.decode("#C1D6E9");

		/**
		 * The main medium color.
		 */
		private static final Color mainMidColor = Color.decode("#A3B8CB");

		/**
		 * The main dark color.
		 */
		private static final Color mainDarkColor = Color.decode("#62778A");

		/**
		 * The main ultra-dark color.
		 */
		private static final Color mainUltraDarkColor = Color.decode("#42576A");

		/**
		 * The foreground color.
		 */
		private static final Color foregroundColor = Color.decode("#000000");

		/**
		 * Creates an active scheme for this skin.
		 */
		public ActiveScheme() {
			super("Nebula Active");
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
		 */
		public Color getForegroundColor() {
			return ActiveScheme.foregroundColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
		 */
		public Color getUltraLightColor() {
			return ActiveScheme.mainUltraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
		 */
		public Color getExtraLightColor() {
			return ActiveScheme.mainExtraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
		 */
		public Color getLightColor() {
			return ActiveScheme.mainLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
		 */
		public Color getMidColor() {
			return ActiveScheme.mainMidColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
		 */
		public Color getDarkColor() {
			return ActiveScheme.mainDarkColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
		 */
		public Color getUltraDarkColor() {
			return ActiveScheme.mainUltraDarkColor;
		}
	}

	/**
	 * Color scheme for default visual state.
	 * 
	 * @author Kirill Grouchnikov
	 */
	protected static class DefaultScheme extends BaseLightColorScheme {
		/**
		 * The main ultra-light color.
		 */
		private static final Color mainUltraLightColor = Color
				.decode("#FBFBFC");

		/**
		 * The main extra-light color.
		 */
		private static final Color mainExtraLightColor = Color
				.decode("#F4F7FC");

		/**
		 * The main light color.
		 */
		private static final Color mainLightColor = Color.decode("#F1F2F4");

		/**
		 * The main medium color.
		 */
		private static final Color mainMidColor = Color.decode("#D6D9DF");

		/**
		 * The main dark color.
		 */
		private static final Color mainDarkColor = Color.decode("#95989E");

		/**
		 * The main ultra-dark color.
		 */
		private static final Color mainUltraDarkColor = Color.decode("#75787E");

		/**
		 * The foreground color.
		 */
		private static final Color foregroundColor = Color.decode("#2A2E36");

		/**
		 * Creates a default scheme for this skin.
		 */
		public DefaultScheme() {
			super("Nebula Default");
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
		 */
		public Color getForegroundColor() {
			return DefaultScheme.foregroundColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
		 */
		public Color getUltraLightColor() {
			return DefaultScheme.mainUltraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
		 */
		public Color getExtraLightColor() {
			return DefaultScheme.mainExtraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
		 */
		public Color getLightColor() {
			return DefaultScheme.mainLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
		 */
		public Color getMidColor() {
			return DefaultScheme.mainMidColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
		 */
		public Color getDarkColor() {
			return DefaultScheme.mainDarkColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
		 */
		public Color getUltraDarkColor() {
			return DefaultScheme.mainUltraDarkColor;
		}
	}

	/**
	 * Color scheme for default visual state.
	 * 
	 * @author Kirill Grouchnikov
	 */
	protected static class DisabledScheme extends BaseLightColorScheme {
		/**
		 * The main ultra-light color.
		 */
		private static final Color mainUltraLightColor = Color
				.decode("#E3EFE9");

		/**
		 * The main extra-light color.
		 */
		private static final Color mainExtraLightColor = Color
				.decode("#DFE2E6");

		/**
		 * Creates a disabled scheme for this skin.
		 */
		public DisabledScheme() {
			super("Nebula Disabled");
		}

		/**
		 * The main light color.
		 */
		private static final Color mainLightColor = Color.decode("#DADDE3");

		/**
		 * The main medium color.
		 */
		private static final Color mainMidColor = Color.decode("#D6D9DF");

		/**
		 * The main dark color.
		 */
		private static final Color mainDarkColor = Color.decode("#C9CCD2");

		/**
		 * The main ultra-dark color.
		 */
		private static final Color mainUltraDarkColor = Color.decode("#BCBFC5");

		/**
		 * The foreground color.
		 */
		private static final Color foregroundColor = Color.decode("#848B98");

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
		 */
		public Color getForegroundColor() {
			return DisabledScheme.foregroundColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
		 */
		public Color getUltraLightColor() {
			return DisabledScheme.mainUltraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
		 */
		public Color getExtraLightColor() {
			return DisabledScheme.mainExtraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
		 */
		public Color getLightColor() {
			return DisabledScheme.mainLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
		 */
		public Color getMidColor() {
			return DisabledScheme.mainMidColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
		 */
		public Color getDarkColor() {
			return DisabledScheme.mainDarkColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
		 */
		public Color getUltraDarkColor() {
			return DisabledScheme.mainUltraDarkColor;
		}
	}

	/**
	 * Color scheme for {@link ComponentState#ROLLOVER_UNSELECTED} state.
	 * 
	 * @author Kirill Grouchnikov
	 */
	protected static class RolloverUnselectedScheme extends
			BaseLightColorScheme {
		/**
		 * The main ultra-light color.
		 */
		private static final Color mainUltraLightColor = Color
				.decode("#FFFFFF");

		/**
		 * The main extra-light color.
		 */
		private static final Color mainExtraLightColor = Color
				.decode("#FDFDFE");

		/**
		 * The main light color.
		 */
		private static final Color mainLightColor = Color.decode("#F7F8FA");

		/**
		 * The main medium color.
		 */
		private static final Color mainMidColor = Color.decode("#E9ECF2");

		/**
		 * The main dark color.
		 */
		private static final Color mainDarkColor = Color.decode("#7A7E86");

		/**
		 * The main ultra-dark color.
		 */
		private static final Color mainUltraDarkColor = Color.decode("#55585E");

		/**
		 * The foreground color.
		 */
		private static final Color foregroundColor = Color.decode("#2A2E36");

		/**
		 * Creates a rollover unselected scheme for this skin.
		 */
		public RolloverUnselectedScheme() {
			super("Nebula Rollover Unselected");
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
		 */
		public Color getForegroundColor() {
			return RolloverUnselectedScheme.foregroundColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
		 */
		public Color getUltraLightColor() {
			return RolloverUnselectedScheme.mainUltraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
		 */
		public Color getExtraLightColor() {
			return RolloverUnselectedScheme.mainExtraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
		 */
		public Color getLightColor() {
			return RolloverUnselectedScheme.mainLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
		 */
		public Color getMidColor() {
			return RolloverUnselectedScheme.mainMidColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
		 */
		public Color getDarkColor() {
			return RolloverUnselectedScheme.mainDarkColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
		 */
		public Color getUltraDarkColor() {
			return RolloverUnselectedScheme.mainUltraDarkColor;
		}
	}

	/**
	 * Color scheme for {@link ComponentState#ROLLOVER_SELECTED} state.
	 * 
	 * @author Kirill Grouchnikov
	 */
	protected static class RolloverSelectedScheme extends BaseLightColorScheme {
		/**
		 * The main ultra-light color.
		 */
		private static final Color mainUltraLightColor = Color
				.decode("#F8FAFC");

		/**
		 * The main extra-light color.
		 */
		private static final Color mainExtraLightColor = Color
				.decode("#E8FDFF");

		/**
		 * The main light color.
		 */
		private static final Color mainLightColor = Color.decode("#D4E9FC");

		/**
		 * The main medium color.
		 */
		private static final Color mainMidColor = Color.decode("#B6CBDE");

		/**
		 * The main dark color.
		 */
		private static final Color mainDarkColor = Color.decode("#3B556D");

		/**
		 * The main ultra-dark color.
		 */
		private static final Color mainUltraDarkColor = Color.decode("#00051D");

		/**
		 * The foreground color.
		 */
		private static final Color foregroundColor = Color.decode("#000000");

		/**
		 * Creates a rollover selected scheme for this skin.
		 */
		public RolloverSelectedScheme() {
			super("Nebula Rollover Selected");
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
		 */
		public Color getForegroundColor() {
			return RolloverSelectedScheme.foregroundColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
		 */
		public Color getUltraLightColor() {
			return RolloverSelectedScheme.mainUltraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
		 */
		public Color getExtraLightColor() {
			return RolloverSelectedScheme.mainExtraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
		 */
		public Color getLightColor() {
			return RolloverSelectedScheme.mainLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
		 */
		public Color getMidColor() {
			return RolloverSelectedScheme.mainMidColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
		 */
		public Color getDarkColor() {
			return RolloverSelectedScheme.mainDarkColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
		 */
		public Color getUltraDarkColor() {
			return RolloverSelectedScheme.mainUltraDarkColor;
		}
	}

	/**
	 * Color scheme for {@link ComponentState#PRESSED_SELECTED} and
	 * {@link ComponentState#PRESSED_UNSELECTED} states.
	 * 
	 * @author Kirill Grouchnikov
	 */
	protected static class PressedScheme extends BaseDarkColorScheme {
		/**
		 * The main ultra-light color.
		 */
		private static final Color mainUltraLightColor = Color
				.decode("#8FA9C0");

		/**
		 * The main extra-light color.
		 */
		private static final Color mainExtraLightColor = Color
				.decode("#7695B2");

		/**
		 * The main light color.
		 */
		private static final Color mainLightColor = Color.decode("#5B89B4");

		/**
		 * The main medium color.
		 */
		private static final Color mainMidColor = Color.decode("#33628C");

		/**
		 * The main dark color.
		 */
		private static final Color mainDarkColor = Color.decode("#1C3851");

		/**
		 * The main ultra-dark color.
		 */
		private static final Color mainUltraDarkColor = Color.decode("#000000");

		/**
		 * The foreground color.
		 */
		private static final Color foregroundColor = Color.decode("#FFFFFF");

		/**
		 * Creates a pressed for this skin.
		 */
		public PressedScheme() {
			super("Nebula Pressed");
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
		 */
		public Color getForegroundColor() {
			return PressedScheme.foregroundColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
		 */
		public Color getUltraLightColor() {
			return PressedScheme.mainUltraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
		 */
		public Color getExtraLightColor() {
			return PressedScheme.mainExtraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
		 */
		public Color getLightColor() {
			return PressedScheme.mainLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
		 */
		public Color getMidColor() {
			return PressedScheme.mainMidColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
		 */
		public Color getDarkColor() {
			return PressedScheme.mainDarkColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
		 */
		public Color getUltraDarkColor() {
			return PressedScheme.mainUltraDarkColor;
		}
	}

	/**
	 * Creates a new <code>Nebula</code> skin.
	 */
	public NebulaSkin() {
		SubstanceColorScheme activeScheme = new ActiveScheme();
		SubstanceColorScheme defaultScheme = new DefaultScheme();
		SubstanceColorScheme rolloverUnselectedScheme = new RolloverUnselectedScheme();
		SubstanceColorScheme pressedScheme = new PressedScheme();
		SubstanceColorScheme rolloverSelectedScheme = new RolloverSelectedScheme();
		SubstanceColorScheme disabledScheme = new DisabledScheme();

		SubstanceColorSchemeBundle defaultSchemeBundle = new SubstanceColorSchemeBundle(
				activeScheme, defaultScheme, disabledScheme);
		defaultSchemeBundle.registerColorScheme(rolloverUnselectedScheme,
				ComponentState.ROLLOVER_UNSELECTED);
		defaultSchemeBundle.registerColorScheme(rolloverSelectedScheme,
				ComponentState.ROLLOVER_SELECTED);
		defaultSchemeBundle.registerColorScheme(pressedScheme,
				ComponentState.PRESSED_SELECTED,
				ComponentState.PRESSED_UNSELECTED, ComponentState.ARMED,
				ComponentState.ROLLOVER_ARMED);

		defaultSchemeBundle.registerHighlightColorScheme(pressedScheme, 0.6f,
				ComponentState.ROLLOVER_UNSELECTED);
		defaultSchemeBundle.registerHighlightColorScheme(pressedScheme, 0.8f,
				ComponentState.SELECTED);
		defaultSchemeBundle.registerHighlightColorScheme(pressedScheme, 0.95f,
				ComponentState.ROLLOVER_SELECTED);
		defaultSchemeBundle.registerHighlightColorScheme(pressedScheme, 0.8f,
				ComponentState.ARMED, ComponentState.ROLLOVER_ARMED);

		registerDecorationAreaSchemeBundle(defaultSchemeBundle,
				DecorationAreaType.NONE);

		this.registerAsDecorationArea(activeScheme.saturate(-0.5),
				DecorationAreaType.PRIMARY_TITLE_PANE,
				DecorationAreaType.SECONDARY_TITLE_PANE,
				DecorationAreaType.HEADER, DecorationAreaType.FOOTER,
				DecorationAreaType.GENERAL);

		this.setPaintingDropShadows(DecorationAreaType.TOOLBAR);

		this.buttonShaper = new ClassicButtonShaper();
		this.gradientPainter = new SubduedGradientPainter();

		MarbleNoiseDecorationPainter decorationPainter = new MarbleNoiseDecorationPainter();
		decorationPainter.setPaintingSeparators(true);
		decorationPainter.setBaseDecorationPainter(new ArcDecorationPainter());
		decorationPainter.setTextureAlpha(0.3f);
		this.decorationPainter = decorationPainter;

		this.highlightPainter = new ClassicHighlightPainter();
		this.borderPainter = new FlatBorderPainter();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.skin.SubstanceSkin#getDisplayName()
	 */
	public String getDisplayName() {
		return NAME;
	}
}
