#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - actors
 *   
 *   This module provides definitions related to actors, which represent
 *   characters in the game.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Implied command modes 
 */
enum ModePlayer, ModeNPC;


/* ------------------------------------------------------------------------ */
/*
 *   A Topic is an object representing some piece of knowledge in the
 *   story.  Actors can use Topic objects in commands such as "ask" and
 *   "tell".
 *   
 *   A physical simulation object can be a Topic through multiple
 *   inheritance.  In addition, a game can define Topic objects for
 *   abstract conversation topics that don't correspond to simulation
 *   objects; for example, a topic could be created for "the meaning of
 *   life" to allow a command such as "ask guru about meaning of life."
 *   
 *   The key distinction between Topic objects and regular objects is that
 *   a Topic can represent an abstract, non-physical concept that isn't
 *   connected to any "physical" object in the simulation.  
 */
class Topic: VocabObject
    /*
     *   Is the topic known?  If this is true, the topic is in scope for
     *   actions that operate on topics, such as "ask about" and "tell
     *   about."  If this is nil, the topic isn't known.  We don't
     *   distinguish among actors; we simply keep track of a single known
     *   status for everyone.
     *   
     *   By default, we mark all topics as known to begin with, which
     *   allows discussion of any topic at any time.  Some authors prefer
     *   to keep track of which topics the player character actually has
     *   reason to know about within the context of the game, making
     *   topics available for conversation only after they become known
     *   for some good reason, such as another character mentioning them
     *   in conversation.  
     */
    isKnown = true
;


/* ------------------------------------------------------------------------ */
/*
 *   FollowInfo - this is an object that tracks an actor's knowledge of
 *   the objects that the actor can follow, which are objects that actor
 *   has witnessed leaving the current location.  We keep track of each
 *   followable object and the direction we saw it depart.  
 */
class FollowInfo: object
    /* the object we can follow */
    obj = nil

    /* the TravelConnector the object traversed to leave */
    connector = nil

    /* 
     *   The source location - this is the location we saw the object
     *   depart.  We keep track of this because an actor can follow an
     *   object only if the actor is starting from the same location where
     *   the actor saw the object depart.  
     */
    sourceLocation = nil

    /*
     *   The action that was performed.  When we follow the actor, we
     *   simply perform the same action that the actor performed.  This
     *   ensures that the "follow" command is treated exactly like any
     *   means of traveling the same route, and that we run all of the
     *   same checks on the travel.  
     */
    origAction = nil
;


/* ------------------------------------------------------------------------ */
/*
 *   Postures.  A posture describes how an actor is internally positioned:
 *   standing, lying, sitting.  We represent postures with objects of
 *   class Posture to make it easier to add new game-specific postures.  
 */
class Posture: object
    /*
     *   The property of the actor's actual container that retrieves the
     *   nominal container for this posture.  When we describe an actor in
     *   this posture, we'll use this to determine what to describe as the
     *   actor's container.  
     */
    nominalContainerProp = nil

    /*
     *   Display the default "I am here" message for an actor in this
     *   posture.  By default, we use the describePosture() routine to
     *   generate the default message using our actorHereProp selector.
     *   
     *   Subclasses using this default implementation should set
     *   actorHereProp to the appropriate property for posture-specific
     *   messages.  
     */
    actorHereDesc(actor) { describePosture(actor, actorHereProp, nil); }
    actorHereProp = nil

    /*
     *   Show a status line addendum for the given actor in this posture.
     *   By default, we use describePosture() to generate the default
     *   status line addendum using our statusProp selector.
     *   
     *   'Room' is the room whose status we are describing.  If the room
     *   is the same as the actor's nominal location for the posture, we
     *   will not show the room name as part of the posture description,
     *   since this would be redundant.  So, if we're sitting in a box,
     *   and we're describing the location as the box, we'll say something
     *   like "Box (sitting)" rather than "Box (sitting in the box)".
     *   
     *   Subclasses using this default implementation should set
     *   statusProp to the appropriate posture-specific message property.  
     */
    statusPosture(actor, room) { describePosture(actor, statusProp, room); }
    statusProp = nil

    /*
     *   Describe an actor in this posture, as part of the actor's
     *   "examine" description. 
     */
    actorPostureDesc(actor)
        { describePosture(actor, actorPostureDescProp, nil); }
    actorPostureDescProp = nil

    /*
     *   Produce a default acknowledgment of a posture change. 
     */
    okayPostureChange(actor)
    {
        local cont;

        /* use the nominal posture container, if possible */
        cont = actor.location.(nominalContainerProp);
        if (cont != nil && libGlobal.playerChar.canSee(cont))
        {
            /* we can see the container, so let it handle the report */
            cont.(okayPostureChangeProp)(actor);
        }
        else
        {
            /* can't see the container, so produce a generic message */
            defaultReport(okayPostureChangeProp);
        }
    }
    okayPostureChangeProp = nil

    /*
     *   Describe an actor in this posture, as part of a list of actors in
     *   a nested room.  
     */
    listActorPosture(actor, pov)
    {
        local cont;
        
        /* 
         *   Use the nominal posture container; if we can't see it from
         *   the point of view doing the describing, don't describe the
         *   actor at all.  
         */
        cont = actor.location.(nominalContainerProp);
        if (cont != nil && pov.canSee(cont))
            cont.(listActorPostureProp)(actor);
    }
    listActorPostureProp = nil

    /*
     *   Produce a list prefix/suffix for a group of actors listed with
     *   their postures.  
     */
    listActorGroupPrefix(loc, cnt)
        { libMessages.(listActorGroupPrefixProp)(loc, cnt); }
    listActorGroupSuffix(loc, cnt)
        { libMessages.(listActorGroupSuffixProp)(loc, cnt); }

    listActorGroupPrefixProp = nil
    listActorGroupSuffixProp = nil

    /* 
     *   Generate a description of the posture for the given actor.
     *   contMsg is the message property to use to display the
     *   description.  If we can see the nominal container, we'll invoke
     *   the property on the nominal container; otherwise, we'll simply
     *   use this as a generic libMessages property.
     *   
     *   'contToIgnore' is a container to ignore.  If the nominal
     *   container that we find for the posture is the same as
     *   contToIgnore, we'll use the generic library message, rather than
     *   going through the container object.
     *   
     *   This is a service routine for the various message generators; it
     *   is not generally invoked directly from unrelated classes.  
     */
    describePosture(actor, contMsg, contToIgnore)
    {
        local cont;
        
        /* 
         *   Describe the actor as here, sitting in the nominal posture
         *   container.  If the nominal container is not visible to the
         *   player character, merely say that the actor is in this
         *   posture without specifying where.  
         */
        cont = actor.location.(nominalContainerProp);
        if (cont != nil
            && cont != contToIgnore
            && libGlobal.playerChar.canSee(cont))
        {
            /* the player can see the container - show it */
            cont.(contMsg)(actor);
        }
        else
        {
            /* 
             *   the player can't see the container - just say we're in
             *   this posture, without saying where 
             */
            libMessages.(contMsg);
        }
    }

    /* 
     *   Try getting the current actor into this posture within the
     *   actor's current location, by running an appropriate implied
     *   command.  
     */
    tryMakingPosture(loc) { }
;

/*
 *   Standing posture - this is the default posture, which an actor
 *   normally uses for travel.  Actors are generally in this posture any
 *   time they are not sitting on something, lying on something, or
 *   similar. 
 */
standing: Posture
    nominalContainerProp = &getNominalStandingContainer
    actorHereProp = &actorStandingHere
    statusProp = &statusStanding
    actorPostureDescProp = &actorStandingDesc
    okayPostureChangeProp = &okayStand

    listActorPostureProp = &listActorStanding
    listActorGroupPrefixProp = &actorStandingGroupPrefix
    listActorGroupSuffixProp = &actorStandingGroupSuffix

    tryMakingPosture(loc) { return tryImplicitAction(Stand); }
;

/*
 *   Sitting posture. 
 */
sitting: Posture
    nominalContainerProp = &getNominalSittingContainer
    actorHereProp = &actorSittingHere
    statusProp = &statusSitting
    actorPostureDescProp = &actorSittingDesc
    okayPostureChangeProp = &okaySit

    listActorPostureProp = &listActorSitting
    listActorGroupPrefixProp = &actorSittingGroupPrefix
    listActorGroupSuffixProp = &actorSittingGroupSuffix

    tryMakingPosture(loc) { return tryImplicitAction(SitOn, loc); }
;

/*
 *   Lying posture. 
 */
lying: Posture
    nominalContainerProp = &getNominalLyingContainer
    actorHereProp = &actorLyingHere
    statusProp = &statusLying
    actorPostureDescProp = &actorLyingDesc
    okayPostureChangeProp = &okayLie

    listActorPostureProp = &listActorLying
    listActorGroupPrefixProp = &actorLyingGroupPrefix
    listActorGroupSuffixProp = &actorLyingGroupSuffix

    tryMakingPosture(loc) { return tryImplicitAction(LieOn, loc); }
;


/* ------------------------------------------------------------------------ */
/*
 *   An Actor is a living person, animal, or other entity with a will of
 *   its own.  Actors can usually be addressed with targeted commands
 *   ("bob, go north"), and with commands like ASK ABOUT, TELL ABOUT, GIVE
 *   TO, and SHOW TO.
 *   
 *   Note that, by default, an Actor can be picked up and moved with
 *   commands like TAKE, PUT IN, and so on.  This is suitable for some
 *   kinds of actors but not for others: it might make sense with a cat or
 *   a small dog, but not with a bank guard or an orc.  For an actor that
 *   can't be taken, use the UntakeableActor or one of its subclasses.
 *   
 *   An actor's contents are the things the actor is carrying or wearing.  
 */
class Actor: Thing, Schedulable, Traveler
    /* flag: we're an actor */
    isActor = true

    /* 
     *   Hide actors from 'all' by default.  The kinds of actions that
     *   normally apply to 'all' and the kinds that normally apply to
     *   actors have pretty low overlap.
     *   
     *   If a particular actor looks a lot like an inanimate object, it
     *   might want to override this to participate in 'all' for most or
     *   all actions.  
     */
    hideFromAll(action) { return true; }

    /* 
     *   Actors are not listed with the ordinary objects in a room's
     *   description.  However, an actor is listed as part of an inventory
     *   description.
     */
    isListed = nil
    isListedInContents = nil
    isListedInInventory = true

    /* the contents of an actor aren't listed in a room's description */
    contentsListed = nil

    /*
     *   Get my listing group for my "I am here" description as part of a
     *   room list.  By default, we'll list with other actors occupying
     *   the same nested room in the same posture.  
     */
    actorHereListWith()
    {
        local group;

        /* get the group for the posture */
        group = location.listWithActorIn(posture);

        /* 
         *   if we have a group, return a list containing the group;
         *   otherwise return an empty list 
         */
        return (group == nil ? [] : [group]);
    }

    /*
     *   Actor "I am here" description.  This is displayed as part of the
     *   description of a room - it describes the actor as being present
     *   in the room.  By default, we display a simple "actor is here"
     *   message; actors might wish to override this to describe what the
     *   actor is doing, or something else more distinctive than the
     *   default message, to make the actor more lifelike.  
     */
    actorHereDesc
    {
        /* 
         *   by default, display the appropriate "I am here" message for
         *   my posture 
         */
        posture.actorHereDesc(self);
    }

    /*
     *   Full description.  By default, we'll show either the pcDesc or
     *   npcDesc, depending on whether we're the current player character
     *   or a non-player character.
     *   
     *   For a non-player character, we'll also call the actor's
     *   holdingDesc after displaying the npcDesc.
     */
    desc
    {
        /* 
         *   show the appropriate messages, depending on whether we're the
         *   player character or a non-player character 
         */
        if (isPlayerChar())
        {
            /* show the player character description */
            pcDesc;
        }
        else
        {
            /* show the non-player character description */
            npcDesc;

            /* show my posture */
            postureDesc;

            /* show my inventory */
            holdingDesc;
        }
    }

    /* the default description when we examine the player character */
    pcDesc { libMessages.pcDesc(self); }

    /* 
     *   Show the description when we examine a non-player character.  By
     *   default, we'll show this as a "default descriptive report," since
     *   it simply says that there's nothing special to say.
     *   
     *   Note that whenever this is overridden with an actual description,
     *   you shouldn't bother to use defaultDescReport - simply display
     *   the descriptive message directly:
     *   
     *   npcDesc = "He's wearing a gorilla costume. " 
     */
    npcDesc { defaultDescReport(&npcDesc, self); }

    /* show my posture */
    postureDesc
    {
        /* show my posture */
        posture.actorPostureDesc(self);
    }

    /* 
     *   Describe my inventory as part of my description - this is only
     *   called when we examine an NPC.  If an NPC doesn't wish to have
     *   its inventory listed as part of its description, it can simply
     *   override this to do nothing.  
     */
    holdingDesc
    {
        /* 
         *   Show my inventory in paragraph ("wide") mode, using the
         *   special listers for actor descriptions. 
         */
        showInventoryWith(nil, holdingDescInventoryLister,
                          holdingDescWearingLister);
    }

    /*
     *   refer to the player character with my player character referral
     *   person, and refer to all other characters in the third person 
     */
    referralPerson { return isPlayerChar() ? pcReferralPerson : ThirdPerson; }

    /* by default, refer to the player character in the second person */
    pcReferralPerson = SecondPerson

    /* determine if I'm the player character */
    isPlayerChar() { return libGlobal.playerChar == self; }

    /*
     *   Implicit command handling style for this actor.  There are two
     *   styles for handling implied commands: "player" and "NPC",
     *   indicated by the enum codes ModePlayer and ModeNPC, respectively.
     *   
     *   In "player" mode, each implied command is announced with a
     *   description of the command to be performed; DEFAULT responses are
     *   suppressed; and failures are shown.  Furthermore, interactive
     *   requests for more information from the parser are allowed.
     *   Transcripts like this result:
     *   
     *   >open door
     *.  (first opening the door)
     *.  (first unlocking the door)
     *.  What do you want to unlock it with?
     *   
     *   In "NPC" mode, implied commands are treated as complete and
     *   separate commands.  They are not announced; default responses are
     *   shown; failures are NOT shown; and interactive requests for more
     *   information are not allowed.  When an implied command fails in NPC
     *   mode, the parser acts as though the command had never been
     *   attempted.
     *   
     *   By default, we return ModePlayer if we're the player character,
     *   ModeNPC if not (thus the respective names of the modes).  Some
     *   authors might prefer to use "player mode" for NPC's as well as for
     *   the player character, which is why the various parts of the parser
     *   that care about this mode consult this method rather than simply
     *   testing the PC/NPC status of the actor.  
     */
    impliedCommandMode() { return isPlayerChar() ? ModePlayer : ModeNPC; }

    /*
     *   An actor can limit the cumulative amount of bulk being held, and
     *   the maximum bulk of any one object held, using bulkCapacity and
     *   maxSingleBulk.  These properties are analogous to the same ones
     *   in Container.
     */
    bulkCapacity = 10000
    maxSingleBulk = 10

    /*
     *   An actor can limit the cumulative amount of weight being held,
     *   using weightCapacity.  By default we make this so large that
     *   there is effectively no limit to how much weight an actor can
     *   carry.  
     */
    weightCapacity = 10000

    /*
     *   Can I own the given object?  By default, an actor can own
     *   anything.  
     */
    canOwn(obj) { return true; }

    /*
     *   Get the preconditions for travel.  By default, we require that
     *   the actor is standing.
     *   
     *   Note that these preconditions apply only when the actor is the
     *   traveler.  If the actor is in a vehicle, so that the vehicle is
     *   the traveler in a given travel operation, the vehicle's
     *   travelerPreCond conditions are used instead of ours.  
     */
    travelerPreCond = [actorStanding]

    /* by default, actors are listed when they arrive aboard a vehicle */
    isListedAboardVehicle = true

    /*
     *   Get the object that's actually going to move when this actor
     *   travels.  In most cases this is simply the actor; but when the
     *   actor is in a vehicle, travel commands move the vehicle, not the
     *   actor: the actor stays in the vehicle while the vehicle moves to
     *   a new location.  We determine this by asking our immediate
     *   location what it thinks about the situation.
     *   
     *   If we have a special traveler explicitly set, it overrides the
     *   traveler indicated by the location.  
     */
    getTraveler()
    {
        /* 
         *   return our special traveler if we have one, otherwise the
         *   traveler indicated by our location
         */
        if (specialTraveler != nil)
            return specialTraveler;
        else
            return location.getTraveler(self);
    }

    /* 
     *   Get the actors involved in travel, when we're acting in our role
     *   as a Traveler.  When the Traveler is simply the Actor, the only
     *   actor involved in the travel is 'self'. 
     */
    getTravelerActors = [self]

    /*
     *   Set the "special traveler."  When this is set, we explicitly
     *   perform travel through this object rather than through the
     *   traveler indicated by our location.  Returns the old value, so
     *   that the old value can be restored when the caller has finished
     *   its need for the special traveler.  
     */
    setSpecialTraveler(traveler)
    {
        local oldVal;

        /* remember the old value so that we can return it */
        oldVal = specialTraveler;

        /* remember the new value */
        specialTraveler = traveler;

        /* return the old value */
        return oldVal;
    }

    /* our special traveler */
    specialTraveler = nil

    /*
     *   Try moving the actor into the given room in preparation for
     *   travel, using pre-condition rules. 
     */
    checkMovingTravelerInto(room, allowImplicit)
    {
        /* try moving the actor into the room */
        return room.checkMovingActorInto(allowImplicit);
    }

    /*
     *   Check to ensure the actor is ready to enter the given nested
     *   room, using pre-condition rules.  By default, we'll ask the given
     *   nested room to handle it.  
     */
    checkReadyToEnterNestedRoom(dest, allowImplicit)
    {
        /* ask the destination to do the work */
        return dest.checkActorReadyToEnterNestedRoom(allowImplicit);
    }

    /*
     *   Travel within a location, as from a room to a contained nested
     *   room.  This should generally be used in lieu of travelTo when
     *   traveling between locations that are related directly by
     *   containment rather than with TravelConnector objects.
     *   
     *   Travel within a location is not restricted by darkness; we assume
     *   that if the nested objects are in scope at all, travel among them
     *   is allowed.
     *   
     *   This type of travel does not trigger calls to travelerLeaving()
     *   or travelerArriving().  To mitigate this loss of notification, we
     *   call actorTravelingWithin() on the source and destination
     *   objects.  
     */
    travelWithin(dest)
    {
        local origin;

        /* if I'm not going anywhere, ignore the operation */
        if (dest == location)
            return;

        /* remember my origin */
        origin = location;
        
        /* notify the source that we're traveling within a room */
        if (origin != nil)
            origin.actorTravelingWithin(origin, dest);

        /* 
         *   if our origin and destination have different effective follow
         *   locations, track the follow 
         */
        if (origin != nil
            && dest != nil
            && origin.effectiveFollowLocation != dest.effectiveFollowLocation)
        {
            /* 
             *   track the follow information; we're not moving along a
             *   connector, so there is no connector associated with the
             *   tracking information 
             */
            sendTrackFollowInfo(nil);
        }

        /* move me to the destination */
        moveInto(dest);

        /* 
         *   recalculate the global sense context for message generation
         *   purposes, since we've moved to a new location 
         */
        gAction.recalcSenseContext();

        /* notify the destination of the interior travel */
        if (dest != nil)
            dest.actorTravelingWithin(origin, dest);
    }

    /*
     *   Check for travel in the dark.  If we're in a dark room, and our
     *   destination is a dark room, ask the connector for guidance.
     *   
     *   Travel connectors normally call this before invoking our
     *   travelTo() method to carry out the travel.  The darkness check
     *   usually must be made before any barrier checks.  
     */
    checkDarkTravel(dest, connector)
    {
        local origin;
        
        /* 
         *   If we're not in the dark in the current location, there's no
         *   need to check for dark-to-dark travel; light-to-dark travel
         *   is always allowed. 
         */
        if (isLocationLit())
            return;

        /* get the origin - this is the traveler's location */
        origin = getTraveler().location;

        /*
         *   Check to see if the connector itself is visible in the dark.
         *   If it is, then allow the travel without restriction.  
         */
        if (connector.isConnectorVisibleInDark(origin, self))
            return;

        /*
         *   We are attempting dark-to-dark travel.  We allow or disallow
         *   this type of travel on a per-connector basis, so ask the
         *   connector to handle it.  If the connector wishes to disallow
         *   the travel, it will display an appropriate failure report and
         *   terminate the command with 'exit'.  
         */
        connector.darkTravel(self, dest);
    }

    /*
     *   Travel to a new location. 
     */
    travelTo(dest, connector, backConnector)
    {
        /* send the request to the traveler */
        getTraveler().travelerTravelTo(dest, connector, backConnector);
    }

    /*
     *   Remember the last door I traveled through.  We use this
     *   information for disambiguation, to boost the likelihood that an
     *   actor that just traveled through a door is referring to the same
     *   door in a subsequent "close" command.  
     */
    rememberLastDoor(obj) { lastDoorTraversed = obj; }

    /*
     *   Remember our most recent travel.  If we know the back connector
     *   (i.e., the connector that reverses the travel we're performing),
     *   then we'll be able to accept a GO BACK command to attempt to
     *   return to the previous location.  
     */
    rememberTravel(origin, dest, backConnector)
    {
        /* remember the destination of the travel, and the connector back */
        lastTravelDest = dest;
        lastTravelBack = backConnector;
    }

    /*
     *   Reverse the most recent travel.  If we're still within the same
     *   destination we reached in the last travel, and we know the
     *   connector we arrived through (i.e., the "back connector" for the
     *   last travel, which reverses the connector we took to get here),
     *   then try traveling via the connector.  
     */
    reverseLastTravel()
    {
        /* 
         *   If we don't know the connector back to our previous location,
         *   we obviously can't reverse the travel.  If we're not still in
         *   the same location as the previous travel's destination, then
         *   we can't reverse the travel either, because the back
         *   connector isn't applicable to our current location.  (This
         *   latter condition could only happen if we've been moved
         *   somewhere without ordinary travel occurring, but this is a
         *   possibility.) 
         */
        if (lastTravelBack == nil
            || lastTravelDest == nil
            || !isIn(lastTravelDest))
        {
            reportFailure(&cannotGoBack);
            exit;
        }

        /* attempt travel via our back connector */
        nestedAction(TravelVia, lastTravelBack);
    }

    /* the last door I traversed */
    lastDoorTraversed = nil

    /* the destination and back connector for our last travel */
    lastTravelDest = nil
    lastTravelBack = nil

    /*
     *   Get the default interlocutor.  By default, we'll address new
     *   conversational commands (ASK ABOUT, TELL ABOUT, SHOW TO) to the
     *   last conversational partner, if that actor is still within range.
     */
    getDefaultInterlocutor()
    {
        /* 
         *   if we've talked to someone before, and we can still talk to
         *   the most recent conversational partner, return that actor;
         *   otherwise we have no default 
         */
        if (lastInterlocutor != nil && canTalkTo(lastInterlocutor))
            return lastInterlocutor;
        else
            return nil;
    }

    /* 
     *   most recent actor that we asked about something, told about
     *   something, or showed something to 
     */
    lastInterlocutor = nil

    /*
     *   Calculate the amount of bulk I'm holding directly.  By default,
     *   we'll simply add up the "actor-encumbering bulk" of each of our
     *   direct contents.
     *   
     *   Note that we don't differentiate here based on whether or not an
     *   item is being worn, or anything else - we deliberately leave such
     *   distinctions up to the getEncumberingBulk routine, so that only
     *   the objects are in the business of deciding how bulky they are
     *   under different circumstances.  
     */
    getBulkHeld()
    {
        local total;

        /* start with nothing */
        total = 0;

        /* add the bulks of directly-contained items */
        foreach (local cur in contents)
            total += cur.getEncumberingBulk(self);

        /* return the total */
        return total;
    }

    /*
     *   Calculate the total weight I'm holding.  By default, we'll add up
     *   the "actor-encumbering weight" of each of our direct contents.
     *   
     *   Note that we deliberately only consider our direct contents.  If
     *   any of the items we are directly holding contain further items,
     *   getEncumberingWeight will take their weights into account; this
     *   frees us from needing any special knowledge of the internal
     *   structure of any items we're holding, and puts that knowledge in
     *   the individual items where it belongs.  
     */
    getWeightHeld()
    {
        local total;

        /* start with nothing */
        total = 0;

        /* add the weights of directly-contained items */
        foreach (local cur in contents)
            total += cur.getEncumberingWeight(self);

        /* return the total */
        return total;
    }

    /*
     *   Try making room to hold the given object.  This is called when
     *   checking the "room to hold object" pre-condition, such as for the
     *   "take" verb.  
     *   
     *   If holding the new object would exceed the our maximum holding
     *   capacity, we'll go through our inventory looking for objects that
     *   can reduce our held bulk with implicit commands.  Objects with
     *   holding affinities - "bags of holding", keyrings, and the like -
     *   can implicitly shuffle the actor's possessions in a manner that
     *   is neutral as far as the actor is concerned, thereby reducing our
     *   active holding load.
     *   
     *   Returns true if an implicit command was attempted, nil if not.  
     */
    tryMakingRoomToHold(obj, allowImplicit)
    {
        local objWeight;
        local objBulk;
        local bagVec; 
        local infoVec;
        
        /* get the amount of weight this will add */
        objWeight = obj.getEncumberingWeight(self);

        /* 
         *   If this object alone is too heavy for us, give up.  We
         *   distinguish this case from the case where the total (of
         *   everything held plus the new item) is too heavy: in the
         *   latter case we can tell the actor that they can pick this up
         *   by dropping something else first, whereas if this item alone
         *   is too heavy, no such advice is warranted. 
         */
        if (objWeight > weightCapacity)
        {
            reportFailure(&tooHeavyForActor, obj);
            exit;
        }

        /* 
         *   if the object would push us over our total carrying weight
         *   limit, give up 
         */
        if (getWeightHeld() + objWeight > weightCapacity)
        {
            reportFailure(&totalTooHeavyFor, obj);
            exit;
        }

        /* get the amount of bulk the object will add */
        objBulk = obj.getEncumberingBulk(self);
        
        /* 
         *   if the object is simply too big to start with, we can't make
         *   room no matter what we do 
         */
        if (objBulk > maxSingleBulk || objBulk > bulkCapacity)
        {
            reportFailure(&tooLargeForActor, obj);
            exit;
        }

        /*
         *   Test what would happen to our bulk if we were to move the
         *   object into our directly held inventory.  Do this by running
         *   a "what if" scenario to test moving the object into our
         *   inventory, and check what effect it has on our held bulk.  If
         *   it fits, we can let the caller proceed without further work.  
         */
        if (obj.whatIfHeldBy({: getBulkHeld()}, self) <= bulkCapacity)
            return nil;

        /*
         *   First, build a list of all of the bags of holding that we're
         *   carrying.  We start with the list of bags because we want to
         *   check each object to see which bag we might want to put it
         *   in.  
         */
        bagVec = new Vector(10);
        getBagsOfHolding(bagVec);

        /*
         *   If I have no bags of holding, or we're not allowed to try
         *   implicit actions, we can't move anything out of our hands
         *   implicitly - show an error and give up.  
         */
        if (bagVec.length() == 0 || !allowImplicit)
        {
            reportFailure(&handsTooFullFor, obj);
            exit;
        }

        /* 
         *   Look at my immediate contents.  For each item, get the item's
         *   bulk, and get each bag's affinity for the item.  Note the bag
         *   with the highest affinity.  Once we've gathered all of this
         *   information, move items in order of highest affinity - this
         *   will ensure that we move items with special affinities for
         *   particular bags first (such as keys to keyrings, credit cards
         *   to wallets, sandwiches to lunch boxes, and so on).  
         */

        /* create a vector to hold information on each child item */
        infoVec = new Vector(contents.length());

        /* look at each child item */
        foreach (local cur in contents)
        {
            local maxAff;
            local bestBag;

            /* find the bag with the highest affinity for this item */
            maxAff = 0;
            bestBag = nil;
            foreach (local bag in bagVec)
            {
                local aff;

                /* get this bag's affinity for this item */                
                aff = bag.affinityFor(cur);

                /* 
                 *   If this is the best so far, note it.  If this bag has
                 *   an affinity of zero for this item, it means it
                 *   doesn't want it at all, so don't even consider it in
                 *   this case. 
                 */
                if (aff != 0 && (bestBag == nil || aff > maxAff))
                {
                    /* this is the best so far - note it */
                    maxAff = aff;
                    bestBag = bag;
                }
            }

            /* 
             *   if we found a bag that wants this item, add it to our
             *   list of results 
             */
            if (bestBag != nil)
                infoVec.append(new BagAffinityInfo(
                    cur, cur.getEncumberingBulk(self),
                    maxAff, bestBag));
        }

        /*
         *   Sort the list in descending order of affinity, and for equal
         *   affinities in order of descending bulk.  This will put the
         *   strongest candidates for implicitly moving into a bag at the
         *   start of the list, so that we can simply work through the
         *   list from front to back.  
         */
        infoVec = infoVec.toList()
                  .sort(SortDesc, {a, b: a.compareAffinityTo(b)});

        /*
         *   Move each object in the list until we have reduced the bulk
         *   sufficiently. 
         */
        foreach (local cur in infoVec)
        {
            /* 
             *   Try moving this object to its bag.  If the bag is itself
             *   inside this object, don't even try, for obvious reasons.  
             */
            if (!cur.bag_.isIn(cur.obj_)
                && cur.bag_.tryPuttingObjInBag(cur.obj_))
            {
                /* 
                 *   this routine tried tried to move the object into the
                 *   bag - check our held bulk to see if we're in good
                 *   enough shape yet
                 */
                if (obj.whatIfHeldBy({: getBulkHeld()}, self) <= bulkCapacity)
                {
                    /* 
                     *   We've met our condition - there's no need to look
                     *   any further.  Return, telling the caller we've
                     *   performed an implicit command.  
                     */
                    return true;
                }
            }
        }
        
        /*
         *   If we get this far, it means that we tried every child object
         *   but failed to find anything that could help.  Explain the
         *   problem and abort the command.  
         */
        reportFailure(&handsTooFullFor, obj);
        exit;
    }

    /*
     *   Check a bulk change of one of my direct contents. 
     */
    checkBulkChangeWithin(obj)
    {
        local objBulk;
        
        /* get the object's new bulk */
        objBulk = obj.getEncumberingBulk(self);
        
        /* 
         *   if this change would cause the object to exceed our
         *   single-item bulk limit, don't allow it 
         */
        if (objBulk > maxSingleBulk || objBulk > bulkCapacity)
        {
            reportFailure(&becomingTooLargeForActor, obj);
            exit;
        }

        /* 
         *   If our total carrying capacity is exceeded with this change,
         *   don't allow it.  Note that 'obj' is already among our
         *   contents when this routine is called, so we can simply check
         *   our current total bulk within.  
         */
        if (getBulkHeld() > bulkCapacity)
        {
            reportFailure(&handsBecomingTooFullFor, obj);
            exit;
        }
    }

    /* 
     *   Next available "holding index" value.  Each time we pick up an
     *   item, we'll assign it our current holding index value and then
     *   increment our value.  This gives us a simple way to keep track of
     *   the order in which we picked up items we're carrying.
     *   
     *   Note that we make the simplifying assumption that an object can
     *   be held by only one actor at a time (multi-location items are
     *   generally not portable), which means that we can use a simple
     *   property in each object being held to store its holding index.  
     */
    nextHoldingIndex = 1

    /* add an object to my contents */
    addToContents(obj)
    {
        /* assign the new object our next holding index */
        obj.holdingIndex = nextHoldingIndex++;

        /* inherit default handling */
        inherited(obj);
    }

    /*
     *   Go to sleep.  This is used by the 'Sleep' action to carry out the
     *   command.  By default, we simply say that we're not sleepy; actors
     *   can override this to cause other actions.  
     */
    goToSleep()
    {
        /* simply report that we can't sleep now */
        mainReport(&cannotSleep);
    }

    /*
     *   My current "posture," which specifies how we're positioned with
     *   respect to our container; this is one of the standard library
     *   posture enum values (Standing, etc.) or another posture added by
     *   the game.  
     */
    posture = standing

    /*
     *   Show our posture, as an addendum to a room's status line report.
     *   'room' is the location whose status line report we're generating;
     *   we'll avoid redundantly naming the room as the nominal container
     *   for the actor as part of the posture description if the nominal
     *   container is the same as the room whose status name we're
     *   showing.  
     */
    statusPosture(room)
    {
        /* ask the posture to provide the addendum */
        posture.statusPosture(self, room);
    }

    /*
     *   Get a default acknowledgment of a change to our posture.  This
     *   should acknowledge the posture so that it tells us the current
     *   posture.  This is used for a command such as "stand up" from a
     *   chair, so that we can report the appropriate posture status in
     *   our acknowledgment; we might end up being inside another nested
     *   container after standing up from the chair, so we might not
     *   simply be standing when we're done.   
     */
    okayPostureChange()
    {
        /* process it through the posture object */
        posture.okayPostureChange(self);
    }

    /*
     *   Describe the actor as part of a list of actors occupying a nested
     *   room.  'pov' is the point of view of the description.
     */
    listActorPosture(pov)
    {
        /* process it through the posture object */
        posture.listActorPosture(self, pov);
    }

    /*
     *   Stand up.  This is used by the 'Stand' action to carry out the
     *   command. 
     */
    standUp()
    {
        /* if we're already standing, say so */
        if (posture == standing)
        {
            reportFailure(&alreadyStanding);
            return;
        }

        /* ask the location to make us stand up */
        location.makeStandingUp();
    }

    /*
     *   Disembark.  This is used by the 'Get out' action to carry out the
     *   command.  By default, we'll let the room handle it.  
     */
    disembark()
    {
        /* let the room handle it */
        location.disembarkRoom();
    }

    /* 
     *   Set our posture to the given status.  By default, we'll simply
     *   set our posture property to the new status, but actors can
     *   override this to handle side effects of the change.
     */
    makePosture(newPosture)
    {
        /* remember our new posture */
        posture = newPosture;
    }

    /* 
     *   Display a description of the actor's location from the actor's
     *   point of view.
     *   
     *   If 'verbose' is true, then we'll show the full description in all
     *   cases.  Otherwise, we'll show the full description if the actor
     *   hasn't seen the location before, or the terse description if the
     *   actor has previously seen the location.  
     */
    lookAround(verbose, explicit)
    {
        /* show a description of my immediate location, if I have one */
        if (location != nil)
            location.lookAroundPov(self, self, verbose, explicit);
    }

    /*
     *   Get the location into which objects should be moved when the
     *   actor drops them with an explicit 'drop' command.  By default, we
     *   return the drop destination of our current container.  
     */
    getDropDestination(objToDrop)
    {
        return (location != nil
                ? location.getDropDestination(objToDrop)
                : nil);
    }

    /*
     *   The senses that determine scope for this actor.  An actor might
     *   possess only a subset of the defined sense.
     *   
     *   By default, we give each actor all of the human senses that we
     *   define, except touch.  In general, merely being able to touch an
     *   object doesn't put the object in scope, because if an object
     *   isn't noticed through some other sense, touch would only make an
     *   object accessible if it's within arm's reach, which for our
     *   purposes means that the object is being held directly by the
     *   actor.  Imagine an actor in a dark room: lots of things might be
     *   touchable in the sense that there's no physical barrier to
     *   touching them, but without some other sense to locate the
     *   objects, the actor wouldn't have any way of knowing where to
     *   reach to touch things, so they're not in scope.  So, touch isn't
     *   a scope sense.  
     */
    scopeSenses = [sight, sound, smell]

    /*
     *   "Sight-like" senses: these are the senses that operate like sight
     *   for the actor, and which the actor can use to determine the names
     *   of objects and the spatial relationships between objects.  These
     *   senses should operate passively, in the sense that they should
     *   tend to collect sensory input continuously and without explicit
     *   action by the actor, the way sight does and the way touch, for
     *   example, does not.  These senses should also operate instantly,
     *   in the sense that the sense can reasonably take in most or all of
     *   a location at one time.
     *   
     *   These senses are used to determine what objects should be listed
     *   in room descriptions, for example.
     *   
     *   By default, the only sight-like sense is sight, since other human
     *   senses don't normally provide a clear picture of the spatial
     *   relationships among objects.  (Touch could with some degree of
     *   effort, but it can't operate passively or instantly, since
     *   deliberate and time-consuming action would be necessary.)
     *   
     *   An actor can have more than one sight-like sense, in which case
     *   the senses will act effectively as one sense that can reach the
     *   union of objects reachable through the individual senses.  
     */
    sightlikeSenses = [sight]

    /* 
     *   Hearing-like senses.  These are senses that the actor can use to
     *   hear objects. 
     */
    hearinglikeSenses = [sound]

    /*
     *   Smell-like senses.  These are senses that the actor can use to
     *   smell objects. 
     */
    smelllikeSenses = [smell]

    /*
     *   Communication senses: these are the senses through which the
     *   actor can communicate directly with other actors through commands
     *   and messages.
     *   
     *   Conceptually, these senses are intended to be only those senses
     *   that the actors would *naturally* use to communicate, because
     *   senses in this list allow direct communications via the most
     *   ordinary game commands, such as "bob, go east".
     *   
     *   If some form of indirect communication is possible via a sense,
     *   but that form is not something the actor would think of as the
     *   most natural, default form of communication, it should *not* be
     *   in this list.  For example, two sighted persons who can see one
     *   another but cannot hear one another could still communicate by
     *   writing messages on pieces of paper, but they would ordinarily
     *   communicate by talking.  In such a case, sound should be in the
     *   list but sight should not be, because sight is not a natural,
     *   default form of communications for the actors.  
     */
    communicationSenses = [sound]
    
    /*
     *   Determine if I can communicate with the given character via a
     *   natural, default form of communication that we share with the
     *   other character.  This determines if I can talk to the other
     *   character.  We'll return true if I can talk to the other actor,
     *   nil if not.
     *   
     *   In order for the player character to issue a command to a
     *   non-player character (as in "bob, go east"), the NPC must be able
     *   to sense the PC via at least one communication sense that the two
     *   actors have in common.
     *   
     *   Likewise, in order for a non-player character to say something to
     *   the player, the player must be able to sense the NPC via at least
     *   one communication sense that the two actors have in common.  
     */
    canTalkTo(actor)
    {
        local common;
        
        /* 
         *   first, get a list of the communications senses that we have
         *   in common with the other actor - we must have a sense channel
         *   via this sense 
         */
        common = communicationSenses.intersect(actor.communicationSenses);

        /* 
         *   if there are no common senses, we can't communicate,
         *   regardless of our physical proximity 
         */
        if (common == [])
            return nil;

        /* 
         *   Determine how well the other actor can sense me in these
         *   senses.  Note that all that matters it that the actor can
         *   hear me, because we're determine if I can talk to the other
         *   actor - it doesn't matter if I can hear the other actor.  
         */
        foreach (local curSense in common)
        {
            local result;

            /* 
             *   determine how well the other actor can sense me in this
             *   sense 
             */
            result = actor.senseObj(curSense, self);

            /* check whether or not this is good enough */
            if (actor.canBeTalkedTo(self, curSense, result))
                return true;
        }

        /* 
         *   if we get this far, we didn't find any senses with a clear
         *   enough communications channel - we can't talk to the other
         *   actor 
         */
        return nil;
    }

    /*
     *   Determine whether or not I can understand an attempt by another
     *   actor to talk to me.  'talker' is the actor doing the talking.
     *   'sense' is the sense we're testing; this will always be a sense
     *   in our communicationSenses list, and will always be a
     *   communications sense we have in common with the other actor.
     *   'info' is a SenseInfo object giving information on the clarity of
     *   the sense path to the other actor.
     *   
     *   We return true if we can understand the communication, nil if
     *   not.  There is no middle ground where we can partially
     *   understand; we can either understand or not.
     *   
     *   Note that this routine is concerned only with our ability to
     *   sense the communication.  The result here should NOT pay any
     *   attention to whether or not we can actually communicate given a
     *   clear sense path - for example, this routine should not reflect
     *   whether or not we have a spoken language in common with the other
     *   actor.
     *   
     *   This is a service method for canTalkTo.  This is broken out as a
     *   separate method so that individual actors can override the
     *   necessary conditions for communications in particular senses.  
     */
    canBeTalkedTo(talker, sense, info)
    {
        /*   
         *   By default, we allow communication if the sense path is
         *   transparent or distant.  We don't care what the sense is,
         *   since we know we'll never be asked about a sense that's not
         *   in our communicationSenses list.  
         */
        return info.trans is in (transparent, distant);
    }

    /*
     *   Flag: we wait for commands issued to other actors to complete
     *   before we get another turn.  If this is true, then whenever we
     *   issue a command to another actor ("bob, go north"), we will not
     *   get another turn until the other actor has finished executing the
     *   full set of commands we issued.
     *   
     *   By default, this is true, which means that we wait for other
     *   actors to finish all of the commands we issue before we take
     *   another turn.  
     *   
     *   If this is set to nil, we'll continue to take turns while the
     *   other actor carries out our commands.  In this case, the only
     *   time cost to us of issuing a command is given by orderingTime(),
     *   which normally takes one turn for issuing a command, regardless
     *   of the command's complexity.  Some games might wish to use this
     *   mode for interesting effects with NPC's carrying out commands in
     *   parallel with the player, but it's an unconventional style that
     *   some players might find confusing, so we don't use this mode by
     *   default.  
     */
    issueCommandsSynchronously = true

    /*
     *   Flag: the "target actor" of the command line automatically reverts
     *   to this actor at the end of a sentence, when this actor is the
     *   issuer of a command.  If this flag is nil, an explicit target
     *   actor stays in effect until the next explicit target actor (or the
     *   end of the entire command line, if no other explicit target actors
     *   are named); if this flag is true, a target actor is in effect only
     *   until the end of a sentence.
     *   
     *   Consider this command line:
     *   
     *   >Bob, go north and get fuel cell. Get log tape.
     *   
     *   If this flag is nil, then the second sentence ("get log tape") is
     *   interpreted as a command to Bob, because Bob is explicitly
     *   designated as the target of the command, and this remains in
     *   effect until the end of the entire command line.
     *   
     *   If this flag is true, on the other hand, then the second sentence
     *   is interpreted as a command to the player character, because the
     *   target actor designation ("Bob,") lasts only until the end of the
     *   sentence.  Once a new sentence begins, we revert to the issuing
     *   actor (the player character, since the command came from the
     *   player via the keyboard).  
     */
    revertTargetActorAtEndOfSentence = nil

    /*
     *   The amount of time, in game clock units, it takes me to issue an
     *   order to another actor.  By default, it takes one unit (which is
     *   usually equal to one turn) to issue a command to another actor.
     *   However, if we are configured to wait for our issued commands to
     *   complete in full, the ordering time is zero; we don't need any
     *   extra wait time in this case because we'll wait the full length
     *   of the issued command to begin with.  
     */
    orderingTime(targetActor)
    {
        return issueCommandsSynchronously ? 0 : 1;
    }

    /*
     *   Wait for completion of a command that we issued to another actor.
     *   The parser calls this routine after each time we issue a command
     *   to another actor.
     *   
     *   If we're configured to wait for completion of orders given to
     *   other actors before we get another turn, we'll set ourselves up
     *   in waiting mode.  Otherwise, we'll do nothing.  
     */
    waitForIssuedCommand(targetActor)
    {
        /* if we can issue commands asynchronously, there's nothing to do */
        if (!issueCommandsSynchronously)
            return;

        /* 
         *   Add an empty pending command at the end of the target actor's
         *   queue.  This command won't do anything when executed; its
         *   purpose is to let us track whether or not the target is still
         *   working on commands we have issued up to this point, which we
         *   can tell by looking to see whether our empty command is still
         *   in the actor's queue.
         *   
         *   Note that we can't simply wait until the actor's queue is
         *   empty, because the actor could acquire new commands while
         *   it's working on our pending commands, and we wouldn't want to
         *   wait for those to finish.  Adding a dummy pending command is
         *   a reliable way of tracking the actor's queue, because any
         *   changes to the target actor's command queue will leave our
         *   dummy command in its proper place until the target actor gets
         *   around to executing it, at which point it will be removed.
         *   
         *   Remember the dummy pending command in a property of self, so
         *   that we can check later to determine when the command has
         *   finished.  
         */
        waitingForActor = targetActor;
        waitingForInfo = new PendingCommandInfo(nil, self, nil, nil);
        targetActor.pendingCommand.append(waitingForInfo);
    }

    /* 
     *   Synchronous command processing: the target actor and dummy
     *   pending command we're waiting for.  When these are non-nil, we
     *   won't take another turn until the given PendingCommandInfo has
     *   been removed from the given target actor's command queue. 
     */
    waitingForActor = nil
    waitingForInfo = nil

    /*
     *   Get the list of objects I can follow.  This is a list of all of
     *   the objects which I have seen departing a location - these are
     *   all in scope for 'follow' commands.  
     */
    getFollowables()
    {
        /* return the list of the objects we know about */
        return followables_.mapAll({x: x.obj});
    }

    /* 
     *   Do I track departing objects for following the given object?
     *   
     *   By default, only the player character tracks objects for
     *   following, since most NPC's will never accept 'follow' commands.
     *   We use this default for efficiency, to avoid storing a bunch of
     *   follow information for NPC's that will never use the information.
     *   If an NPC is to accept 'follow' commands, it must override this
     *   to return true.
     *   
     *   Note that an NPC that can go into "follow mode" to follow an
     *   actor around must return true, because follow mode uses regular
     *   "follow" commands to tag along.  
     */
    wantsFollowInfo(obj)
    {
        /* 
         *   by default, the player character tracks everyone for
         *   following, and no one else does 
         */
        return isPlayerChar();
    }

    /*
     *   Receive notification that an object is leaving its current
     *   location.  Actors (and possibly other objects) will broadcast
     *   this notification to all Actor objects connected in any way by
     *   containment when they move under their own power (such as with
     *   Actor.travelTo) to a new location.  We'll keep tracking
     *   information if we are configured to keep tracking information for
     *   the given object and we can see the given object.  Note that this
     *   is called when the object is still at the source end of the
     *   travel - the important thing is that we see the object departing.
     *   
     *   'obj' is the object that is seen to be leaving, and 'conn' is the
     *   TravelConnector it is taking.
     *   
     *   Note that this notification is sent only to actors with some sort
     *   of containment connection to the actor that's moving, because a
     *   containment connection is necessary for there to be a sense
     *   connection.  
     */
    trackFollowInfo(obj, conn)
    {
        local info;
        
        /* 
         *   If we're not tracking the given object, or we can't see the
         *   given object, ignore the notification.  In addition, we
         *   obviously have no need to track ourselves.x  
         */
        if (obj == self || !wantsFollowInfo(obj) || !canSee(obj))
            return;

        /* 
         *   If we already have a FollowInfo for the given object, re-use
         *   the existing one; otherwise, create a new one and add it to
         *   our tracking list. 
         */
        info = followables_.valWhich({x: x.obj == obj});
        if (info == nil)
        {
            /* we don't have an existing one - create a new one */
            info = new FollowInfo();
            info.obj = obj;

            /* add it to our list */
            followables_ += info;
        }

        /* remember information about the travel */
        info.connector = conn;
        info.sourceLocation = gActor.getTraveler().location;
        info.origAction = gAction;
    }

    /*
     *   Get information on what to do to make this actor follow the given
     *   object.  This returns a FollowInfo object that reports our last
     *   knowledge of the given object's location and departure, or nil if
     *   we don't know anything about how to follow the actor.  
     */
    getFollowInfo(obj)
    {
        return followables_.valWhich({x: x.obj == obj});
    }

    /*
     *   By default, all actors are followable.
     */
    verifyFollowable()
    {
        return true;
    }

    /*
     *   Verify a "follow" command being performed by this actor.  
     */
    actorVerifyFollow(obj)
    {
        /* 
         *   check to see if we're in the same effective follow location
         *   as the target; if we are, it makes no sense to follow the
         *   target, since we're already effectively at the same place 
         */
        if (obj.location != nil
            && (location.effectiveFollowLocation
                == obj.location.effectiveFollowLocation))
        {
            /*
             *   We're in the same location as the target.  If we're the
             *   player character, this makes no sense, because the player
             *   character can't go into follow mode (as that would take
             *   away the player's ability to control the player
             *   character).  If we're an NPC, though, this simply tells
             *   us to go into follow mode for the target, so there's
             *   nothing wrong with it.  
             */
            if (isPlayerChar)
            {
                /* 
                 *   The target is right here, but we're the player
                 *   character, so it makes no sense for us to go into
                 *   follow mode.  If we can see the target, complain that
                 *   it's already here; if not, we can only assume it's
                 *   here, but we can't know for sure.  
                 */
                if (canSee(obj))
                    illogicalNow(&followAlreadyHere);
                else
                    illogicalNow(&followAlreadyHereInDark);
            }
        }
        else if (!canSee(obj))
        {
            /* 
             *   The target isn't here, and we can't see it from here, so
             *   we must want to follow it to its current location.  Get
             *   information on how we will follow the target.  If there's
             *   no such information, we obviously can't do any following
             *   because we never saw the target go anywhere in the first
             *   place.  
             */
            if (getFollowInfo(obj) == nil)
            {
                /* we've never heard of the target */
                illogicalNow(&followUnknown);
            }
        }
    }

    /*
     *   Carry out a "follow" command being performed by this actor.  
     */
    actorActionFollow(obj)
    {
        local canSeeObj;

        /* note whether or not we can see the target */
        canSeeObj = canSee(obj);
        
        /* 
         *   If we're not the NPC, check to see if this is a follow-mode
         *   request; otherwise, try to go to the location of the target.
         *   If we're already in follow mode, we don't need to bother
         *   going into follow mode, obviously.  
         */
        if (!isPlayerChar && canSeeObj && followingActor != obj)
        {
            /* let them know we're going to follow the actor now */
            reportAfter(&okayFollowMode);

            /* go into follow mode */
            followingActor = obj;

            /* 
             *   if we're already in the target's effective follow
             *   location, that's all we need to do 
             */
            if (location.effectiveFollowLocation
                == obj.location.effectiveFollowLocation)
                return;
        }

        /* if we can see the target simply go where the target is */
        if (canSeeObj)
        {
            /* 
             *   We have no information, so we will only have made it past
             *   verification if we can see the other actor from our
             *   current location.  Try moving to the other actor's
             *   effective follow location.  
             */
            obj.location.effectiveFollowLocation.checkMovingActorInto(true);

            /*
             *   Since the 'check' routine will do its work through
             *   implicit actions, we will not have a real report for the
             *   action it performed.  So, generate an acknowledgment that
             *   we have arrived at the actor's location. 
             */
            defaultReport(&okayFollowInSight,
                          location.effectiveFollowLocation);
        }
        else
        {
            local info;
            
            /* get the information on how to follow the target */
            info = getFollowInfo(obj);
            
            /*
             *   Before we can follow the target, we must be in the same
             *   effective location that the target was in when we
             *   observed the target leaving.  
             */
            if (location.effectiveFollowLocation
                != info.sourceLocation.effectiveFollowLocation)
            {
                /* 
                 *   Try moving into the same location, by invoking the
                 *   pre-condition handler for moving me into the
                 *   effective follow location from our memory of the
                 *   actor's travel.  We *could* run this as an actual
                 *   precondition, but it's easier to run it here now that
                 *   we've sorted out exactly what we want to do.  
                 */
                info.sourceLocation.effectiveFollowLocation.
                    checkMovingActorInto(true);
            }
        
            /* 
             *   run the command that the actor we're following originally
             *   used for this travel as a nested command for the actor
             *   doing the following 
             */
            nestedActionAgain(info.origAction);
        }
    }

    /*
     *   Our list of followable information.  Each entry in this list is a
     *   FollowInfo object that tracks a particular followable.  
     */
    followables_ = []

    /*
     *   Determine if the actor recognizes the given object as a "topic,"
     *   which is an object that represents some knowledge the actor can
     *   use in conversations, consultations, and the like.
     *   
     *   By default, we'll recognize any Topic object marked as known, and
     *   we'll recognize any game object that has been marked as seen.
     *   Games might wish to override this in some cases to limit an
     *   actor's knowledge according to what the actor has experienced of
     *   the setting or story.  
     */
    knowsTopic(obj)
    {
        /* we know anything that's a Topic */
        if (obj.ofKind(Topic) && obj.isKnown)
            return true;

        /* we know of any simulation object that's been seen */
        if (obj.ofKind(Thing) && obj.seenBy(self))
            return true;

        /* other objects we don't know about */
        return nil;
    }

    /*
     *   Determine if the given object is a likely topic for an action
     *   performed by this actor.  By default, we'll return true if the
     *   topic has ever been seen, nil if not.  
     */
    isLikelyTopic(obj)
    {
        /* if the object has ever been seen, it's a possible topic */
        return obj.seenBy(self);
    }

    /*
     *   Receive notification that a command is being carried out in our
     *   presence. 
     */
    beforeAction()
    {
        /*
         *   If another actor is trying to take something in my inventory,
         *   by default, do not allow it. 
         */
        if (gActor != self
            && (gActionIs(Take) || gActionIs(TakeFrom))
            && gDobj.isIn(self))
        {
            /* check to see if we want to allow this action */
            checkTakeFromInventory(gActor, gDobj);
        }
    }

    /*
     *   Check to see if we want to allow another actor to take something
     *   from my inventory.  By default, we won't allow it - we'll always
     *   fail the command.  
     */
    checkTakeFromInventory(actor, obj)
    {
        /* don't allow it - show an error and terminate the command */
        mainReport(&willNotLetGo, self, obj);
        exit;
    }

    /*
     *   Receive notification that I'm performing an action.  This is
     *   called on the actor performing the action before the action is
     *   actually carried out.  
     */
    actorAction()
    {
    }

    /*
     *   Build a list of the objects that are explicitly registered to
     *   receive notification when I'm the actor in a command.
     */
    getActorNotifyList()
    {
        return actorNotifyList;
    }

    /*
     *   Add an item to our registered notification items.  These items
     *   are to receive notifications when we're the actor performing a
     *   command.
     *   
     *   Items can be added here if they must be notified of actions
     *   performed by the actor even when the items aren't connected by
     *   containment with the actor at the time of the action.  All items
     *   connected to the actor by containment are automatically notified
     *   of each action; only items that must receive notification even
     *   when not in scope need to be registered here.  
     */
    addActorNotifyItem(obj)
    {
        actorNotifyList += obj;
    }

    /* remove an item from the registered notification list */
    removeActorNotifyItem(obj)
    {
        actorNotifyList -= obj;
    }

    /* our list of registered actor notification items */
    actorNotifyList = []

    /*
     *   Get the ambient light level in the visual senses at this actor.
     *   This is the ambient level at the actor.  
     */
    getVisualAmbient()
    {
        local ret;

#ifdef SENSE_CACHE
        local cache;

        /* check for a cached value */
        if ((cache = libGlobal.actorVisualAmbientCache) != nil
            && (ret = cache[self]) != nil)
        {
            /* found a cached entry - use it */
            return ret;
        }
#endif

        /* get the maximum ambient level at self for my sight-like senses */
        ret = senseAmbientMax(sightlikeSenses);

#ifdef SENSE_CACHE
        /* if caching is active, cache our result for next time */
        if (cache != nil)
            cache[self] = ret;
#endif

        /* return the result */
        return ret;
    }

    /*
     *   Determine if my location is lit for my sight-like senses.
     */
    isLocationLit()
    {
        /* 
         *   Check for a simple, common case before doing the full
         *   sense-path calculation: if our location is providing its own
         *   light to its interior, then the location is lit.  Most simple
         *   rooms are always lit.  
         */
        if (sightlikeSenses.indexOf(sight) != nil
            && location != nil
            && location.brightness > 1
            && location.transSensingOut(sight) == transparent)
            return true;

        /* 
         *   We don't have the simple case of light directly from our
         *   location, so run the full sense path check and get our
         *   maximum visual ambience level.  If it's above the "self-lit"
         *   level of 1, then we can see. 
         */
        return (getVisualAmbient() > 1);
    }

    /*
     *   Get the best (most transparent) sense information for one of our
     *   visual senses to the given object.  
     */
    bestVisualInfo(obj)
    {
        local best;

        /* we don't have a best value yet */
        best = nil;

        /* check each sight-like sense */
        foreach (local sense in sightlikeSenses)
        {
            local info;
            
            /* get the information for the object in this sense */
            info = senseObj(sense, obj);

            /* 
             *   if we don't have one at all yet, or this is the best so
             *   far, keep this one 
             */
            if (info != nil
                && (best == nil
                    || transparencyCompare(info.trans, best.trans) > 0))
                best = info;
        }

        /* return the best one we found */
        return best;
    }

    /*
     *   Build a list of all of the objects of which an actor is aware.
     *   
     *   An actor is aware of an object if the object is within reach of
     *   the actor's senses, and has some sort of presence in that sense.
     *   Note that both of these conditions must be true for at least one
     *   sense possessed by the actor; an object that is within earshot,
     *   but not within reach of any other sense, is in scope only if the
     *   object is making some kind of noise.
     *   
     *   In addition, objects that the actor is holding (i.e., those
     *   contained by the actor directly) are always in scope, regardless
     *   of their reachability through any sense.  
     */
    scopeList()
    {
        local lst;

        /* we have nothing in our master list yet */
        lst = new Vector(32);

        /* oneself is always in one's own scope list */
        lst.append(self);

        /* iterate over each sense */
        foreach (local sense in scopeSenses)
        {
            /* 
             *   get the list of objects with a presence in this sense
             *   that can be sensed from our point of view, and and append
             *   it to our master list 
             */
            lst.appendUnique(sensePresenceList(sense));
        }

        /* add all of the items we are directly holding */
        lst.appendUnique(contents);

        /* 
         *   ask each of our direct contents to add any contents of their
         *   own that are in scope by virtue of their containers being in
         *   scope 
         */
        foreach (local cur in contents)
            cur.appendHeldContents(lst);

        /* add any items that are specially in scope in the location */
        if (location != nil)
            lst.appendUnique(location.getExtraScopeItems(self));

        /* return the result */
        return lst.toList();
    }

    /*
     *   Determine if I can see the given object.  This returns true if
     *   the object can be sensed at all in one of my sight-like senses,
     *   nil if not.  
     */
    canSee(obj)
    {
        /* try each sight-like sense */
        foreach (local sense in sightlikeSenses)
        {
            /* 
             *   if I can sense the object in this sense, I can sense the
             *   object 
             */
            if (senseObj(sense, obj).trans != opaque)
                return true;
        }

        /* we didn't find any sight-like sense where we can see the object */
        return nil;
    }

    /*
     *   Determine if I can hear the given object. 
     */
    canHear(obj)
    {
        /* try each hearling-like sense */
        foreach (local sense in hearinglikeSenses)
        {
            /* 
             *   if I can sense the object in this sense, I can sense the
             *   object 
             */
            if (senseObj(sense, obj).trans != opaque)
                return true;
        }
        
        /* we found no hearing-like sense that lets us hear the object */
        return nil;
    }

    /*
     *   Determine if I can smell the given object. 
     */
    canSmell(obj)
    {
        /* try each hearling-like sense */
        foreach (local sense in smelllikeSenses)
        {
            /* 
             *   if I can sense the object in this sense, I can sense the
             *   object 
             */
            if (senseObj(sense, obj).trans != opaque)
                return true;
        }
        
        /* we found no smell-like sense that lets us hear the object */
        return nil;
    }

    /*
     *   Determine if I can touch the given object.  By default, we can
     *   always touch our immediate container; otherwise, we can touch
     *   anything with a touch path that we can traverse.
     */
    canTouch(obj)
    {
        local path;
        local result;

#ifdef SENSE_CACHE
        local key = [self, obj];
        local cache;
        local info;

        /* if we have a cache, check the cache for the desired data */
        if ((cache = libGlobal.canTouchCache) != nil
            && (info = cache[key]) != nil)
        {
            /* if we cached the canTouch result, return it */
            if (info.propDefined(&canTouch))
                return info.canTouch;

            /* 
             *   we didn't calculate the canTouch result, but we at least
             *   have a cached path that we can avoid recalculating 
             */
            path = info.touchPath;
        }
        else
        {
            /* we don't have a cached path, so calculate it anew */
            path = getTouchPathTo(obj);
        }
#endif
        
        /* if there's no 'touch' path, we can't touch the object */
        if (path == nil)
        {
            /* there's no path */
            result = nil;
        }
        else
        {
            /* we have a path - check to see if we can reach along it */
            result = traversePath(path,
                {ele, op: ele.canTouchViaPath(self, obj, op)});
        }

#ifdef SENSE_CACHE
        /* if caching is active, cache our result */
        if (cache != nil)
        {
            /* if we don't already have a cache entry, create one */
            if (info == nil)
                cache[key] = info = new CanTouchInfo(path);

            /* save our canTouch result in the cache entry */
            info.canTouch = result;
        }
#endif

        /* return the result */
        return result;
    }

    /*
     *   Find the object that prevents us from touching the given object.
     */
    findTouchObstructor(obj)
    {
        /* cache 'touch' sense path information */
        cacheSenseInfo(connectionTable(), touch);
        
        /* return the opaque obstructor for the sense of touch */
        return findOpaqueObstructor(touch, obj);
    }

    /*
     *   Find the object that prevents us from seeing the given object. 
     */
    findVisualObstructor(obj)
    {
        /* try to find an opaque obstructor in one of our visual senses */
        foreach (local sense in sightlikeSenses)
        {
            local obs;

            /* cache path information for this sense */
            cacheSenseInfo(connectionTable(), sense);
            
            /* if we find an obstructor in this sense, return it */
            if ((obs = findOpaqueObstructor(sense, obj)) != nil)
                return obs;
        }

        /* we didn't find any obstructor */
        return nil;
    }

    /*
     *   Build a table of full sensory information for all of the objects
     *   visible to the actor through the actor's sight-like senses.
     *   Returns a lookup table with the same set of information as
     *   senseInfoTable().  
     */
    visibleInfoTable()
    {
        /* return objects visible from my own point of view */
        return visibleInfoTableFromPov(self);
    }

    /*
     *   Build a table of full sensory information for all of the objects
     *   visible to me from a particular point of view through my
     *   sight-like senses.  
     */
    visibleInfoTableFromPov(pov)
    {
        local tab;

        /* we have no master table yet */
        tab = nil;

        /* iterate over each sense */
        foreach (local sense in sightlikeSenses)
        {
            local cur;
            
            /* get information for all objects for the current sense */
            cur = pov.senseInfoTable(sense);

            /* merge the table so far with the new table */
            tab = mergeSenseInfoTable(cur, tab);
        }

        /* return the result */
        return tab;
    }

    /*
     *   Build a lookup table of the objects that can be sensed for the
     *   purposes of taking inventory.  We'll include everything in the
     *   normal visual sense table, plus everything directly held.  
     */
    inventorySenseInfoTable()
    {
        local visInfo;
        local cont;
        local ambient;
        local info;
        
        /*   
         *   Start with the objects visible to the actor through the
         *   actor's sight-like senses.  
         */
        visInfo = visibleInfoTable();

        /* get the ambient light level at the actor */
        if ((info = visInfo[self]) != nil)
            ambient = info.ambient;
        else
            ambient = 0;
        
        /*   
         *   We'll assume that the actor has at some point taken note of
         *   each item being directly carried (by virtue of having picked
         *   it up at some point in the past) and can still identify each
         *   by touch, even if it's too dark to see it.
         *   
         *   Likewise, add items within our direct contents that are
         *   considered equally held.  
         */
        cont = new Vector(32);
        foreach (local cur in contents)
        {
            /* add this item from our contents */
            cont.append(cur);

            /* add its contents that are themselves equally as held */
            cur.appendHeldContents(cont);
        }

        /* 
         *   Make a fully-sensible entry for each of our held items.  We
         *   can simply replace any existing entry in the table that we
         *   got from the visual senses, since a fully transparent entry
         *   will be at least as good as anything we got from the normal
         *   visual list. 
         */
        foreach (local cur in cont)
            visInfo[cur] = new SenseInfo(cur, transparent, nil, ambient);

        /* return the table */
        return visInfo;
    }

    /*
     *   Show what the actor is carrying.
     */
    showInventory(tall)
    {
        /* 
         *   show our inventory with our default listers as given by our
         *   inventory/wearing lister properties 
         */
        showInventoryWith(tall, inventoryLister, wearingLister);
    }

    /*
     *   Show what the actor is carrying, using the given listers.
     *   
     *   Note that this method must be overridden if the actor does not
     *   use a conventional 'contents' list property to store its full set
     *   of contents.  
     */
    showInventoryWith(tall, inventoryLister, wearingLister)
    {
        local infoTab;

        /* get the table of objects sensible for inventory */
        infoTab = inventorySenseInfoTable();

        /* list in the appropriate mode ("wide" or "tall") */
        if (tall)
        {
            /* 
             *   show the list of items being carried, recursively
             *   displaying the contents; show items being worn in-line
             *   with the regular listing 
             */
            inventoryLister.showList(self, self, contents,
                                     ListTall | ListRecurse,
                                     0, infoTab, nil);
        }
        else
        {
            local carrying;
            local wearing;

            /* 
             *   paragraph style ("wide") inventory - go through the
             *   direct contents, and separate the list into what's being
             *   carried and what's being worn 
             */
            carrying = new Vector(32);
            wearing = new Vector(32);
            foreach (local cur in contents)
            {
                if (cur.isWornBy(self))
                    wearing.append(cur);
                else
                    carrying.append(cur);
            }

            /* convert the carrying and wearing vectors to lists */
            carrying = carrying.toList();
            wearing = wearing.toList();
            
            /* show the list of items being carried */
            inventoryLister.showList(self, self, carrying, ListRecurse,
                                     0, infoTab, nil);

            /* show the list of items being worn */
            if (wearing.length() != 0)
            {
                /* show the items being worn */
                wearingLister.showList(self, self, wearing, ListRecurse, 0,
                                       infoTab, nil);
            }
        }

        /* mention sounds coming from inventory items */
        inventorySense(sound, inventoryListenLister);

        /* mention odors coming from inventory items */
        inventorySense(smell, inventorySmellLister);
    }

    /*
     *   Add to an inventory description a list of things we notice
     *   through a specific sense.
     */
    inventorySense(sense, lister)
    {
        local infoTab;
        local presenceList;
        
        /* get the information table for the desired sense */
        infoTab = senseInfoTable(sense);
        
        /* 
         *   get the list of everything with a presence in this sense that
         *   I'm carrying 
         */
        presenceList = senseInfoTableSubset(infoTab,
            {obj, info: obj.isIn(self) && obj.(sense.presenceProp)});

        /* add a paragraph break */
        cosmeticSpacingReport('<.p>');
        
        /* list the items */
        lister.showList(self, nil, presenceList, 0, 0, infoTab, nil);
    }

    /*
     *   The Lister object that we use for inventory listings.  By
     *   default, we use actorInventoryLister, but this can be overridden
     *   if desired to use a different listing style.  
     */
    inventoryLister = actorInventoryLister

    /*
     *   The Lister object that we use for inventory listings of items
     *   being worn.  
     */
    wearingLister = actorWearingLister

    /*
     *   The Lister objects for inventory and worn-item listings, for use
     *   in a full description of the actor.  By default, we use the "long
     *   form" inventory lister, on the assumption that most actors have
     *   relatively lengthy descriptive text.  This can be overridden to
     *   use other formats; the short-form lister, for example, is useful
     *   for actors with only brief descriptions.  
     */
    holdingDescInventoryLister = actorHoldingDescInventoryListerLong
    holdingDescWearingLister = actorHoldingDescWearingLister

    /*
     *   Perform library pre-initialization on the actor 
     */
    initializeActor()
    {
        /* set up an empty pending command list */
        pendingCommand = new Vector(5);

        /* create a default inventory lister if we don't have one already */
        if (inventoryLister == nil)
            inventoryLister = actorInventoryLister;

        /* create a wearing lister if we don't have one already */
        if (wearingLister == nil)
            wearingLister = actorWearingLister;

        /* create a table of antecedents */
        antecedentTable = new LookupTable(8, 8);
    }

    /*
     *   Note conditions before an action or other event.  By default, we
     *   note our location and light/dark status, so that we comment on
     *   any change in the light/dark status after the event if we're
     *   still in the same location.  
     */
    noteConditionsBefore()
    {
        /* note our original location and light/dark status */
        locationBefore = location;
        locationLitBefore = isLocationLit();
    }

    /*
     *   Note conditions after an action or other event.  By default, if
     *   we are still in the same location we were in when
     *   noteConditionsBefore() was last called, and the light/dark status
     *   has changed, we'll mention the change in light/dark status. 
     */
    noteConditionsAfter()
    {
        /* 
         *   If our location hasn't changed but our light/dark status has,
         *   note the new status.  We don't make any announcement if the
         *   location has changed, since the travel routine will
         *   presumably have shown us the new location's light/dark status
         *   implicitly as part of the description of the new location
         *   after travel. 
         */
        if (location == locationBefore
            && isLocationLit() != locationLitBefore)
        {
            /* consider this the start of a new turn */
            "<.commandsep>";

            /* note the change with a new 'NoteDarkness' action */
            newActorAction(self, NoteDarkness);

            /* 
             *   start another turn, in case this occurred during an
             *   implicit action or the like 
             */
            "<.commandsep>";
        }
    }

    /* conditions we noted in noteConditionsBefore() */
    locationBefore = nil
    locationLitBefore = nil

    /* let the actor have a turn as soon as the game starts */
    nextRunTime = 0

    /* 
     *   Scheduling order - this determines the order of execution when
     *   several items are schedulable at the same game clock time.
     *   
     *   We choose a scheduling order that schedules actors in this
     *   relative order:
     *   
     *   100 player character, ready to execute
     *.  200 NPC, ready to execute
     *.  300 player character, idle
     *.  400 NPC, idle
     *   
     *   An "idle" actor is one that is waiting for another character to
     *   complete a command, or an NPC with no pending commands to
     *   perform.  (For the player character, it doesn't matter whether or
     *   not there's a pending command, because if the PC has no pending
     *   command, we ask the player for one.)
     *   
     *   This ordering ensures that each actor gets a chance to run each
     *   turn, but that actors with work to do go first, and other things
     *   being equal, the player character goes ahead of NPC's.  
     */
    scheduleOrder = 100

    /* calculate the scheduling order */
    calcScheduleOrder()
    {
        /* determine if we're ready to run */
        if (readyForTurn())
            scheduleOrder = isPlayerChar() ? 100 : 200;
        else
            scheduleOrder = isPlayerChar() ? 300 : 400;

        /* return the scheduling order */
        return scheduleOrder;
    }

    /*
     *   Determine if we're ready to do something on our turn.  We're
     *   ready to do something if we're not waiting for another actor to
     *   finish doing something and either we're the player character or
     *   we already have a pending command in our command queue.  
     */
    readyForTurn()
    {
        /* 
         *   if we're waiting for another actor, we're not ready to do
         *   anything 
         */
        if (checkWaitingForActor())
            return nil;

        /* 
         *   if we're the player character, we're always ready to take a
         *   turn as long as we're not waiting for another actor (which we
         *   now know we're not), because we can either execute one of our
         *   previously queued commands, or we can ask for a new command
         *   to perform 
         */
        if (isPlayerChar())
            return true;

        /* 
         *   if we have something other than placeholders in our command
         *   queue, we're ready to take a turn, because we can execute the
         *   next command in our queue 
         */
        if (pendingCommand.indexWhich({x: x.hasCommand()}) != nil)
            return true;

        /* 
         *   we have no specific work to do, so we're not ready for our
         *   next turn 
         */
        return nil;
    }

    /*
     *   Check to see if we're waiting for another actor to do something.
     *   Return true if so, nil if not.  If we've been waiting for another
     *   actor, and the actor has finished the task we've been waiting for
     *   since the last time we checked, we'll clean up our internal state
     *   relating to the wait and return nil.  
     */
    checkWaitingForActor()
    {
        local idx;

        /* if we're not waiting for an actor, simply return nil */
        if (waitingForActor == nil)
            return nil;

        /* 
         *   We're waiting for an actor to complete a command.  Check to
         *   see if the completion marker is still in the actor's queue;
         *   if it's not, then the other actor has already completed our
         *   task.  If the completion marker is in the other actor's
         *   queue, but it's the first thing in the queue, then we're also
         *   done waiting, because we're not actually waiting for the
         *   completion marker but instead for the tasks that were ahead
         *   of it in the queue.  
         */
        idx = waitingForActor.pendingCommand.indexOf(waitingForInfo);
        if (idx != nil && idx != 1)
        {
            /* 
             *   The marker is still in the queue, and there's at least
             *   one other command ahead of it, so the other actor hasn't
             *   finished the task we've been waiting for.  Tell the
             *   caller that we are indeed still waiting for someone.  
             */
            return true;
        }

        /*
         *   The other actor has disposed of our end-marker (or is about
         *   to, because it's the next thing left in the actor's queue),
         *   so it has finished with all of the commands we have been
         *   waiting for.  However, if I haven't caught up in game clock
         *   time with the actor I've been waiting for, I'm still waiting.
         */
        if (waitingForActor.nextRunTime > nextRunTime)
            return true;

        /* we're done waiting - forget our wait status information */
        waitingForActor = nil;
        waitingForInfo = nil;

        /* tell the caller we're no longer waiting for anyone */
        return nil;
    }

    /* the action the actor performed most recently */
    mostRecentAction = nil

    /*
     *   Add busy time.  An action calls this when we are the actor
     *   performing the action, and the action consumes game time.  This
     *   marks us as busy for the given time units.  
     */
    addBusyTime(action, units)
    {
        /* note the action being performed */
        mostRecentAction = action;

        /* adjust the next run time by the busy time */
        nextRunTime += units;
    }

    /*
     *   When it's our turn and we don't have any command to perform,
     *   we'll call this routine, which can perform a scripted operation
     *   if desired.  
     */
    idleTurn()
    {
        /* 
         *   do nothing, but consume a turn, so we're not ready to run
         *   again until the next game time increment 
         */
        ++nextRunTime;
    }

    /*
     *   Receive notification that this is a non-idle turn.  This is
     *   called whenever a command in our pending command queue is about
     *   to be executed.
     *   
     *   This method need not do anything at all, since the caller will
     *   take care of running the pending command.  The purpose of this
     *   method is to take care of any changes an actor wants to make when
     *   it receives an explicit command, as opposed to running its own
     *   autonomous activity.
     *   
     *   By default, we cancel follow mode if it's in effect.  It usually
     *   makes sense for an explicit command to interrupt follow mode;
     *   follow mode is usually started by an explicit command in the
     *   first place, so it is usually sensible for a new command to
     *   replace the one that started follow mode.
     */
    nonIdleTurn()
    {
        /* by default, cancel follow mode */
        followingActor = nil;
    }

    /*
     *   If we're following an actor, this keeps track of the actor we're
     *   following.  NPC's can use this to follow around another actor
     *   whenever possible.  
     */
    followingActor = nil

    /*
     *   Handle a situation where we're trying to follow an actor but
     *   can't.  By default, this simply cancels our follow mode.
     *   
     *   Actors might want to override this to be more tolerant.  For
     *   example, an actor might want to wait until five turns elapse to
     *   give up on following, in case the target actor returns after a
     *   brief digression; or an actor could stay in follow mode until it
     *   received other instructions, or found something better to do.  
     */
    cannotFollow()
    {
        /* 
         *   by default, simply cancel follow mode by forgetting about the
         *   actor we're following
         */
        followingActor = nil;
    }

    /*
     *   Execute one "turn" - this is a unit of time passing.  The player
     *   character generally is allowed to execute one command in the
     *   course of a turn; a non-player character with a programmed task
     *   can perform an increment of the task.  
     */
    executeTurn()
    {
        /* 
         *   always start a new command visually when a new actor is
         *   taking over 
         */
        "<.commandsep>";

        /*
         *   If we have a pending response, and we're in a position to
         *   deliver it, our next work is to deliver the pending response.
         */
        if (pendingResponse != nil && canTalkTo(pendingResponse.issuer_))
        {
            /* 
             *   We have a pending response, and the command issuer from
             *   the pending response can hear us now, so we can finally
             *   deliver the response.
             *   
             *   If the issuer is the player character, send to the player
             *   using our deferred message generator; otherwise, call the
             *   issuer's notification routine, since it's an NPC-to-NPC
             *   notification.  
             */
            if (pendingResponse.issuer_.isPlayerChar())
            {
                /* 
                 *   we're notifying the player - use the deferred
                 *   message generator 
                 */
                getParserDeferredMessageObj().(pendingResponse.prop_)(
                    self, pendingResponse.args_...);
            }
            else
            {
                /* it's an NPC-to-NPC notification - notify the issuer */
                pendingResponse.issuer_.notifyIssuerParseFailure(
                    self, pendingResponse.prop_, pendingResponse.args_);
            }
        }

        /* check to see if we're waiting for another actor to do something */
        if (checkWaitingForActor())
        {
            /* 
             *   we're still waiting, so there's nothing for us to do;
             *   take an idle turn and return 
             */
            idleTurn();
            return true;
        }

        /* 
         *   if we're the player character, and we have no pending
         *   commands to execute, our next task will be to read and
         *   execute a command 
         */
        if (pendingCommand.length() == 0 && isPlayerChar())
        {
            local toks;
            
            /* execute any pre-command-prompt daemons */
            eventManager.executePrompt();
            
            /* read a command line and get the resulting token list */
            toks = readMainCommandTokens(rmcCommand);
            
            /* 
             *   If it came back nil, it means that the input was fully
             *   processed in pre-parsing; this means that we don't have
             *   any more work to do on this turn, so we can simply end
             *   our turn now.
             */
            if (toks == nil)
                return true;
            
            /* retrieve the token list from the command line */
            toks = toks[2];
            
            /* 
             *   Add it to our pending command queue.  Since we read the
             *   command from the player, and we're the player character,
             *   we treat the command as coming from myself.  
             *   
             *   Since this is a newly-read command line, we're starting a
             *   new sentence.  
             */
            addPendingCommand(true, self, toks);
        }

        /*
         *   Check to see if we have any pending command to execute.  If
         *   so, our next task is to execute the pending command.  
         */
        if (pendingCommand.length() != 0)
        {
            local cmd;
            
            /* remove the first pending command from our queue */
            cmd = pendingCommand[1];
            pendingCommand.removeElementAt(1);
            
            /* if this is a real command, note the non-idle turn */
            if (cmd.hasCommand())
                nonIdleTurn();
            
            /* execute the first pending command */
            cmd.executePending(self);

            /* 
             *   We're done with this turn.  If we no longer have any
             *   pending commands, tell the scheduler to refigure the
             *   execution order, since another object might now be ready
             *   to run ahead of our idle activity.  
             */
            if (pendingCommand.indexWhich({x: x.hasCommand()}) == nil)
                return nil;
            else
                return true;
        }

        /*
         *   If we're following an actor, and the actor isn't in sight,
         *   see if we can catch up.  
         */
        if (followingActor != nil
            && (followingActor.location.effectiveFollowLocation
                != location.effectiveFollowLocation))
        {
            local info;
            
            /* see if we have enough information to follow */
            info = getFollowInfo(followingActor);

            /* 
             *   Check to see if we have enough information to follow the
             *   actor.  We can only follow if we saw the actor depart at
             *   some point, and we're in the same location where we last
             *   saw the actor depart.  (We have to be in the same
             *   location, because we follow by performing the same
             *   command we saw the actor perform when we last saw the
             *   actor depart.  Repeating the command will obviously be
             *   ineffective unless we're in the same location as the
             *   actor was.)  
             */
            if (info != nil)
            {
                /* 
                 *   We know how to follow the actor.  Go into our sense
                 *   context, and perform the same command we saw the
                 *   actor perform. 
                 */
                callWithSenseContext(self, sight,
                    {: newActorAction(self, Follow, followingActor), 0 });

                /* 
                 *   if we failed to track the actor, note that we are
                 *   unable to follow the actor 
                 */
                if (location.effectiveFollowLocation
                    != followingActor.location.effectiveFollowLocation)
                {
                    /* note that we failed to follow the actor */
                    cannotFollow();
                }

                /* we're done with this turn */
                return true;
            }
            else
            {
                /* 
                 *   we don't know how to follow this actor - call our
                 *   cannot-follow handler 
                 */
                cannotFollow();
            }
        }

        /* we have no pending work to perform, so take an idle turn */
        idleTurn();

        /* no change in scheduling priority */
        return true;
    }

    /*
     *   By default, all actors are likely command targets.  This should
     *   be overridden for actors who are obviously not likely to accept
     *   commands of any kind.
     *   
     *   This is used to disambiguate target actors in commands, so this
     *   should provide an indication of what should be obvious to a
     *   player, because the purpose of this information is to guess what
     *   the player is likely to take for granted in specifying a target
     *   actor.
     */
    isLikelyCommandTarget = true

    /*
     *   Determine if we should accept a command.  'issuingActor' is the
     *   actor who issued the command: if the player typed the command on
     *   the command line, this will be the player character actor.
     *   
     *   This routine performs only the simplest check, since it doesn't
     *   have access to the specific action being performed.  This is
     *   intended as a first check, to allow us to bypass noun resolution
     *   if the actor simply won't accept any command from the issuer.
     *   
     *   Returns true to accept a command, nil to reject it.  If this
     *   routine returns nil, and the command came from the player
     *   character, a suitable message should be displayed.
     *   
     *   Note that most actors should not override this routine simply to
     *   express the will of the actor to accept a command, since this
     *   routine performs a number of checks for the physical ability of
     *   the actor to execute a command from the issuer.  To determine
     *   whether or not the actor should obey physically valid commands
     *   from the issuer, override obeyCommand().  
     */
    acceptCommand(issuingActor)
    {
        /* if we're the current player character, accept any command */
        if (isPlayerChar())
            return true;

        /* if the other actor can't hear us, we can't talk to it */
        if (issuingActor != self && !issuingActor.canTalkTo(self))
        {
            /* the other actor can't hear us */
            notifyParseFailure(issuingActor, &commandNotHeard, []);

            /* tell the caller that the command cannot proceed */
            return nil;
        }

        /* if I'm busy doing something else, say so */
        if (nextRunTime > Schedulable.gameClockTime)
        {
            /* tell the issuing actor I'm busy */
            notifyParseFailure(issuingActor, &refuseCommandBusy,
                               [issuingActor]);

            /* tell the caller to abandon the command */
            return nil;
        }

        /* check to see if I have other work to perform first */
        if (!acceptCommandBusy(issuingActor))
            return nil;

        /* we can accept the command; check if we want to */
        return obeyCommand(issuingActor);
    }

    /*
     *   Check to see if I'm busy with pending commands, and if so,
     *   whether or not I should accept a new command.  Returns true if we
     *   should accept a command, nil if not.  If we return nil, we must
     *   notify the issuer of the rejection.
     *   
     *   By default, we won't accept a command if we have any work
     *   pending.  
     */
    acceptCommandBusy(issuingActor)
    {
        /* if we have any pending commands, don't accept a new command */
        if (pendingCommand.length() != 0)
        {
            /* 
             *   if we have only commands from the same issuer pending,
             *   cancel all of the pending commands and accept the new
             *   command instead 
             */
            foreach (local info in pendingCommand)
            {
                /* 
                 *   if this is from a different issuer, don't accept a
                 *   new command 
                 */
                if (info.issuer_ != issuingActor)
                {
                    /* tell the other actor that we're busy */
                    notifyParseFailure(issuingActor, &refuseCommandBusy,
                                       [issuingActor]);

                    /* tell the caller to abandon the command */
                    return nil;
                }
            }

            /* 
             *   all of the pending commands were from the same issuer, so
             *   presumably the issuer wants to override those commands;
             *   remove the old ones from our pending queue
             */
            pendingCommand.removeRange(1, pendingCommand.length());
        }

        /* we didn't find any problems */
        return true;
    }

    /*
     *   Determine if we want to obey a command from the given actor.  We
     *   only get this far when we determine that it's possible for us to
     *   accept a command, given the sense connections between us and the
     *   issuing actor, and our game time status.
     *   
     *   This routine should display an appropriate message and return nil
     *   if the command is not to be accepted, and should simply return
     *   true to accept a command from the actor.
     *   
     *   By default, we'll refuse commands from other actors.
     *   
     *   Note that actors that override this might also need to override
     *   wantsFollowInfo(), since an actor that accepts "follow" commands
     *   will need to keep track of the movements of other actors if it is
     *   to carry out any following.  
     */
    obeyCommand(issuingActor)
    {
        /* 
         *   we're not the player character, so don't accept anything -
         *   display a message and indicate failure 
         */
        getParserMessageObj().refuseCommand(self, issuingActor);
        return nil;
    }
    
    /* 
     *   Add a command to our pending command list.  The new command is
     *   specified as a list of tokens to be parsed, and it is added after
     *   any commands already in our pending list.  
     */
    addPendingCommand(startOfSentence, issuer, toks)
    {
        /* add a descriptor to the pending command list */
        pendingCommand.append(
            new PendingCommandInfo(startOfSentence, issuer, toks, nil));
    }

    /* 
     *   Insert a command at the head of our pending command list.  The
     *   new command is specified as a list of tokens to parse, and it is
     *   inserted into our pending command list before any commands
     *   already in the list.  
     */
    addFirstPendingCommand(startOfSentence, issuer, toks)
    {
        /* add a descriptor to the start of our list */
        pendingCommand.insertAt(
            1, new PendingCommandInfo(startOfSentence, issuer, toks, nil));
    }

    /*
     *   Add a resolved action to our pending command list.  The new
     *   command is specified as a resolved Action object; it is added
     *   after any commands already in our list. 
     */
    addPendingAction(startOfSentence, issuer, action, [objs])
    {
        /* add a descriptor to the pending command list */
        pendingCommand.append(
            new PendingCommandInfo(startOfSentence, issuer, nil,
                                   action, objs...));
    }

    /*
     *   Insert a resolved action at the start of our pending command
     *   list.  The new command is specified as a resolved Action object;
     *   it is added before any commands already in our list.  
     */
    addFirstPendingAction(startOfSentence, issuer, action, [objs])
    {
        /* add a descriptor to the pending command list */
        pendingCommand.insertAt(
            1, new PendingCommandInfo(startOfSentence, issuer, nil,
                                      action, objs...));
    }


    /* pending commands - this is a list of PendingCommandInfo objects */
    pendingCommand = nil

    /* 
     *   pending response - this is a single PendingResponseInfo object,
     *   which we'll deliver as soon as the issuing actor is in a position
     *   to hear us 
     */
    pendingResponse = nil

    /* 
     *   get the library message object for a parser message addressed to
     *   the player character 
     */
    getParserMessageObj()
    {
        /* 
         *   If I'm the player character, use the player character message
         *   object; otherwise, use the default non-player character
         *   message object.
         *   
         *   To customize parser messages from a particular actor, create
         *   an object based on npcMessages, and override this routine in
         *   the actor so that it returns the custom object rather than
         *   the standard npcMessages object.  To customize messages for
         *   ALL of the NPC's in a game, simply modify npcMessages itself,
         *   since it's the default for all non-player characters.  
         */
        return isPlayerChar() ? playerMessages : npcMessages;
    }

    /*
     *   Get the deferred library message object for a parser message
     *   addressed to the player character.  We only use this to generate
     *   messages deferred from non-player characters.  
     */
    getParserDeferredMessageObj() { return npcDeferredMessages; }

    /*
     *   Get the library message object for action responses.  This is
     *   used to generate library responses to verbs.  
     */
    getActionMessageObj()
    {
        /* 
         *   return the default player character or NPC message object,
         *   depending on whether I'm the player or not; individual actors
         *   can override this to supply actor-specific messages for
         *   library action responses 
         */
        return isPlayerChar() ? playerActionMessages: npcActionMessages;
    }

    /* 
     *   Notify an issuer that a command sent to us resulted in a parsing
     *   failure.  We are meant to reply to the issuer to let the issuer
     *   know about the problem.  messageProp is the libGlobal message
     *   property describing the error, and args is a list with the
     *   (varargs) arguments to the message property.  
     */
    notifyParseFailure(issuingActor, messageProp, args)
    {
        /* check who's talking to whom */
        if (issuingActor.isPlayerChar())
        {
            /*
             *   The player issued the command.  If the command was
             *   directed to an NPC (i.e., we're not the player), check to
             *   see if the player character is in scope from our
             *   perspective.  
             */
            if (issuingActor != self && !canTalkTo(issuingActor))
            {
                /* 
                 *   The player issued the command to an NPC, but the
                 *   player is not capable of hearing the NPC's response.
                 */
                cannotRespondToCommand(issuingActor, messageProp, args);
            }
            else
            {
                /* 
                 *   generate a message using the appropriate message
                 *   generator object 
                 */
                getParserMessageObj().(messageProp)(self, args...);
            }
        }
        else
        {
            /*
             *   the command was issued from one NPC to another - notify
             *   the issuer of the problem, but don't display any
             *   messages, since this interaction is purely among the
             *   NPC's 
             */
            issuingActor.
                notifyIssuerParseFailure(self, messageProp, args);
        }
    }

    /*
     *   We have a parser error to report to the player, but we cannot
     *   respond at the moment because the player is not capable of
     *   hearing us (there is no sense path for our communications senses
     *   from us to the player actor).  Defer reporting the message until
     *   later.
     */
    cannotRespondToCommand(issuingActor, messageProp, args)
    {
        /* 
         *   Remember the problem for later deliver.  If we already have a
         *   deferred response, forget it - just report the latest
         *   problem.  
         */
        pendingResponse =
            new PendingResponseInfo(issuingActor, messageProp, args);

        /*
         *   Some actors might want to override this to start searching
         *   for the player character.  We don't have any generic
         *   mechanism to conduct such a search, but a game that
         *   implements one might want to make use of it here.  
         */
    }

    /*
     *   Receive notification that a command we sent to another NPC
     *   failed.  This is only called when one NPC sends a command to
     *   another NPC; this is called on the issuer to let the issuer know
     *   that the target can't perform the command because of the given
     *   resolution failure.
     *   
     *   By default, we don't do anything here, because we don't have any
     *   default code to send a command from one NPC to another.  Any
     *   custom NPC actor that sends a command to another NPC actor might
     *   want to use this to deal with problems in processing those
     *   commands.  
     */
    notifyIssuerParseFailure(targetActor, messageProp, args)
    {
        /* by default, we do nothing */
    }

    /*
     *   Antecedent lookup table.  Each actor keeps its own table of
     *   antecedents indexed by pronoun type, so that we can
     *   simultaneously have different antecedents for different pronouns.
     */
    antecedentTable = nil

    /* look up a pronoun's value */
    getPronounAntecedent(typ)
    {
        return antecedentTable[typ];
    }

    /* set a pronoun's antecedent value */
    setPronounAntecedent(typ, val)
    {
        antecedentTable[typ] = val;
    }

    /*
     *   Copy pronoun antecedents from the given actor.  This should be
     *   called whenever an actor issues a command to us, so that pronouns
     *   in the command are properly resolved relative to the issuer.  
     */
    copyPronounAntecedentsFrom(issuer)
    {
        /* copy every element from the issuer's table */
        issuer.antecedentTable.forEachAssoc(
            {key, val: antecedentTable[key] = val });
    }

    /* -------------------------------------------------------------------- */
    /*
     *   Verb processing 
     */

    /* show a "take from" message as indicating I don't have the dobj */
    takeFromNotInMessage = &takeFromNotInActor

    dobjFor(Take)
    {
        verify()
        {
            /* we can't take ourselves, obviously */
            if (self == gActor)
                illogical(&takingSelf);
            
            /* inherit default handling */
            inherited();
        }
    }

    dobjFor(Drop)
    {
        verify()
        {
            /* we can't drop ourselves, obviously */
            if (self == gActor)
                illogical(&droppingSelf);
            
            /* inherit default handling */
            inherited();
        }
    }

    dobjFor(Kiss)
    {
        preCond = [touchObj]
        verify()
        {
            /* cannot kiss oneself */
            if (gActor == self)
                illogical(&cannotKissSelf);
        }
        action() { mainReport(&cannotKissActor); }
    }

    dobjFor(AskFor)
    {
        preCond = [canTalkToObj]
        verify() { }
        action() { mainReport(&noResponseFrom, self); }
    }

    dobjFor(TalkTo)
    {
        preCond = [canTalkToObj]
        verify() { }
        action()
        {
            /* use the normal greeting routine */
            greetingsFrom(gActor);
        }
    }

    iobjFor(GiveTo)
    {
        verify() { }
        check()
        {
            /* 
             *   If the indirect object is the issuing actor, rephrase
             *   this as "issuer, ask actor for dobj".
             */
            if (gIssuingActor == self)
                replaceActorAction(self, AskFor, gActor, gDobj);
        }
        action()
        {
            /* note that I'm the last conversational partner */
            gActor.lastInterlocutor = self;

            /* by default, actors are uninterested in receiving objects */
            mainReport(&notInterested, self);
        }
    }

    iobjFor(ShowTo)
    {
        verify() { }
        action()
        {
            /* note that I'm the last conversational partner */
            gActor.lastInterlocutor = self;

            /* we have no response by default */
            mainReport(&notInterested, self);
        }
    }

    dobjFor(AskAbout)
    {
        verify()
        {
            /* it makes no sense to ask oneself about something */
            if (gActor == self)
                illogical(&cannotAskSelf);
        }
        action()
        {
            /* note that I'm the last conversational partner */
            gActor.lastInterlocutor = self;

            /* we don't have any response by default */
            mainReport(&noResponseFrom, self);
        }
    }

    dobjFor(TellAbout)
    {
        verify()
        {
            /* it makes no sense to tell oneself about something */
            if (gActor == self)
                illogical(&cannotTellSelf);
        }
        check()
        {
            /* 
             *   if the direct object is the issuing actor, rephrase this
             *   as "issuer, ask actor about iobj" 
             */
            if (gDobj == gIssuingActor)
                replaceActorAction(gIssuingActor, AskAbout, gActor, gTopic);
        }
        action()
        {
            /* note that I'm the last conversational partner */
            gActor.lastInterlocutor = self;

            /* we don't have any response by default */
            mainReport(&noResponseFrom, self);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   An UntakeableActor is one that can't be picked up and moved. 
 */
class UntakeableActor: Actor, Fixed
    /* use customized messages for some 'Fixed' methods */
    cannotTakeMsg = &cannotTakeActor
    cannotMoveMsg = &cannotMoveActor
    cannotPutMsg = &cannotPutActor

    /* 
     *   even though we act like a Fixed, we don't count as a Fixed for
     *   listing purposes 
     */
    contentsInFixedIn(loc) { return nil; }
;


/*
 *   A Person is an actor that represents a human character.  This is just
 *   an UntakeableActor with some custom versions of the messages for
 *   taking and moving the actor.  
 */
class Person: UntakeableActor
    /* customize the messages for trying to take or move me */
    cannotTakeMsg = &cannotTakePerson
    cannotMoveMsg = &cannotMovePerson
    cannotPutMsg = &cannotPutPerson

    /* 
     *   use a fairly large default bulk, since people are usually fairly
     *   large compared with the sorts of items that one carries around 
     */
    bulk = 10
;

/* ------------------------------------------------------------------------ */
/*
 *   Pending response information structure 
 */
class PendingResponseInfo: object
    construct(issuer, prop, args)
    {
        issuer_ = issuer;
        prop_ = prop;
        args_ = args;
    }

    /* the issuer of the command (and target of the response) */
    issuer_ = nil

    /* the message property and argument list for the message */
    prop_ = nil
    args_ = []
;

/*
 *   Pending Command Information structure 
 */
class PendingCommandInfo: object
    construct(startOfSentence, issuer, toks, action, [objs])
    {
        startOfSentence_ = startOfSentence;
        issuer_ = issuer;
        tokens_ = toks;
        action_ = action;
        objs_ = objs;
    }

    /*
     *   Check to see if this pending command item has a command to
     *   perform.  This returns true if we have a command, nil if we're
     *   just a queue placeholder without any actual command to execute.  
     */
    hasCommand()
    {
        /* if we have either tokens or an action object, we have a command */
        return (tokens_ != nil || action_ != nil);
    }

    /*
     *   Execute the pending command. 
     */
    executePending(targetActor)
    {
        /*
         *   The way we execute the command depends on how the command is
         *   specified: if it's specified with a token list, we must parse
         *   the tokens and execute the command; if it's specified as a
         *   resolved action, we must simply execute the action.
         *   
         *   Note that we can have a pending command that has neither an
         *   action nor a token list.  Empty pending commands can be used
         *   as queue markers, for purposes such as waiting for an actor
         *   to finish a set of commands before allowing another task to
         *   proceed.  We simply ignore empty pending commands.
         *   
         *   No matter what kind of command we have, if the target actor
         *   isn't the player character, we must execute the command in a
         *   sight sensory context for the target actor; this will ensure
         *   that any messages we generate will be displayed by default
         *   only if the player character can see the actor performing the
         *   action.  
         */
        if (tokens_ != nil)
        {
            /*
             *   It's a token list, so we must parse and execute the
             *   tokens.  
             */
            callWithSenseContext(
                targetActor.isPlayerChar() ? nil : targetActor, sight,
                executeCommand, targetActor, issuer_,
                tokens_, startOfSentence_);
        }
        else if (action_ != nil)
        {
            /*
             *   It's a resolved action, so simply invoke the action's
             *   main execution method. 
             */
            try
            {
                /* run the action */
                newActionObj(CommandTranscript, issuer_, targetActor,
                             action_, objs_...);
            }
            catch (TerminateCommandException tcExc)
            {
                /* 
                 *   the command cannot proceed; simply abandon the
                 *   command action here 
                 */
            }
        }
    }
    
    /* the issuer of the command */
    issuer_ = nil

    /* 
     *   The token list for the command, and the action.  Note that these
     *   are mutually exclusive: a pending command is specified either as
     *   a token list to parse or as a pre-resolved action object.  If a
     *   token list is provided, the action is ignored.  
     */
    tokens_ = nil
    action_ = nil

    /* the resolved objects for the action (used if action_ is used) */
    objs_ = nil

    /* we're at the start of a "sentence" */
    startOfSentence_ = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   Set the current player character 
 */
setPlayer(actor)
{
    /* remember the new player character */
    libGlobal.playerChar = actor;

    /* set the root global point of view to this actor */
    setRootPOV(actor);
}

