-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001,2002  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.IO_Exceptions;
with Ada.Strings.Fixed;
with Ada.Strings.Unbounded;
with Ada.Text_IO;
with Configuration;
with Expanding_Array;
with Externals;             use Externals;
with Externals.Mail;
with Externals.Simple;
with Menus;                 use Menus;
with Misc;                  use Misc;
with Readline;

package body Attachments is

   Attachment_List_Initial_Size : constant Positive := 50;

   type Menu_Array_Ptr is access all Attachlist_Menus.MNA;

   -- Add an attachment to the list....
   procedure Add_Attachment (Filename  : in     UBS;
                             List      : in out Attachment_List) is
      Match : Boolean := False;
   begin
      Debug("+Add_Attachment");
      -- Is it a duplicate?
      Debug("Add_Attachment: Checking for duplicates");
      declare
         use type UBS;
      begin
         Debug("Add_Attachment: Approaching duplicates loop");
         for I in 1..List.Count loop
            Debug("Add_Attachment: In duplicates loop");
            Match := Match or Filename = UAP.Value(List.AA, I);
         end loop;
      end;
      if not Match then
         -- Add the attachment.
         Debug("Add_Attachment: Adding an attachment");
         List.Count := List.Count + 1;
         UAP.Set(List.AA, List.Count, Filename);
         Debug("Add_Attachment: Done adding an attachment");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Attachments.Add_Attachment");
         raise;
   end Add_Attachment;

   -- Remove a key.  We then shuffle the array down.
   -- It will only remove the last key if there are multiple instances.
   procedure Remove_Attachment (Filename  : in     UBS;
                                List      : in out Attachment_List) is
      A_Num : Integer;
      A_Found : Boolean := False;
      use type UBS;
   begin
      for I in 1..List.Count loop
         if Filename = UAP.Value(List.AA, I) then
            A_Found := True;
            A_Num := I;
         end if;
      end loop;
      if A_Found then
         -- Shuffle the array over it.
         for I in A_Num .. List.Count - 1 loop
            UAP.Set(List.AA, I, UAP.Value(List.AA, I+1));
         end loop;
         -- Reduce the number of keys.
         List.Count := List.Count - 1;
      else
         raise Attachment_Not_Found;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Attachments.Remove_Attachment");
         raise;
   end Remove_Attachment;

   -- List the keys and edit them as appropriate (include removing a key
   -- from that list, and adding one from the keyring.
   procedure List (List : in out Attachment_List) is
      A_Head_Length : constant Positive := 74;
      A_Menu : Menu_Array_Ptr;

      procedure Generate_A_List is
      begin
         Debug("+Generate_A_List");
         -- Create a menu with the prefix of these attachments.
         A_Menu := new Attachlist_Menus.MNA(1..List.Count);
         for I in 1..List.Count loop
            declare
               S : String := ToStr(UAP.Value(List.AA, I));
            begin
               if S'Length < A_Head_Length then
                  A_Menu(I) := ToUBS(S);
               else
                  -- Get the first 10, then ..., then A_Head_Length-13 of the last characters.
                  A_Menu(I) := ToUBS(S(S'First..S'First+9)
                                     & "..."
                                     & S(S'Last-A_Head_Length+14..S'Last));
               end if;
            end;
         end loop;
         Debug("-Generate_A_List");
      exception
         when others =>
            Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                                 "Exception raised in Attachments.List.Generate_A_List");
            raise;
      end Generate_A_List;

      Selection    : Attachlist_Menus.CLM_Return;
      Selected     : Integer;
   begin
      Debug("+List");
      Generate_A_List;
  List_Keys_Loop:
      loop
         Selection := Attachlist_Menu(A_Menu.all);
         if not Selection.Is_Num then
            case Selection.I is
               when Done =>
                  exit List_Keys_Loop;
               when Add => -- Add filename.
                  declare
                     New_Filename : UBS;
                     Alt_Filename : UBS;
                     use Ada.Text_IO;
                  begin
                     New_Line(1);
                     Put_Line("Add attachment:");
                     New_Filename
                       := ToUBS(Readline.Get_String("Type filename: ",
                                                    Enable_Tab_Completion => True));
                     -- Deal with the last space that we'll often get.
                     declare
                        NFN : String := ToStr(New_Filename);
                     begin
                        if NFN(NFN'Last) = ' ' then
                           Alt_Filename := ToUBS(NFN(NFN'First..NFN'Last-1));
                        else
                           Alt_Filename := New_Filename;
                        end if;
                     end;
                     -- Does it exist?
                     if Externals.Simple.Test_R(ToStr(New_Filename)) then
                        Add_Attachment(New_Filename, List);
                        Generate_A_List;
                     elsif Externals.Simple.Test_R(ToStr(Alt_Filename)) then
                        Add_Attachment(Alt_Filename, List);
                        Generate_A_List;
                     else
                        Put_Line("`" & ToStr(New_Filename) & "' is not readable or does not exist (if you've used a `~', try expanding it)");
                     end if;
                  end;
            end case;
         else
            -- Remove the given number.
            Selected := Selection.N;
            Remove_Attachment(UAP.Value(List.AA, Selected),
                              List);
            Generate_A_List;
         end if;
      end loop List_Keys_Loop;
      Debug("-List_Keys");
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Attachments.List");
         raise;
   end List;

   procedure Empty_Attachmentlist (List : in out Attachment_List) is
   begin
      UAP.Create(List.AA, Attachment_List_Initial_Size);
      List.Count := 0;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Attachments.Empty_Attachmentlist");
         raise;
   end;

   function Count (List : in Attachment_List) return Natural is
   begin
      return List.Count;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Attachments.Count");
         raise;
   end Count;

   procedure Replace_Tmpfile (Tmpfile           : in String;
                              List              : in out Attachment_List;
                              Internal_Mimetype :    out UBS) is
      AL : UBS_Array(1..Count(List)+1);
      TF : String := Temp_File_Name("attrtf");
   begin
      -- Do nothing if the attachment list is empty.
      if List.Count /= 0 then
         -- Now, construct a list of items.
         AL(1) := ToUBS(Tmpfile);
         for I in 1..List.Count loop
            AL(I+1) := UAP.Value(List.AA, I);
         end loop;
         -- And call mimeconstruct.
         Externals.Mail.Mimeconstruct_Mixed(AL, TF);
         -- Then replace the original tmpfile.
         Externals.Mail.Extract_Body(TF, Tmpfile);
         -- We guess that the mimetype is in fact multipart/mixed.
      else
         Internal_Mimetype := ToUBS("");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Attachments.Replace_Tmpfile");
         raise;
   end Replace_Tmpfile;

end Attachments;


