package org.apache.turbine.services.jsp;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and 
 *    "Apache Turbine" must not be used to endorse or promote products 
 *    derived from this software without prior written permission. For 
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without 
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// Java Stuff 
import java.io.File;
import java.io.IOException;

// Java Servlet Classes
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.RequestDispatcher;
import javax.servlet.http.HttpServletRequest;

// Turbine Stuff
import org.apache.turbine.util.Log;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.TurbineException;

import org.apache.turbine.services.jsp.util.JspLink;
import org.apache.turbine.services.InitializationException;
import org.apache.turbine.services.TurbineBaseService;
import org.apache.turbine.services.servlet.TurbineServlet;
import org.apache.turbine.services.template.BaseTemplateEngineService;
import org.apache.turbine.services.template.TurbineTemplate;

import org.apache.velocity.runtime.configuration.Configuration;

/**
 * This is a Service that can process JSP templates from within a Turbine 
 * screen.
 *
 * @author <a href="mailto:john.mcnally@clearink.com">John D. McNally</a>
 * @author <a href="mailto:jvanzyl@apache.org">Jason van Zyl</a>
 * @author <a href="mailto:dlr@finemaltcoding.com">Daniel Rall</a>
 */
public class TurbineJspService extends BaseTemplateEngineService
    implements JspService
{
    /** The base path prepended to filenames given in arguments */
    private String[] paths;

    /** The buffer size for the output stream. */
    private int bufferSize;

    /**
     * Performs early initialization of this Turbine service.
     */
    public void init(ServletConfig config) throws InitializationException
    {
        try
        {
            initJsp();
            registerConfiguration("jsp");
            setInit(true);
        }
        catch (Exception e)
        {
            throw new InitializationException(
                "TurbineJspService failed to initialize", e);
        }
    }
    
    /**
     * Adds some convenience objects to the request.  For example an instance
     * of JspLink which can be used to generate links to other templates.
     * 
     * @param RunData the turbine rundata object 
     */
    public void addDefaultObjects(RunData data)
    {
        HttpServletRequest req = data.getRequest();
        req.setAttribute(LINK, new JspLink(data));
        req.setAttribute(RUNDATA, data);
    }

    /**
     * The buffer size
     * 
     */
    public int getDefaultBufferSize()
    {
        return bufferSize;
    }

    /**
     * Process the request 
     * 
     * @param RunData 
     * @param String the filename of the template.
     * @throws TurbineException Any exception trown while processing will be
     *         wrapped into a TurbineException and rethrown.
     */
    public void handleRequest(RunData data, String filename) 
        throws TurbineException
    {
        handleRequest(data, filename, false);
    }
    
    /**
     * Process the request 
     * 
     * @param RunData 
     * @param String the filename of the template.
     * @param boolean whether to perform a forward or include.
     * @throws TurbineException Any exception trown while processing will be
     *         wrapped into a TurbineException and rethrown.
     */
    public void handleRequest(RunData data, String filename, boolean isForward) 
        throws TurbineException
    {                                
        // get the RequestDispatcher for the JSP
        RequestDispatcher dispatcher = data.getServletContext()
            .getRequestDispatcher(paths[0] + filename);
        
        try
        {
            if (isForward)
            {
                // forward the request to the JSP
                dispatcher.forward( data.getRequest(), data.getResponse() ); 
            }
            else
            {
                data.getOut().flush();
                // include the JSP
                dispatcher.include( data.getRequest(), data.getResponse() ); 
            }
        }
        catch(Exception e)
        {
            // as JSP service is in Alpha stage, let's try hard to send the error
            // message to the browser, to speed up debugging
            try
            {
                data.getOut().print("Error encountered processing a template: "+filename); 
                e.printStackTrace(data.getOut());
            }
            catch(IOException ignored) 
            {
            }    

            // pass the exception to the caller according to the general
            // contract for tamplating services in Turbine
            throw new TurbineException(
                "Error encountered processing a template:" + filename, e);
        }
    }
   
    /**
     * This method sets up the template cache.
     */
    private void initJsp() throws Exception
    {
        ServletContext context = TurbineServlet.getServletContext();
        Configuration config = getConfiguration();

        /*
         * Use the turbine template service to translate
         * the template paths.
         */
        paths = TurbineTemplate.translateTemplatePaths(
            config.getStringArray("templates"));
        
        /* not working properly - removing until I can investigate (jdm)
        if (path.startsWith("/"))
        {
            path = path.substring(1);
        }
        if (path.length() > 0 && !path.endsWith("/"))
        {
            path += "/";
        }
        */
        
        bufferSize = config.getInt("buffer.size", 8192);
    
        /*
         * Register with the template service.
         */
        registerConfiguration("jsp");
    }

    /**
     * Determine whether a given template exists. This service
     * currently only supports file base template hierarchies
     * so we will use the utility methods provided by
     * the template service to do the searching.
     *
     * @param String template
     * @return boolean
     */
    public boolean templateExists(String template)
    {
        return TurbineTemplate.templateExists(template, paths);
    }              
}
