#!/bin/bash
#
# 	multistation-audio-refresh.sh
#
#	Author:
#		Jason Kim <jason@userful.com>
#
# 	Copyright (c) 2002-2006 by Userful Corporation
#
# 	DESCRIPTION
#		  This program regenerates /etc/usbaudio using HAL on every bootup and
#		keyboard assignment.
#
# 	Called from: multistation-audio-config.sh, multistation-devices-refresh.sh

logfile="/var/log/userful-start.log"
CONFIG_FILE="/etc/usbaudio"
ProgramName=`basename $0`

if [ ! -e /etc/X11/stations ]; then
    echo "`date +"%b %d %T"` $ProgramName: /etc/X11/stations file is missing" >> $logfile
    exit 0
fi

if [ ! -e $CONFIG_FILE ]; then
	touch $CONFIG_FILE
fi

# Remove all error line (not contain 5 items)
awk '{if (NF == 5) print $0}' $CONFIG_FILE > $CONFIG_FILE.old

# Create empty /etc/usbaudio
rm -f $CONFIG_FILE
touch $CONFIG_FILE
chmod a+r $CONFIG_FILE

# Exit if there is no sound device
if [ ! -d /sys/class/sound ]; then
	echo "`date +"%b %d %T"` $ProgramName: No audio device found." >> $logfile
	exit 0
fi

#----------------------------------------------
# Utilize HAL system to find alsa audio devices
# TODO: need to handle internal audio
#----------------------------------------------

# find udis for every playback devices
AUDIO_UDIS=`hal-find-by-property --key "alsa.type" --string "playback"`

for UDI in $AUDIO_UDIS
do
	# Find alsa card number
	ALSA_CARD=`hal-get-property --udi "$UDI" --key "alsa.card"`

    # Some audio devices, especailly builtins, have more than one playback device.
    # It will create more than one same entry in /etc/usbaudio file
    ALSA_DEVICE=`hal-get-property --udi "$UDI" --key "alsa.device"`
    if [[ "$ALSA_DEVICE" != "0" ]] ; then
        continue
    fi

	# Find device vendor name
	PARENT_UDI=`hal-get-property --udi "$UDI" --key "info.parent"`
	PARENT_SUBSYSTEM=`hal-get-property --udi "$PARENT_UDI" --key "linux.subsystem"`

	if [ "$PARENT_SUBSYSTEM" == "sound" ]; then
        # Parent's subsystem is sound in the newer version of HAL 
        DEV_VENDOR=`hal-get-property --udi "$PARENT_UDI" --key "sound.card_id"`

	elif [ "$PARENT_SUBSYSTEM" == "usb" ]; then
		# USB audio device
		DEV_VENDOR=`hal-get-property --udi "$PARENT_UDI" --key "usb.vendor"`

	elif [ "$PARENT_SUBSYSTEM" == "pci" ]; then
		# PCI audio device
		DEV_VENDOR=`hal-get-property --udi "$PARENT_UDI" --key "pci.vendor"`

	else
		# Unknown type audio device
		DEV_VENDOR=`hal-get-property --udi "$PARENT_UDI" --key "info.vendor"`
	fi

	if [ "$DEV_VENDOR" == "" ]; then
		DEV_VENDOR="Unkown"
	else # remove blank
		DEV_VENDOR=`echo $DEV_VENDOR | tr "[[:blank:]]" "_"`
	fi
	
	# Find device sys path
	DEV_PATH=`hal-get-property --udi "$PARENT_UDI" --key "linux.sysfs_path"`

	#
	# Check if there is a manual configuration for the current device.
   	# If there is no previouse manual configuration, follow the screen number of a keyboard
	# on the same USB path

	PRE_ASSIGNED_SCREEN=`grep -m 1 $DEV_PATH $CONFIG_FILE.old | awk '{print $3}'`
	PRE_ASSIGNED_FLAG=`grep -m 1 $DEV_PATH $CONFIG_FILE.old | awk '{print $4}'`

	#echo "DEV_PATH=$DEV_PATH, assinged-screen=$PRE_ASSIGNED_SCREEN, flag=$PRE_ASSIGNED_FLAG" >> $logfile

	if [ "$PRE_ASSIGNED_SCREEN" != "" -a "$PRE_ASSIGNED_FLAG" == "W" ] ; then
   		# If the current audio device was configured manually before, follow the old configuration.
		echo "`date +"%b %d %T"` $ProgramName: Found $DEV_VENDOR($PARENT_SUBSYSTEM) on $CONFIG_FILE."  >> $logfile
		ASSIGNED_SCREEN="$PRE_ASSIGNED_SCREEN"
		ASSIGNED_FLAG="W"

		if [ "${ASSIGNED_SCREEN:0:4}" == "Srn:" ]; then
			SCREEN_NUM=`echo $ASSIGNED_SCREEN | awk -F':' '{print $2}' | awk -F'.' '{print $1}'`
		else
			SCREEN_NUM="-1"
		fi

	else
        if [ "$PARENT_SUBSYSTEM" == "sound" ]; then
            TMP_UDI=`hal-get-property --udi "$PARENT_UDI" --key "info.parent"`
            TMP_SUBSYSTEM=`hal-get-property --udi "$TMP_UDI" --key "linux.subsystem"`
        fi

		if [ "$PARENT_SUBSYSTEM" == "usb" -o "$TMP_SUBSYSTEM" == "usb" ]; then
			# Find screen number base on the current audio device path.
			SCREEN_NUM=`/opt/userful/bin/get-screen-num-from-devpath -T 12 -P $DEV_PATH 2>/dev/null`

			if [ "$SCREEN_NUM" == "-1" ]; then
				# Couldn't find screen number for the current audio device
				echo "`date +"%b %d %T"` $ProgramName: Fails to auto-assign hw:$ALSA_CARD. Disabled." >> $logfile
				ASSIGNED_SCREEN="Disabled"
			else
				ASSIGNED_SCREEN="Srn:$SCREEN_NUM.0"
			fi

		else # disable internal audio device by default.
			ASSIGNED_SCREEN="Disabled"
		fi

		ASSIGNED_FLAG="R/W"
	fi

	if [ "$ASSIGNED_SCREEN" != "Disabled" ]; then

		# Check if there is any audio device already assigned to the same screen.
		# If then, remove the old device.
		CONFLICT=`grep -c $ASSIGNED_SCREEN $CONFIG_FILE`

   		if [ $CONFLICT -gt 0 ]; then
			echo "`date +"%b %d %T"` $ProgramName: Another audio device already assigned screen $ASSIGNED_SCREEN" >> $logfile
			sed -i -e "/$ASSIGNED_SCREEN/d" $CONFIG_FILE
			#perl -pi -e "s/$ASSIGNED_SCREEN/Disabled/g" $CONFIG_FILE
		fi

		#Deal with popup message when hotplug USB audio device only when station is log in.
   		if [ -f /etc/stations ] && [ -f /etc/startat ]; then
			STARTAT=`cat /etc/startat`
			REAL_STATION_NUM=`expr $SCREEN_NUM + $STARTAT`

           	if [ -e /opt/userful/locks/username$REAL_STATION_NUM ]; then
           		echo 1 > /opt/userful/locks/audio$REAL_STATION_NUM
			fi
		fi
	fi

    ##
    ## Reset PARENT_SUBSYSTEM for the newer linux system
    if [ "$PARENT_SUBSYSTEM" == "sound" ]; then
        PARENT_SUBSYSTEM="$TMP_SUBSYSTEM"
    fi

	# add current device to the config file /etc/usbaudio
	echo "`date +"%b %d %T"` $ProgramName: Assign audio device $DEV_VENDOR(hw:$ALSA_CARD) to $ASSIGNED_SCREEN." >> $logfile
	echo "$DEV_VENDOR($PARENT_SUBSYSTEM) hw:$ALSA_CARD $ASSIGNED_SCREEN $ASSIGNED_FLAG $DEV_PATH" >> $CONFIG_FILE

done

# Assign Disabled devices into the first detected unassinged station.

while read DISABLED_DEVICE 
do
	PRE_ASSIGNED_FLAG=`echo $DISABLED_DEVICE | awk '{print $4}'`
    #echo "$DISABLED_DEVICE, $PRE_ASSIGNED_FLAG" 
    
    if [[ "$PRE_ASSIGNED_FLAG" == "W" ]]; then
        # Disabled by manual configuration, ignore it.
        continue
    fi

    # Find the first station without audio device assigned.
    MAX_STATION=`cat /etc/X11/stations`
    for (( i=0; i < $MAX_STATION; i++ )); do

        if grep "Srn:$i.0" $CONFIG_FILE > /dev/null 2>&1; then
            # An audio device is assigned to Screen $i
            continue

        else
	        ALSA_NAME=`echo $DISABLED_DEVICE | awk '{print $1}'`
	        ALSA_NUMBER=`echo $DISABLED_DEVICE | awk '{print $2}'`
	        ALSA_PATH=`echo $DISABLED_DEVICE | awk '{print $5}'`

            # Remove current device
            sed -i -e "/$ALSA_NUMBER/d" $CONFIG_FILE

            # Add current device with screen information
            echo "$ALSA_NAME $ALSA_NUMBER Srn:$i.0 R/W $ALSA_PATH" >> $CONFIG_FILE
            break
        fi
    done

done < <(grep "Disabled" $CONFIG_FILE)


# Add for support other multiuser devices

#if [ -x /opt/userful/bin/multiuserdevices_init.sh ]; then
#   /opt/userful/bin/multiuserdevices_init.sh
#fi
#***
