/*
Copyright (C) 2002-2003 Victor Luchits

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
#include "client.h"

static cgame_export_t *cge;

static mempool_t *cl_gamemodulepool;
static void *module_handle;

//======================================================================

// CL_GameModule versions of the CM functions passed to the game module
// they only add sv.cms as the first parameter

//======================================================================

static inline int CL_GameModule_CM_NumInlineModels( void ) {
	return CM_NumInlineModels( cl.cms );
}

static inline int CL_GameModule_CM_PointContents( vec3_t p, struct cmodel_s *cmodel ) {
	return CM_PointContents( cl.cms, p, cmodel );
}

static inline int CL_GameModule_CM_TransformedPointContents( vec3_t p, struct cmodel_s *cmodel, vec3_t origin, vec3_t angles ) {
	return CM_TransformedPointContents( cl.cms, p, cmodel, origin, angles );
}

static inline void CL_GameModule_CM_BoxTrace( trace_t *tr, vec3_t start, vec3_t end, vec3_t mins, vec3_t maxs,
	struct cmodel_s *cmodel, int brushmask ) {
	CM_BoxTrace( cl.cms, tr, start, end, mins, maxs, cmodel, brushmask );
}

static inline void CL_GameModule_CM_TransformedBoxTrace( trace_t *tr, vec3_t start, vec3_t end, vec3_t mins, vec3_t maxs,
	struct cmodel_s *cmodel, int brushmask, vec3_t origin, vec3_t angles ) {
	CM_TransformedBoxTrace( cl.cms, tr, start, end, mins, maxs, cmodel, brushmask, origin, angles );
}

static inline struct cmodel_s *CL_GameModule_CM_InlineModel( int num ) {
	return CM_InlineModel( cl.cms, num );
}

static inline void CL_GameModule_CM_InlineModelBounds( struct cmodel_s *cmodel, vec3_t mins, vec3_t maxs ) {
	CM_InlineModelBounds( cl.cms, cmodel, mins, maxs );
}

static inline struct cmodel_s *CL_GameModule_CM_ModelForBBox( vec3_t mins, vec3_t maxs ) {
	return CM_ModelForBBox( cl.cms, mins, maxs );
}

static inline qboolean CL_GameModule_CM_AreasConnected( int area1, int area2 ) {
	return CM_AreasConnected( cl.cms, area1, area2 );
}

static inline void CL_GameModule_CM_SetAreaPortalState( int portalnum, int area, int otherarea, qboolean open ) {
	CM_SetAreaPortalState( cl.cms, portalnum, area, otherarea, open );
}

static inline int CL_GameModule_CM_BoxLeafnums( vec3_t mins, vec3_t maxs, int *list, int listsize, int *topnode ) {
	return CM_BoxLeafnums( cl.cms, mins, maxs, list, listsize, topnode );
}

static inline int CL_GameModule_CM_LeafCluster( int leafnum ) {
	return CM_LeafCluster( cl.cms, leafnum );
}

static inline int CL_GameModule_CM_LeafArea( int leafnum ) {
	return CM_LeafArea( cl.cms, leafnum );
}

static inline char *CL_GameModule_CM_LoadMapMessage( char *name, char *message, int size ) {
	return CM_LoadMapMessage( cl.cms, name, message, size );
}

//======================================================================

/*
===============
CL_GameModule_Error
===============
*/
static void CL_GameModule_Error( char *msg ) {
	Com_Error( ERR_DROP, "%s", msg );
}

/*
===============
CL_GameModule_Print
===============
*/
static void CL_GameModule_Print( char *msg ) {
	Com_Printf( "%s", msg );
}

/*
===============
CL_GameModule_GetConfigString
===============
*/
static void CL_GameModule_GetConfigString( int i, char *str, int size )
{
	if( i < 0 || i >= MAX_CONFIGSTRINGS )
		Com_DPrintf( S_COLOR_RED "CL_GameModule_GetConfigString: i > MAX_CONFIGSTRINGS" );
	if( !str || size <= 0 )
		Com_DPrintf( S_COLOR_RED "CL_GameModule_GetConfigString: NULL string" );
	Q_strncpyz( str, cl.configstrings[i], size );
}

/*
===============
CL_GameModule_NET_GetCurrentState
===============
*/
static void CL_GameModule_NET_GetCurrentState( int *incomingAcknowledged, int *outgoingSequence, int *outgoingSent )
{  
	if( incomingAcknowledged )
		*incomingAcknowledged = cls.ucmdAcknowledged;
	if( outgoingSequence )
		*outgoingSequence = cls.ucmdHead;
	if( outgoingSent )
		*outgoingSent = cls.ucmdSent;
}

/*
===============
CL_GameModule_NET_GetUserCmd
===============
*/
static void CL_GameModule_NET_GetUserCmd( int frame, usercmd_t *cmd ) {
	if( cmd )
		*cmd = cl.cmds[frame & CMD_MASK];
} 

/*
===============
CL_GameModule_NET_GetCurrentUserCmdNum
===============
*/
static int CL_GameModule_NET_GetCurrentUserCmdNum( void ) {
	return cls.ucmdHead;
} 

/*
===============
CL_GameModule_R_RegisterWorldModel
===============
*/
static void CL_GameModule_R_RegisterWorldModel( char *model ) {
	R_RegisterWorldModel( model, cl.cms ? CM_VisData( cl.cms ) : NULL );
}

#ifdef ZEROTHREEAPI
typedef struct
{
	// drops to console a client game error
	void			(*Error)( char *msg );

	// console messages
	void			(*Print)( char *msg );

	// frame stuff
	unsigned int	(*LastFrameTime)( void );
	qboolean		(*FrameSnapWaiting)( void );
	void			(*GetFrameSnap)( frame_t **frame, frame_t **lerpframe );

	// dynvars
	dynvar_t	*(*Dynvar_Create)( const char *name, qboolean console, dynvar_getter_f getter, dynvar_setter_f setter );
	void		(*Dynvar_Destroy)( dynvar_t *dynvar );
	dynvar_t	*(*Dynvar_Lookup)( const char *name );
	const char	*(*Dynvar_GetName)( dynvar_t *dynvar );
	dynvar_get_status_t (*Dynvar_GetValue)( dynvar_t *dynvar,void **value );
	dynvar_set_status_t (*Dynvar_SetValue)( dynvar_t *dynvar,void *value );
	void		(*Dynvar_AddListener)( dynvar_t *dynvar, dynvar_listener_f listener );
	void		(*Dynvar_RemoveListener)( dynvar_t *dynvar, dynvar_listener_f listener );

	// console variable interaction
	cvar_t			*(*Cvar_Get)( const char *name, const char *value, int flags );
	cvar_t			*(*Cvar_Set)( const char *name, const char *value );
	void			(*Cvar_SetValue)( const char *name, float value );
	cvar_t			*(*Cvar_ForceSet)( const char *name, const char *value );	// will return 0 0 if not found
	float			(*Cvar_VariableValue)( const char *name );
	char			*(*Cvar_VariableString)( const char *name );

	void			(*Cmd_TokenizeString)( char *text );
	int				(*Cmd_Argc)( void );
	char			*(*Cmd_Argv)( int arg );
	char			*(*Cmd_Args)( void );	// concatenation of all argv >= 1

	void			(*Cmd_AddCommand)( char *name, void(*cmd)( void ) );
	void			(*Cmd_RemoveCommand)( char *cmd_name );
	void			(*Cmd_ExecuteText)( int exec_when, const char *text );
	void			(*Cmd_Execute)( void );

	// files will be memory mapped read only
	// the returned buffer may be part of a larger pak file,
	// or a discrete file from anywhere in the quake search path
	// a -1 return means the file does not exist
	// NULL can be passed for buf to just determine existance
	int				(*FS_FOpenFile)( const char *filename, int *filenum, int mode );
	int				(*FS_Read)( void *buffer, size_t len, int file );
	int				(*FS_Write)( const void *buffer, size_t len, int file );
	int				(*FS_vPrintf)( int file, const char *format, va_list argptr ); // wsw -- PLX
	int				(*FS_Tell)( int file );
	int				(*FS_Seek)( int file, int offset, int whence );
	int				(*FS_Eof)( int file );
	int				(*FS_Flush)( int file );
	void			(*FS_FCloseFile)( int file );
	int				(*FS_GetFileList)( const char *dir, const char *extension, char *buf, size_t bufsize );
	const char		*(*FS_FirstExtension)( const char *filename, char **extensions, int num_extensions );
	qboolean		(*FS_IsPureFile)( const char *filename );

	// key bindings
	char			*(*Key_GetBindingBuf)( int binding );
	char			*(*Key_KeynumToString)( int keynum );

	void			(*GetConfigString)( int i, char *str, int size );
	unsigned int	(*Milliseconds)( void );
	qboolean		(*DownloadRequest)( const char *filename, qboolean requestpak );

	void			(*NET_GetUserCmd)( int frame, usercmd_t *cmd );
	int				(*NET_GetCurrentUserCmdNum)( void );
	void			(*NET_GetCurrentState)( int *incomingAcknowledged, int *outgoingSequence, int *outgoingSent );

	// refresh system
	void			(*R_UpdateScreen)( void );
	int				(*R_GetClippedFragments)( vec3_t origin, float radius, vec3_t axis[3], int maxfverts, vec3_t *fverts, int maxfragments, fragment_t *fragments );
	void			(*R_ClearScene)( void );
	void			(*R_AddEntityToScene)( entity_t *ent );
	void			(*R_AddLightToScene)( vec3_t org, float intensity, float r, float g, float b, struct shader_s *shader );
	void			(*R_AddPolyToScene)( poly_t *poly );
	void			(*R_AddLightStyleToScene)( int style, float r, float g, float b );
	void			(*R_RenderScene)( refdef_t *fd );
	void			(*R_RegisterWorldModel)( char *name );
	void			(*R_ModelBounds)( struct model_s *mod, vec3_t mins, vec3_t maxs );
	struct model_s		*(*R_RegisterModel)( char *name );
	struct shader_s 	*(*R_RegisterPic)( char *name );
	struct shader_s 	*(*R_RegisterSkin)( char *name );
	struct skinfile_s 	*(*R_RegisterSkinFile)( char *name );
	qboolean		(*R_LerpTag)( orientation_t *orient, struct model_s *mod, int oldframe, int frame, float lerpfrac, const char *name );
	void			(*R_SetCustomColor)( int num, int r, int g, int b );
	void			(*R_LightForOrigin)( vec3_t origin, vec3_t dir, vec4_t ambient, vec4_t diffuse, float radius );
	void			(*R_DrawStretchPic)( int x, int y, int w, int h, float s1, float t1, float s2, float t2, vec4_t color, struct shader_s *shader );
	void			(*R_TransformVectorToScreen)( refdef_t *rd, vec3_t in, vec2_t out );
	int				(*R_SkeletalGetNumBones)( struct model_s *mod, int *numFrames );
	int				(*R_SkeletalGetBoneInfo)( struct model_s *mod, int bone, char *name, int size, int *flags );
	void			(*R_SkeletalGetBonePose)( struct model_s *mod, int bone, int frame, bonepose_t *bonepose );

	// collision detection
	int				(*CM_NumInlineModels)( void );
	struct cmodel_s	*(*CM_InlineModel)( int num );
	struct cmodel_s	*(*CM_ModelForBBox)( vec3_t mins, vec3_t maxs );
	void			(*CM_BoxTrace)( trace_t *tr, vec3_t start, vec3_t end, vec3_t mins, vec3_t maxs, struct cmodel_s *cmodel, int brushmask );
	void			(*CM_TransformedBoxTrace)( trace_t *tr, vec3_t start, vec3_t end, vec3_t mins, vec3_t maxs, struct cmodel_s *cmodel, int brushmask, vec3_t origin, vec3_t angles );
	int				(*CM_PointContents)( vec3_t p, struct cmodel_s *cmodel );
	int				(*CM_TransformedPointContents)( vec3_t p, struct cmodel_s *cmodel, vec3_t origin, vec3_t angles );
	void			(*CM_InlineModelBounds)( struct cmodel_s *cmodel, vec3_t mins, vec3_t maxs );
	char			*(*CM_LoadMapMessage)( char *name, char *message, int size );

	// sound system
	struct sfx_s	*(*S_RegisterSound)( const char *name );
	void			(*S_StartFixedSound)( struct sfx_s *sfx, const vec3_t origin, int entchannel, float fvol, float attenuation );
	void			(*S_StartRelativeSound)( struct sfx_s *sfx, int entnum, int entchannel, float fvol, float attenuation );
	void			(*S_StartGlobalSound)( struct sfx_s *sfx, int entchannel, float fvol );
	void			(*S_Update)( const vec3_t origin, const vec3_t velocity, const vec3_t v_forward, const vec3_t v_right, const vec3_t v_up );
	void			(*S_AddLoopSound)( struct sfx_s *sfx, int entnum, float fvol, float attenuation );
	void			(*S_StartBackgroundTrack)( const char *intro, const char *loop );
	void			(*S_StopBackgroundTrack)( void );

	// fonts
	struct mufont_s *(*SCR_RegisterFont)( char *name );
	void			(*SCR_DrawString)( int x, int y, int align, const char *str, struct mufont_s *font, vec4_t color );
	int				(*SCR_DrawStringWidth)( int x, int y, int align, const char *str, int maxwidth, struct mufont_s *font, vec4_t color );
	size_t			(*SCR_strHeight)( struct mufont_s *font );
	size_t			(*SCR_strWidth)( const char *str, struct mufont_s *font, int maxlen );
	size_t			(*SCR_StrlenForWidth)( const char *str, struct mufont_s *font, size_t maxwidth );

	// managed memory allocation
	struct mempool_s *(*Mem_AllocPool)( const char *name, const char *filename, int fileline );
	void			*(*Mem_Alloc)( struct mempool_s *pool, size_t size, const char *filename, int fileline );
	void			(*Mem_Free)( void *data, const char *filename, int fileline );
	void			(*Mem_FreePool)( struct mempool_s **pool, const char *filename, int fileline );
	void			(*Mem_EmptyPool)( struct mempool_s *pool, const char *filename, int fileline );
} cgame_import03_t;

/*
===============
CL_GameModule_LastFrameTime
===============
*/
static unsigned int CL_GameModule_LastFrameTime( void )
{
	if( cl.receivedSnapNum <= 0 || !cl.frames[cl.receivedSnapNum&UPDATE_MASK].valid )
		return 0;

	return cl.frames[cl.receivedSnapNum&UPDATE_MASK].serverTime;
}

/*
===============
CL_GameModule_FrameSnapWaiting
===============
*/
static qboolean CL_GameModule_FrameSnapWaiting( void )
{
	return (cl.receivedSnapNum > 0 && cl.frames[cl.receivedSnapNum&UPDATE_MASK].valid && cl.currentSnapNum != cl.receivedSnapNum);
}

/*
===============
CL_GameModule_GetFrameSnap
===============
*/
static void CL_GameModule_GetFrameSnap( frame_t **frame, frame_t **lerpframe )
{
	int start, i;

	assert( frame );
	assert( lerpframe );

	*frame = NULL;
	*lerpframe = NULL;

	/*if( !cl.lastFrame || !cl.lastFrame->valid )
		return;*/

	if( cl.receivedSnapNum <= 0 || !cl.frames[cl.receivedSnapNum&UPDATE_MASK].valid )
		return;

	/*if( cl.curFrame && cl.curFrame->valid ) {
		start = cl.curFrame->serverFrame + 1;
		if( start < cl.lastFrame->serverFrame - UPDATE_BACKUP + 1 )
			start = cl.lastFrame->serverFrame - UPDATE_BACKUP + 1;
	} else {
		start = cl.lastFrame->serverFrame - UPDATE_BACKUP + 1;
	}*/
	if( cl.currentSnapNum > 0 && cl.frames[cl.currentSnapNum&UPDATE_MASK].valid ) {
		start = cl.currentSnapNum + 1;
		if( start < cl.receivedSnapNum - UPDATE_BACKUP + 1 )
			start = cl.receivedSnapNum - UPDATE_BACKUP + 1;
	} else {
		start = cl.receivedSnapNum - UPDATE_BACKUP + 1;
	}

	// find the frame
	for( i = start; i <= cl.receivedSnapNum; i++ )
	{
		if( cl.frames[i & UPDATE_MASK].serverFrame == i && cl.frames[i & UPDATE_MASK].valid ) {
			//cl.curFrame = *frame = &cl.frames[i & UPDATE_MASK];
			*frame = &cl.frames[i & UPDATE_MASK];
			cl.currentSnapNum = i;
			break;
		}
	}
	assert( i <= cl.receivedSnapNum );

	// find closest frame in front of it, for lerping
	for( i = cl.currentSnapNum - 1; i > cl.currentSnapNum - UPDATE_BACKUP; i-- )
	{
		if( cl.frames[i & UPDATE_MASK].serverFrame == i && cl.frames[i & UPDATE_MASK].valid ) {
			*lerpframe = &cl.frames[i & UPDATE_MASK];
			break;
		}
	}
}

int CL_GameModule_FS_GetFileList_03( const char *dir, const char *extension, char *buf, size_t bufsize ) {
	return FS_GetFileList( dir, extension, buf, bufsize, 0, 0 );
}
#endif

/*
===============
CL_GameModule_MemAlloc
===============
*/
static void *CL_GameModule_MemAlloc( mempool_t *pool, size_t size, const char *filename, int fileline ) {
	return _Mem_Alloc( pool, size, MEMPOOL_CLIENTGAME, 0, filename, fileline );
}

/*
===============
CL_GameModule_MemFree
===============
*/
static void CL_GameModule_MemFree( void *data, const char *filename, int fileline ) {
	_Mem_Free( data, MEMPOOL_CLIENTGAME, 0, filename, fileline );
}

/*
===============
CL_GameModule_MemAllocPool
===============
*/
static mempool_t *CL_GameModule_MemAllocPool( const char *name, const char *filename, int fileline ) {
	return _Mem_AllocPool( cl_gamemodulepool, name, MEMPOOL_CLIENTGAME, filename, fileline );
}

/*
===============
CL_GameModule_MemFreePool
===============
*/
static void CL_GameModule_MemFreePool( mempool_t **pool, const char *filename, int fileline ) {
	_Mem_FreePool( pool, MEMPOOL_CLIENTGAME, 0, filename, fileline );
}

/*
===============
CL_GameModule_MemEmptyPool
===============
*/
static void CL_GameModule_MemEmptyPool( mempool_t *pool, const char *filename, int fileline ) {
	_Mem_EmptyPool( pool, MEMPOOL_CLIENTGAME, 0, filename, fileline );
}

/*
===============
CL_GameModule_InitImportStruct
===============
*/
#define CL_GameModule_InitImportStruct(import) \
( \
	import.Error = CL_GameModule_Error, \
	import.Print = CL_GameModule_Print, \
/* 0.3 compat stuff */ \
	import.LastFrameTime = CL_GameModule_LastFrameTime, \
	import.FrameSnapWaiting = CL_GameModule_FrameSnapWaiting, \
	import.GetFrameSnap = CL_GameModule_GetFrameSnap, \
/* dynvars */ \
	import.Dynvar_Create = Dynvar_Create, \
	import.Dynvar_Destroy = Dynvar_Destroy, \
	import.Dynvar_Lookup = Dynvar_Lookup, \
	import.Dynvar_GetName = Dynvar_GetName, \
	import.Dynvar_GetValue = Dynvar_GetValue, \
	import.Dynvar_SetValue = Dynvar_SetValue, \
	import.Dynvar_AddListener = Dynvar_AddListener, \
	import.Dynvar_RemoveListener = Dynvar_RemoveListener, \
 \
	import.Cvar_Get = Cvar_Get, \
	import.Cvar_Set = Cvar_Set, \
	import.Cvar_SetValue = Cvar_SetValue, \
	import.Cvar_ForceSet = Cvar_ForceSet, \
	import.Cvar_VariableString = Cvar_VariableString, \
	import.Cvar_VariableValue = Cvar_VariableValue, \
 \
	import.Cmd_TokenizeString = Cmd_TokenizeString, \
	import.Cmd_Argc = Cmd_Argc, \
	import.Cmd_Argv = Cmd_Argv, \
	import.Cmd_Args = Cmd_Args, \
 \
	import.Cmd_AddCommand = Cmd_AddCommand, \
	import.Cmd_RemoveCommand = Cmd_RemoveCommand, \
	import.Cmd_ExecuteText = Cbuf_ExecuteText, \
	import.Cmd_Execute = Cbuf_Execute, \
 \
	import.FS_FOpenFile = FS_FOpenFile, \
	import.FS_Read = FS_Read, \
	import.FS_Write = FS_Write, \
	import.FS_vPrintf = FS_vPrintf, \
	import.FS_Tell = FS_Tell, \
	import.FS_Seek = FS_Seek, \
	import.FS_Eof = FS_Eof, \
	import.FS_Flush = FS_Flush, \
	import.FS_FCloseFile = FS_FCloseFile, \
/*	import.FS_GetFileList = FS_GetFileList,*/ \
	import.FS_FirstExtension = FS_FirstExtension, \
	import.FS_IsPureFile = FS_IsPureFile, \
 \
	import.Key_GetBindingBuf = Key_GetBindingBuf, \
	import.Key_KeynumToString = Key_KeynumToString, \
 \
	import.GetConfigString = CL_GameModule_GetConfigString, \
	import.Milliseconds = Sys_Milliseconds, \
	import.DownloadRequest = CL_DownloadRequest, \
 \
	import.NET_GetUserCmd = CL_GameModule_NET_GetUserCmd, \
	import.NET_GetCurrentUserCmdNum = CL_GameModule_NET_GetCurrentUserCmdNum, \
	import.NET_GetCurrentState = CL_GameModule_NET_GetCurrentState, \
 \
	import.R_UpdateScreen = SCR_UpdateScreen, \
	import.R_GetClippedFragments = R_GetClippedFragments, \
	import.R_ClearScene = R_ClearScene, \
	import.R_AddEntityToScene = R_AddEntityToScene, \
	import.R_AddLightToScene = R_AddLightToScene, \
	import.R_AddPolyToScene = R_AddPolyToScene, \
	import.R_AddLightStyleToScene = R_AddLightStyleToScene, \
	import.R_RenderScene = R_RenderScene, \
	import.R_RegisterWorldModel = CL_GameModule_R_RegisterWorldModel, \
	import.R_ModelBounds = R_ModelBounds, \
	import.R_RegisterModel = R_RegisterModel, \
	import.R_RegisterPic = R_RegisterPic, \
	import.R_RegisterSkin = R_RegisterSkin, \
	import.R_RegisterSkinFile = R_RegisterSkinFile, \
	import.R_LerpTag = R_LerpTag, \
	import.R_LightForOrigin = R_LightForOrigin, \
	import.R_SetCustomColor = R_SetCustomColor, \
	import.R_DrawStretchPic = R_DrawStretchPic, \
	import.R_TransformVectorToScreen = R_TransformVectorToScreen, \
	import.R_SkeletalGetNumBones = R_SkeletalGetNumBones, \
	import.R_SkeletalGetBoneInfo = R_SkeletalGetBoneInfo, \
	import.R_SkeletalGetBonePose = R_SkeletalGetBonePose, \
 \
	import.CM_NumInlineModels = CL_GameModule_CM_NumInlineModels, \
	import.CM_InlineModel = CL_GameModule_CM_InlineModel, \
	import.CM_BoxTrace = CL_GameModule_CM_BoxTrace, \
	import.CM_TransformedBoxTrace = CL_GameModule_CM_TransformedBoxTrace, \
	import.CM_PointContents = CL_GameModule_CM_PointContents, \
	import.CM_TransformedPointContents = CL_GameModule_CM_TransformedPointContents, \
	import.CM_ModelForBBox = CL_GameModule_CM_ModelForBBox, \
	import.CM_InlineModelBounds = CL_GameModule_CM_InlineModelBounds, \
	import.CM_LoadMapMessage = CL_GameModule_CM_LoadMapMessage, \
 \
	import.S_RegisterSound = CL_SoundModule_RegisterSound, \
	import.S_StartFixedSound = CL_SoundModule_StartFixedSound, \
	import.S_StartRelativeSound = CL_SoundModule_StartRelativeSound, \
	import.S_StartGlobalSound = CL_SoundModule_StartGlobalSound, \
	import.S_Update = CL_SoundModule_Update, \
	import.S_AddLoopSound = CL_SoundModule_AddLoopSound, \
	import.S_StartBackgroundTrack = CL_SoundModule_StartBackgroundTrack, \
	import.S_StopBackgroundTrack = CL_SoundModule_StopBackgroundTrack, \
 \
	import.SCR_RegisterFont = SCR_RegisterFont, \
	import.SCR_DrawString = SCR_DrawString, \
	import.SCR_DrawStringWidth = SCR_DrawStringWidth, \
	import.SCR_strHeight = SCR_strHeight, \
	import.SCR_strWidth = SCR_strWidth, \
	import.SCR_StrlenForWidth = SCR_StrlenForWidth, \
\
	import.Mem_Alloc = CL_GameModule_MemAlloc, \
	import.Mem_Free = CL_GameModule_MemFree, \
	import.Mem_AllocPool = CL_GameModule_MemAllocPool, \
	import.Mem_FreePool = CL_GameModule_MemFreePool, \
	import.Mem_EmptyPool = CL_GameModule_MemEmptyPool \
)

/*
===============
CL_GameModule_Init
===============
*/
void CL_GameModule_Init( void )
{
	int apiversion, oldState;
	unsigned int start;
	cgame_import_t import;

	// unload anything we have now
	CL_SoundModule_StopAllSounds();
	CL_GameModule_Shutdown();

	cl_gamemodulepool = Mem_AllocPool( NULL, "Client Game Progs" );

	CL_GameModule_InitImportStruct( import );
	import.FS_GetFileList = FS_GetFileList;

#ifdef CGAME_HARD_LINKED
	{
		EXTERN_API_FUNC void *GetCGameAPI( void * );
		cge = (cgame_export_t *)GetCGameAPI( &import );
	}
#else
	cge = (cgame_export_t *)Com_LoadGameLibrary( "cgame", "GetCGameAPI", &module_handle, &import, cls.sv_pure );
#endif
	if( !cge )
		Com_Error( ERR_DROP, "Failed to load client game DLL" );

	apiversion = cge->API();
	if( apiversion != CGAME_API_VERSION ) {
#ifdef ZEROTHREEAPI
		if( apiversion == 32 ) {
			cgame_import03_t import03;

			Com_UnloadGameLibrary( &module_handle );

			CL_GameModule_InitImportStruct( import03 );
			import03.FS_GetFileList = CL_GameModule_FS_GetFileList_03;

			cge = (cgame_export_t *)Com_LoadGameLibrary( "cgame", "GetCGameAPI", &module_handle, &import03, cls.sv_pure );
			if( !cge )
				Com_Error( ERR_DROP, "Failed to load client game DLL" );
			goto load;
		}
#endif

#ifndef CGAME_HARD_LINKED
		Com_UnloadGameLibrary( &module_handle );
#endif
		Mem_FreePool( &cl_gamemodulepool );
		cge = NULL;
		Com_Error( ERR_DROP, "Client game is version %i, not %i", apiversion, CGAME_API_VERSION );
	}

#ifdef ZEROTHREEAPI
load:
#endif

	oldState = cls.state;
	cls.state = CA_LOADING;

	start = Sys_Milliseconds ();
#ifdef ZEROTHREEAPI
	cls.cgameAPIversion = apiversion;
	cge->Init( cl.playernum, viddef.width, viddef.height, cls.demoplaying, cls.sv_pure );
#else
	cge->Init( cl.playernum, viddef.width, viddef.height, cls.demoplaying, cls.sv_pure, PROTOCOL_VERSION );
#endif
	Com_DPrintf( "CL_GameModule_Init: %.2f seconds\n", (float)(Sys_Milliseconds () - start) * 0.001f );

	cls.state = oldState;
	cls.cgameActive = qtrue;

	// check memory integrity
	Mem_CheckSentinelsGlobal ();

	CL_SoundModule_SoundsInMemory ();

	Sys_SendKeyEvents ();	// pump message loop
}

/*
===============
CL_GameModule_Shutdown
===============
*/
void CL_GameModule_Shutdown (void)
{
	if ( !cge )
		return;

	cls.cgameActive = qfalse;

	cge->Shutdown ();
#ifndef CGAME_HARD_LINKED
	Com_UnloadGameLibrary( &module_handle );
#endif
	Mem_FreePool( &cl_gamemodulepool );
	cge = NULL;
}

/*
==============
CL_GameModule_InGameMenu
==============
*/
void CL_GameModule_InGameMenu( void )
{
	if( cge )
		cge->InGameMenu();
}

/*
==============
CL_GameModule_GetEntitySoundOrigin
==============
*/
void CL_GameModule_GetEntitySpatilization( int entNum, vec3_t origin, vec3_t velocity )
{
	if( cge )
		cge->GetEntitySpatilization( entNum, origin, velocity );
}

/*
===============
CL_GameModule_ConfigString
===============
*/
void CL_GameModule_ConfigString( int number, char *value )
{
	if( cge )
		cge->ConfigString( number, value );
}

/*
===============
CL_GameModule_SetSensitivityScale
===============
*/
float CL_GameModule_SetSensitivityScale( const float sens )
{
	if( cge )
		return cge->SetSensitivityScale( sens );
	else
		return 1.0f;
}

/*
===============
CL_GameModule_NewFrameSnap
===============
*/
qboolean CL_GameModule_NewSnapshot( int pendingSnapshot ) {
	frame_t	*currentSnap, *newSnap;
	if( cge ) {
		currentSnap = ( cl.currentSnapNum <= 0 ) ? NULL : &cl.frames[cl.currentSnapNum & UPDATE_MASK];
		newSnap = &cl.frames[pendingSnapshot & UPDATE_MASK];
		cge->NewFrameSnapshot( newSnap, currentSnap );
		return qtrue;
	}
	return qfalse;
}

/*
===============
CL_GameModule_RenderView
===============
*/
void CL_GameModule_RenderView( float stereo_separation )
{
	if( cge ) {
#ifdef ZEROTHREEAPI
		cl.cgametime = cge->RenderView( cls.frametime, cls.realframetime, cls.realtime, cl.serverTime, stereo_separation );
#else
		cge->RenderView( cls.frametime, cls.realframetime, cls.realtime, cl.serverTime, stereo_separation );
#endif
	}
}

/*
===============
CL_GameModule_Trace
===============
*/
void CL_GameModule_Trace( trace_t *tr, vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end, int passent, int contentmask )
{
	if( tr && cge )
		cge->Trace( tr, start, mins, maxs, end, passent, contentmask );
}
