/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "ui_local.h"

#define REF_OPENGL	0
#define REF_3DFX	1
#define REF_POWERVR	2
#define REF_VERITE	3

static void Video_MenuInit( void );

/*
=======================================================================

VIDEO MENU

=======================================================================
*/

static menuframework_s	s_video_menu;


static void BrightnessCallback( menucommon_t *menuitem )
{
	trap_Cvar_SetValue( "r_gamma", menuitem->curvalue/10.0 );
}

static void ResetDefaults( menucommon_t *menuitem )
{
	Video_MenuInit();
}

static void ExtensionsCallback( menucommon_t *menuitem )
{
	trap_Cmd_ExecuteText( EXEC_APPEND, "menu_glext" );
}

static void ApplyChanges( menucommon_t *unused )
{
	menucommon_t *menuitem;

	menuitem = UI_MenuItemByName( "m_video_r_picmip" );
	trap_Cvar_SetValue( "r_picmip", 6 - menuitem->curvalue );

	menuitem = UI_MenuItemByName( "m_video_r_skymip" );
	trap_Cvar_SetValue( "r_skymip", 3 - menuitem->curvalue );

	menuitem = UI_MenuItemByName( "m_video_r_subdivisions" );
	trap_Cvar_SetValue( "r_subdivisions", (8 - menuitem->curvalue)*4 );

	menuitem = UI_MenuItemByName( "m_video_vid_fullscreen" );
	trap_Cvar_SetValue( "vid_fullscreen", menuitem->curvalue );

	menuitem = UI_MenuItemByName( "m_video_r_mode" );
	trap_Cvar_SetValue( "r_mode", menuitem->curvalue );

	menuitem = UI_MenuItemByName( "m_video_r_colorbits" );
	trap_Cvar_SetValue( "r_colorbits", 16 * menuitem->curvalue );

	menuitem = UI_MenuItemByName( "m_video_r_texturebits" );
	trap_Cvar_SetValue( "r_texturebits", 16 * menuitem->curvalue );

	menuitem = UI_MenuItemByName( "m_video_r_detailtextures" );
	trap_Cvar_SetValue( "r_detailtextures", menuitem->curvalue );

	menuitem = UI_MenuItemByName( "m_video_r_texturemode" );
	trap_Cvar_Set( "r_texturemode", menuitem->curvalue ? "GL_LINEAR_MIPMAP_LINEAR" : "GL_LINEAR_MIPMAP_NEAREST" );

#ifndef UI_GLEXT_MENU
	menuitem = UI_MenuItemByName( "m_video_glext_anisotropic" );
	if( menuitem )
		trap_Cvar_SetValue( "gl_ext_texture_filter_anisotropic", menuitem->curvalue ? (1<<menuitem->curvalue) : 0 );
#endif

#ifdef CGAMEGETLIGHTORIGIN
	menuitem = UI_MenuItemByName( "m_video_cg_shadows" );
	trap_Cvar_SetValue( "cg_shadows", menuitem->curvalue );
	trap_Cvar_SetValue( "r_stencilbits", (menuitem->curvalue > 1)*8 );  //  2 is SHADOWS_STENCIL
#else
	menuitem = UI_MenuItemByName( "m_video_cg_shadows" );
	trap_Cvar_SetValue( "r_shadows", menuitem->curvalue );
	trap_Cvar_SetValue( "r_stencilbits", menuitem->curvalue != 0) * 8 );  //  2 is SHADOWS_STENCIL
#endif
	menuitem = UI_MenuItemByName( "m_video_r_bloom" );
	trap_Cvar_SetValue( "r_bloom", menuitem->curvalue );

	menuitem = UI_MenuItemByName( "m_video_r_swapinterval" );
	trap_Cvar_SetValue( "r_swapinterval", menuitem->curvalue );

	trap_Cmd_ExecuteText( EXEC_APPEND, "vid_restart\n" );
	trap_Cmd_Execute();
}

/*
** Video_MenuInit
*/
static void Video_MenuInit( void )
{
	menucommon_t *menuitem;
	char *gl_driver = trap_Cvar_VariableString( "gl_driver" );
	int yoffset = 0;
#ifndef UI_GLEXT_MENU
	int i, logtwo;
#endif
	char custom_resolution[64];
	static char **resolutions;
	static char *refs[] =
	{
		"[default OpenGL]",
		"[3Dfx OpenGL   ]",
		"[PowerVR OpenGL]",
//		"[Rendition OpenGL]",
		0
	};
	static char *filter_names[] =		{ "bilinear", "trilinear", 0			};
	static char *colordepth_names[] =	{ "desktop", "16 bits", "32 bits", 0	};
	static char *texturebits_names[] =	{ "default", "16", "32", 0				};
#ifdef CGAMEGETLIGHTORIGIN
	static char *shadows_names[] =		{ "no", "simple", "stencil", 0			};
#endif

	qboolean wideScreen;

	if( !resolutions ) { // count video modes
		int i, width, height;

		for( i = 0; trap_VID_GetModeInfo( &width, &height, &wideScreen, i ); i++ );

		resolutions = UI_Malloc( sizeof( char * ) * (i + 1) );

		for( i = 0; trap_VID_GetModeInfo( &width, &height, &wideScreen, i ); i++ )
		{
			Q_snprintfz( custom_resolution, sizeof( custom_resolution ), "%s%i x %i", (wideScreen ? "W " : ""), width, height );
			resolutions[i] = UI_CopyString( custom_resolution );
		}
		resolutions[i] = NULL;
	}

	s_video_menu.nitems = 0;

	// title
	menuitem = UI_InitMenuItem( "m_video_title1", "VIDEO OPTIONS", 0, yoffset, MTYPE_SEPARATOR, ALIGN_CENTER_TOP, uis.fontSystemBig, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	yoffset += trap_SCR_strHeight( menuitem->font );

	yoffset += trap_SCR_strHeight( menuitem->font );

#ifdef UI_GLEXT_MENU
	menuitem = UI_InitMenuItem( "m_video_glext", "OpenGL Extensions", 0, yoffset, MTYPE_ACTION, ALIGN_CENTER_TOP, uis.fontSystemMedium, ExtensionsCallback );
	Menu_AddItem( &s_video_menu, menuitem );
	yoffset += trap_SCR_strHeight( menuitem->font );

	yoffset += trap_SCR_strHeight( menuitem->font );
#endif

	menuitem = UI_InitMenuItem( "m_video_gl_driver", "driver", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	if( !Q_stricmp( gl_driver, "3dfxgl" ) )
		UI_SetupSpinControl( menuitem, refs, REF_3DFX );
	else if( !Q_stricmp( gl_driver, "pvrgl" ) )
		UI_SetupSpinControl( menuitem, refs, REF_POWERVR );
	else if( !Q_stricmp( gl_driver, "opengl32" ) )
		UI_SetupSpinControl( menuitem, refs, REF_OPENGL );
	else
		UI_SetupSpinControl( menuitem, refs, REF_OPENGL );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_mode", "resolution", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSpinControl( menuitem, resolutions, trap_Cvar_VariableValue( "r_mode" ) );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_gamma", "brightness", 0, yoffset, MTYPE_SLIDER, ALIGN_RIGHT_TOP, uis.fontSystemSmall, BrightnessCallback );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSlider( menuitem, 12, bound((int)(trap_Cvar_VariableValue("r_gamma") * 10.0f), 5, 13), 5, 13 );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_vid_fullscreen", "fullscreen", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSpinControl( menuitem, noyes_names, trap_Cvar_VariableValue( "vid_fullscreen" ) != 0 );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_colorbits", "color depth", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	if( !Q_stricmp( trap_Cvar_VariableString( "r_colorbits" ), "16" ) )
		UI_SetupSpinControl( menuitem, colordepth_names, 1 );
	else if( !Q_stricmp( trap_Cvar_VariableString( "r_colorbits" ), "32" ) )
		UI_SetupSpinControl( menuitem, colordepth_names, 2 );
	else
		UI_SetupSpinControl( menuitem, colordepth_names, 0 );
	yoffset += trap_SCR_strHeight( menuitem->font );
	
	menuitem = UI_InitMenuItem( "m_video_r_texturebits", "texture bits", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	if( !Q_stricmp( trap_Cvar_VariableString( "r_texturebits" ), "16" ) )
		UI_SetupSpinControl( menuitem, texturebits_names, 1 );
	else if( !Q_stricmp( trap_Cvar_VariableString( "r_texturebits" ), "32" ) )
		UI_SetupSpinControl( menuitem, texturebits_names, 2 );
	else
		UI_SetupSpinControl( menuitem, texturebits_names, 0 );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_detailtextures", "detail textures", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSpinControl( menuitem, offon_names, trap_Cvar_VariableValue( "r_detailtextures" ) != 0 );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_picmip", "texture quality", 0, yoffset, MTYPE_SLIDER, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSlider( menuitem, 12, 6-trap_Cvar_VariableValue( "r_picmip" ), 0, 6 );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_skymip", "sky quality", 0, yoffset, MTYPE_SLIDER, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSlider( menuitem, 12, 3-trap_Cvar_VariableValue( "r_skymip" ), 0, 3 );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_subdivisions", "curves geometric detail", 0, yoffset, MTYPE_SLIDER, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSlider( menuitem, 12, (8-(trap_Cvar_VariableValue( "r_subdivisions" )/4)), 0, 7 );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_texturemode", "texture filter", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	if( !Q_stricmp( trap_Cvar_VariableString( "r_texturemode" ), "GL_LINEAR_MIPMAP_NEAREST" ) )
		UI_SetupSpinControl( menuitem, filter_names, 0 );
	else
		UI_SetupSpinControl( menuitem, filter_names, 1 );
	yoffset += trap_SCR_strHeight( menuitem->font );

#ifndef UI_GLEXT_MENU
	for( i = trap_Cvar_VariableValue( "gl_ext_max_texture_filter_anisotropic" ), logtwo = 0; (i/=2) > 0; logtwo++ );
	if( logtwo ) {
		int maxfilter = logtwo;
		for( i = trap_Cvar_VariableValue( "gl_ext_texture_filter_anisotropic" ), logtwo = 0; (i/=2) > 0; logtwo++ );
		menuitem = UI_InitMenuItem( "m_video_glext_anisotropic", "texture anisotropic filter", 0, yoffset, MTYPE_SLIDER, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
		UI_SetupSlider( menuitem, 12, logtwo, 0, maxfilter );
		Menu_AddItem( &s_video_menu, menuitem );
		yoffset += trap_SCR_strHeight( menuitem->font );
	}
#endif

	menuitem = UI_InitMenuItem( "m_video_cg_shadows", "shadows", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
#ifdef CGAMEGETLIGHTORIGIN
	UI_SetupSpinControl( menuitem, shadows_names, trap_Cvar_VariableValue( "cg_shadows" ) );
#else
	UI_SetupSpinControl( menuitem, noyes_names, trap_Cvar_VariableValue( "r_shadows" ) );
#endif
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_bloom", "light bloom", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSpinControl( menuitem, noyes_names, trap_Cvar_VariableValue( "r_bloom" ) );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_r_swapinterval", "vertical sync", 0, yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, NULL );
	Menu_AddItem( &s_video_menu, menuitem );
	UI_SetupSpinControl( menuitem, noyes_names, trap_Cvar_VariableValue( "r_swapinterval" ) );
	yoffset += trap_SCR_strHeight( menuitem->font );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_reset", "reset to defaults", 0, yoffset, MTYPE_ACTION, ALIGN_CENTER_TOP, uis.fontSystemSmall, ResetDefaults );
	Menu_AddItem( &s_video_menu, menuitem );
	yoffset += trap_SCR_strHeight( menuitem->font );

	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_video_apply", "apply", 16, yoffset, MTYPE_ACTION, ALIGN_LEFT_TOP, uis.fontSystemBig, ApplyChanges );
	Menu_AddItem( &s_video_menu, menuitem );

	menuitem = UI_InitMenuItem( "m_video_back", "back", -16, yoffset, MTYPE_ACTION, ALIGN_RIGHT_TOP, uis.fontSystemBig, M_genericBackFunc );
	Menu_AddItem( &s_video_menu, menuitem );
	yoffset += trap_SCR_strHeight( menuitem->font );

	Menu_Center( &s_video_menu );
	Menu_Init( &s_video_menu );
}

/*
================
Video_MenuDraw
================
*/
static void Video_MenuDraw( void )
{
	/*
	** move cursor to a reasonable starting position
	*/
	Menu_AdjustCursor( &s_video_menu, 1 );

	/*
	** draw the menu
	*/
	Menu_Draw( &s_video_menu );
}

/*
================
Video_MenuKey
================
*/
static const char *Video_MenuKey( int key )
{
	return Default_MenuKey( &s_video_menu, key );
}

/*
================
Video_MenuCharEvent
================
*/
static const char *Video_MenuCharEvent( int key )
{
	return Default_MenuCharEvent( &s_video_menu, key );
}

void M_Menu_Video_f( void )
{
	Video_MenuInit();
	M_PushMenu( &s_video_menu, Video_MenuDraw, Video_MenuKey, Video_MenuCharEvent );
}
