/***************************************************************************

NEC MC-8123 encryption emulation

Decryption tables provided by David Widel
Decryption algorithm by Nicola Salmoria

The NEC MC-8123 is a Z80 with built-in encryption.
It contains some battery-backed RAM, when the battery dies the CPU stops
working.

It seems that there are only 0x1a0 possible ways to encrypt a byte (a strange
value...). 0xd0 are used for data, 0xd0 for opcodes. It is also possible not
to encrypt bytes, this is done for example by Altered Beast for data above 8000.
The internal RAM only contains the code of the table to use at every address
(only 12 bits of the address are used, so the encryption repeats).

Known games that use this CPU are:

CPU #  Status   Game                     Notes
-------- --- ------------------------ ------------------
317-5012     Ganbare Chinsan Ooshoubu  NEC MC-8123A
317-0014     DakkoChan Jansoh
317-0029     Block Gal                 NEC MC-8123B 651
317-0030     Perfect Billiard
317-0042     Opa Opa
317-0043 [3] Wonder Boy Monster Land
317-0054 [2] Shinobi (sound CPU)       NEC MC-8123B 652
317-0057     Fantasy Zone 2
317-0064     Ufo Senshi Yohko Chan
317-0066 [1] Altered Beast (sound CPU)
???      [3] Ninja Kid II (sound CPU)

[1] Complete, but some uncertain opcode tables above 8000
[2] Data complete for address 0000-7FFF, opcodes partial
[3] Tables derived by comparison with bootleg version

***************************************************************************/

#include "driver.h"


static UINT8 key_0043[0x2000] =
{
	/* data */
	0x24,0xa5,0x78,0x04,0x01,0xcc,0x10,0x06,0x48,0xc3,0x0c,0x70,0xce,0xb2,0x82,0x4b,
	0x6f,0x98,0xca,0x03,0x7c,0x45,0x06,0x75,0x14,0x1a,0x9b,0x39,0xb7,0xcc,0x54,0x66,
	0x9f,0x43,0x2e,0x71,0x86,0x86,0x1b,0x5e,0xc7,0x22,0xa5,0x03,0xa6,0x52,0x58,0x02,
	0xc0,0x65,0x63,0x60,0x22,0x16,0x45,0x52,0x3e,0xb3,0x73,0x74,0x2a,0x2e,0xcc,0x29,
	0x54,0x0d,0x19,0x05,0x65,0xab,0x6e,0xc0,0x1e,0x65,0xb2,0x75,0x77,0xc1,0x97,0x81,
	0xa6,0x60,0x5e,0x33,0xc5,0x10,0x78,0x2f,0xce,0x10,0xa4,0x15,0x67,0x37,0x3a,0x41,
	0xce,0xb5,0x6c,0x8d,0x75,0x11,0xbd,0xb5,0x23,0x5a,0x39,0x31,0x38,0x69,0x8a,0xca,
	0x99,0x48,0x33,0xbf,0x32,0x0e,0x81,0x0a,0x04,0x1d,0xc2,0x2d,0x84,0x63,0xcb,0x10,
	0x15,0xb6,0xb3,0x16,0x94,0x17,0xa5,0xb3,0x1f,0x10,0x30,0x53,0x9d,0xa5,0xbf,0xa4,
	0x36,0x17,0xca,0x94,0x86,0x86,0x9c,0x2f,0x6c,0x78,0x22,0x09,0x60,0x7a,0x84,0x07,
	0x17,0x1f,0xb1,0x1c,0x02,0x60,0x54,0xc4,0x50,0x42,0xab,0x8b,0x2f,0x11,0x55,0x6b,
	0x9a,0xcd,0x0c,0x7a,0xc8,0x41,0x54,0x52,0x01,0xb2,0x0a,0x99,0xab,0xc0,0xa0,0x90,
	0x0f,0xa9,0xce,0x28,0x29,0x0f,0x68,0x56,0xc0,0x47,0x6c,0xb9,0xcd,0x7d,0xcb,0x28,
	0x10,0x1a,0xbb,0x6c,0x7d,0xb0,0x29,0x98,0x33,0x11,0x64,0x35,0x5b,0x78,0x64,0xb6,
	0x29,0xcb,0x5e,0x1f,0x67,0x19,0xa4,0x2c,0x1f,0x43,0x86,0x42,0xb3,0x7c,0xc8,0xb3,
	0x9e,0x13,0x86,0x6b,0xc6,0x8d,0x64,0x54,0x24,0x1c,0x4e,0x04,0x29,0x17,0x42,0x02,
	0x8a,0x33,0x02,0x92,0x90,0x6f,0x84,0x74,0x79,0x76,0xcb,0x2e,0xc5,0x32,0x43,0x07,
	0x6a,0x40,0xc4,0x1c,0x63,0x95,0x05,0x88,0x07,0x79,0x4f,0xac,0xb4,0x90,0x05,0x8b,
	0x27,0x0c,0xab,0x8c,0x0f,0xce,0x9e,0x95,0x31,0x2d,0x18,0xa0,0x80,0x90,0x0b,0x6f,
	0x8d,0xbf,0xa0,0xc7,0x42,0xa7,0x3d,0x16,0x90,0x14,0x11,0x11,0xa5,0x4d,0xcf,0x9a,
	0xc1,0x34,0x46,0x5d,0x47,0x84,0x14,0x24,0x45,0xce,0xb1,0x0c,0x66,0x1b,0x02,0x62,
	0xbc,0x30,0x99,0x59,0x82,0xb3,0x5e,0x14,0x78,0x63,0x40,0xc6,0x0c,0xc0,0x92,0xc5,
	0x1f,0x39,0x1c,0x09,0x47,0xbf,0xae,0x1f,0x9b,0x29,0x03,0x67,0x09,0xaf,0xc1,0xa1,
	0xcc,0x21,0x49,0xc6,0x42,0xa9,0x67,0x4a,0x07,0xae,0x22,0x23,0x4d,0x97,0x18,0xa5,
	0x74,0x7a,0x0e,0xc4,0x03,0x41,0x32,0x0f,0x40,0x10,0x6f,0xbe,0x1a,0x8a,0xc9,0xcf,
	0x3d,0x75,0x66,0x69,0xc7,0x18,0x19,0x10,0x8e,0x19,0x23,0x1f,0x35,0x55,0x07,0x7e,
	0xc4,0x0f,0x0e,0xc2,0x6e,0x66,0xce,0x36,0x1f,0x51,0x72,0x7e,0xc1,0xcd,0x04,0x8a,
	0xc6,0x0f,0x36,0x72,0x9d,0x66,0x2a,0x16,0x9f,0x01,0xc3,0x22,0x1b,0x29,0x5c,0x7f,
	0xce,0x75,0x3a,0x6e,0x89,0x34,0x46,0x87,0x5d,0x36,0x50,0xc2,0xc2,0xc4,0x94,0x99,
	0x07,0xce,0xc5,0x8d,0xa8,0x07,0x61,0x8a,0x01,0xcc,0x1a,0x3f,0x12,0xaa,0x08,0x13,
	0x83,0x07,0x11,0xb8,0x05,0xcb,0x06,0x68,0x6f,0x05,0x97,0xce,0x5d,0xcf,0x15,0x16,
	0xc9,0x01,0x6c,0x6f,0xcd,0x30,0x56,0x54,0x0e,0xad,0xbe,0x02,0x7a,0xab,0x05,0xb1,
	0x30,0x42,0x93,0x7b,0x0e,0xc0,0x3e,0x2e,0x1f,0xa6,0x65,0x0d,0x33,0x41,0xb2,0xb1,
	0x1f,0xbc,0x0a,0x81,0xcf,0x80,0x9a,0xa1,0x14,0x0e,0xb8,0x04,0x10,0xc3,0x2a,0x1d,
	0x37,0x17,0x0c,0xab,0x8a,0xcf,0x1d,0x89,0x18,0xcf,0x43,0x03,0xc4,0x2f,0x80,0x82,
	0x18,0x7c,0xcf,0x07,0xcb,0x34,0x12,0x6a,0xce,0x34,0x13,0xa2,0xa2,0x31,0x7b,0x6b,
	0x53,0xc1,0xa2,0xca,0xce,0xa2,0x7c,0xc4,0x86,0x44,0x8c,0x95,0x13,0x38,0x17,0xc6,
	0x55,0x7b,0x4d,0x72,0x06,0x20,0x00,0x35,0x3c,0x08,0xc0,0x97,0x0e,0xb4,0xa0,0x80,
	0x10,0xba,0x3e,0xca,0x8e,0x3f,0x8d,0x9f,0xa3,0xc6,0x80,0x86,0x27,0x3c,0xa0,0xb3,
	0xca,0x2d,0x31,0x27,0x4a,0xa5,0x0b,0x21,0x3b,0x52,0x4f,0x9f,0xc4,0x18,0x71,0xac,
	0x5c,0xc6,0x2a,0x88,0x8d,0xc7,0x33,0xc1,0x1b,0x51,0x64,0x69,0x57,0x20,0xb9,0x98,
	0x71,0xa9,0x79,0x1a,0x00,0x32,0x05,0xc5,0x34,0x1b,0x22,0xa5,0x91,0x13,0xc5,0xae,
	0x40,0x5c,0xc7,0x0b,0x7c,0x84,0x36,0x31,0x1c,0xc4,0x81,0x22,0x2b,0x95,0x1d,0x04,
	0x11,0x65,0x3d,0x23,0x12,0x96,0x01,0xc2,0x2a,0x3b,0x54,0x46,0x12,0x0f,0xa1,0xbe,
	0xbb,0xa7,0x28,0x30,0x1d,0x53,0x13,0x38,0x43,0x77,0x8f,0x76,0x02,0x12,0x9c,0x97,
	0xb5,0xa2,0x9d,0x9a,0xb1,0x2d,0x13,0x08,0xc3,0x24,0xac,0x55,0x34,0x17,0x6b,0x30,
	0x09,0x78,0x99,0x2e,0x11,0xcb,0x2c,0x04,0x1f,0x33,0x42,0xce,0xa6,0xc4,0x88,0x98,
	0xcf,0x22,0x54,0xac,0x0e,0x8e,0xa7,0x56,0x61,0x7f,0xc2,0xa5,0xc1,0x21,0x1f,0x8d,
	0xbc,0xbd,0x68,0xb9,0xa9,0x3c,0xc9,0x22,0x46,0x0a,0xc3,0x7d,0x9d,0x65,0xbd,0xc2,
	0x2c,0x79,0x8a,0x79,0x63,0x8d,0xc3,0x05,0x67,0x2d,0x05,0xad,0xaf,0x63,0x98,0x3c,
	0xb7,0x42,0x87,0xb1,0x08,0x75,0x37,0x0f,0x7b,0x1d,0x33,0x11,0xc6,0xca,0x4b,0x04,
	0x2e,0x44,0x16,0xcc,0x21,0x50,0x41,0x45,0x01,0xc3,0x14,0x5a,0xb6,0x39,0x3f,0x41,
	0x8a,0x2e,0x36,0x4b,0xcc,0x4d,0x92,0x1c,0xcf,0x16,0x20,0x85,0x66,0x9e,0xcf,0x8d,
	0xa1,0x14,0x10,0x59,0xc1,0x09,0xab,0x96,0x5c,0x24,0x15,0x99,0x02,0x3f,0xa4,0x49,
	0xaf,0x70,0x0f,0x1c,0x90,0xc2,0x4f,0xae,0x0c,0x1c,0x5e,0xb8,0xc1,0x28,0xb7,0x59,
	0xc4,0x1f,0x4a,0x75,0xba,0x6d,0x80,0x8d,0x78,0xbd,0x5e,0xb8,0x44,0x35,0x87,0x5d,
	0x3f,0x49,0x1f,0x23,0x3e,0x04,0x89,0xbf,0x5b,0xcb,0x15,0x56,0x08,0xc7,0x0e,0x78,
	0x0e,0x28,0x14,0x14,0x6e,0xca,0x3e,0xb1,0x96,0x81,0x20,0x82,0x6c,0x25,0x06,0xc9,
	0x6e,0x8c,0x24,0x1d,0x0f,0x05,0xab,0xcb,0x11,0x77,0x14,0x1a,0xa4,0x28,0x13,0x66,
	0xcc,0x55,0x16,0xc1,0x72,0x32,0x13,0x66,0x63,0xc3,0x33,0x82,0x31,0x2f,0x44,0x25,
	0x07,0x98,0x95,0xce,0x4b,0x6b,0xc2,0x04,0x4a,0x50,0x05,0x99,0x16,0xb0,0x2c,0xc7,
	0x18,0xc4,0x88,0xa6,0xca,0x82,0x1e,0x7b,0x99,0x77,0x79,0x75,0x89,0x1e,0x11,0x6d,
	0xc3,0xae,0x74,0x5c,0x6e,0x8b,0x0d,0x99,0x0f,0x07,0x78,0xc7,0x1a,0xb8,0x82,0x76,
	0x99,0x14,0x54,0xca,0x40,0x07,0xb1,0x49,0x0c,0x07,0x1b,0xb0,0x7c,0xa4,0xbc,0x3d,
	0x9f,0xaa,0x2c,0x9f,0x9e,0x54,0x41,0x64,0x79,0xc5,0x0a,0x1d,0xcf,0x8f,0xaf,0xb4,
	0x93,0x7c,0x98,0xc3,0x02,0xaf,0xc5,0x60,0x8d,0x95,0xc4,0x49,0x65,0x49,0x06,0x6b,
	0x82,0x7a,0x0d,0x08,0x4c,0x3c,0x65,0x8e,0x54,0xc6,0x4d,0x06,0x77,0xbc,0x41,0x18,
	0xc9,0x11,/*0x15,*/0x12,0x01,0x68,0xbc,0x4b,0x52,0x03,0x56,0x19,0x6c,0x76,0xb4,0x71,0x6f,
	0xb5,0xbd,0x60,0x16,0x6b,0x94,0x14,0x5b,0x76,0xb4,0xad,0x1e,0x15,0x68,0xcd,0x69,
	0xc5,0x49,0x04,0x06,0x95,0x75,0xbf,0x4d,0x73,0xc6,0x0c,0x12,0x74,0xc7,0x07,0x7f,
	0x86,0x02,0x9d,0xcb,0xb3,0x1b,0xc9,0x44,0x29,0x45,0x60,0xc6,0x34,0x4f,0xa1,0x1c,
	0x14,0x08,0x4a,0xc9,0x49,0x6c,0x8a,0x77,0xa8,0x17,0x24,0x1a,0x2f,0x97,0x74,0x77,
	0x1e,0x64,0x9f,0x8b,0x19,0xbb,0x62,0xa2,0x24,0x02,0x96,0xb2,0xc6,0xb5,0x21,0x67,
	0x80,0x23,0x04,0x55,0x13,0xb9,0xc1,0xb2,0x50,0x14,0x44,0xbd,0x62,0x85,0xc0,0x59,
	0xcf,0x85,0x6c,0x54,0x3e,0xce,0x58,0x33,0x08,0x96,0xc1,0x0f,0x20,0x10,0x18,0xce,
	0x37,0x87,0x12,0x1a,0x2f,0x21,0x46,0x48,0xa3,0x2f,0xac,0x89,0x5c,0x8f,0xc4,0xae,
	0x30,0x5f,0x09,0x2c,0x15,0xc9,0xa3,0x8b,0x23,0x13,0x18,0x74,0x8a,0xcc,0x49,0x98,
	0x75,0xc9,0xb8,0xcb,0x18,0xa4,0x80,0x2a,0xc7,0x41,0x8c,0x5b,0x84,0x64,0x93,0x5b,
	0xa2,0x3f,0x55,0xa0,0x79,0x9a,0x9f,0x70,0xcf,0x56,0x8f,0x92,0x0e,0x8a,0x19,0xc0,
	0x5d,0x5f,0x0b,0xb2,0x92,0x16,0x7c,0xac,0x9c,0x85,0x3e,0x1b,0xae,0x34,0x53,0xbe,
	0xaa,0x04,0xb2,0x23,0xbf,0x91,0xb0,0x79,0xaf,0x84,0x2f,0x5a,0x24,0xb1,0xc4,0xcf,
	0x41,0xab,0xbc,0xbb,0x7b,0x20,0x30,0x35,0x1f,0xaf,0xbd,0x88,0xa1,0xc0,0x3b,0x05,
	0x85,0x23,0xcf,0xc1,0x3b,0x42,0x0c,0xb8,0x1b,0x7a,0xcd,0x5f,0x04,0x79,0x2f,0x36,
	0x24,0x80,0x3e,0x47,0x19,0x9c,0x7d,0x8f,0x81,0x13,0xba,0x2d,0xcb,0x02,0x9e,0x13,
	0x80,0xac,0x5f,0x76,0x56,0x97,0x6f,0xa9,0xb0,0x60,0xcc,0x17,0x4c,0xad,0x7a,0xba,
	0x2e,0x2c,0x4f,0xcb,0x53,0x68,0x5c,0x06,0xa3,0xcb,0x7d,0xa2,0xbc,0x5c,0xcb,0x13,
	0xca,0x9b,0x17,0xc8,0xb8,0x37,0x0d,0x13,0xb3,0x9c,0x97,0x97,0xac,0x3f,0x3c,0x36,
	0xb1,0x0b,0x1f,0xbb,0x93,0xb9,0xc7,0x38,0x5f,0x8a,0x87,0xc1,0xbd,0x40,0x36,0x58,
	0x27,0x46,0x07,0x9d,0x97,0xbe,0x3a,0x80,0x09,0x28,0x61,0x73,0x6b,0xc2,0x14,0x8b,
	0x5e,0x43,0x47,0x90,0x7a,0x51,0xc7,0x00,0x3f,0x71,0xc6,0x3e,0x43,0x40,0x42,0xce,
	0x58,0x54,0x41,0xa0,0x01,0x50,0x0a,0x20,0x1e,0x9d,0xbe,0x14,0x4b,0x17,0x1a,0xa0,
	0x50,0xc8,0x3e,0x57,0xca,0x61,0xc8,0x5f,0x4f,0x97,0xc3,0x12,0x67,0x1a,0x0e,0x0e,
	0x01,0xc2,0xc5,0x27,0xcb,0x02,0x5c,0xce,0xa3,0x04,0xbf,0x94,0x09,0x91,0x03,0x66,
	0x0e,0x67,0xba,0x1a,/*0x1e,*/0xc6,0x40,0x9a,0x87,0x03,0xb1,0x3d,0x46,0xc8,0x44,0x38,0xc6,
	0x7a,0x70,0xb0,0x0d,0xca,0x55,0x36,0x7d,0x6a,0xab,0x2c,0x05,0xc0,0xc3,0x1f,0x86,
	0xa9,0x3c,0x0b,0x02,0x54,0x72,0x78,0xab,0xb7,0xaf,0x9d,0xc0,0x2c,0xce,0x0d,0x10,
	0x95,0xa6,0x43,0x8b,0x35,0x90,0xc1,0x0d,0x35,0x9f,0x9c,0x24,0x6e,0x7e,0xb7,0xb9,
	0x7a,0xcf,0xbe,0x1b,0x97,0x31,0x3a,0x1b,0x0d,0x3a,0x08,0x32,0x23,0x18,0x7f,0x56,
	0x1d,0x9e,0x15,0xc6,0xa7,0x93,0x9c,0x4e,0x87,0x82,0xa8,0x4f,0x82,0xcd,0x27,0x0c,
	0x76,0xcd,0x2e,0x4c,0x19,0x13,0x75,0x9e,0x7d,0xba,0xcf,0xaa,0x13,0x91,0x08,0xc8,
	0xa3,0x2f,0xa2,0x12,0x06,0x10,0x29,0xc9,0x0a,0xcc,0x52,0x14,0x75,0xc9,0x6e,0xc0,
	0xc7,0x58,0x51,0x88,0xc9,0x54,0x10,0xc9,0x04,0x0c,0x86,0x4f,0x85,0x7e,0xb4,0x74,
	0x8a,0x90,0xc9,0x5f,0x6c,0x6c,0x53,0x77,0x73,0xa0,0x26,0x1a,0x74,0x0b,0x24,0x33,
	0x53,0x29,0x1f,0x0b,0x55,0x47,0xcb,0xa1,0x8b,0x15,0x70,0x67,0x63,0x11,0x9f,0x19,
	0xbd,0x53,0xc9,0x08,0xaa,0x1e,0x81,0x17,0xcd,0x16,0x1c,0xc0,0xa8,0x02,0x40,0x29,
	0xa7,0x51,0xc4,0x08,0x1e,0xa4,0x01,0xaf,0x78,0xa9,0x84,0x38,0x05,0x87,0x24,0x1e,
	0x6a,0xa2,0x29,0xcb,0x47,0xce,0xcc,0xc2,0x50,0x09,0x17,0x66,0x0f,0xbd,0x63,0x77,
	0xbc,0x2e,0x3b,0x0c,0x6c,0xc0,0xa8,0x74,0xcc,0x5b,0xa6,0x60,0x6e,0xa3,0x8f,0xb9,
	0x58,0x15,0x6a,0xc8,0x1e,0x7c,0x6d,0xba,0x05,0xca,0xbc,0x0f,0x73,0x11,0x60,0x61,
	0x01,0xbe,0x36,0x2b,0x08,0x6f,0x86,0x11,0x99,0x1e,0xb6,0x0e,0xc3,0xc8,0x03,0xc9,
	0x44,0x43,0x0e,0x96,0x1f,0xb4,0x1f,0xb6,0xbe,0x0b,0x16,0x37,0xb1,0x47,0xc9,0x18,
	0x46,0x0b,0x38,0x8a,0x73,0x1d,0x1f,0x31,0x59,0x64,0x1b,0x28,0x51,0x93,0xa2,0x15,
	0x02,0xb0,0xb3,0x4a,0x49,0x2f,0x0c,0x10,0xa7,0x65,0x04,0xb4,0x80,0xa3,0x19,0x03,
	0x16,0xba,0x33,0x53,0xcf,0x07,0xcc,0xbe,0x90,0x4e,0xa5,0x54,0xba,0x51,0xc1,0x3c,
	0x97,0xc5,0x26,0x82,0x82,0x90,0x49,0xbc,0x55,0x0f,0x1a,0xc6,0xb5,0x8e,0x0b,0x17,
	0xca,0x3e,0x87,0xae,0x06,0x94,0x28,0x1c,0x27,0x59,0x40,0x2f,0x0b,0x8a,0x1a,0x1d,
	0xa0,0x41,0x25,0x0b,0x89,0x4d,0x3b,0x0c,0x0d,0x1a,0x03,0xbf,0x7f,0x15,0xa4,0x34,
	0xaa,0x47,0x93,0x2a,0x17,0x02,0x67,0x78,0xc3,0x25,0xc2,0x1b,0x06,0x0c,0x1e,0xb1,
	0x13,0x07,0x77,0xc9,0xa7,0x76,0x76,0x56,0xb5,0x1d,0x07,0x38,0x0a,0x0f,0x07,0x95,
	0xcf,0x1f,0x23,0x8b,0x39,0x5b,0xa6,0xc7,0x12,0x19,0x03,0xb5,0x77,0x01,0x61,0x6b,
	0x75,0x17,0xcb,0xc3,0x1b,0x50,0x54,0x91,0x8c,0x35,0x9d,0x90,0x7d,0x7b,0x15,0xc6,
	0xc2,0x4f,0x64,0xad,0x9d,0xb0,0x6c,0x80,0x1c,0x1e,0x47,0x15,0x7b,0x68,0x86,0x52,
	0x21,0x5f,0x30,0x62,0x8e,0x06,0x65,0x94,0x0f,0x51,0x02,0x35,0x0a,0xad,0xab,0x06,
	0x86,0x3e,0x7e,0xaf,0xbf,0xa9,0xaf,0xa8,0x3a,0x0b,0x47,0xac,0x02,0xc8,0x10,0x9c,
	0x45,0xcf,0x02,0x75,0xc9,0x2e,0x1f,0x68,0xc2,0x46,0xc6,0x8d,0x96,0xb6,0x94,0x5e,
	0x42,0xbc,0x0e,0x56,0x09,0x7b,0x5b,0xc4,0x0b,0x8a,0x31,0x18,0x31,0xbb,0x8a,0x72,
	0x9b,0xb2,0x70,0xab,0xab,0xc1,0x0e,0x22,0x7d,0xb6,0x1c,0x12,0x6e,0x45,0xaf,0xca,
	0xaa,0x0b,0x3e,0x36,0x37,0x1b,0x0a,0x5e,0x71,0x03,0x7d,0x01,0x65,0x2a,0x33,0x4b,

	0x8b,0x55,0x5f,0x14,0x1d,0x96,0x15,0x3b,0x03,0x39,0x22,0x2a,0xc1,0x02,0x27,0xc6,
	0x13,0xb8,0x05,0x8d,0xc0,0x11,0xb2,0x43,0xb4,0xc0,0x8e,0x7b,0x2f,0xac,0x53,0x1f,
	0x0e,0x05,0x6b,0xb7,0x49,0x04,0x16,0x78,0x34,0x20,0x0d,0x59,0x7e,0x86,0x06,0x62,
	0xa9,0x2e,0x69,0x03,0x62,0x9c,0xa1,0x8c,0x2e,0x01,0x4b,0xbb,0x84,0x06,0xcf,0x2e,
	0x02,0xa0,0xbc,0x51,0x8a,0xbc,0x3a,0x50,0x08,0xc2,0x06,0x08,0xa6,0x03,0x6f,0x66,
	0x79,0x59,0x1d,0xa6,0x80,0xa3,0x38,0xaa,0x40,0x31,0x19,0x03,0x71,0x79,0xc1,0x82,
	0x7c,0xc7,0x62,0x64,0x28,0x13,0xa5,0x47,0x01,0xa4,0x14,0xcf,0x22,0x09,0x02,0xc1,
	0xb5,0x1f,0x15,0x7d,0xcd,0x59,0x53,0x13,0xce,0x49,0xc4,0x3b,0x43,0x5d,0x4d,0xcc,
	0x57,0xc3,0xc2,0x76,0xc6,0x1e,0x9c,0x15,0x60,0x63,0x6d,0xc6,0x97,0xc3,0x53,0x1f,
	0xbc,0x59,0x27,0x1a,0xcc,0x37,0x25,0xa8,0x16,0xb5,0xaf,0x5d,0x26,0x03,0x1c,0xb2,
	0xa6,0x7c,0xc3,0x9d,0x56,0x0d,0x5e,0x00,0x89,0x8b,0x44,0x4d,0xc0,0x74,0x6f,0xb2,
	0x2e,0xce,0x39,0x47,0x40,0x0f,0x24,0x80,0x62,0xcc,0x3e,0xc5,0x04,0x66,0xaf,0x65,
	0x01,0x11,0xc4,0x7d,0x09,0x1e,0x5b,0xce,0x14,0xc5,0xc1,0x60,0x84,0xc9,0x39,0x3a,
	0x66,0x6e,0x90,0x75,0x5b,0x52,0x0a,0x65,0x77,0x7a,0x12,0xa6,0x5a,0x28,0x6d,0x90,
	0x0e,0x0e,0x77,0xac,0x03,0xb6,0x0c,0x3e,0xc7,0x51,0x9a,0xaa,0x9b,0x64,0x2d,0x69,
	0x07,0x71,0xc9,0x34,0x41,0x6a,0x8a,0xb6,0x55,0x99,0x83,0x07,0x7e,0x93,0x13,0xae,
	0x8e,0xcb,0x14,0xc4,0x0e,0x63,0xb3,0x10,0x14,0x48,0xcb,0x87,0x26,0x30,0x37,0x76,
	0xb1,0x4f,0xbb,0xc6,0x00,0x30,0x04,0x40,0x1f,0x60,0x9f,0xc1,0xc6,0x5f,0xba,0x71,
	0x72,0xba,0x09,0x6f,0x81,0x30,0x1a,0x83,0x9e,0x97,0x92,0x8a,0x43,0x01,0xcf,0x43,
	0x83,0x7a,0x63,0x12,0xa0,0xc6,0xca,0x66,0x73,0x84,0x34,0x29,0x4e,0x1b,0x89,0x0c,
	0x52,0x34,0x55,0x0f,0x56,0x69,0x13,0xb5,0xbb,0xb5,0xc0,0x18,0x81,0x17,0xa2,0x2a,
	0x2d,0x03,0xa2,0x8d,0x97,0x24,0x17,0x80,0x0a,0x9b,0x09,0x27,0x4e,0x87,0x4c,0x2f,
	0x14,0x9b,0x7b,0x9b,0x08,0x3c,0x21,0xc5,0x13,0x91,0x02,0xb0,0x43,0x3c,0x66,0x13,
	0xb1,0x1a,0xcc,0x03,0x4c,0x37,0x4d,0xcb,0x2d,0x14,0x78,0xa4,0xb3,0x6f,0xc2,0x2d,
	0x57,0xa6,0x7e,0xc0,0xb8,0x3a,0xc8,0x92,0x09,0x4f,0xa5,0xbe,0x0e,0x6c,0x96,0x3a,
	0x47,0xbf,0x22,0x3d,0x1f,0x7b,0x67,0xb7,0x11,0x65,0x78,0x8d,0xcc,0x10,0x85,0x18,
	0x0e,0xce,0x02,0xb3,0x18,0x10,0x15,0x76,0x78,0x65,0x37,0xb3,0x9e,0x10,0x40,0x43,
	0x8c,0x56,0x29,0x4a,0x1e,0x8b,0xb2,0x93,0x19,0x4f,0x5e,0xb7,0x10,0x96,0x26,0x6d,
	0x9a,0x96,0x26,0x8f,0x64,0x34,0xaa,0x5e,0x29,0x49,0xce,0x79,0xab,0xc6,0x0b,0x7a,
	0xce,0x1b,0x68,0x4f,0xc3,0x1e,0x0d,0x00,0x54,0xc8,0x04,0xc0,0xc4,0x46,0x13,0x9c,
	0x32,0x5e,0x18,0xa6,0x8a,0x46,0x10,0x4b,0x0a,0x5e,0x71,0x91,0x67,0xb1,0xa4,0x07,
	0xbe,0x3c,0x58,0x43,0x48,0x3e,0x04,0xcd,0x02,0xc7,0xb5,0x89,0x3e,0x49,0x2d,0x1e,
	0xc2,0x23,0x1d,0x27,0x7e,0x63,0x11,0x67,0xb6,0xaf,0x4e,0x0f,0x4a,0x4e,0x03,0x6a,
	0xa0,0xa0,0x4f,0xc4,0x01,0xa2,0xc6,0xce,0xa8,0x54,0x0b,0x0b,0x2a,0x23,0x27,0x7c,
	0xc0,0x3b,0xb5,0x49,0xc5,0x72,0x91,0xc9,0x61,0x82,0x2b,0x46,0x27,0x4b,0x40,0x41,
	0x35,0x10,0x06,0x33,0x32,0x99,0x3a,0xc6,0xca,0x73,0x09,0x36,0x92,0x86,0x0d,0x0b,
	0x0f,0xc5,0x94,0xcc,0x5e,0x06,0x78,0x6b,0x66,0x0a,0xc6,0x72,0xcc,0x38,0x0f,0x8e,
	0x19,0x60,0x96,0xc5,0x9a,0x73,0x93,0xa0,0x46,0x0e,0x15,0x1d,0xc5,0x69,0x8c,0x01,
	0xc9,0x62,0x5b,0xac,0x3f,0x5f,0x02,0x07,0x05,0xa6,0x1d,0x83,0xc9,0x13,0xa2,0x65,
	0x30,0x05,0x9d,0xca,0x0a,0xa0,0xac,0x4d,0x44,0x17,0x13,0xc0,0x01,0x69,0x6b,0x81,
	0x33,0x5a,0x58,0x1e,0x0d,0x33,0x06,0x95,0x05,0xac,0x1f,0x51,0x74,0x38,0x16,0x80,
	0x23,0x1c,0xc8,0xb5,0x5f,0xcc,0x3b,0x4e,0x5a,0x01,0x55,0x04,0x1e,0x86,0x45,0x96,
	0x20,0x36,0x1d,0x02,0xc2,0x11,0x84,0xbf,0x01,0xa3,0xcf,0xb8,0x30,0x4b,0x53,0x82,
	0xcd,0x10,0x76,0xc3,0x36,0x22,0x1a,0x11,0x50,0x1d,0xa4,0xce,0xc8,0x27,0x44,0xc6,
	0x40,0x49,0x98,0x05,0x1e,0xce,0x17,0x72,0xb8,0x61,0xb6,0x23,0x0a,0x6a,0xb0,0xba,
	0x54,0x0b,0x7b,0x96,0x52,0x2b,0x71,0x98,0x76,0x54,0x0c,0x0e,0xce,0x1e,0xc4,0x16,
	0x01,0x1a,0x01,0x65,0x30,0x13,0x19,0x33,0x87,0x66,0x9b,0x06,0xc6,0x07,0xad,0xcf,
	0xbc,0xcf,0x57,0x4d,0x7e,0xb4,0xa4,0x46,0x20,0x5a,0x8e,0x1c,0x7c,0x9c,0x24,0x2e,
	0x3d,0x9a,0x02,0x09,0x64,0x98,0xc3,0x3d,0x19,0x19,0x17,0x9c,0x1d,0x7a,0xcb,0x03,
	0x0e,0xab,0xc0,0x15,0xcf,0xa3,0x1b,0x2e,0x7e,0x7a,0x69,0x03,0xb0,0x6d,0xb0,0xc3,
	0x0d,0x45,0x0e,0x01,0x96,0x17,0xc0,0x0b,0x97,0x3b,0x05,0x8e,0x1b,0x43,0x17,0x93,
	0x26,0x23,0x44,0x40,0x2b,0x3a,0x8f,0x3b,0x38,0x47,0x90,0x2a,0x3d,0x18,0x88,0x5d,
	0xad,0x72,0xae,0x02,0x83,0x42,0x96,0x04,0xbe,0xbb,0x8e,0x05,0x89,0x3a,0x4e,0x4e,
	0x0c,0x78,0x4d,0xc7,0x27,0x06,0x14,0x6e,0xcf,0x7f,0x92,0x2c,0x71,0x2d,0xac,0x32,
	0x3e,0x08,0xab,0x7c,0x1d,0x01,0xb9,0x46,0x40,0xc3,0x58,0x41,0x9c,0x13,0xcf,0x08,
	0x1f,0x62,0x8c,0x0a,0x70,0x01,0xb9,0x74,0xc9,0x7d,0x88,0xbc,0x16,0x2f,0x62,0x9e,
	0x05,0x5e,0x26,0xc8,0x23,0x06,0x49,0xc7,0x29,0x09,0x3f,0x8f,0x5e,0x2a,0xc5,0x54,
	0xc5,0x0a,0xb4,0x0b,0xb3,0x42,0x91,0x0c,0xcd,0x43,0x6b,0x4a,0xbe,0x0c,0xca,0xad,
	0x7f,0x78,0x73,0x80,0x76,0x3a,0xbc,0x31,0x40,0x16,0x5e,0x0b,0x04,0x43,0x0c,0x4a,
	0xc8,0x12,0x56,0x13,0xc6,0x17,0x91,0x93,0xcb,0x2e,0x1f,0x98,0x15,0x71,0x1d,0x7b,
	0x7a,0xb0,0x01,0xc9,0x24,0xa3,0x14,0xb6,0x9a,0x1e,0x7b,0x6c,0x95,0x76,0x02,0x24,
	0x17,0x19,0x2a,0xa5,0x76,0xb8,0xcb,0xa3,0x59,0xa3,0xc7,0x4c,0x4c,0x8e,0x4b,0x54,
	0xce,0x27,0xa9,0x67,0x8e,0x67,0x8f,0x58,0x62,0x48,0xa3,0x89,0x07,0x01,0xcf,0x77,
	0xc3,0x90,0x7c,0x67,0x2f,0x00,0x18,0x82,0xca,0x19,0x2f,0x52,0x0e,0x8c,0x1d,0xb0,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0x05,0x64,0x6f,0x01,0x13,0x14,0x5c,0xb5,0xa3,0x62,0x54,0x6a,0x36,0x20,0x1a,0x6e,
	0x44,0x07,0x27,0x09,0xb9,0x6d,0x4e,0x44,0xba,0xa5,0x7e,0x1c,0xc1,0x2f,0x0f,0x6d,
	0x08,0x3e,0x32,0x15,0xb9,0x9a,0x71,0x4a,0x34,0x1a,0x1e,0xb9,0xc4,0x95,0x28,0x1f,
	0x99,0x31,0x10,0x92,0xb2,0xad,0x59,0x1d,0xad,0x4f,0x5b,0x04,0xc4,0x2f,0x56,0x9a,
	0x1c,0x6e,0x9f,0x3f,0xcd,0x14,0x8b,0xc0,0x89,0x97,0x13,0x42,0x1a,0xca,0x5f,0x05,
	0xc9,0x16,0xc6,0x09,0xc2,0x67,0x9a,0x0b,0x98,0x1e,0xa4,0x93,0x78,0x53,0x7e,0x4c,
	0xc9,0x27,0x03,0x02,0x48,0x1c,0x5b,0x24,0x0e,0x73,0x87,0x5b,0x48,0xc4,0x32,0x7a,
	0x02,0xcd,0x06,0xce,0x9c,0x04,0x94,0x18,0x85,0x9a,0x27,0xc8,0x36,0xad,0x8c,0xca,
	0x9d,0x29,0x2e,0x42,0x61,0x93,0x58,0x16,0x14,0x9c,0x41,0x0c,0x60,0xcf,0x1e,0x50,
	0x00,0x2e,0xab,0xca,0xb0,0x37,0x9f,0x11,0x28,0x4c,0x16,0xac,0x7e,0x3e,0x56,0x90,
	0x10,0x92,0x2b,0x02,0x30,0x99,0x8f,0x93,0x30,0x95,0x00,0x3e,0x68,0x4b,0x0a,0x5e,
	0x69,0x72,0xc4,0xc6,0xca,0x1a,0x04,0x70,0x88,0x36,0xcb,0x86,0x37,0xb4,0x82,0x85,
	0x54,0x45,0xc4,0x14,0x70,0xc5,0x6b,0x6c,0x14,0xa3,0x45,0x79,0xb2,0xcc,0x0c,0x1f,
	0x84,0x5d,0x17,0xab,0x0c,0x17,0x32,0xce,0x4b,0x33,0x41,0x1b,0xc8,0x3b,0x06,0x62,
	0x63,0xcf,0x94,0x71,0x9e,0xc9,0x17,0x73,0xcd,0x9b,0x13,0xc5,0x86,0x1d,0x4c,0x55,
	0x0e,0x93,0x53,0x13,0x7a,0x1f,0x00,0x0e,0x0a,0x62,0x4a,0xca,0x39,0x47,0x00,0x84,
	0x4a,0x6a,0x13,0x1a,0x6e,0x10,0xce,0x18,0x84,0x47,0xce,0x46,0x25,0x1c,0xb9,0x79,
	0x3f,0x9e,0x00,0x8b,0x02,0x95,0x49,0x07,0xc9,0x1d,0xc5,0x4f,0x1b,0xab,0x40,0x5e,
	0x15,0x60,0x61,0x74,0x8a,0xc1,0xc9,0x1b,0x69,0x60,0x7c,0x33,0x56,0x04,0x6b,0xb4,
	0xc6,0x3b,0x05,0x8c,0x06,0xa1,0xae,0x2d,0x69,0xb6,0x8c,0x2a,0xc0,0x4d,0xb8,0xc0,
	0x5a,0x88,0x58,0x2f,0x27,0x0d,0xc3,0xaf,0x11,0x49,0x7d,0xc9,0x20,0xca,0x1b,0xcb,
	0x90,0x61,0x22,0xc3,0xa7,0x0a,0x10,0x1b,0x3a,0xbf,0x9f,0x07,0x0d,0x24,0x13,0x3d,
	0xc4,0x44,0x35,0x17,0x9e,0x5a,0xb3,0xa7,0xa3,0x2d,0x19,0x7a,0x0a,0xa7,0x93,0xcb,
	0xc4,0x01,0x38,0x26,0x0e,0xbc,0x86,0x01,0xcc,0xc7,0x4c,0x58,0x0c,0xa8,0xb9,0xc0,
	0xc8,0x1e,0x0e,0x74,0xcd,0x8f,0x0c,0xc3,0x11,0x92,0x5a,0x13,0x97,0xb5,0xc2,0x74,
	0x45,0x8f,0x3a,0x4c,0x1c,0x18,0xcb,0x67,0x71,0x5d,0x0d,0x98,0x88,0x0f,0x8a,0x55,
	0x48,0x01,0x25,0x3b,0x49,0x8e,0x28,0x31,0x0f,0xbc,0x8d,0x08,0x28,0xb5,0xb5,0x19,
	0x85,0x56,0x20,0x7a,0x03,0x19,0x8f,0x49,0xbe,0xc8,0x6d,0x42,0x28,0x6e,0x8d,0x28,
	0x85,0x85,0x58,0x28,0x34,0x47,0x0d,0x7a,0x0a,0xc0,0x9c,0x48,0x88,0x85,0x9f,0x24,
	0x30,0xce,0x11,0xc1,0x22,0xc4,0xb5,0x42,0xa5,0x10,0x60,0xb8,0xc2,0x57,0x7a,0x15,
	0x3c,0x6f,0x6d,0x70,0xb3,0xb1,0x02,0xbf,0xc3,0xa5,0x51,0x2c,0x2c,0x70,0xcf,0x70,
	0xa0,0xb9,0x99,0x24,0x91,0xc8,0xc5,0xa6,0x70,0x58,0x4b,0x34,0x32,0x8e,0x47,0x09,
	0x4c,0x1a,0x7b,0xc6,0x07,0x4e,0x0b,0xc6,0x60,0xc0,0x8f,0x1a,0x3c,0x94,0xb7,0xcc,
	0x99,0x03,0xb0,0xac,0x06,0x91,0x9c,0x45,0xcc,0xcb,0x1e,0x4b,0x13,0x4a,0x2c,0xa6,
	0x3f,0x7d,0xab,0x18,0x67,0xcf,0x6d,0x18,0x4b,0x0b,0xa0,0x7f,0x74,0x10,0x14,0x71,
	0xc9,0x83,0x34,0x40,0x41,0x0f,0x74,0xc8,0x98,0x69,0x2d,0x14,0x30,0x09,0xbe,0xa0,
	0x72,0x50,0x1b,0xc2,0x7e,0x59,0x45,0xa8,0x9e,0x96,0x0e,0x05,0x5f,0xac,0x48,0xa1,
	0x5b,0x7e,0x5e,0x90,0xc6,0x6b,0x81,0x3e,0x95,0x3c,0x42,0x0b,0x0a,0x05,0x09,0x2d,
	0x9f,0x27,0x21,0x07,0xa3,0xa7,0x83,0x29,0x0b,0xa8,0x3d,0x1c,0x17,0x03,0x18,0x77,
	0x9a,0x76,0x28,0x4c,0x50,0x65,0x10,0x7f,0x89,0x18,0x5b,0xc0,0xc8,0xcc,0x98,0x0c,
	0x21,0x0e,0xa9,0x38,0x9e,0x0a,0xc2,0x17,0x3a,0xc6,0x80,0x33,0x11,0x1a,0x4e,0x1b,
	0x37,0x76,0x59,0x1c,0xc3,0x5d,0xc5,0x8b,0x06,0x45,0x8d,0xc4,0x25,0x31,0x00,0x88,
	0xb7,0x27,0x9f,0xc7,0xc7,0x22,0xb0,0x19,0xab,0x68,0xba,0xa4,0x05,0xa9,0xca,0xba,
	0xb6,0x7e,0x10,0x22,0x9c,0x5e,0x56,0x64,0x8d,0x0a,0x83,0x41,0xcc,0x34,0x16,0x06,
	0x8c,0x50,0x00,0xc3,0x58,0x0f,0x94,0x0a,0xce,0x2d,0x6a,0x5a,0x07,0xc4,0xab,0x0f,
	0x9d,0x83,0xcd,0x2e,0x97,0x1c,0xcb,0x84,0x38,0x85,0x65,0x3f,0x3b,0x05,0x0e,0xa8,
	0x2a,0x7d,0x1e,0x06,0x78,0x14,0x59,0x20,0x15,0x1b,0x62,0x4f,0x1e,0x19,0x01,0x9b,
	0x34,0x03,0x6c,0x1d,0xc8,0x17,0x6d,0x2a,0x8e,0x0a,0x48,0xac,0x40,0x6d,0x37,0x1d,
	0x8d,0x1a,0x05,0x39,0x39,0x4c,0x52,0x31,0xb9,0xa9,0xa2,0x9e,0x55,0x3a,0xcf,0x06,
	0x99,0x06,0xcc,0x8b,0xbc,0x9f,0x16,0xb5,0xb8,0x6f,0x7a,0x06,0xc0,0x46,0x07,0x16,
	0x6d,0x02,0x9e,0x38,0xb4,0x70,0x96,0x8a,0x36,0x14,0xc8,0x1c,0x7d,0x9b,0xce,0xce,
	0x11,0x1e,0x04,0x16,0x63,0x66,0x86,0x05,0x0c,0x6c,0xb7,0x56,0x0b,0x6d,0x13,0xc2,
	0x58,0x58,0x0c,0xc4,0x53,0xc7,0x24,0x4f,0x35,0xc2,0xcf,0x94,0x02,0xac,0x71,0x9b,
	0x20,0xcb,0xbf,0xcd,0x79,0x80,0x1d,0x69,0x19,0x91,0x87,0x2b,0xa2,0x3a,0x52,0x61,
	0x25,0xc5,0xa2,0xa0,0xcc,0x04,0x0e,0x7e,0xc3,0x36,0x2d,0x98,0xaa,0xaa,0xcc,0x30,
	0x3a,0x82,0x65,0x5c,0x03,0x11,0x5e,0xa1,0xad,0x13,0xa2,0x95,0xc0,0xc1,0xc8,0x66,
	0x0e,0x06,0xcf,0x6f,0x65,0x3a,0x9b,0x06,0xad,0x13,0x14,0x33,0x15,0xc5,0x12,0xae,
	0x38,0x9c,0x13,0xcc,0x7c,0x06,0x05,0xcd,0xc3,0x5a,0x04,0x4f,0x2b,0x13,0x54,0x97,
	0x35,0xcc,0x8f,0x08,0x03,0x1a,0x80,0x2e,0x19,0x2d,0x72,0x23,0x46,0x4f,0x8d,0x39,
	0x22,0xa0,0x55,0x1d,0xcc,0xc3,0xcd,0x0a,0x35,0x6b,0xa8,0x95,0x3c,0x1b,0xb5,0xcb,
	0x58,0x41,0x9f,0x8d,0x3c,0x1b,0x9c,0x4b,0x0c,0x2d,0x9a,0x00,0x21,0x59,0x35,0x05,
	0x48,0x1b,0x1e,0xbf,0x6d,0x1a,0xc0,0x36,0x36,0x46,0x1b,0x17,0x51,0x48,0x48,0x61,
	0x53,0xa1,0x3f,0x29,0x1a,0x70,0x0a,0x98,0x06,0x60,0x77,0x34,0x7f,0xb0,0x01,0x8b,
	0xce,0x8f,0x26,0x91,0x26,0x5d,0xbc,0x89,0x59,0x4a,0x03,0x69,0x0f,0x70,0x61,0x1a,
	0x1c,0xab,0xba,0xc4,0xaa,0x5e,0xbc,0x28,0x0f,0x6e,0x5b,0xc1,0x9a,0x7a,0x11,0x13,
	0x28,0x67,0xc0,0x53,0x07,0x88,0x5f,0x2b,0xa0,0xa0,0x28,0xc2,0x1b,0x74,0x20,0x71,
	0x55,0x50,0x10,0xa0,0x30,0x6a,0x91,0xa7,0xac,0x9a,0xa8,0x2f,0x0a,0x79,0x1f,0x93,
	0x92,0x10,0x8b,0x88,0x2b,0x0a,0x58,0x01,0xbd,0x86,0x13,0x53,0xa5,0xb4,0x7c,0x50,
	0x0d,0x8b,0x13,0x0b,0xc2,0x51,0x4c,0x5f,0x30,0x95,0x0f,0x31,0x0c,0xc0,0x06,0x62,
	0x79,0x96,0x14,0x25,0xcd,0x97,0x0b,0xcc,0x67,0xcd,0x13,0xb0,0x12,0xa2,0xae,0x37,
	0xbd,0x11,0x16,0xca,0xc3,0x8e,0x5c,0x14,0xb9,0x9b,0x4f,0x47,0x1c,0x92,0xbf,0x66,
	0x5f,0xc0,0xb4,0xb6,0xc8,0x0a,0xcd,0x62,0x42,0x20,0xc1,0x44,0xa3,0x37,0xa5,0x0e,
	0x4b,0x45,0x32,0x61,0x1e,0xcd,0x78,0x22,0xb3,0x12,0x91,0x89,0x1e,0x4a,0x13,0x6a,
	0x24,0x49,0x06,0x9d,0x45,0x20,0xbf,0x90,0x1b,0xb7,0x1a,0x1c,0x0e,0x75,0x8c,0x37,
	0xc5,0x47,0x16,0x19,0x1f,0x02,0x61,0x04,0x16,0x8d,0x83,0xc6,0xc1,0x1d,0x81,0x6e,
	0xca,0x13,0x0c,0x12,0x9c,0x56,0x14,0xa0,0xb2,0x09,0x61,0x56,0x1c,0x73,0x83,0x50,
	0xb3,0x12,0x9e,0x06,0x4c,0x44,0x98,0xcd,0x34,0x0a,0x28,0xa1,0x11,0x09,0xaa,0x91,
	0x9b,0x36,0x59,0x5a,0x34,0x1c,0x74,0x5e,0x13,0x7d,0x46,0x5b,0x37,0x8a,0x5d,0xcb,
	0x5f,0x14,0x5b,0x65,0x97,0xba,0xa8,0x67,0xc9,0x92,0x17,0x68,0xcd,0xa8,0x9a,0x1b,
	0x5b,0x8a,0x35,0x66,0x22,0x1d,0x4c,0x7a,0x3f,0x39,0x37,0x57,0x08,0x0b,0xaa,0x2a,
	0x1b,0x69,0xbb,0x72,0x01,0x67,0x81,0x07,0xb6,0xcd,0x7e,0x11,0x13,0xc3,0x7d,0xcb,
	0x9c,0xc0,0x2c,0xc5,0x06,0xcc,0x00,0x02,0xc2,0x30,0x0a,0xc2,0x5f,0x45,0x00,0xc5,
	0x12,0x84,0x9d,0x19,0x31,0x43,0x14,0xc4,0x35,0x5a,0x1c,0x0a,0x70,0x01,0x8c,0x0d,
	0x41,0x1e,0xca,0x10,0x17,0x4b,0x1f,0xb6,0xae,0x08,0x12,0x06,0x72,0x20,0x3f,0x07,
	0xcf,0x49,0x3b,0x21,0x2a,0x64,0x74,0xc1,0x9a,0x1d,0xb9,0x77,0x68,0x75,0x15,0x4d,
	0x0b,0xa9,0x91,0x8e,0x00,0xcf,0x32,0xb7,0x68,0x10,0x00,0x85,0x07,0x13,0x8b,0x07,
	0xc7,0x05,0x13,0x1a,0xbb,0x44,0x41,0x31,0x4d,0xcb,0xc4,0x0d,0x68,0xc2,0x24,0x59,
	0x5c,0xc1,0x21,0x49,0xbc,0xc4,0x2b,0x01,0x81,0xc3,0x98,0x13,0x72,0xca,0xb6,0x38,
	0x12,0x78,0x16,0xb2,0x85,0x5b,0x27,0x11,0xc9,0xce,0x5a,0x30,0xca,0x7f,0xce,0x4f,
	0x38,0x15,0x07,0x0b,0x2d,0x5f,0x97,0x23,0x1c,0x96,0x5c,0x25,0x34,0xa9,0xc5,0x03,
	0x2f,0x97,0xcf,0xb8,0x61,0x12,0x7f,0xb5,0x8b,0x1f,0x5a,0x0e,0x40,0x06,0xc4,0x5b,
	0x5b,0x34,0xc4,0x4a,0x8f,0x3b,0x2a,0xcf,0xcc,0x1b,0xb8,0x5c,0x53,0x02,0x4a,0x07,
	0xa8,0xa1,0x63,0xb7,0xa8,0x17,0x93,0xb2,0xb9,0x94,0xb1,0x42,0xb7,0xc7,0x50,0x19,
	0x33,0x1d,0xb9,0x1a,/*0x1e,*/0x1c,0x44,0x7d,0xce,0x04,0x83,0x2c,0x4f,0x1b,0x54,0x1e,0xc1,
	0xb3,0xbb,0x5a,0x71,0x91,0x2c,0x0e,0x16,0xc0,0xb5,0xc6,0x1b,0x4a,0x95,0x30,0xc1,
	0xa2,0xa1,0xcf,0x30,0x63,0x87,0x00,0x13,0x7c,0xa5,0x47,0x55,0x10,0xc8,0x89,0x2a,
	0x65,0x0e,0x19,0x8b,0x14,0xce,0x14,0x66,0x56,0x03,0x5e,0x49,0xcc,0x15,0x4d,0x0b,
	0x24,0x63,0x2d,0x19,0xa4,0x6f,0x3a,0x77,0x6e,0x8f,0x1d,0xc0,0x02,0x95,0x49,0xcf,
	0x76,0x15,0x9c,0x0b,0xb0,0x83,0x87,0x48,0x9f,0x18,0xa1,0x9e,0x52,0x49,0x3b,0xc3,
	0x42,0x52,0x3c,0x7b,0xc0,0x05,0x2f,0xc3,0xcb,0x75,0x51,0x6f,0x92,0x97,0x59,0x03,
	0xc8,0x91,0xbe,0x98,0x29,0x33,0x13,0xa4,0x01,0x2e,0x2c,0x14,0x58,0x44,0x98,0xaa,
	0xb9,0x3d,0xc6,0x1e,0x4f,0x5c,0x42,0x14,0xcc,0x88,0x55,0x9f,0x5e,0xb6,0xc4,0x13,
	0x35,0xbe,0xc0,0x08,0x4f,0x25,0x80,0x80,0xcb,0x05,0x1d,0xcb,0x17,0x71,0x3d,0xba,
	0xae,0x72,0x32,0x11,0x1e,0x8e,0x45,0x30,0x6b,0x9b,0x7a,0x32,0x9c,0x01,0x94,0x38,
	0x66,0x0b,0xc5,0xa4,0xad,0x60,0x0d,0x72,0x0a,0x9e,0x70,0x66,0x11,0xca,0xc7,0xba,
	0x92,0xc7,0x77,0x74,0x64,0xbe,0x69,0x00,0x56,0x0c,0x01,0x3f,0x44,0x0a,0xa9,0x13,
	0x5e,0x04,0x77,0xaa,0x7c,0xcc,0x03,0xc2,0x10,0x0f,0x6c,0x15,0x62,0x3b,0x0f,0xaa,
	0x86,0x1d,0xc5,0xc8,0x50,0xb7,0xc0,0x49,0xbc,0x87,0x76,0xc0,0x84,0x17,0x03,0x96,
	0x00,0x32,0xa7,0x15,0x99,0x4e,0x11,0x53,0x7e,0x0d,0x5a,0x2a,0xc0,0xc4,0xc3,0x1b,
	0xb5,0xb7,0x72,0x7f,0xcb,0x19,0xa3,0x07,0x55,0xc3,0x30,0x31,0x1d,0xcf,0x7e,0x1a,
	0xa3,0x8e,0x10,0xcd,0x34,0x39,0x0b,0x39,0x43,0xb7,0xb8,0xb8,0xc2,0x0b,0xb1,0x82,
	0x6a,0xcb,0xb6,0xcb,0x03,0x2a,0xbf,0x7f,0x0a,0x07,0x43,0x10,0xce,0x2a,0x87,0x7d,
	0x81,0x76,0x0c,0xcf,0x9b,0x76,0x0b,0xcf,0x68,0x45,0x26,0xbf,0xc1,0x0e,0x1e,0xa8,
	0x06,0xa7,0x18,0xa9,0x8a,0x61,0xc1,0x2f,0x5a,0x11,0x51,0x69,0xc1,0x9e,0x05,0xcd,
	0x2e,0x5c,0xc1,0x3d,0x06,0xa9,0x0e,0x17,0x17,0xcd,0x03,0x93,0x19,0x72,0x75,0x3f,
	0x75,0xaa,0xcf,0xcb,0x4b,0xb3,0xbb,0x1a,0x67,0xb2,0x47,0x0c,0x07,0x93,0xcd,0xbf,
	0x90,0x87,0x40,0x21,0x17,0x4c,0x12,0xb0,0x61,0x67,0x49,0x45,0x5b,0x31,0x38,0x36,
	0x05,0x94,0x04,0xa0,0x54,0xbe,0xc0,0x0c,0x24,0x4d,0x30,0x11,0x03,0x2e,0x1f,0x74,
	0x4e,0x0e,0x9b,0x4f,0x73,0x4a,0x43,0x79,0x10,0x6b,0xca,0xb8,0x4f,0x1a,0xb2,0xaf,
	0x87,0x0f,0xaf,0x1e,0x83,0x74,0x0f,0xb3,0x2a,0x78,0x15,0xc9,0x38,0x9c,0x6c,0x94,
	0x15,0x5b,0xba,0xbb,0x81,0x78,0xc4,0x0e,0xcd,0x73,0x94,0x12,0x59,0x5f,0xba,0x66,
	0x16,0xce,0x78,0x39,0x7f,0x38,0x65,0x7f,0x94,0x1d,0xbd,0x54,0x07,0x03,0xc6,0x3a,
	0xc5,0x20,0x75,0xa1,0x5c,0xcf,0xcd,0x0a,0x9b,0x10,0x04,0x9c,0x4d,0x8f,0x8d,0x39,
	0x46,0x16,0x82,0xb0,0x13,0x65,0x63,0xb3,0xc8,0x14,0x50,0x04,0x07,0x09,0xc8,0x12,
	0xc6,0x04,0xcd,0x87,0x43,0xcf,0xc3,0x75,0xbd,0x71,0x67,0xcd,0x0d,0xbb,0x88,0x55,
	0x07,0xb6,0x89,0x50,0x02,0x42,0x66,0x10,0xc1,0x00,0xa9,0xce,0xc5,0x7b,0xa9,0xa0,
	0x94,0xcd,0xa5,0xb6,0x6d,0x8c,0x40,0x9c,0xc4,0x35,0x9d,0x24,0xcb,0x32,0xc2,0x04,
	0x19,0xb7,0x19,0x73,0xc5,0x82,0x4e,0x0d,0x01,0x04,0xa7,0x11,0x2a,0x8b,0x22,0x21,

	0x38,0x7b,0xa9,0x7f,0x47,0x6f,0xc3,0x06,0x0f,0x9c,0x1f,0x36,0x1b,0xae,0x27,0x01,
	0x92,0x1d,0x1a,0x64,0x1e,0x9d,0x1f,0xc3,0x4e,0x23,0xb1,0xb8,0xcb,0x7c,0x66,0x1c,
	0x34,0x1f,0x1b,0x0b,0xc2,0x3b,0x84,0x9a,0xc9,0xc9,0x43,0xb3,0xcd,0x0e,0xca,0xa8,
	0x27,0x75,0x9b,0x72,0x19,0x01,0x07,0x1d,0x82,0xbb,0x8f,0x10,0x03,0x16,0x59,0x09,
	0x7b,0xad,0x7b,0x9b,0x26,0xcb,0xca,0x04,0x2f,0x5f,0x13,0xbf,0xa8,0x07,0x03,0x37,
	0xc1,0x95,0x02,0x82,0xb9,0x69,0x04,0x45,0x3e,0x0c,0x85,0x16,0x17,0x11,0xac,0x2f,
	0x5a,0x85,0xc3,0x52,0x1c,0x15,0x59,0x73,0x9a,0x49,0x79,0x20,0xbe,0x19,0x89,0x95,
	0x3c,0xaf,0x13,0x6c,0xac,0xcc,0x19,0x8f,0x93,0xbd,0xa4,0x1e,0x38,0x60,0x23,0x51,
	0x13,0xc9,0x3e,0x56,0x53,0x08,0x1e,0xc1,0x1b,0xc3,0xb9,0xac,0xb8,0x4a,0x95,0x50,
	0x54,0x39,0xc9,0x32,0x8e,0x73,0xbb,0x91,0xb4,0xc9,0x43,0x17,0x17,0x33,0x3b,0x46,
	0x63,0x21,0x08,0x05,0x58,0x0a,0xc8,0xcb,0x20,0x74,0x14,0x0f,0xae,0x22,0xba,0x3a,
	0xad,0xa4,0xa9,0x71,0x70,0x24,0x14,0xcd,0xc6,0x17,0x9e,0x0f,0x03,0x18,0x0a,0xc5,
	0xa9,0x66,0x20,0x3f,0x9f,0x54,0x13,0xc7,0x18,0x1b,0xcc,0x02,0xc9,0x1d,0xa3,0x7f,
	0x93,0xb3,0x05,0x56,0x08,0xcd,0xa2,0x22,0x80,0x43,0x0e,0x87,0x95,0x1d,0x65,0x16,
	0x15,0xc2,0x39,0x02,0x94,0x0f,0x3f,0x45,0x14,0x87,0x21,0x3d,0x3b,0x7b,0x86,0x11,
	0x4f,0x0f,0x04,0x99,0xbe,0xc1,0x61,0x15,0x60,0x07,0x02,0x06,0x4f,0x63,0xb2,0x7e,
	0x53,0x1f,0x71,0x1b,0x1b,0x36,0x64,0xba,0x8f,0x3e,0x3e,0x01,0x6b,0x5f,0x18,0x40,
	0x2b,0xc4,0x6d,0x35,0x44,0x1d,0x81,0x20,0x8d,0x73,0xcc,0x85,0x1e,0x13,0x06,0xb3,
	0x61,0x09,0xc8,0xaf,0xb0,0x7e,0x2a,0xa6,0x4a,0x5b,0x13,0x21,0x0d,0xcf,0x11,0x8e,
	0x2c,0x09,0xb7,0x74,0x39,0x15,0x3d,0x61,0xcb,0xc7,0x49,0x0b,0x2c,0x97,0x1c,0x3d,
	0x53,0xb1,0x5e,0x95,0x1c,0x92,0x7e,0x1e,0xc0,0x4b,0x26,0x14,0xc6,0x20,0xcf,0x43,
	0x52,0x9a,0xb8,0xce,0x12,0xc4,0x1c,0x56,0x2c,0x47,0x5b,0x0b,0x9a,0x3c,0x53,0x7a,
	0x83,0x20,0x78,0x69,0x77,0x70,0x11,0x6f,0xc0,0x05,0xc6,0x02,0xcc,0x70,0x0a,0x60,
	0x90,0x03,0x9c,0x19,0xba,0x74,0x75,0xc5,0xcb,0x03,0x0b,0x51,0x1e,0x4a,0x46,0x3e,
	0xcc,0x73,0xc9,0x09,0x9a,0x18,0x07,0xaa,0x0a,0x9e,0x50,0x12,0xa8,0xac,0x1d,0xa7,
	0x17,0x1f,0x4a,0x75,0xa4,0x13,0xad,0xcb,0xb7,0x7e,0x14,0x0f,0xcf,0xc0,0x8f,0x12,
	0xb6,0x86,0x1e,0xc4,0x1d,0x9a,0x07,0x7a,0x50,0x18,0x81,0x04,0x7b,0xa7,0xb9,0xce,
	0xa5,0x27,0x0d,0x85,0x17,0x9a,0x21,0xce,0x29,0x18,0x52,0x0b,0x2b,0xa9,0x96,0x0a,
	0x6b,0x60,0x0f,0x64,0x15,0x13,0x4c,0xc0,0xcc,0x80,0x6e,0x33,0x37,0x82,0x62,0x76,
	0x07,0x30,0x11,0xad,0x86,0x45,0x5e,0x9a,0x14,0xbd,0xb6,0x87,0x18,0x2a,0x13,0x03,
	0x59,0x4a,0x11,0x27,0xca,0xb9,0xcc,0xa7,0x0c,0x16,0xb8,0x77,0x9f,0xaf,0x90,0xa8,
	0x35,0x16,0xc2,0x0f,0xb7,0xaf,0x73,0x64,0x0e,0xca,0x53,0x4f,0x17,0x07,0xc0,0x0f,
	0x4c,0x17,0xc8,0xbc,0xbb,0xbc,0x1a,0x7c,0x89,0x4d,0x8a,0x39,0x4a,0x18,0xa2,0x6f,
	0x0d,0x9c,0x09,0x84,0x0b,0xc8,0xce,0x2a,0xc4,0x12,0x64,0x97,0x5b,0xaa,0xc1,0xa1,
	0x7c,0xa8,0x5c,0x18,0x48,0xc9,0xc0,0x4b,0x6a,0x1b,0x8c,0x63,0xc2,0x6e,0x48,0x0a,
	0x0e,0x4d,0x5f,0x8f,0x3e,0xb2,0x63,0x69,0x96,0x16,0x21,0x0f,0x4c,0x4d,0x33,0x35,
	0x65,0x18,0x8f,0x60,0x97,0xb7,0x56,0x9c,0x01,0x32,0x2d,0x80,0x01,0x5d,0x76,0xa0,
	0x00,0xb9,0x44,0x4e,0x17,0x96,0x0f,0x4a,0x4e,0x9d,0x4e,0x1f,0xb2,0x7b,0x05,0x6e,
	0x04,0xa2,0x25,0x9e,0x2e,0xaa,0xca,0x2b,0x1d,0x49,0x5f,0x88,0x2b,0xc3,0x59,0x9d,
	0xcd,0xc7,0x9a,0x4d,0x65,0x51,0x1d,0xa8,0xa6,0x68,0x5d,0x51,0x4d,0x9a,0x9f,0x26,
	0x28,0xa7,0x67,0xa8,0x05,0x15,0xc6,0x51,0xbc,0xcc,0x6c,0x88,0x0e,0x68,0x8c,0x14,
	0xca,0x3b,0xa0,0x0c,0xbd,0x11,0x6c,0xc9,0x76,0x07,0x07,0x1f,0xbf,0x79,0x8e,0x04,
	0x54,0xc0,0xaa,0xbd,0x96,0x16,0xcc,0xcd,0xc1,0x25,0xb1,0x80,0xc3,0x82,0x9d,0x30,
	0x15,0x58,0x50,0x24,0x10,0xc0,0x19,0x5a,0x5f,0xc5,0x9d,0x0f,0xcd,0x75,0x51,0x08,
	0x0f,0xbc,0x62,0x09,0x6b,0xc9,0xc4,0x81,0x4d,0xc6,0x46,0x63,0xcc,0xbb,0x5a,0x5b,
	0x37,0x34,0x7b,0xcc,0x68,0x9f,0x03,0x33,0x0f,0x1c,0x67,0x97,0x0f,0xcf,0xc0,0x4b,
	0x1a,0x80,0xcc,0x0a,0x63,0x6d,0x1f,0x1e,0x92,0x29,0x08,0x39,0x08,0x2f,0x19,0x39,
	0x7c,0x98,0x9d,0x51,0x15,0x1c,0x81,0x97,0x36,0x2d,0x58,0x4f,0x14,0xa0,0x13,0x09,
	0xae,0x11,0x19,0xcf,0x45,0x12,0x9d,0x14,0x2b,0x40,0x20,0x77,0x6e,0x17,0x8a,0x71,
	0xbd,0x81,0x13,0x71,0x55,0x02,0xa4,0x83,0xba,0x67,0xa4,0x87,0x5f,0x3c,0x67,0xcc,
	0x08,0x26,0x49,0x52,0x76,0x17,0x1d,0x85,0x12,0x51,0x1f,0x33,0x0f,0xb6,0xa6,0x01,
	0x77,0x24,0x14,0x3c,0x0c,0x38,0x91,0x0a,0x70,0x77,0x6a,0x0b,0xbc,0x78,0x62,0x0c,
	0xc8,0xad,0x84,0x0f,0x3a,0x1e,0x97,0x07,0x3f,0x1b,0x88,0x04,0x35,0x0f,0x00,0x8d,
	0x11,0x3d,0x3e,0x0a,0x00,0x08,0x07,0xc4,0x2b,0xc3,0x26,0x0f,0xc4,0x16,0xa5,0x8b,
	0xc3,0xba,0x21,0x3f,0xcc,0x8f,0x41,0x9e,0x2f,0x07,0x7f,0x12,0x05,0x19,0x9b,0x40,
	0x14,0x62,0x05,0xc9,0x06,0x12,0x5a,0xbe,0xca,0x45,0x2b,0x51,0x57,0x5e,0xb2,0x9d,
	0x3b,0x39,0x30,0xc9,0x06,0x9b,0x55,0x21,0x25,0xcf,0x69,0x02,0x19,0x67,0x1d,0xab,
	0x7e,0xb3,0x8c,0x3f,0xac,0xb0,0x96,0x41,0x8a,0x66,0x44,0x0b,0xcd,0x0e,0xc7,0x3b,
	0x1b,0x55,0x1a,0x0a,0x3e,0x00,0x5d,0x0f,0x0a,0x1a,0x03,0x14,0x40,0xc6,0x0c,0x8b,
	0x18,0x75,0x1b,0x0f,0x07,0x5a,0x50,0x5a,0x28,0x7d,0x0b,0xa1,0x1e,0x4e,0x58,0x52,
	0x3e,0x5c,0x3c,0x3e,0x78,0x49,0x07,0xb6,0x61,0x70,0x21,0x11,0x84,0x1b,0xc5,0x7c,
	0xb8,0x7e,0x52,0xcc,0xcf,0x35,0x29,0x1a,0x7e,0x91,0x85,0x65,0x30,0x74,0x57,0x97,
	0x4c,0x1b,0xcb,0x5f,0x4c,0xcb,0x5a,0x55,0x14,0xc6,0x6b,0xcb,0x0e,0x0c,0x6f,0xc6,
	0xc8,0x1e,0x6b,0x0a,0x6a,0xc0,0x9d,0x4c,0x44,0x26,0x4f,0x0d,0x19,0x0e,0xbe,0x3f,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};


static UINT8 key_0054[0x2000] =
{
	/* data */
	0x15,0x3c,0xc1,0x0f,0x3f,0x9b,0x91,0x9b,0xc5,0xa2,0x1e,0xae,0x46,0x27,0xc7,0x73,
	0x59,0xbd,0xbf,0x10,0xc6,0x0f,0xb9,0x3b,0x63,0x91,0x02,0x17,0x15,0x84,0x20,0xc1,
	0x76,0xc1,0x01,0xb9,0x4f,0xae,0x5b,0x3a,0x25,0x2b,0x0b,0x36,0xb0,0x94,0x22,0xb5,
	0x9c,0x1e,0x60,0x0f,0xc0,0xa8,0x10,0x78,0x50,0x15,0xb6,0x31,0x94,0x37,0x79,0x1a,
	0xcd,0x3e,0x78,0x21,0x3a,0x17,0x46,0x39,0x1e,0x8f,0xc9,0x50,0x4a,0x92,0x6f,0xbf,
	0x1e,0x74,0xcd,0x77,0x9f,0xaf,0xa6,0xc6,0x8e,0x35,0x86,0xb2,0x13,0x85,0x0c,0x6a,
	0xc2,0x6e,0x14,0x0d,0x0f,0x07,0x14,0x67,0xc2,0x62,0x04,0x5c,0x02,0x7f,0xcc,0xbe,
	0x02,0x2f,0x5b,0x17,0x3f,0x7e,0x2c,0x17,0x97,0x01,0xc7,0xa6,0xc9,0xc3,0x64,0x2c,
	0x3c,0xae,0x6e,0x6e,0x89,0x1a,0x11,0x82,0x1b,0xc9,0xcf,0xb4,0x47,0x53,0x57,0x7c,
	0xb6,0x94,0xce,0x63,0x87,0xbe,0x39,0x07,0x86,0x07,0x12,0x19,0x09,0x53,0x12,0x91,
	0x77,0x81,0xcf,0x8b,0xc3,0x44,0x08,0xcb,0x0d,0xbf,0x03,0x0d,0x5b,0x8d,0x18,0xc8,
	0x01,0xb8,0xcb,0x02,0x74,0x53,0x0d,0xc1,0x1a,0x0b,0x1d,0xb4,0x2c,0xa4,0xa6,0x45,
	0xa5,0x2e,0x56,0x1a,0x6a,0xc6,0x9d,0x54,0xa0,0xc8,0xb8,0x96,0x89,0x4e,0xbe,0x0c,
	0x02,0xb4,0x9a,0x30,0x0c,0xc4,0x5c,0x87,0x0d,0x3e,0x45,0x1f,0x7f,0x70,0x15,0x30,
	0x0a,0x28,0x40,0x11,0x9f,0x79,0x87,0x22,0x0f,0xa3,0xaa,0x9c,0xb0,0xcc,0x7a,0x8d,
	0x69,0x16,0xcb,0xb2,0xc4,0x8a,0xa0,0xad,0x83,0x7b,0x2a,0x1b,0x19,0x94,0x74,0xca,
	0x1f,0xc5,0xca,0x7e,0x70,0x8e,0xc7,0xc5,0x9a,0x56,0x73,0xad,0x1d,0x36,0xc3,0x1f,
	0x4a,0xc0,0x87,0x47,0xb2,0xcf,0x5e,0xc4,0xb4,0x6b,0xca,0x90,0x1e,0x05,0x0e,0x76,
	0x7e,0x3a,0xac,0x54,0xaf,0x44,0x89,0x13,0x77,0xa3,0xa9,0x98,0x0f,0x96,0x09,0x07,
	0x02,0x68,0xba,0x4b,0xc7,0xb4,0x98,0xaf,0x1e,0x85,0x1c,0x2c,0x77,0xc3,0x5c,0x55,
	0x00,0xa1,0x6e,0x60,0x1f,0xc9,0x1e,0x64,0x55,0x8b,0xc2,0xcd,0x67,0x7d,0xca,0x15,
	0x9e,0x27,0x1c,0x4f,0x64,0xc8,0x13,0x10,0x91,0xad,0xa0,0xa5,0x31,0x8b,0x91,0x52,
	0x0c,0x6d,0x9d,0x98,0x0b,0xc0,0xb6,0x9e,0x1e,0x4f,0x7b,0xb6,0x45,0x4d,0x01,0x12,
	0x31,0xbd,0x84,0x36,0xc7,0x53,0x69,0xb4,0x54,0x40,0x17,0x78,0x2a,0x64,0xc2,0x6e,
	0xc5,0x02,0x7a,0xb6,0xab,0xc2,0x2b,0x28,0xcf,0x0c,0x09,0x29,0xc0,0x2e,0x14,0x67,
	0x08,0xbc,0x2c,0xcc,0xcf,0x17,0x4f,0xab,0x8e,0x9e,0xc1,0x80,0x8f,0x62,0x4b,0x0a,
	0x0f,0xbb,0xc3,0x2c,0x7b,0x34,0xbf,0x7f,0x1d,0x25,0x1b,0x14,0x6b,0x50,0x06,0x30,
	0x98,0x28,0x1d,0xa0,0x06,0x6c,0x45,0x04,0xc8,0xac,0xa7,0xcc,0x4d,0x0c,0xb0,0xaf,
	0x12,0x99,0x17,0xcf,0x9a,0x92,0x79,0xb3,0xb1,0x41,0x03,0x65,0x4c,0xb4,0xc5,0x69,
	0xa3,0x65,0x07,0x03,0x51,0x5a,0x62,0x5c,0x30,0x04,0xc0,0x1a,0x0f,0x5b,0xbb,0x1a,
	0x82,0x0b,0x2a,0xcd,0x64,0x2d,0xc3,0xca,0x48,0x11,0xb8,0x9e,0x0d,0x70,0x10,0x10,
	0x90,0x5a,0x17,0x25,0x84,0x48,0x06,0x85,0x5a,0x6b,0x14,0x42,0x16,0x89,0x03,0x64,
	0x95,0x03,0x7d,0x31,0x74,0xc3,0x36,0x0e,0x13,0x69,0x44,0x76,0x19,0x65,0x10,0x1c,
	0x43,0x5b,0x63,0x01,0x3f,0xc9,0x19,0x09,0x2e,0x65,0x25,0x9a,0x47,0x74,0xbd,0x47,
	0xae,0x17,0xa5,0xc9,0xa5,0x8b,0xcc,0x20,0x9c,0x1c,0x15,0x79,0xc8,0xc5,0x86,0x11,
	0xc9,0xae,0x0e,0x13,0x17,0x73,0xb2,0x28,0x1a,0x42,0xa6,0x7f,0x83,0xb0,0x37,0x7c,
	0x83,0x06,0x38,0x28,0x47,0x99,0x0f,0x84,0x14,0x87,0x74,0x79,0xaa,0x30,0xb6,0x19,
	0xc6,0xc8,0xb4,0x57,0x12,0x2b,0x74,0xc7,0x28,0x46,0x2e,0x4f,0x63,0x0e,0x5d,0x02,
	0x1f,0x0c,0x11,0x61,0x6b,0x1c,0x72,0x0c,0xa3,0x09,0x5f,0x2b,0x80,0x67,0x7d,0x9d,
	0x0a,0x9c,0x8c,0x1d,0xc9,0x4c,0x1a,0x3a,0x01,0x1a,0x6d,0x16,0xc3,0x6e,0x33,0x13,
	0x15,0x98,0x26,0x69,0x6a,0x94,0x02,0xae,0xcd,0xa0,0x04,0x56,0x7e,0x12,0x6a,0xc8,
	0x34,0xcb,0x11,0x85,0xc2,0x71,0x15,0x16,0x64,0x05,0x28,0x0b,0xc6,0x5a,0xce,0x19,
	0xc7,0x3b,0xa8,0x90,0x85,0xb1,0xba,0xb1,0x59,0x91,0x9d,0x13,0xcd,0x4e,0x0a,0x52,
	0x1b,0x4b,0x7d,0x6f,0x1e,0x01,0x35,0x1b,0x7c,0x1c,0x29,0x12,0xcf,0x55,0x6a,0xcd,
	0x3a,0x0d,0x1e,0x14,0x07,0xce,0x71,0x01,0x62,0x10,0xb3,0x01,0xce,0x3d,0xc3,0x68,
	0x27,0x92,0x3d,0xcc,0x26,0xa6,0x3d,0x02,0x88,0x7a,0x00,0xa8,0x37,0xc0,0x28,0x4d,
	0x8d,0xca,0x86,0xb8,0x47,0x0f,0x8c,0x32,0x15,0x63,0x4a,0x12,0xaf,0x1f,0x1e,0x47,
	0xbb,0x09,0x23,0xce,0x13,0xc3,0x63,0x90,0xb5,0x58,0x19,0xc8,0x04,0x13,0x10,0x49,
	0x32,0x17,0xbe,0xcb,0x0e,0xc3,0x83,0x4e,0x50,0x66,0x2d,0x12,0x19,0x3e,0x66,0x23,
	0xc1,0xc0,0xbb,0x44,0x06,0x13,0x56,0xab,0xc2,0x85,0x36,0xcb,0x52,0x14,0x7f,0x5f,
	0xac,0x47,0x0b,0x89,0x01,0x84,0x49,0x69,0xb9,0xb2,0xce,0xcf,0x59,0x8e,0xc5,0x10,
	0xcb,0xce,0x04,0x60,0x40,0xca,0xb8,0x77,0xcc,0x93,0xc2,0x6f,0x92,0x1e,0xb4,0x27,
	0x32,0x17,0x1a,0x60,0xb9,0x05,0x04,0x8d,0xa3,0x44,0x83,0xa0,0x1e,0xca,0x68,0x93,
	0xbf,0x15,0x9e,0x9b,0xb9,0x9f,0x77,0xa0,0xcd,0x09,0x05,0x0d,0xa0,0xc2,0xc5,0x6e,
	0x8d,0xa2,0xc2,0x5a,0x9b,0x80,0x60,0x2c,0x02,0x1a,0x4e,0x6f,0x79,0xc9,0xa6,0x41,
	0x25,0x7e,0x10,0x6b,0xb9,0x94,0x76,0x1e,0x6a,0x03,0xc2,0xc6,0x35,0xc6,0x65,0x9c,
	0x3a,0x50,0xc9,0x66,0x48,0x4c,0x96,0xb6,0x9d,0x0f,0x9d,0xb7,0x77,0x59,0x4f,0xa7,
	0x0f,0x2b,0xbd,0xba,0x1d,0x14,0x09,0x81,0xcd,0x95,0x94,0x53,0x17,0xb3,0x5a,0x05,
	0xc7,0x11,0x38,0xc6,0x25,0x0d,0x02,0x38,0x26,0x19,0x17,0x38,0xad,0x13,0x95,0x36,
	0xac,0x0b,0x0c,0x53,0x9b,0xab,0x35,0xcb,0x9b,0x24,0x16,0x69,0x8c,0xac,0x7b,0x15,
	0xc4,0x31,0x86,0x4a,0x7f,0x05,0x23,0xb0,0x37,0xac,0x72,0x7e,0x94,0x8d,0x0c,0x1e,
	0xbf,0x10,0x80,0x04,0xbb,0x3a,0x1d,0x42,0x88,0x34,0xc9,0x1d,0xa5,0x1f,0xc2,0x1f,
	0x18,0x6b,0x97,0x1c,0x05,0x20,0x3f,0x62,0x9c,0x1f,0xce,0x89,0x19,0xc9,0xa9,0xcc,
	0x79,0xac,0xc9,0x21,0x49,0x0e,0xa6,0x23,0xb8,0xa0,0x02,0x43,0x5a,0xc8,0x06,0x83,
	0x4a,0x32,0x48,0x3e,0x55,0x26,0xc9,0x08,0x02,0xb3,0x9a,0x52,0x95,0x51,0x51,0xc7,
	0xcd,0xa1,0x71,0x05,0x93,0x61,0x1f,0x6d,0x48,0x81,0x98,0xcd,0xa7,0x1a,0x08,0xae,
	0x14,0xa4,0x41,0x82,0xc8,0xc1,0x24,0x35,0xa1,0x1d,0x4b,0x96,0x18,0x57,0x0b,0x4d,
	0xb6,0xc1,0x59,0xc7,0xb8,0xa1,0x8d,0x96,0xac,0x39,0x96,0x57,0x8e,0x7c,0x31,0x03,
	0x9f,0x48,0x1f,0x60,0x1e,0x00,0x11,0x6f,0x2a,0x84,0xb6,0x34,0x44,0xc8,0x14,0x4c,
	0x39,0x67,0x9d,0x68,0xc7,0xb9,0xca,0x32,0x3b,0x71,0x8a,0xca,0xb7,0x8d,0x1e,0x0b,
	0x33,0xce,0x03,0x48,0xc6,0x67,0x11,0xb7,0x2a,0x88,0x04,0x70,0x7a,0x0d,0x67,0x75,
	0x26,0x0d,0xc2,0x0b,0x10,0x0d,0x8f,0x4b,0x82,0x59,0x4c,0x26,0xc6,0x37,0xba,0x99,
	0x36,0x35,0xa4,0x5f,0x67,0x38,0x05,0xa1,0x22,0x14,0x49,0x17,0x88,0x5d,0x09,0x5a,
	0xa5,0xc2,0x0e,0x67,0x92,0x08,0x60,0xbf,0x3c,0x63,0x73,0xcc,0x99,0x02,0xc3,0x87,
	0xaf,0x3d,0xa8,0x24,0x35,0x94,0x0f,0x37,0x37,0x45,0x93,0x14,0x64,0xc3,0x2e,0xca,
	0x30,0x79,0xc4,0x9c,0xc3,0xc0,0x17,0x9a,0x20,0x25,0x3c,0x1a,0x39,0x59,0x7a,0x05,
	0x51,0x4c,0xb2,0x74,0x2e,0x1b,0x1c,0x68,0xcb,0xbe,0x2b,0x39,0xb3,0x8a,0x96,0x55,
	0x4f,0x08,0x30,0x14,0x34,0xb9,0x90,0x65,0x03,0x16,0xb9,0x34,0x29,0x42,0x9d,0xc9,
	0x5b,0x31,0x34,0x86,0x26,0x24,0x07,0x19,0xcf,0x1e,0x06,0xb2,0x68,0x97,0x08,0x03,
	0x81,0x62,0x01,0x6a,0x4b,0xbd,0x1d,0x77,0x0e,0xaf,0xb3,0x06,0x9b,0xc3,0x6e,0x9c,
	0x4e,0x02,0x75,0x9b,0x98,0x3f,0x42,0x0e,0x53,0x18,0xb8,0xc7,0x88,0x1e,0xb8,0x80,
	0x1d,0x76,0xc8,0x61,0x30,0xa4,0x58,0x50,0x0a,0x3b,0x08,0x6c,0x11,0x1b,0xac,0x57,
	0x1e,0xae,0x30,0x3b,0xb1,0x93,0x0a,0x1a,0x07,0x36,0x3c,0x9a,0x09,0x4e,0x59,0x12,
	0x7a,0x8f,0x33,0x8f,0x1c,0x5a,0x48,0x1c,0x36,0x86,0x48,0x2b,0xb1,0x62,0x7e,0x8c,
	0x64,0x74,0x61,0x2d,0x1a,0x23,0x0f,0x45,0xce,0x09,0x0a,0x40,0xad,0x8f,0xce,0xab,
	0xc5,0x0f,0x86,0x05,0x25,0x6a,0xcf,0x4b,0xb3,0x77,0xce,0x3c,0xca,0x8d,0x9f,0xad,
	0xb6,0x8b,0x35,0x59,0x1e,0x82,0x14,0x23,0x74,0x71,0xad,0x6e,0x04,0xac,0x1c,0x9c,
	0x76,0x86,0x1c,0x88,0x31,0x93,0x0a,0x96,0xcf,0xab,0x4e,0xb5,0x4b,0x05,0x6a,0x06,
	0x39,0xc9,0x92,0xa3,0x8d,0xaa,0x41,0x83,0x80,0xad,0xb7,0x05,0x3f,0x23,0x68,0x24,
	0x11,0x02,0xc9,0x8d,0x0e,0x60,0xb8,0x11,0x89,0x80,0x49,0x0b,0xac,0x4d,0xc3,0x6d,
	0xa7,0x58,0x0a,0x30,0x6a,0x2c,0x3e,0xc7,0xc4,0xcf,0x39,0xb7,0xb7,0x73,0x97,0xb2,
	0xaf,0x32,0x2c,0xc8,0x0d,0x01,0x03,0x1a,0x9c,0x35,0x1e,0x7b,0x0c,0x4e,0x89,0x1f,
	0x20,0xb6,0x88,0x99,0x31,0xbb,0x6d,0xca,0x4e,0xc3,0x5f,0x02,0xc8,0x20,0x2d,0x7f,
	0x2a,0x9c,0x48,0x06,0x33,0x6b,0x5e,0x8d,0x0a,0x33,0x8b,0x1d,0x61,0xc9,0x0f,0x02,
	0x28,0x42,0x33,0x1e,0x39,0x03,0x76,0x54,0xc0,0x58,0x17,0xa4,0x42,0x03,0xab,0x40,
	0x0a,0x87,0xb3,0x3d,0xb8,0x4f,0xb8,0x03,0x1d,0x52,0x93,0x16,0x0b,0xcc,0x17,0xba,
	0xc6,0x0a,0xc1,0x7d,0x1a,0xa1,0x77,0x6c,0xc5,0x8e,0x25,0xa6,0x2d,0x48,0xac,0x4e,
	0xcf,0xb0,0x5f,0x08,0x9d,0x14,0x0e,0xc1,0x5e,0x04,0x70,0x1d,0x8d,0x17,0xae,0x0f,
	0x16,0x64,0xb3,0x1c,0x39,0x3d,0xbe,0xab,0x1b,0xbf,0x4d,0xc0,0x9f,0x1b,0x18,0x58,
	0x12,0x89,0x6e,0x32,0x16,0x14,0x05,0x20,0x96,0x66,0x7b,0x4e,0xab,0x91,0x3a,0xa2,
	0x39,0x0b,0x04,0x15,0x07,0x09,0x9c,0xba,0x0f,0xc5,0x9f,0xab,0x25,0x0a,0x24,0x89,
	0x1d,0xb8,0x38,0x20,0x65,0x8b,0x3c,0xb8,0x2f,0xbc,0x0b,0xaf,0x2f,0x1b,0x2e,0x27,
	0xa0,0x5b,0x95,0x41,0x8f,0xc4,0x0e,0x20,0x9e,0xcf,0x46,0x4e,0x1c,0x8c,0x72,0xbd,
	0xc0,0x19,0x0b,0xc8,0x02,0x84,0x11,0xad,0x07,0xb8,0xb5,0x60,0x04,0x05,0xa9,0xc7,
	0x74,0x81,0x45,0x16,0x8e,0x30,0x59,0x19,0x4e,0x99,0xcc,0xc7,0x4f,0x6c,0x00,0x6d,
	0x08,0x0c,0x17,0xcd,0x08,0x9d,0x77,0x71,0x08,0xca,0x55,0x51,0x0e,0xc7,0x72,0x0f,
	0xa2,0x8f,0x9c,0x3e,0xb9,0x09,0x24,0x19,0xcd,0x15,0x0d,0x5a,0xc1,0x9d,0x12,0xb7,
	0xc1,0x25,0x7f,0x10,0x2f,0xcf,0x1f,0xce,0x05,0xc9,0x19,0x17,0xc4,0x38,0xc8,0xb9,
	0x05,0x11,0x11,0x0c,0x63,0xa9,0x6e,0x39,0xc5,0x5c,0xba,0x62,0xa3,0x6d,0x08,0x1c,
	0x72,0xb4,0xac,0x44,0xb2,0x3b,0xb7,0xa3,0x6d,0x6b,0xa0,0xc7,0x2d,0x8f,0x8e,0x52,
	0x8e,0x17,0x29,0xb3,0x2b,0x41,0x38,0x7b,0x83,0x14,0x69,0x44,0x6f,0x46,0x69,0x16,
	0x0d,0xa6,0xcc,0x0d,0x0e,0x3c,0x2c,0xa0,0x93,0x12,0x3c,0x88,0x2c,0x20,0xa5,0x26,
	0x2f,0x18,0x0e,0xc5,0x91,0xca,0x42,0xbb,0xc5,0x3c,0x9e,0xc3,0xb6,0x09,0x3c,0x25,
	0x5b,0x88,0x77,0x10,0xbe,0x52,0x2b,0xa8,0x9e,0x0c,0x5f,0x06,0xac,0x44,0xba,0x10,
	0x07,0x4d,0x9b,0x19,0x08,0x21,0x92,0x40,0xae,0x19,0x50,0xa2,0x09,0x55,0x04,0xa3,
	0xb3,0xc1,0x74,0xa2,0x16,0x7c,0x8e,0xa4,0x2b,0x0d,0x13,0x0b,0x30,0x95,0x9a,0x56,
	0x1a,0x23,0xc5,0xbd,0x7c,0xa6,0xc9,0x04,0x86,0x34,0x93,0x39,0x46,0x8a,0xbd,0x8a,
	0xc8,0x0e,0x0f,0x9d,0x5d,0x52,0xa6,0x0a,0x79,0xb6,0x89,0x9a,0x8f,0x67,0x10,0xac,
	0x1e,0x91,0x19,0xc4,0x21,0x85,0x18,0x57,0x25,0x52,0xc4,0x64,0x6d,0x10,0x36,0x65,
	0xc2,0x62,0xb0,0x24,0x4e,0x04,0x14,0x83,0x8c,0x9d,0x5c,0x1d,0x28,0x8f,0x2a,0xc1,
	0x5c,0xc9,0x58,0x14,0xb2,0x7d,0x05,0x30,0x0b,0xca,0x0b,0x1a,0x9d,0x1e,0x12,0x1e,
	0x8f,0x05,0x4f,0x88,0xae,0x1e,0x5c,0x25,0x2a,0xcd,0x73,0x0b,0xc7,0xb7,0xa6,0x32,
	0x3d,0xc4,0x06,0x90,0x97,0xa9,0x13,0x88,0xb7,0x2e,0x22,0x1a,0x6c,0x0c,0x3c,0xcb,
	0x6a,0x60,0x27,0x74,0x66,0xc6,0xa8,0x01,0x71,0xcb,0x09,0x15,0x01,0xb2,0xc2,0x19,
	0x04,0x1a,0x13,0xc7,0x2f,0x19,0x2c,0x06,0x9b,0x1f,0xaa,0x64,0x8c,0x8d,0x41,0x6f,
	0x70,0x5d,0xa9,0xc6,0xcc,0xaa,0x84,0xb4,0xc5,0x7c,0xbe,0x1b,0x09,0xa3,0xc0,0x02,
	0xca,0x94,0xcb,0x93,0x1b,0x13,0x3a,0x07,0x80,0x62,0x0d,0x0a,0x17,0x9d,0x02,0x7b,
	0x06,0x33,0xb1,0x53,0xcc,0x0a,0x13,0x5f,0x3e,0x35,0x03,0x93,0x5f,0xc3,0x38,0x85,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0x18, 0  ,0x56, 0  ,0x1b, 0  ,0xb1, 0  ,0x30, 0  ,0x11, 0  ,0x36, 0  ,0x9b, 0  ,
	0x55, 0  ,0x92, 0  ,0xbf, 0  ,0x09, 0  ,0xb7, 0  ,0x1d, 0  ,0x31, 0  ,0x50, 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,0xca, 0  ,
	0x93, 0  ,0xc0, 0  ,0x65, 0  ,0x78, 0  ,0x3b, 0  ,0x60, 0  ,0xca, 0  ,0xca, 0  ,
	0x61, 0  ,0xc9, 0  ,0xb7, 0  ,0x09, 0  ,0xcb, 0  ,0x94, 0  ,0xb5, 0  ,0x13, 0  ,
	0x48, 0  ,0x25, 0  ,0xac, 0  ,0xb7, 0  ,0xa8, 0  ,0x1e, 0  ,0x5e, 0  ,0xcf, 0  ,
	0x88, 0  ,0x90, 0  ,0x09, 0  ,0x03, 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,0x1b, 0  ,
	0x25, 0  ,0x2f, 0  ,0x1a, 0  ,0xcf, 0  ,0x61, 0  ,0x84, 0  ,0x9b, 0  ,0x83, 0  ,
	0x22, 0  ,0x61, 0  ,0xb4, 0  ,0x0c, 0  ,0x11, 0  ,0x57, 0  ,0x63, 0  ,0x65, 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,0xac, 0  ,
	0xc7, 0  ,0x17, 0  ,0x05, 0  ,0x04, 0  ,0x8f, 0  ,0x04, 0  ,0xb4, 0  ,0x7a, 0  ,
	0x85, 0  ,0x05, 0  ,0x00, 0  ,0xcb, 0  ,0x3d, 0  ,0xb9, 0  ,0xc7, 0  ,0x1c, 0  ,
	0xc8, 0  ,0x90, 0  ,0x07, 0  ,0xaa, 0  ,0xcf, 0  ,0x7c, 0  ,0x8b, 0  ,0xc0, 0  ,
	0xbc, 0  ,0x15, 0  ,0x48, 0  ,0xc1, 0  ,0x1e, 0  ,0x65, 0  ,0x10, 0  ,0xc8, 0  ,
	0x49, 0  ,0x40, 0  ,0x0f, 0  ,0x53, 0  ,0xa7, 0  ,0x88, 0  ,0x16, 0  ,0x6a, 0  ,
	0x1a, 0  ,0x70, 0  ,0xcb, 0  ,0x77, 0  ,0xa3, 0  ,0x4d, 0  ,0x0b, 0  ,0x4d, 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	0x2f, 0  ,0xc2, 0  ,0x9a, 0  ,0x1e, 0  , 0  , 0  , 0  , 0  , 0  , 0  ,0x6e, 0  ,
	0x60, 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,0x70, 0  ,
	0x3d, 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,0x9b, 0  ,
	0x86, 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,0x12, 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};


static UINT8 key_0066[0x2000] =
{
	/* data */
	0xc3,0x18,0x2d,0xa8,0xc2,0x56,0x5d,0x11,0x5f,0x0b,0x09,0x81,0x53,0xc9,0x48,0x27,
	0x10,0xcf,0x07,0xb5,0x86,0x84,0x53,0xac,0xbe,0x2b,0x10,0x64,0x5e,0x1d,0x69,0x6e,
	0x79,0xc3,0x85,0x79,0x7a,0x62,0xc4,0x54,0x4a,0xc0,0x4c,0x67,0x8e,0x04,0x13,0x0f,
	0x22,0x54,0x98,0x6a,0x8f,0x2c,0x0a,0x79,0x61,0x11,0xc5,0xbd,0x93,0x35,0x09,0x8a,
	0x87,0x1e,0xa9,0x60,0x80,0x00,0x20,0x38,0x9b,0x0f,0xbc,0x11,0x7c,0x93,0x98,0x0c,
	0x6b,0x76,0x05,0x73,0x46,0xbb,0x6c,0x7c,0x2e,0xb3,0xc2,0x45,0x98,0x9d,0x5a,0xa0,
	0x52,0x09,0x59,0xad,0x55,0xa2,0x3b,0x33,0x63,0x9c,0x27,0x4a,0xc8,0x39,0x15,0x1b,
	0xa6,0x31,0x02,0xb0,0x1b,0xa8,0x0e,0xab,0x1c,0x3a,0x28,0x2d,0xcf,0x08,0xba,0xc2,
	0x4f,0xcc,0x7d,0xc1,0x02,0xa0,0x4b,0xc3,0x85,0x3e,0x28,0x1d,0x11,0x06,0x1e,0xb7,
	0xb0,0x76,0x5e,0x2d,0x2f,0x19,0xcf,0x5b,0x6f,0x92,0x34,0x12,0xc7,0x35,0x9b,0x37,
	0x51,0x68,0x25,0x22,0x9b,0x42,0xa1,0xa7,0x0e,0xca,0x21,0x08,0xb1,0x66,0x85,0x01,
	0xa7,0x02,0x22,0x09,0x7c,0x94,0x25,0x9e,0x43,0xb6,0x4c,0x28,0x0a,0x3e,0x06,0x68,
	0x96,0x1c,0x0b,0x7b,0x90,0x15,0x62,0x07,0xb8,0x45,0x4b,0x0f,0x2a,0x02,0xc6,0x23,
	0x20,0x43,0x10,0xad,0x4c,0x1f,0x1f,0x7c,0x5e,0x04,0x1d,0x06,0x09,0xc0,0x03,0xc0,
	0x3c,0xac,0x18,0xa0,0x13,0x73,0x8a,0x10,0x08,0x0e,0xa5,0xbd,0x73,0x55,0xbc,0x13,
	0x98,0xc0,0x5f,0xb8,0xc5,0x46,0x1c,0xc2,0x72,0x7e,0x61,0xb4,0xc6,0x4d,0x14,0xae,
	0xb9,0x8c,0x1b,0xb5,0xce,0x13,0xa6,0x15,0x78,0x7c,0x6e,0xcd,0x43,0xc4,0x94,0x56,
	0x57,0x22,0x02,0x71,0x71,0xcd,0x0f,0x19,0x00,0x67,0x8d,0xce,0x59,0x1a,0x17,0x0e,
	0xa8,0x36,0x0a,0x01,0x13,0xa7,0xc2,0xc8,0x58,0x0e,0x98,0x79,0x40,0x30,0x74,0x83,
	0xc2,0x3b,0x11,0x2e,0x1b,0x3f,0x0a,0x18,0xa6,0x49,0xc0,0x72,0x84,0x7c,0x29,0x26,
	0x3d,0xc6,0x87,0x21,0x77,0x91,0x99,0x7a,0x45,0x09,0xa6,0x95,0x7e,0x3b,0xbd,0x98,
	0x01,0xa4,0x80,0x0c,0xc4,0x87,0x97,0x0f,0x06,0xb5,0xc3,0x3b,0x3c,0x1e,0xc5,0x1a,
	0x70,0xc5,0xc8,0x17,0xcb,0x00,0x06,0x90,0x65,0x96,0xcc,0x5a,0x14,0xa6,0x5b,0xc5,
	0x16,0x4a,0xcd,0x53,0x1b,0xab,0xcc,0xb9,0x8b,0x99,0x8e,0x5f,0xbb,0x0d,0xb3,0xcf,
	0x5a,0x08,0xbc,0x40,0x42,0xc1,0x07,0x5d,0x26,0x0b,0x9b,0x51,0xa6,0x42,0x3d,0x16,
	0xcd,0x05,0xca,0x8c,0x9c,0x59,0xbe,0xc9,0x90,0x23,0x1d,0x5c,0xae,0x86,0x76,0x94,
	0xc5,0x8d,0x86,0x83,0x76,0xcc,0x14,0x14,0xad,0x46,0x12,0x37,0x16,0x11,0x26,0x3a,
	0x10,0x73,0x81,0x7c,0x97,0x61,0x86,0x65,0x03,0xcd,0x0a,0x99,0x2f,0x0b,0x73,0x8d,
	0x61,0xb9,0x78,0x8a,0x23,0x28,0x13,0x22,0x19,0xc7,0x07,0x78,0x89,0x73,0x4c,0x06,
	0x93,0x07,0x6b,0xcc,0x0a,0x64,0x68,0x95,0xca,0x48,0x59,0x7f,0x70,0x50,0xb4,0x50,
	0x1f,0x0b,0xb9,0xc1,0xb6,0x82,0x29,0x67,0x27,0x2f,0xc4,0x80,0x17,0x0d,0xb0,0x2b,
	0xa1,0x11,0x1d,0xba,0x0c,0xbf,0x4c,0x07,0x07,0x1c,0xc3,0x4c,0xc3,0x56,0x1b,0xca,
	0x75,0x5b,0x06,0xa5,0xb0,0xc1,0xa6,0x9d,0x95,0x17,0xac,0x03,0xa7,0x6d,0xca,0x15,
	0x83,0x42,0x0c,0x08,0xac,0x45,0x59,0x77,0x9a,0x2e,0xa3,0xc1,0x3d,0x30,0x2f,0x1b,
	0x09,0x2f,0x3a,0x20,0x14,0xb3,0xb1,0x1c,0xab,0x2b,0x0e,0xc4,0x93,0x9f,0x6e,0xb3,
	0x00,0x28,0x0f,0x64,0x9c,0x2f,0x35,0xc1,0x05,0x58,0x03,0x71,0x95,0x10,0x0d,0x8c,
	0x44,0x36,0x16,0x8a,0x7e,0x7a,0x01,0xc3,0x4f,0xc1,0x24,0x03,0x00,0xc0,0x3b,0x2a,
	0xcb,0x31,0xbb,0x50,0x26,0x24,0x78,0x4a,0x06,0xa8,0x00,0x1f,0xa9,0x41,0xaf,0x95,
	0xbc,0x52,0xb6,0x5f,0x7b,0x73,0x33,0x1a,0x6d,0xb1,0x1e,0x67,0xc5,0xcd,0x49,0x6f,
	0xc8,0x4b,0xcd,0x5b,0xc8,0x5a,0xce,0x6a,0x76,0x75,0x15,0xc4,0x47,0x06,0xb8,0x1a,
	0x37,0x0d,0xc0,0xc8,0x41,0xb2,0x13,0x09,0x74,0x76,0xc1,0xc1,0xcc,0x8c,0x54,0x4b,
	0x2a,0xa3,0x75,0x65,0xc5,0x1d,0xaa,0x23,0x06,0x60,0xc2,0x1f,0x71,0x01,0x61,0xba,
	0x3e,0x7d,0x6e,0x01,0xb7,0x56,0x5e,0xc8,0x0c,0x63,0x94,0x5c,0x38,0x09,0x89,0x5e,
	0x82,0x6e,0x95,0x24,0x18,0x7d,0x1e,0xb4,0x43,0xab,0x76,0x9c,0x74,0x80,0x3e,0x1c,
	0xce,0x9d,0x39,0x02,0x99,0x20,0x83,0x51,0xa8,0xb2,0x9f,0x66,0x82,0x0a,0x7f,0x7e,
	0xb4,0x9a,0x52,0xab,0x8f,0x8f,0x24,0xa3,0xc4,0xc1,0x80,0x95,0x1c,0xc7,0x1c,0x68,
	0x45,0xca,0x41,0xc9,0x31,0x10,0x9a,0x94,0x9e,0x48,0x58,0x31,0xb6,0x4f,0xc9,0x00,
	0x10,0xa6,0x51,0x09,0xbb,0xce,0x0e,0x32,0x22,0x35,0x5d,0x0a,0xae,0xce,0x53,0x4e,
	0x0a,0xcd,0x42,0xc9,0x49,0xc5,0x3c,0x24,0x30,0x0b,0xb4,0x14,0x18,0xa5,0xa0,0x6e,
	0xc8,0x27,0x27,0x0b,0xc3,0xc5,0x8a,0x2b,0x0b,0xbc,0x8e,0x91,0x9e,0x5b,0x7c,0x9c,
	0x89,0x93,0x5e,0x06,0x12,0xbf,0xc8,0x00,0x2f,0xa1,0x72,0x2c,0x13,0x06,0x7b,0x9b,
	0x78,0x8d,0x69,0x9c,0x86,0x49,0xa4,0x1c,0x32,0x08,0x24,0x6b,0x70,0x79,0x0a,0x0d,
	0x63,0x29,0x07,0x05,0x3a,0xa8,0xcf,0x49,0x1d,0x22,0x7b,0x58,0x00,0x89,0x57,0x16,
	0xc0,0x18,0x0d,0x19,0x9f,0xcc,0x42,0x1b,0x7b,0x68,0xc6,0x05,0xc2,0x34,0x09,0x1e,
	0xcd,0x11,0x8a,0x80,0x02,0x2d,0x1c,0x16,0xa9,0x98,0x49,0xaa,0x05,0x90,0xb1,0x07,
	0xac,0xcd,0x5a,0x1a,0x4e,0x11,0x05,0x0e,0xa6,0x50,0xcb,0x4a,0x1d,0xa9,0x94,0x4f,
	0xa1,0xb7,0x01,0xc3,0xb0,0x70,0x9d,0x01,0xba,0x49,0xb7,0x79,0xc6,0x07,0x8b,0x04,
	0x0a,0x1a,0xbc,0x59,0x0f,0x02,0x7f,0x87,0xb5,0x53,0x0c,0x2a,0x18,0x09,0x0b,0x0c,
	0x13,0x2a,0x16,0xce,0xce,0x52,0xbb,0x92,0x61,0xc6,0x9b,0x70,0x6a,0xca,0x31,0x17,
	0x24,0x19,0x35,0x0c,0x2d,0xc7,0x07,0xcc,0x05,0x65,0x9d,0x31,0x0b,0x35,0x4e,0x3c,
	0x10,0x7f,0x76,0x93,0x6e,0xc5,0x1f,0xc3,0x3c,0x15,0x31,0xab,0x80,0x03,0xc3,0x3b,
	0x1a,0xcc,0x2e,0x3e,0x1d,0x04,0xa7,0xbc,0x21,0x34,0xb9,0x5d,0xb9,0x8f,0xc7,0x4e,
	0x01,0x43,0x12,0x6b,0x2b,0x6d,0x65,0xc0,0x18,0x16,0xcd,0xcb,0x4b,0x77,0x56,0xc1,
	0xc1,0x47,0xc0,0x46,0xab,0x83,0x83,0x84,0x13,0xab,0x1b,0xa5,0xcd,0x0c,0x3f,0xaa,
	0xcc,0x48,0x4b,0x01,0x49,0x1f,0x1d,0x91,0x45,0xc7,0x5e,0x25,0x04,0xc5,0x13,0xb7,
	0x96,0x94,0x45,0xa8,0xbc,0x29,0x04,0x78,0x40,0x0d,0x79,0x72,0x65,0xcd,0x95,0x65,
	0x6a,0x7e,0xce,0x4a,0x5c,0xce,0x5a,0x77,0x9e,0x14,0x03,0x1b,0x30,0x4a,0x8a,0x76,
	0x9f,0x2e,0x1a,0x65,0x71,0x01,0xcb,0xbf,0x81,0x27,0x1d,0x9a,0x97,0x0a,0xad,0x7c,
	0x90,0x10,0x22,0x3a,0x89,0x1b,0xb8,0x01,0x6c,0x81,0x8a,0x1c,0x7b,0x66,0x11,0x6f,
	0x50,0xb9,0x70,0x6c,0x3c,0xb1,0xcc,0x5b,0x8a,0x8f,0x44,0xa2,0x4c,0x1d,0x4f,0xaf,
	0x43,0xa6,0x39,0x31,0x73,0x8e,0x25,0x5c,0xc6,0x2b,0x05,0x0b,0xa4,0x23,0x12,0xb2,
	0x0b,0xaa,0x1e,0xa9,0x08,0x28,0x2a,0x3f,0xc5,0x18,0xbe,0xcc,0x51,0xc2,0x61,0xcb,
	0x10,0x12,0xa2,0x55,0xc9,0x95,0x2c,0x2a,0x0d,0x01,0x16,0x0a,0xb3,0xb2,0x66,0x40,
	0x3f,0x2d,0x09,0xc1,0x4d,0x7f,0x80,0x36,0x06,0xc9,0x27,0x89,0x35,0x47,0x74,0x37,
	0x30,0x89,0x54,0xa5,0xa3,0x1e,0xc0,0x33,0x18,0xb5,0x7a,0x95,0x15,0xa3,0x12,0x30,
	0x1d,0x6c,0x86,0x37,0x8c,0x5d,0xb4,0x5a,0x2a,0x1a,0x2c,0x16,0x74,0x84,0x08,0x1f,
	0x67,0x82,0x05,0x7e,0x13,0xba,0x5b,0x55,0x1b,0x38,0x12,0xcc,0x21,0x22,0x5d,0x04,
	0xaf,0x5a,0x0f,0x0f,0x6c,0x40,0x14,0x0d,0x16,0x1d,0xce,0x17,0xce,0x3e,0xa8,0x0c,
	0xa2,0x03,0x6f,0x9a,0x04,0x18,0x1e,0xa7,0xbf,0x6f,0x43,0xb8,0x03,0x8a,0xca,0x49,
	0xba,0xcf,0x50,0x08,0xc8,0x62,0x6e,0x71,0xb0,0xcc,0x53,0xac,0xbd,0x9c,0x0b,0xb7,
	0xc0,0x03,0xa4,0x05,0x68,0x6c,0x72,0xc7,0x5d,0xce,0x86,0x48,0x41,0x30,0x12,0x6d,
	0x6d,0xc7,0x1b,0x09,0x10,0x77,0x9d,0xc0,0x4f,0x0e,0x07,0x1e,0xaa,0x24,0x1a,0x15,
	0x03,0xa3,0xcc,0xc2,0x46,0x1e,0x8a,0x65,0x56,0x32,0x64,0x93,0xcc,0x39,0x01,0x3c,
	0x0b,0x3d,0x1a,0x0c,0xa4,0x57,0xce,0x62,0x94,0x6c,0x3b,0x34,0x2f,0xcc,0x97,0x33,
	0x6b,0x83,0x8b,0x6a,0x5b,0x1d,0xa4,0x87,0x6e,0x39,0xbf,0x8a,0x15,0xa0,0x90,0x1c,
	0xca,0x97,0x85,0x1b,0x16,0xb7,0xcd,0x39,0x3e,0x0a,0xcb,0x0e,0x60,0xcf,0xc6,0x07,
	0xc1,0x10,0x16,0x82,0x75,0x84,0xc6,0x44,0x00,0xa4,0x4d,0xcb,0x02,0x5c,0xc3,0x45,
	0x0b,0xa9,0xc2,0xbd,0x9b,0x8b,0x9e,0x49,0xb9,0x19,0xb1,0xc1,0x44,0x18,0xb8,0x56,
	0x54,0xcf,0x13,0x4b,0x34,0x1f,0x89,0x47,0xa4,0x54,0x2f,0x02,0xc3,0x11,0xc4,0x9c,
	0x8e,0x4f,0xbc,0xc3,0x82,0x21,0x0d,0x42,0xac,0x96,0x66,0x86,0xcf,0x9d,0x96,0x93,
	0x66,0xc6,0xaf,0x50,0x06,0x35,0x02,0x01,0x34,0x28,0x04,0x6f,0x91,0x6c,0x85,0x71,
	0x96,0x75,0x17,0xc3,0x1a,0x8b,0x2d,0x1f,0x6f,0x9d,0x71,0xbd,0x68,0x9a,0x21,0x2a,
	0x03,0x30,0x09,0xc9,0x13,0x68,0x99,0x6f,0x5a,0x16,0x81,0x13,0x7b,0xc2,0x1a,0x78,
	0x74,0x87,0xc4,0x5e,0x4f,0x63,0x60,0x4e,0xb0,0x4e,0x0f,0x1f,0xbd,0xcf,0xb4,0x92,
	0x3b,0x77,0x25,0x2d,0x10,0xca,0x90,0x07,0x19,0xb2,0x29,0xa5,0x01,0x0d,0xbe,0x1c,
	0xbb,0x5a,0x13,0x13,0x08,0xc9,0x5a,0xcd,0x48,0x0b,0xc0,0x69,0x4d,0x16,0xa7,0xb2,
	0xcf,0xa4,0x8f,0x87,0x07,0xa8,0x17,0xa3,0x7d,0xc4,0x05,0x93,0x54,0x1c,0x18,0xa8,
	0x5b,0x4f,0x6b,0x88,0x3c,0xa1,0xcb,0x2f,0x32,0x2d,0x0b,0x1d,0x2d,0x28,0x22,0xb1,
	0xb5,0x08,0xa9,0x29,0x1e,0xce,0x81,0x8d,0x72,0xb1,0x10,0x2a,0x1b,0x78,0x8e,0x2d,
	0x27,0xcf,0x11,0x46,0x17,0x6d,0x87,0x04,0x19,0x9c,0x52,0x24,0x02,0x9a,0x6e,0x6a,
	0x15,0xcd,0x51,0xcf,0x26,0x17,0x10,0xce,0x39,0x38,0xc1,0x23,0xb9,0x4e,0x34,0x26,
	0x68,0x5c,0x16,0xaa,0x10,0x0f,0xad,0x5f,0xab,0x87,0xb8,0x4c,0xb4,0x49,0x67,0x6f,
	0x31,0x0e,0x7d,0xb5,0x0a,0x77,0xcb,0xc7,0x57,0x7f,0xc2,0x55,0xc7,0x4d,0xc6,0x44,
	0xc4,0x76,0x66,0x69,0x05,0xca,0x59,0x16,0xba,0x0e,0x35,0x19,0xca,0xc2,0x5f,0xb6,
	0x03,0x1d,0x64,0x66,0xcb,0xcf,0xc6,0x9c,0x4a,0x55,0x38,0xa1,0x69,0x75,0xcf,0x0d,
	0xae,0x21,0x16,0x7c,0xcc,0x0f,0x6d,0x15,0x71,0xbe,0x2c,0x61,0x72,0x15,0xb3,0x48,
	0x40,0xc6,0x1c,0x73,0x86,0x42,0x3a,0x1d,0x99,0x40,0x92,0x72,0x87,0x26,0x0c,0x61,
	0x0a,0xb0,0x5d,0xa9,0x66,0x8e,0x64,0x90,0x2c,0x08,0xc4,0x8f,0x2b,0x12,0x8b,0x22,
	0x93,0x47,0xaa,0xb6,0x8d,0x7a,0x92,0x1a,0x63,0x6e,0xb0,0x88,0x4c,0xa9,0x9f,0x9f,
	0x26,0xa1,0xce,0xcb,0x90,0x87,0x08,0xcd,0x08,0x74,0x5b,0xc0,0x5f,0xc7,0x23,0x04,
	0x88,0x86,0x8c,0x5e,0x46,0x23,0xb4,0x51,0xc7,0x10,0x04,0xa4,0x47,0x1d,0xb9,0xc4,
	0x1e,0x20,0x30,0x27,0x4b,0x1a,0xac,0xc0,0x45,0x58,0x1a,0xc7,0x54,0x10,0xc7,0x57,
	0xcb,0x3e,0x26,0x32,0x1f,0xb0,0x0c,0xa7,0xa2,0x72,0xc2,0x25,0x25,0x1f,0xcd,0xcf,
	0x9a,0x29,0x1f,0x10,0xb8,0x9e,0x83,0x8c,0x4d,0x6c,0x8e,0x99,0x81,0x40,0x16,0x06,
	0xbb,0xc6,0x10,0x2d,0xa5,0x62,0x2e,0x03,0x16,0x67,0x89,0x68,0x9d,0x79,0x8e,0x96,
	0x57,0xa0,0x08,0x20,0x18,0x26,0x7b,0x60,0x65,0x1a,0x19,0x73,0x3b,0x13,0x11,0x28,
	0xaa,0xc1,0x57,0x0d,0x30,0x67,0x46,0x10,0x99,0x41,0x02,0xce,0x0c,0x19,0x09,0x8d,
	0xc6,0x54,0x0b,0x67,0x74,0xcc,0x11,0xc8,0x36,0x1d,0x0a,0xc3,0x01,0x9a,0x90,0x12,
	0x3f,0x08,0x02,0xad,0x8a,0x57,0xae,0x11,0x82,0xb5,0x13,0xa8,0xc7,0x44,0x0e,0x58,
	0x01,0x11,0x1e,0xa4,0x4e,0xc5,0x5c,0x0d,0xad,0x86,0x51,0xa5,0xb3,0x15,0xc9,0xb2,
	0x60,0x8f,0x15,0xbe,0x57,0xb3,0x65,0xc8,0x13,0x9b,0x14,0x1a,0x0e,0xb8,0x4f,0x1b,
	0x12,0x63,0x97,0xb7,0x45,0x1c,0x38,0x0c,0x1d,0x1f,0x1c,0x03,0x38,0x02,0xc0,0xc0,
	0x4c,0xb9,0x80,0x71,0xc8,0x89,0x60,0x76,0xc4,0x23,0x07,0x26,0x09,0x27,0x1c,0x3f,
	0xcd,0x13,0xc2,0x11,0x75,0x8f,0x23,0x1f,0x27,0x58,0x3e,0x04,0x63,0x66,0x81,0x72,
	0xcf,0x0f,0xc9,0x3e,0x05,0x23,0xa9,0x90,0x17,0xc9,0x39,0x0e,0xc2,0x3c,0x2c,0x0d,
	0x14,0xa3,0xb8,0x33,0x36,0xbd,0x4b,0xbd,0x9f,0xc9,0x58,0x15,0x5d,0x06,0x7b,0x29,
	0x7d,0x75,0xca,0x0c,0x02,0xc3,0xc5,0x55,0x6b,0x48,0xcf,0xcb,0x59,0xce,0x50,0xa9,
	0x93,0x93,0x94,0x03,0xa9,0x0b,0xa7,0xc7,0x1c,0x3d,0xae,0xc6,0x5e,0x55,0x15,0x57,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,

	/* opcodes */
	0x34,0xaf,0x3f,0x9f,0x81,0x20,0x92,0x31,0x67,0xc1,0xc1,0xcc,0x0f,0x67,0x07,0x69,
	0x8b,0x10,0x5e,0x6a,0x86,0x3d,0x34,0x19,0x2c,0x03,0x01,0xcf,0x38,0x85,0x2f,0x42,
	0x18,0x9d,0x0f,0x71,0xc2,0xc6,0x38,0x7a,0x6e,0x4c,0x18,0xa6,0x2d,0x15,0xa1,0xbe,
	0xa5,0x8b,0xc9,0xa5,0xac,0xb8,0x9e,0x25,0x37,0x80,0x2b,0xaf,0xca,0x0c,0x1b,0x07,
	0x47,0x25,0xc6,0xa8,0xc5,0x5f,0x02,0xa5,0xb1,0x19,0x8f,0x73,0xc5,0x5c,0x01,0xcc,
	0xcd,0x16,0x6b,0xb0,0xc0,0x08,0x51,0x53,0xcd,0x07,0x72,0x19,0xb2,0xc5,0xc6,0x04,
	0xa9,0xaa,0x0d,0xbd,0x23,0x7f,0xb6,0xb2,0x49,0x7d,0x86,0x3a,0xc6,0xcf,0x29,0x68,
	0x39,0x01,0x30,0x7b,0x24,0x74,0x44,0x4e,0xb9,0xc0,0x42,0x37,0x88,0x52,0x1d,0x13,
	0x64,0x58,0x0a,0x60,0x13,0x6a,0x06,0x77,0x14,0x45,0x5b,0x5e,0x9d,0x00,0x7e,0x86,
	0x32,0x8a,0xb3,0x91,0x0a,0x68,0x36,0x97,0xc9,0x41,0x5b,0x15,0x19,0x0e,0x16,0x6d,
	0x60,0xaa,0x2d,0x5e,0x48,0x11,0x81,0x28,0xbf,0xca,0x5d,0x1a,0x89,0x5c,0xcd,0x4a,
	0x20,0xa0,0x54,0x47,0xcd,0x27,0x69,0x65,0x06,0x92,0x50,0x00,0x61,0xae,0xc9,0x09,
	0x65,0x0a,0x47,0x01,0xb2,0xc8,0x54,0xc4,0x26,0x6c,0x2b,0x5b,0x02,0x41,0x0e,0xa0,
	0xcc,0x14,0x03,0xad,0xce,0x1d,0xb8,0x0f,0x62,0x3a,0x36,0x07,0x4d,0x0a,0x98,0x46,
	0x51,0x26,0x18,0x7b,0x2b,0x17,0x17,0xb2,0x2d,0x0c,0x15,0x0e,0x01,0x80,0x0b,0x80,
	0x57,0x7a,0x10,0x68,0x1b,0xbd,0xcc,0x18,0x00,0x06,0x79,0x73,0xbd,0x24,0x72,0x1b,
	0xc6,0x9c,0x2c,0x62,0x93,0x23,0x14,0x9a,0xbc,0xba,0xb1,0x70,0x98,0x2a,0x1c,0x66,
	0x63,0xce,0x13,0x71,0x8c,0x1b,0x64,0x1d,0xa4,0xb2,0xbe,0x9b,0x26,0x9e,0xc8,0x3d,
	0x3c,0x47,0x0a,0xb5,0xb5,0x9b,0x07,0x11,0x08,0xaf,0xc1,0x8c,0x30,0x12,0x1f,0x06,
	0x6a,0x4b,0x02,0x09,0x1b,0x65,0x86,0x94,0x31,0x06,0xc6,0xa5,0x3f,0x59,0xa2,0xcb,
	0x86,0x44,0x19,0x49,0x13,0x40,0x02,0x10,0x64,0x2e,0x80,0xbc,0xc6,0xb2,0x5a,0x43,
	0x56,0x98,0xcd,0x50,0xab,0xc1,0xc9,0xac,0x3a,0x01,0x64,0xc7,0xba,0x44,0x73,0xc6,
	0x09,0x78,0xc0,0x04,0x82,0xcd,0xc3,0x07,0x0e,0x71,0x8b,0x44,0x57,0x16,0x8f,0x12,
	0xb4,0x93,0x88,0x1f,0x9f,0x08,0x0e,0xce,0xa7,0xcc,0x96,0x29,0x1c,0x64,0x28,0x8f,
	0x1e,0x37,0x87,0x38,0x13,0x77,0x8a,0x63,0xc3,0xc9,0xca,0x2c,0x7f,0x05,0x7d,0x81,
	0x29,0x00,0x72,0x3f,0x27,0x8d,0x0f,0x34,0x43,0x03,0xcd,0x20,0x64,0x27,0x56,0x1e,
	0x87,0x0d,0x8e,0xce,0xc0,0x30,0x6e,0x99,0xce,0x46,0x15,0x35,0x66,0xc2,0xaa,0xc8,
	0x93,0xc1,0xc2,0xcb,0xaa,0x96,0x1c,0x1c,0x7b,0x23,0x1a,0x4a,0x1e,0x19,0x43,0x45,
	0x18,0xbd,0xcf,0xb2,0xc3,0xb1,0xc2,0xa7,0x0b,0x87,0x02,0xc9,0x48,0x03,0xbd,0xc1,
	0xb1,0x63,0xa4,0xcc,0x46,0x5b,0x1b,0x47,0x11,0x89,0x0f,0xa4,0xc7,0xbd,0x2b,0x0e,
	0xc5,0x0f,0xa9,0x8a,0x02,0xa6,0xa0,0xc7,0x8e,0x2f,0x30,0xbb,0xb4,0x21,0x70,0x21,
	0x17,0x03,0x63,0x8d,0x6c,0xc4,0x5a,0xaf,0x42,0x48,0x82,0xc0,0x1f,0x05,0x60,0x4c,
	0x69,0x19,0x15,0x7e,0x04,0x6f,0x2b,0x0f,0x0f,0x14,0x97,0x2b,0x8b,0x3d,0x13,0x92,
	0xa3,0x28,0x0e,0x79,0x60,0x8d,0x64,0xcf,0xc7,0x1f,0x7a,0x0b,0x65,0xb7,0x8e,0x1d,
	0xcb,0x27,0x04,0x00,0x7a,0x3a,0x30,0xab,0xc2,0x49,0x75,0x91,0x56,0x59,0x48,0x13,
	0x01,0x48,0x45,0x51,0x1c,0x7d,0x61,0x14,0x77,0x4c,0x06,0x9e,0xc5,0xcb,0xbe,0x7d,
	0x08,0x5b,0x07,0xa6,0xc0,0x48,0x5c,0x8d,0x0d,0x31,0x0b,0xb5,0xc7,0x18,0x05,0xce,
	0x3b,0x4b,0x1e,0xcc,0xba,0xac,0x09,0x8b,0x32,0x8d,0x55,0x0b,0x08,0x80,0x44,0x4d,
	0x9f,0x58,0x7f,0x21,0x43,0x55,0xa4,0x37,0x0e,0x6a,0x08,0x17,0x6b,0x3e,0x67,0xc7,
	0x72,0x39,0x6c,0x2c,0xad,0xbd,0x4e,0x12,0xb7,0x61,0x16,0xaf,0x8f,0x9b,0x2e,0xbf,
	0x94,0x36,0x9b,0x28,0x88,0x29,0x90,0xa8,0xaa,0xa3,0x1d,0x82,0x22,0x0e,0x62,0x12,
	0x4a,0x05,0x9c,0x94,0x3e,0x7c,0x1b,0x01,0xa2,0xaa,0x91,0x8d,0x96,0xce,0x25,0x36,
	0x4d,0x75,0xa3,0xa7,0x93,0x15,0x76,0x46,0x0e,0xb0,0x86,0x17,0xb5,0x09,0xb1,0x7e,
	0x41,0xb3,0xbe,0x09,0x6d,0x3d,0x2d,0x88,0x04,0xb9,0xc8,0x35,0x53,0x01,0xc7,0x2d,
	0xc4,0xbe,0xc7,0x55,0x10,0xb3,0x16,0x70,0x26,0x77,0xaa,0xc0,0xa2,0xc0,0x41,0x14,
	0x90,0xcf,0x52,0x0a,0xc9,0x51,0xcb,0x20,0x6a,0x7c,0xcb,0xae,0xc4,0x02,0xbb,0xba,
	0x70,0xc2,0x39,0x77,0xc5,0xc5,0x55,0x75,0x9e,0x91,0xc0,0xc7,0x14,0x95,0x14,0xa0,
	0x3a,0x92,0x3e,0x85,0x58,0x18,0xc2,0xc8,0xc4,0x2f,0x31,0x58,0x6c,0x32,0x85,0x08,
	0x18,0x64,0x20,0x01,0x7f,0x90,0x06,0x4f,0x47,0x5c,0x34,0x02,0x66,0x8c,0x38,0x33,
	0x02,0x9b,0x27,0x85,0x2e,0x8f,0x57,0x55,0x59,0x03,0x70,0x1c,0x10,0x79,0x68,0xbe,
	0x94,0x42,0x42,0x03,0x8b,0x93,0xcc,0x4c,0x03,0x72,0xca,0xc1,0xc4,0x28,0xb2,0xc0,
	0xc7,0xc5,0x2d,0x0e,0x1a,0x6f,0x88,0x08,0x48,0x69,0xbc,0x5f,0x1b,0x0e,0xad,0xcd,
	0xa4,0xc1,0xa1,0xc0,0xc2,0x2e,0x78,0x14,0x4f,0x00,0x55,0xa9,0xb4,0xa5,0x02,0x05,
	0xb9,0x5a,0x0f,0x0d,0x45,0x6a,0x81,0x2e,0x15,0x47,0xad,0x31,0x08,0xc7,0x3c,0x1e,
	0x80,0x10,0x05,0x11,0xcb,0x96,0x27,0x13,0xad,0xa0,0x98,0x0d,0x9a,0x5d,0x01,0x16,
	0x87,0x19,0xcc,0xc0,0x0a,0x5e,0x14,0x1e,0x6b,0xc6,0x2e,0x76,0x0d,0xce,0x61,0x0f,
	0x7a,0x9b,0x29,0x12,0x33,0x19,0x0d,0x06,0x64,0x21,0x83,0x37,0x15,0x6b,0xc8,0x32,
	0x69,0x6d,0x09,0x97,0x60,0xb4,0xcf,0x09,0x7e,0x2e,0x6d,0xa5,0x84,0x0f,0xc3,0x0c,
	0x02,0x12,0x72,0x30,0x07,0x0a,0xbb,0xcd,0x71,0x38,0x04,0x4d,0x10,0x01,0x03,0x04,
	0x1b,0x4d,0x1e,0x8c,0x8c,0x39,0x7f,0xca,0xb1,0x84,0xcd,0xb4,0xa8,0x8e,0x58,0x1f,
	0x55,0x11,0x5c,0x04,0x5e,0x95,0x0f,0x8a,0x0d,0xa7,0xcf,0x58,0x03,0x5c,0x33,0x57,
	0x18,0xbb,0xaa,0xc5,0xbe,0x93,0x17,0x97,0x57,0x1d,0x58,0x77,0xc0,0x0b,0x97,0x44,
	0x12,0x8a,0x49,0x41,0x15,0x0c,0x65,0x72,0x50,0x5d,0x63,0x34,0x63,0xc5,0x89,0x33,
	0x09,0x26,0x1a,0xa9,0x4c,0xb7,0xa7,0x9c,0x10,0x1e,0x87,0x83,0x36,0xab,0x3d,0x8d,
	0x91,0x22,0x80,0x23,0x77,0xcb,0xcb,0xc6,0x1b,0x77,0x13,0x79,0x9b,0x04,0x40,0x76,
	0x96,0x2f,0x36,0x09,0x2e,0x17,0x15,0xc1,0x3a,0x95,0x2d,0x54,0x0c,0x8f,0x1b,0x6d,
	0xcc,0xc8,0x3a,0x6a,0x72,0x5a,0x0c,0xa4,0x3f,0x05,0xa5,0xbc,0xa7,0x9b,0xc7,0xa7,
	0xa8,0xba,0x8c,0x37,0x35,0x90,0x29,0xab,0xc4,0x1c,0x0b,0x13,0x59,0x37,0xcc,0xaa,
	0xcb,0x49,0x12,0xa7,0xb5,0x09,0x9f,0x6f,0xcf,0x42,0x15,0xc2,0xc3,0x02,0x7b,0xb2,
	0xce,0x18,0x47,0x45,0xc7,0x13,0x62,0x09,0xb6,0xcf,0xcc,0x14,0xad,0xae,0x19,0xbf,
	0x21,0x63,0xb4,0xb6,0x57,0x61,0x96,0x28,0xcc,0xc5,0x3b,0x74,0x2b,0x15,0x32,0x67,
	0x26,0x64,0x52,0x58,0xbd,0xca,0x54,0x35,0x98,0x4c,0x0d,0x03,0x78,0x46,0x1a,0x7c,
	0x03,0x76,0x16,0x6b,0x00,0x5b,0x4d,0x40,0x8f,0x10,0x6e,0x96,0x20,0x9a,0xb1,0x83,
	0x18,0x1a,0x74,0x24,0x85,0xc7,0x5f,0x4d,0x05,0x09,0x1e,0x02,0x7d,0x7c,0xae,0x3f,
	0x40,0x5e,0x01,0x91,0x2a,0xbb,0xc0,0x4b,0x0e,0x99,0x42,0xc7,0x5c,0x22,0xa2,0x4a,
	0x59,0xc7,0x25,0x79,0x75,0x16,0x80,0x4e,0x10,0x71,0xac,0xc7,0x1d,0x75,0x1a,0x59,
	0x15,0xb6,0xc2,0x4a,0xce,0x34,0x70,0x29,0x4d,0x12,0x5f,0x1e,0xa2,0xc6,0x00,0x17,
	0xaf,0xc4,0x0d,0xba,0x1b,0x7e,0x28,0x24,0x13,0x53,0x1a,0x8a,0x50,0x47,0x34,0x0c,
	0x67,0x29,0x07,0x07,0xb6,0x3f,0x1c,0x05,0x1e,0x15,0x90,0x1f,0x90,0x41,0x6a,0x04,
	0x74,0x0b,0xbf,0xc2,0x0c,0x10,0x16,0x65,0x6f,0xbf,0x26,0x62,0x0b,0xcc,0x8e,0x2e,
	0x7e,0x81,0x21,0x00,0x88,0xb8,0xbe,0xb5,0x60,0x8a,0x38,0x7a,0x73,0xc0,0x03,0x6d,
	0x9c,0x0b,0x78,0x0d,0xa0,0xb6,0xbc,0x89,0x34,0x8c,0xc2,0x2f,0x3e,0x59,0x1a,0xb7,
	0xb7,0x89,0x13,0x01,0x18,0xab,0xcf,0x9c,0x32,0x06,0x0f,0x16,0x76,0x55,0x12,0x1d,
	0x0b,0x75,0x96,0x86,0x23,0x16,0xcc,0xa7,0x3d,0x4f,0xa6,0xc5,0x96,0x52,0x09,0x57,
	0x03,0x56,0x12,0x04,0x78,0x3c,0x90,0xb8,0xc8,0xb6,0x44,0x5d,0x48,0x8a,0xc3,0x4e,
	0xa9,0xcb,0xc3,0xa8,0x28,0x15,0x78,0xcd,0xbe,0x52,0x6f,0xcc,0x1d,0x68,0xce,0x14,
	0x92,0xc3,0xc9,0x13,0x1e,0x6d,0x9b,0x52,0x41,0x02,0x9f,0x06,0xb0,0x81,0x98,0x0f,
	0x8d,0x18,0x1e,0xc4,0xa3,0xc6,0x84,0x3b,0x08,0x78,0x2a,0x9f,0x0a,0x35,0x97,0x3a,
	0x03,0x6b,0x9a,0x73,0xcd,0xc3,0xc4,0x2e,0x63,0x11,0x61,0x91,0x3b,0x10,0x62,0x3d,
	0x25,0x9d,0x1b,0x36,0x5d,0x17,0xc7,0x22,0x78,0x25,0x48,0x0a,0x97,0x19,0x9e,0xc0,
	0xca,0x32,0x72,0x8b,0xc4,0x50,0x05,0x27,0x7a,0xcc,0xae,0xc2,0x81,0xcf,0xcc,0xc5,
	0xae,0x84,0x67,0x21,0x0e,0x5c,0x0a,0x09,0x5d,0x5b,0x0c,0xbf,0xc1,0xb6,0xc9,0xb5,
	0xcc,0xa3,0x1f,0x97,0x12,0xc3,0x5e,0x17,0xbf,0xcf,0xb5,0x73,0xa0,0xc2,0x50,0x4d,
	0x0b,0x59,0x01,0x99,0x1b,0xa0,0xc9,0xbf,0x29,0x1e,0xcf,0x1b,0xad,0x9a,0x12,0xa4,
	0xa2,0xcd,0x9e,0x2d,0x32,0xb9,0xb0,0x33,0x60,0x33,0x07,0x17,0x73,0x9d,0x70,0xca,
	0x44,0xab,0x54,0x5e,0x18,0x92,0xce,0x0f,0x11,0x7c,0x5a,0x79,0x09,0x05,0x6e,0x14,
	0x7f,0x29,0x1b,0x1b,0x00,0x85,0x29,0x9b,0x2f,0x03,0x80,0xa1,0x2a,0x1e,0x65,0x7c,
	0x9d,0x78,0xc5,0xcd,0x0f,0x6a,0x1f,0x75,0xb3,0x9e,0x0d,0xc5,0x25,0x14,0x10,0x6a,
	0x28,0x32,0xa9,0xc8,0x57,0x69,0x83,0x48,0x4f,0x5e,0x03,0x15,0x5e,0x5b,0x47,0x61,
	0x71,0x00,0x6b,0x5a,0x16,0x8c,0xcf,0xc1,0xbc,0x61,0x18,0x4d,0x13,0xa4,0xca,0x5e,
	0x42,0x9d,0x19,0x23,0x1f,0xb7,0xcd,0x0c,0x11,0xc0,0x39,0x55,0x0a,0xc2,0xbe,0xa8,
	0x1d,0x9b,0x20,0x9d,0x43,0x1f,0x18,0x90,0x52,0x53,0x8d,0x46,0x63,0x33,0x5d,0x43,
	0xa0,0x35,0x1e,0x76,0x18,0x07,0x7b,0x2c,0x77,0xcd,0x62,0x2b,0x70,0x2e,0xaf,0xbf,
	0x58,0x06,0xb3,0x71,0x02,0xab,0x9f,0x89,0x3c,0xbb,0x86,0x24,0x89,0x2a,0x98,0x3b,
	0x82,0xaa,0xae,0xa1,0x0d,0x92,0x30,0x1e,0x7e,0x06,0x5c,0x11,0x8e,0x86,0x2c,0x6c,
	0x0b,0x15,0xa6,0xae,0x83,0x9d,0x84,0xc0,0x37,0x24,0x53,0x69,0xa1,0xa3,0x81,0x05,
	0x66,0x50,0x1e,0xb2,0x96,0x07,0xb7,0x1d,0xb5,0x6e,0x5f,0xb1,0xbc,0x1d,0x7d,0x2f,
	0x3f,0x98,0x14,0xbd,0xc2,0x27,0x45,0x15,0xc9,0x3f,0xca,0xbc,0xcd,0x43,0x04,0xb1,
	0x02,0x60,0x34,0x6b,0xae,0xca,0xa6,0xce,0x5f,0x00,0x82,0xc5,0x4c,0x1a,0xc3,0x47,
	0xc5,0x22,0x76,0x6c,0xc1,0xac,0xca,0x12,0xb9,0xbe,0x60,0xc8,0x2b,0x6b,0xcb,0xcb,
	0x43,0x69,0x8c,0x83,0xce,0xcd,0x00,0x87,0x00,0xa2,0x28,0x80,0x2c,0x95,0x46,0x0c,
	0xc8,0xc2,0xce,0x2d,0x23,0x46,0x70,0x20,0x95,0x18,0x0c,0x78,0x22,0x15,0x63,0x82,
	0x16,0x51,0x59,0x42,0x36,0x12,0x7a,0x9c,0x3a,0x31,0x12,0x89,0x25,0x18,0x95,0x3c,
	0x9f,0x41,0x43,0x4f,0x17,0x60,0x04,0x65,0x74,0xbc,0x86,0x54,0x54,0x17,0x9b,0x81,
	0xc2,0x5a,0x17,0x18,0x62,0xc4,0xcb,0xce,0x2a,0xb6,0xca,0xc9,0xcf,0x3f,0x1e,0x0e,
	0x7f,0x98,0x18,0x5e,0x79,0xb8,0x49,0x0b,0x1e,0xaf,0xc7,0xa0,0xcf,0xa5,0xca,0xcc,
	0x3c,0x68,0x00,0x51,0x10,0x43,0xad,0xb0,0xa7,0x12,0x11,0xbd,0x44,0x1b,0x19,0x5b,
	0x76,0x91,0x3c,0x05,0x59,0xaf,0x23,0x18,0xc9,0x3e,0x0a,0x90,0x04,0x11,0x01,0xc1,
	0x84,0x25,0x03,0xaf,0xa2,0x8a,0x19,0x88,0x4b,0x15,0x02,0x97,0x09,0xc2,0xce,0x1a,
	0x40,0x00,0x0a,0x7b,0xcc,0x3c,0x66,0x19,0xc4,0x71,0x1b,0x6a,0x89,0x3b,0x06,0x31,
	0x09,0x19,0x16,0x78,0x33,0x93,0x35,0x05,0x7b,0xc2,0x20,0x79,0x7d,0x1d,0x85,0x7c,
	0xb0,0xc5,0x1d,0x6e,0x3c,0x7d,0xa7,0x94,0x1b,0xcd,0x1c,0x12,0x06,0x62,0x32,0x13,
	0x1a,0xb9,0xc3,0x6d,0x3a,0x14,0x53,0x04,0x15,0x17,0x14,0x0b,0x53,0x0a,0x9c,0x9c,
	0x2b,0x63,0xc0,0xb5,0x94,0xc7,0xb0,0xaa,0x9e,0x46,0x0f,0x43,0x01,0x42,0x14,0x40,
	0x87,0x1b,0x9a,0x19,0xa3,0xc5,0x46,0x17,0x42,0x31,0x41,0x0c,0xb9,0xae,0xcf,0xbc,
	0x81,0x07,0x85,0x41,0x0d,0x46,0x6b,0xce,0x1f,0x85,0x52,0x06,0x9a,0x57,0x5f,0x05,
	0x1c,0x75,0x62,0x4e,0x4b,0x73,0x36,0x73,0xcb,0x99,0x31,0x1d,0x34,0x0e,0xad,0x5a,
	0xb3,0xa3,0x8e,0x04,0x0a,0x97,0x93,0x24,0xa9,0x2f,0x9d,0x83,0x30,0x90,0x21,0x6b,
	0xc5,0xc5,0xc8,0x0b,0x6b,0x03,/*0x07,*/0x65,0x89,0x14,0x56,0x66,0x84,0x2d,0x24,0x1d,0x3c,
	0x03,/*0x07,*/0x05,0xcb,0x28,0x87,0x3f,0x4a,0x1c,0x9f,0x0b,0x7d,0xc6,0xc2,0x28,0x76,0x62,
	0x44,0x1c,0xa0,0x3d,0x11,0xa7,0xb8,0xa3,0x89,0xcd,0xa3,0xaa,0xbe,0x9c,0x35,0x27,
	0x82,0x3b,0xa9,0xce,0x1f,0x03,/*0x07,*/0x4f,0x35,0xc2,0xae,0xc1,0x57,0x02,/*0x06,*/0xa3,0xb7,0x1d,
	0x8d,0x7f,0xc1,0x54,0x05,0xc8,0xc9,0x12,0x67,0xb6,0xc4,0x08,/*0x0c,*/0x59,0x5b,0xc9,0x03,/*0x07,*/
	0x7e,0x1d,0xb4,0xc1,0xc2,0x00,0xaf,0xac,0x09,/*0x0d,*/0xbb,0x33,0x73,0xb0,0xb4,0x41,0x71,
	0x84,0x2a,0xc2,0xcb,0x39,0x64,0x29,0x05,0x20,0x77,0x34,0x78,0x4c,0x46,0xbf,0xc4,
	0x4a,0x27,0x8a,0x5a,0x19,0x17,0x68,0x50,0x0e,0x6c,0x17,0x66,0x02,/*0x06,*/0x7b,0x10,0x4d,
	0x53,0x56,0x9f,0x04,0x72,0x84,0x22,0x88,0xb5,0x93,0x08,/*0x0c,*/0x0e,0x64,0x26,0x95,0xcd,
	0x49,0x53,0x11,0x1d,0x0a,0x12,0x61,0x6c,0xac,0x3d,0x56,0x40,0x15,0x83,0x38,0xb9,
	0xce,0x55,0x1e,0x8b,0x54,0xc9,0x42,0x30,0xa6,0x5c,0x4f,0xc9,0x37,0x65,0x69,0x02,/*0x06,*/
	0x90,0x58,0x04,0x6d,0xa8,0xcd,0x09,/*0x0d,*/0x69,0x0e,0x4f,0x05,0xb4,0xcc,0x5c,0xc0,0x36,
	0x60,0x3b,0x53,0x02,/*0x06,*/0x49,0x0a,0xa6,0xc8,0x10,0x03,/*0x07,*/0xab,0xca,0x19,0xbe,0x0b,0x6e,
	0x2a,0x26,0x03,/*0x07,*/0x45,0x0e,0x9a,0x4e,0x59,0x36,0x1c,0x77,0x3b,0x13,0x13,0xb4,0x3d,
	0x11,0x0a,0x05,0x82,0x0f,0x82,0x5f,0x76,0x14,0x64,0x1f,0xbb,0xc8,0x1c,0x04,0x02,/*0x06,*/
	0x75,0x7f,0xbb,0x34,0x7e,0x1f,0xc2,0x9e,0x3c,0x6e,0x91,0x33,0x10,0x98,0xba,0xbc,
	0xb7,0x7c,0x9a,0x3a,0x18,0x6a,0x6f,0xca,0x17,0x7d,0x8e,0x1f,0x68,0x19,0xa2,0xb4,
	0xb8,0x99,0x36,0x9c,0xcc,0x2d,0x2c,0x4f,0x0e,0xb3,0xb3,0x99,0x03,/*0x07,*/0x15,0x08,/*0x0c,*/0xa9,
	0xc5,0x8e,0x20,0x16,0x1b,0x02,/*0x06,*/0x66,0x43,0x02,/*0x06,*/0x09,/*0x0d,*/0x1f,0x69,0x84,0x96,0x21,0x02,/*0x06,*/
	0xc2,0xa3,0x2f,0x51,0xa4,0xcf,0x84,0x4c,0x1d,0x41,0x17,0x48,0x02,/*0x06,*/0x14,0x68,0x3e,
	0x82,0xba,0xc2,0xb4,0x52,0x4b,0x5e,0x9a,0xc9,0x58,0xad,0xc5,0xcd,0xaa,0x2a,0x05,
	0x68,0xc3,0xbc,0x4c,0x7f,0xc2,0x09,/*0x0d,*/0x74,0xc4,0x00,0x80,0xc9,0xc7,0x03,/*0x07,*/0x0a,0x7d,
	0x89,0x4c,0x5f,0x12,0x8d,0x16,0xb2,0x91,0x8a,0x1b,0x9d,0x08,/*0x0c,*/0x0a,0xca,0xa1,0xc8,
	0x94,0x39,0x18,0x68,0x38,0x8d,0x1a,0x27,0x85,0x28,0x17,0x7b,0x88,0x6f,0xc7,0xcd,
	0xce,0x3c,0x73,0x01,0x71,0x83,0x39,0x04,0x7e,0x2f,0x37,0x8f,0x0b,0x24,0x4b,0x03,/*0x07,*/
	0xc9,0x30,0x68,0x37,0x5e,0x1a,0x85,0x09,/*0x0d,*/0x8c,0xca,0xc4,0x20,0x62,0x9b,0xca,0x4e,
	0x11,0x25,0x6a,0xc6,0xac,0xcc,0x91,0xc5,0xc6,0xcf,0xac,0x94,0x18,0x18,0x77,0x33,
	0x1e,0x42,0x1a,0x1d,0x4b,0x4d,0x1c,0xbb,0xcb,0xb4,0xc7,0xb7,0xc6,0xa1,0x0f,0x85,
	0x02,/*0x06,*/0xcd,0x40,0x03,/*0x07,*/0xbb,0xc5,0xb7,0x6f,0xa2,0xc8,0x4e,0x53,0x1f,0x4f,0x15,0x8b,
	0x0b,0xa2,0xc3,0xbb,0x3b,0x0a,0xc1,0x0b,0xaf,0x88,0x02,/*0x06,*/0xa0,0xa6,0xc3,0x8c,0x3f,
	0x20,0xbd,0xb2,0x31,0x7c,0x31,0x13,0x03,/*0x07,*/0x6f,0x8f,0x60,0xc0,0x52,0xa9,0x4a,0x40,
	0x08,/*0x0c,*/0x80,0xc4,0x1b,0x01,0x6c,0x44,0x65,0x1d,0x11,0x72,0x00,0x63,0x3b,0x0b,0x0b,
	0x10,0x95,0x3b,0x89,0x2d,0x17,0x90,0xa5,0x38,0x0a,0x75,0x6c,0x8f,0x68,0xcb,0xc3,
	0x1b,0x76,0x0f,0x69,0xb1,0x8c,0x19,0xcf,0x37,0x00,0x04,0x76,0x2a,0x20,0xad,0xc6,
	0x41,0x79,0x93,0x5e,0x51,0x40,0x17,0x05,0x40,0x4d,0x59,0x18,0x71,0x6d,0x10,0x7b,
	0x44,0x02,/*0x06,*/0x9c,0xc1,0xcf,0xb8,0x71,0x08,/*0x0c,*/0x53,0x03,/*0x07,*/0xa0,0xc4,0x40,0x54,0x8f,0x09,/*0x0d,*/
	0x21,0x0f,0xb3,0xc3,0x1c,0x01,0xca,0x2b,0x43,0x1a,0xc8,0xbc,0xaa,0x09,/*0x0d,*/0x89,0x22,
	0x8f,0x5d,0x0f,0x08,/*0x0c,*/0x82,0x4c,0x45,0x9d,0x50,0x73,0x31,0x4b,0x5d,0xa2,0x27,0x0a,
	0x66,0x08,/*0x0c,*/0x13,0x67,0x2e,0x6b,0xc3,0x7e,0x29,0x60,0x3c,0xab,0xbb,0x46,0x16,0xb1,
	0x6d,0x12,0xa9,0x8d,0x99,0x3e,0xb9,0x96,0x26,0x99,0x38,0x8a,0x39,0x92,0xae,0xac,
	0xa5,0x19,0x80,0x32,0x0a,0x6e,0x16,0x42,0x01,0x9e,0x96,0x2e,0x70,0x1f,0x05,0xa4,
	0xac,0x93,0x8f,0x94,0xca,0x35,0x26,0x45,0x79,0xa5,0xa1,0x91,0x11,0x7a,0x4e,0x0a,
	0xb6,0x84,0x13,0xb3,0x09,/*0x0d,*/0xb7,0x72,0x49,0xb5,0xb8,0x09,/*0x0d,*/0x61,0x2d,0x3d,0x8a,0x00,
	0xbf,0xcc,0x25,0x5b,0x05,0xc3,0x3d,0xc0,0xb8,0xc3,0x5d,0x14,0xb5,0x12,0x7c,0x36,
	0x7b,0xac,0xc4,0xa4,0xc4,0x49,0x10,0x92,0xcb,0x5a,0x0e,0xcd,0x59,0xcf,0x30,0x66,
	0x70,0xcf,0xa8,0xc0,0x02,/*0x06,*/0xbd,0xbc,0x7c,0xc6,0x29,0x7b,0xc1,0xc1,0x5d,0x79,0x9c,
	0x93,0xc4,0xc3,0x10,0x97,0x10,0xa6,0x2a,0x90,0x2e,0x87,0x50,0x1c,0xc6,0xcc,0xc0,
	0x3f,0x21,0x50,0x60,0x22,0x87,0x08,/*0x0c,*/0x1c,0x68,0x30,0x05,0x73,0x92,0x02,/*0x06,*/0x47,0x4f,
	0x54,0x24,0x02,/*0x06,*/0x6a,0x8e,0x28,0x23,0x02,/*0x06,*/0x99,0x37,0x08,/*0x0c,*/0x87,0x3e,0x8d,0x5f,0x5d,
	0x51,0x03,/*0x07,*/0x7c,0x18,0x14,0x75,0x64,0xb8,0x96,0x4a,0x4a,0x03,/*0x07,*/0x89,0x91,0xc8,0x44,
	0x03,/*0x07,*/0x08,/*0x0c,*/0x7e,0xce,0xc5,0xc0,0x38,0xb4,0xc4,0xc3,0xc1,0x3d,0x0a,0x1e,0x63,0x8a,
	0x08,/*0x0c,*/0x40,0x65,0xba,0x57,0x1f,0x0a,0xab,0xc9,0xa2,0xc5,0xa7,0xc4,0xc6,0x3e,0x74,
	0x10,0x47,0x04,0x5d,0xaf,0xb2,0xa3,0x02,/*0x06,*/0x01,0xbf,0x52,0x0b,0x09,/*0x0d,*/0x4d,0x66,0x83,
	0x3e,0x11,0x4f,0xab,0x21,0x08,/*0x0c,*/0xc3,0x2c,0x1a,0x82,0x14,0x01,0x15,0xcf,0x94,0x37,
	0x17,0xab,0xa6,0x9a,0x09,/*0x0d,*/0x98,0x55,0x05,0x12,0x85,0x1d,0xc8,0xc4,0x0e,0x56,0x10,
	0x1a,0x67,0xc2,0x3e,0x7a,0x09,/*0x0d,*/0xca,0x6d,0x0b,0x76,0x99,0x39,0x16,0x23,0x1d,0x09,/*0x0d,*/
	0x02,/*0x06,*/0x68,0x31,0x81,0x27,0x11,0x67,0xcc,0x22,0x65,0x61,0x09,/*0x0d,*/0x95,0x6c,0xb2,0xcb,
	0x09,/*0x0d,*/0x72,0x3e,0x61,0xa3,0x86,0x0b,0xc7,0x02,/*0x06,*/0x16,0x7e,0x20,0x03,/*0x07,*/0x0e,0xbd,0xc9,
	0x7d,0x28,0x00,0x45,0x14,0x05,0x03,/*0x07,*/0x00,0x1f,0x45,0x1a,0x8e,0x8e,0x29,0x73,0xce,
	0xb7,0x86,0xc9,0xb2,0xae,0x8c,0x50,0x1b,0x5d,0x15,0x54,0x00,0x56,0x97,0x0b,0x88,
	0x09,/*0x0d,*/0xa1,0xcb,0x50,0x03,/*0x07,*/0x54,0x23,0x5f,0x1c,0xbd,0xac,0xc1,0xb8,0x91,0x13,0x95,
	0x5f,0x19,0x50,0x7b,0xc4,0x0f,0x95,0x4c,0x16,0x88,0x41,0x49,0x11,0x69,0x7e,0x58,
	0x55,0x6f,0x24,0x6f,0xc1,0x8b,0x23,0x09,/*0x0d,*/0x36,0x1e,0xaf,0x44,0xb1,0xa1,0x9e,0x14,
	0x1a,0x85,0x81,0x26,0xad,0x2d,0x8f,0x93,0x32,0x82,0x33,0x7b,0xcf,0xcf,0xc2,0x1f,
	0x7b,0x17,0x75,0x99,0x00,0x48,0x7a,0x94,0x3f,0x26,0x09,/*0x0d,*/0x3e,0x13,0x11,0xc5,0x2a,
	0x97,0x3d,0x5c,0x8d,0x1f,0x61,0xc8,0xcc,0x2a,0x66,0x7e,0x52,0x08,/*0x0c,*/0xa2,0x2f,0x01,
	0xa3,0xba,0xa1,0x99,0xc3,0xa1,0xae,0xbc,0x8e,0x27,0x25,0x92,0x39,0xad,0xc0,0x18,
	0x0f,0x17,0x51,0x27,0xc8,0xac,0xcf,0x41,0x16,0xa1,0xb3,0x09,/*0x0d,*/0x9d,0x63,0xcb,0x4a,
	0x11,0xc6,0xc7,0x02,/*0x06,*/0x77,0xb4,0xca,0x1c,0x4f,0x4d,0xc3,0x17,0x6e,0x09,/*0x0d,*/0xb0,0xcb,
	0xc8,0x10,0xab,0xa8,0x1d,0xb9,0x31,0x6f,0xb2,0xb0,0x5f,0x6d,0x94,0x38,0xc8,0xc1,
	0x2b,0x78,0x3b,0x11,0x22,0x6b,0x36,0x68,0x5a,0x50,0xbb,0xce,0x5c,0x25,0x9a,0x44,
	0x09,/*0x0d,*/0x03,/*0x07,*/0x74,0x4e,0x1e,0x70,0x03,/*0x07,*/0x7a,0x12,0x67,0x04,0x53,0x45,0x48,0x8d,0x14,
	0x62,0x94,0x30,0x98,0xb7,0x81,0x1c,0x1e,0x78,0x34,0x87,0xc3,0x57,0x45,0x01,0x09,/*0x0d,*/
	0x1a,0x02,/*0x06,*/0x71,0x70,0xa8,0x2f,0x48,0x56,0x05,0x93,0x3a,0xbd,0xc4,0x43,0x0a,0x9b,
	0x4a,0xc3,0x54,0x32,0xa4,0x42,0x51,0xc3,0x35,0x75,0x79,0x12,0x82,0x46,0x14,0x7d,
	0xaa,0xc3,0x19,0x79,0x1e,0x51,0x11,0xb0,0xc6,0x42,0xca,0x24,0x7c,0x39,0x45,0x16,
	0x57,0x1a,0xa4,0xc2,0x04,0x13,0xa9,0xc0,0x09,/*0x0d,*/0xbc,0x1f,0x72,0x38,0x34,0x17,0x5b,
	0x1e,0x88,0x58,0x4f,0x24,0x08,/*0x0c,*/0x6b,0x39,0x03,/*0x07,*/0x03,/*0x07,*/0xb0,0x2f,0x18,0x01,0x1a,0x11,
	0x92,0x1b,0x92,0x49,0x66,0x00,0x78,0x0f,0xb9,0xc6,0x08,/*0x0c,*/0x14,0x12,0x69,0x63,0xb9,
	0x36,0x6e,0x0f,0xc8,0x8c,0x3e,0x72,0x83,0x31,0x04,0x8a,0xbe,0xb8,0xb3,0x6c,0x88,
	0x28,0x08,/*0x0c,*/0x76,0x7f,0xc4,0x03,/*0x07,*/0x61,0x9e,0x0f,0x74,0x09,/*0x0d,*/0xa6,0xb0,0xba,0x8b,0x24,
	0x8e,0xc6,0x3f,0x2e,0x51,0x1e,0xb1,0xb1,0x8b,0x17,0x05,0x1c,0xad,0xcb,0x9e,0x22,
	0x02,/*0x06,*/0x0b,0x12,0x7a,0x5d,0x16,0x19,0x0f,0x79,0x94,0x84,0x33,0x12,0xc8,0xa1,0x2d,
	0x47,0xa0,0xc1,0x94,0x5a,0x09,/*0x0d,*/0x5f,0x03,/*0x07,*/0x5e,0x16,0x00,0x74,0x2c,0x92,0xbe,0xcc,
	0xb0,0x4c,0x55,0x40,0x88,0xc7,0x46,0xaf,0xcf,0xc7,0xae,0x38,0x11,0x74,0xc9,0xb8,
	0x5a,0x63,0xc8,0x19,0x64,0xca,0x10,0x90,0xc7,0xcd,0x17,0x1a,0x61,0x99,0x5a,0x49,
	0x02,/*0x06,*/0x9d,0x02,/*0x06,*/0xb6,0x83,0x9a,0x0b,0x8f,0x1c,0x1a,0xc0,0xa5,0xc2,0x86,0x2b,0x08,/*0x0c,*/
	0x74,0x3a,0x9d,0x0e,0x25,0x95,0x2a,0x03,/*0x07,*/0x67,0x98,0x7f,0xc9,0xc7,0xc0,0x3e,0x6f,
	0x15,0x6d,0x93,0x2b,0x14,0x6e,0x2d,0x35,0x9f,0x1f,0x26,0x55,0x13,0xc3,0x32,0x74,
	0x35,0x40,0x0e,0x95,0x1d,0x9c,0xc4,0xce,0x22,0x7e,0x89,0xc0,0x58,0x01,0x37,0x76,
	0xc8,0xa8,0xc6,0x83,0xcb,0xc8,0xc1,0xa8,0x86,0x08,/*0x0c,*/0x08,/*0x0c,*/0x6b,0x31,0x0a,0x54,0x0e,
	0x09,/*0x0d,*/0x55,0x53,0xb9,0xc5,0xb0,0xcd,0xb3,0xc8,0xa5,0x1b,0x95,0x16,0xc7,0x56,0x13,
	0xb9,0xcb,0xb3,0x7f,0xa6,0xc6,0x58,0x45,0x0f,0x51,0x05,0x9b,0x1f,0xa6,0xcd,0xb9,
	0x39,0x1a,0xcb,0x1f,0xab,0x98,0x16,0xa2,0xa4,0xc9,0x9c,0x3d,0x22,0xbf,0xb6,0x23,
	0x6c,0x23,0x03,/*0x07,*/0x13,0x7f,0x9f,0x7c,0xce,0x4c,0xad,0x5c,0x56,0x1c,0x90,0xca,0x0b,
	0x15,0x70,0x52,0x75,0x09,/*0x0d,*/0x01,0x62,0x10,0x73,0x39,0x1f,0x1f,0x04,0x87,0x39,0x99,
	0x3f,0x03,/*0x07,*/0x82,0xa7,0x3a,0x1a,0x69,0x70,0x9f,0x74,0xc1,0xc9,0x0b,0x66,0x1b,0x79,
	0xb5,0x9c,0x09,/*0x0d,*/0xc1,0x35,0x10,0x14,0x66,0x38,0x22,0xaf,0xcc,0x5f,0x65,0x81,0x40,
	0x47,0x56,0x03,/*0x07,*/0x11,0x56,0x53,0x4f,0x08,/*0x0c,*/0x6d,0x7d,0x04,0x67,0x52,0x12,0x8e,0xcb,
	0xc5,0xba,0x6d,0x1c,0x45,0x17,0xa2,0xce,0x56,0x4a,0x9f,0x1d,0x33,0x1b,0xb1,0xc9,
	0x08,/*0x0c,*/0x15,0xc4,0x29,0x5d,0x0e,0xc6,0xb8,0xae,0x19,0x99,0x30,0x9f,0x4b,0x1b,0x1c,
	0x92,0x5a,0x5b,0x8f,0x4e,0x6f,0x23,0x55,0x4b,0xa6,0x25,0x1a,0x7a,0x1c,0x03,/*0x07,*/0x77,
	0x3c,0x7b,0xc9,0x6e,0x3b,0x7c,0x3e,0xa9,0xb9,0x50,0x02,/*0x06,*/0xb5,0x7d,0x02,/*0x06,*/0xad,0x9d,
	0x8b,0x2c,0xbd,0x84,0x34,0x8b,0x3a,0x9a,0x2b,0x80,0xac,0xa8,0xa7,0x09,/*0x0d,*/0x90,0x20,
	0x1a,0x72,0x02,/*0x06,*/0x54,0x15,0x8c,0x84,0x3c,0x60,0x0f,0x11,0xa0,0xa8,0x81,0x9f,0x86,
	0xc4,0x27,0x34,0x5b,0x65,0xa7,0xa5,0x83,0x01,0x6a,0x58,0x1a,0xb4,0x94,0x03,/*0x07,*/0xb1,
	0x19,0xb3,0x62,0x57,0xb7,0xba,0x19,0x71,0x3f,0x2f,0x9a,0x10,0xbb,0xc6,0x37,0x4d,
	0x11,0xcd,0x2f,0xce,0xba,0xc9,0x4b,0x00,0xb7,0x02,/*0x06,*/0x6c,0x24,0x67,0xa8,0xce,0xa0,
	0xca,0x57,0x04,0x80,0xc1,0x44,0x1e,0xc7,0x4f,0xc1,0x32,0x7a,0x60,0xc5,0xaa,0xce,
	0x16,0xbf,0xb8,0x6c,0xcc,0x3b,0x67,0xcf,0xcf,0x4b,0x65,0x8e,0x81,0xca,0xc9,0x04,
	0x85,0x04,0xa4,0x38,0x82,0x3c,0x97,0x4e,0x08,/*0x0c,*/0xcc,0xc6,0xca,0x3d,0x33,0x4e,0x7c,
	0x30,0x97,0x1c,0x74,0x32,0x11,0x6f,0x80,0x12,0x59,0x51,0x4a,0x26,0x16,0x76,0x9e,
	0x2a,0x21,0x16,0x8b,0x35,0x1c,0x97,0x2c,0x9d,0x49,0x4b,0x47,0x13,0x6c,0x08,/*0x0c,*/0x00,
	0x69,0x78,0xba,0x84,0x5c,0x5c,0x13,0x99,0x83,0xc6,0x52,0x13,0x1c,0x6e,0xc0,0xcf,
	0xca,0x3a,0xb0,0xce,0xcd,0xcb,0x2f,0x1a,0x0a,0x73,0x9a,0x1c,0x56,0x75,0xbe,0x41,
	0x0f,0x1a,0xa9,0xc3,0xa6,0xcb,0xa3,0xce,0xc8,0x2c,0x64,0x04,0x59,0x14,0x4b,0xab,
	0xb6,0xa1,0x16,0x15,0xbb,0x4c,0x1f,0x1d,0x53,0x7a,0x93,0x2c,0x01,0x51,0xa9,0x33,
	0x1c,0xcd,0x2e,0x0e,0x92,0x00,0x15,0x05,0xc5,0x86,0x35,0x03,/*0x07,*/0xa9,0xa4,0x88,0x1d,
	0x8a,0x43,0x11,0x02,/*0x06,*/0x95,0x09,/*0x0d,*/0xc6,0xca,0x1e,0x48,0x04,0x0e,0x77,0xc8,0x2c,0x6a,
	0x1d,0xc0,0x7d,0x1f,0x66,0x8b,0x2b,0x02,/*0x06,*/0x21,0x09,/*0x0d,*/0x1d,0x12,0x74,0x23,0x91,0x25,
	0x01,0x77,0xc6,0x30,0x75,0x71,0x19,0x87,0x70,0xb6,0xc1,0x19,0x62,0x2c,0x71,0xa1,
	0x96,0x1f,0xc9,0x18,0x16,0x02,/*0x06,*/0x6e,0x22,0x17,0x1e,0xbf,0xc7,0x61,0x2a,0x10,0x5b,
	0x00,0x11,0x13,0x10,0x0f,0x5b,0x0e,0x9e,0x9e,0x3b,0x6f,0xc4,0xb3,0x96,0xc3,0xb6,
	0xac,0x9c,0x4e,0x0b,0x4b,0x05,0x4a,0x10,0x48,0x85,0x1f,0x98,0x1d,0xa5,0xc1,0x4e,
	0x13,0x4a,0x21,0x49,0x08,/*0x0c,*/0xbf,0xa8,0xcb,0xba,0x83,0x03,/*0x07,*/0x87,0x49,0x09,/*0x0d,*/0x4e,0x67,
	0xca,0x1b,0x87,0x5a,0x02,/*0x06,*/0x98,0x5f,0x57,0x01,0x18,0x79,0x6e,0x46,0x43,0x7f,0x26,
	0x7f,0xcf,0x9b,0x21,0x19,0x24,0x0a,0xab,0x52,0xb5,0xa5,0x8c,0x00,0x0e,0x95,0x91,
	0x34,0xaf,0x3f,0x9f,0x81,0x20,0x92,0x31,0x67,0xc1,0xc1,0xcc,0x0f,0x67,0x03,/*0x07,*/0x69,
};


static UINT8 key_ninjakid2[0x2000] =
{
	/* data */
	0x0e,0x38,0x11,0x10,0x40,0x02,0x27,0x06,0xc6,0x8d,0x16,0x15,0xbd,0xc4,0x0a,0x77,
	0x0e,0x2e,0xc0,0x15,0x28,0x68,0xc2,0xcb,0x60,0xa2,0x68,0xae,0x50,0x91,0x2d,0xce,
	0x4c,0x0b,0x1f,0x22,0xc2,0x9e,0x2d,0x82,0xbe,0x7c,0x81,0x1b,0x2c,0x1e,0x65,0xa0,
	0x8d,0x99,0x26,0x72,0x69,0xc3,0x01,0x98,0x99,0x06,0x3e,0x28,0x0c,0xc1,0x2c,0xcc,
	0x4e,0x12,0x06,0x87,0xb4,0xc6,0x14,0x83,0x1d,0x47,0x16,0x70,0xa7,0x77,0xab,0xbc,
	0x06,0xa9,0x1c,0x33,0xc3,0x24,0x05,0x87,0x52,0x65,0x52,0x09,0xba,0x68,0x92,0x80,
	0x6f,0x67,0x36,0x85,0x8c,0x54,0x57,0x00,0x95,0x89,0xc4,0xa0,0x1b,0x57,0x5e,0xba,
	0x0c,0x22,0xc9,0xc2,0xb8,0x98,0xc1,0x43,0x9b,0xc6,0x6c,0x34,0xad,0x9d,0x9a,0xc6,
	0x03,0xc3,0xb2,0x32,0x9d,0x01,0x41,0xb5,0x0c,0x53,0x68,0x2e,0x91,0x1d,0x0e,0xa6,
	0x2b,0xc7,0xc1,0x11,0x36,0x7b,0x40,0x06,0xa6,0xc8,0xb1,0xc7,0xc0,0x0c,0x70,0x66,
	0x4d,0x06,0x0d,0x14,0x02,0x9d,0x2a,0x0b,0x0f,0xa1,0x46,0x92,0x06,0x19,0x13,0x7b,
	0xa4,0xcf,0xab,0x8e,0xcf,0x15,0x1e,0x1e,0x86,0x01,0x7b,0xc2,0x31,0x8d,0x86,0x8e,
	0x11,0xc2,0x80,0xbd,0x04,0xb1,0x0e,0x7a,0x24,0x26,0x97,0x23,0xbb,0x08,0x2d,0x2f,
	0x21,0xc0,0x8d,0x6b,0xa7,0x67,0xa0,0x6d,0x19,0x2c,0x66,0x03,0xae,0x64,0xa6,0x04,
	0x2d,0x4e,0xac,0x08,0x5e,0x1d,0xc1,0x2e,0xa8,0x07,0x1d,0xc7,0xcc,0xc1,0x5c,0x50,
	0x16,0x08,0xc2,0x6d,0x14,0x79,0x1a,0xc0,0x3b,0x8d,0x7b,0x41,0x90,0x89,0xb8,0xc1,
	0x95,0x9a,0x01,0x14,0xcf,0x0a,0xce,0x0f,0x32,0x70,0x8b,0x90,0x6b,0x39,0x11,0xa9,
	0x1d,0x3f,0x82,0x6a,0x82,0x14,0x02,0x11,0x06,0x3a,0xcc,0xb9,0x08,0xad,0x72,0x89,
	0x02,0x28,0xcf,0xc3,0x6c,0x72,0xa3,0xcb,0xcd,0xa5,0x16,0x35,0x8b,0xce,0xb2,0x0b,
	0x03,0x4c,0xb8,0x4e,0x9f,0x9d,0xcc,0x5d,0x74,0x42,0xce,0x36,0xb1,0x72,0x2b,0x11,
	0x63,0x30,0x31,0xbc,0xba,0xb2,0x47,0xad,0xca,0xa8,0x3a,0x0e,0xbc,0x68,0x66,0xc6,
	0x97,0x64,0x0c,0x63,0x39,0xa9,0x86,0x91,0x9e,0x5a,0x0b,0x1a,0xb8,0x1b,0x0c,0x06,
	0x70,0x41,0x4a,0x0b,0x13,0x01,0xa9,0x81,0x78,0x8b,0xcb,0x7c,0x0b,0xaf,0x58,0xa3,
	0x53,0xc4,0x17,0xc0,0x9c,0x8c,0x8e,0x52,0x07,0x8c,0x5a,0x71,0xa1,0xcb,0xbf,0x76,
	0x6b,0x96,0x55,0xcb,0x65,0x56,0x01,0x64,0x2a,0xb8,0x18,0xbf,0x41,0xc3,0x3a,0x59,
	0x83,0x9c,0x93,0x2a,0x0a,0x94,0x19,0xcf,0x62,0x73,0x93,0x90,0xb9,0xca,0x98,0x09,
	0x18,0x8f,0x4a,0x21,0x6a,0x84,0x35,0x11,0x70,0xc4,0x97,0xa6,0x5c,0x5f,0x64,0xbb,
	0x84,0x3b,0x86,0x1c,0x1d,0xa5,0x37,0xb7,0x18,0x01,/*0x05,*/0xa4,0x07,0x04,0xa6,0x03,0x41,
	0xad,0x73,0x31,0x65,0x01,/*0x05,*/0x0a,0x82,0x04,0xa1,0x42,0x3d,0x13,0x65,0xc0,0x87,0x93,
	0x89,0x41,0x51,0x77,0xc2,0x35,0x27,0x02,0x45,0xa8,0x10,/*0x14,*/0x21,0x5b,0xc3,0xc5,0x3d,
	0xb7,0x0b,/*0x0f,*/0x6b,0xc6,0x9c,0xc6,0x15,0xcf,0x39,0x93,0x1c,0x4a,0x8e,0x1d,0x1e,0xa4,
	0x07,0x1d,0x06,0x2d,0xb2,0x43,0x40,0x4f,0x42,0x19,0x01,0x02,0x88,0x03,0xa3,0x0b,/*0x0f,*/
	0x6e,0x13,0x01,0x5c,0x5b,0x03,0x09,0xb4,0x6e,0xcc,0xc2,0xc7,0xc2,0x4a,0x8f,0x84,
	0xb0,0x7b,0x02,0x9b,0xb4,0x6e,0x78,0x54,0x52,0x8f,0x1f,0x3e,0x56,0xc6,0x09,0x51,
	0x70,0x21,0x80,0x2a,0x8c,0xcd,0x44,0x67,0x14,0xb2,0x70,0xc5,0x06,0x0b,/*0x0f,*/0xc8,0x8a,
	0x60,0x1b,0x52,0xb3,0xb7,0xca,0x1a,/*0x1e,*/0x63,0x3d,0x53,0x0c,0x48,0x2a,0x50,0x0b,/*0x0f,*/0x03,
	0x94,/*0x98,*/0x80,0x64,0x58,0x00,/*0x04,*/0x07,0xcc,0xa5,0x7b,0x12,0x1d,0x68,0x81,0xac,0x25,0x79,
	0x9e,0x5a,0x23,0x15,0x11,/*0x15,*/0x82,0x3d,0x01,0x11,0x0c,0x6c,0x6a,0x4b,0xb0,0x61,0xca,
	0xcc,0x56,0x1b,/*0x1f,*/0xca,0xcf,0x3b,0x50,0x14,0x9b,0x23,0xc4,0xcf,0x3b,0x9b,0x18,0x81,
	0x3c,0x15,0xa9,0x1b,0x1f,0x33,0x33,0x0f,0x38,0xc5,0xcd,0x54,0x0a,/*0x0e,*/0x49,0xc7,0x95,
	0x6c,0xb2,0x9b,0x76,0x7d,0x65,0x17,0x3a,0x31,0xb8,0xa8,0x6a,0xad,0x1b,0xb5,0x18,
	0x51,0x0b,0x89,0x99,0xc9,0x05,0xc1,0x2c,0xc0,0x12,0x15,0x10,0xc2,0x10,0xa5,0x08,
	0x14,0x06,0x2e,0x8b,0xa9,0x8e,0x02,0x5e,0x71,0x40,0x14,0x0e,0xb3,0x15,0xc9,0x13,
	0x7a,0x65,0x58,0xb9,0x98,0xc3,0xcd,0xa8,0xb7,0x9a,0x16,0x9e,0x78,0x8f,0x9a,0x15,
	0x06,0x44,0x70,0x38,0x19,0x7c,0xce,0x5e,0xcd,0x1d,0x4c,0x10,0xcc,0x95,0xc2,0x96,
	0xca,0x7b,0xbe,0x0f,0x31,0x51,0x54,0xc4,0xc3,0x50,0x62,0x55,0x0e,0x03,0x4d,0x13,
	0x33,0x9e,0xc7,0x57,0x0a,0x65,0x23,0x03,0xac,0x6e,0x04,0x20,0x20,0x5b,0xc6,0x6e,
	0x18,0x84,0x8f,0x0f,0x78,0x75,0x84,0x10,0x42,0x93,0x6f,0xc4,0x0c,0x70,0x8a,0xc6,
	0x11,0x06,0xc2,0x14,0x17,0x65,0x4d,0x5e,0xa0,0xca,0xcd,0xcd,0x10,0x66,0x7f,0x66,
	0x2f,0x28,0x24,0xcb,0x55,0x4f,0x65,0x45,0x37,0x84,0xc0,0xc2,0x3b,0x2b,0x50,0x5d,
	0x00,0xa6,0x30,0x8e,0x13,0xce,0x02,0xbb,0x1a,0x46,0x0e,0x2c,0x44,0x39,0x47,0x2d,
	0x90,0x08,0x7e,0x7e,0x64,0x63,0xc4,0x1c,0x1a,0x1b,0x3c,0xa9,0xcf,0x00,0x4e,0x97,
	0x63,0x4d,0x49,0x46,0x0e,0x74,0x08,0x31,0xcf,0x63,0x32,0xa4,0x3b,0x18,0x63,0x67,
	0x4b,0x63,0xb3,0x53,0x21,0xa5,0x01,0x6a,0x04,0x3b,0x76,0x69,0xa8,0x11,0x98,0xcb,
	0xc8,0xc1,0x3d,0xc2,0x1d,0x42,0xb7,0xc1,0x32,0xb8,0xa4,0x48,0x0d,0x22,0xb5,0x25,
	0xba,0xbb,0x12,0x06,0xba,0x34,0x97,0xc5,0x7d,0x0a,0x70,0x0c,0xaa,0x77,0x20,0x7a,
	0x5d,0x4b,0xc0,0x22,0xc2,0x39,0x59,0x7b,0x0a,0x2f,0x3f,0x27,0x39,0x37,0xb1,0x58,
	0x0d,0xc4,0x7f,0xa7,0x35,0x0e,0x47,0x69,0x49,0x06,0xa7,0x60,0xc1,0x77,0x7a,0x6f,
	0x02,0x3b,0x25,0x5c,0x8a,0x73,0xbf,0x0c,0x80,0x28,0x26,0x8c,0x4c,0x53,0x93,0x08,
	0xc7,0x27,0xcf,0x05,0x0b,0x05,0x4d,0x0f,0x05,0x3a,0xc5,0x37,0x1a,0x40,0xc5,0x10,
	0x7f,0x37,0x37,0x48,0x19,0x57,0xbb,0xcc,0x4e,0xab,0x02,0x8d,0x1c,0x10,0x36,0x0f,
	0x31,0xcb,0x63,0x14,0xac,0x27,0x2f,0x6b,0x1a,0x8f,0x20,0x3d,0x49,0x1b,0x61,0x5e,
	0x88,0x08,0xce,0xba,0x14,0xc3,0x3d,0xc9,0xa9,0x02,0x82,0x3d,0x53,0x02,0xc2,0xc0,
	0x0a,0x0e,0x5a,0xab,0xa6,0x5c,0x01,0x06,0x38,0xb5,0xc5,0x42,0x32,0x02,0x7c,0x5b,
	0x15,0x48,0x13,0x08,0xab,0x29,0xae,0x42,0x98,0x81,0x60,0xcb,0xa9,0x9f,0xba,0xa1,
	0xb9,0x58,0xc8,0x21,0x8c,0x49,0x89,0x87,0x70,0x51,0xa7,0x68,0xad,0x65,0x04,0x46,
	0x6f,0x13,0x04,0x72,0x50,0x0f,0x3d,0xca,0xcb,0x38,0x65,0x7a,0x6e,0xbd,0xc0,0x92,
	0x40,0x1e,0xac,0xa8,0x05,0x87,0x75,0x18,0x70,0x5a,0x70,0xa1,0xaa,0x16,0x73,0x1c,
	0x16,0x26,0x12,0xa2,0x06,0xb4,0xbb,0x98,0x88,0x07,0x49,0xca,0x18,0x4b,0xcf,0x8b,
	0x11,0x1f,0x08,0x65,0x5b,0xc9,0xa8,0x43,0x38,0x32,0x76,0x8e,0x04,0x87,0xc9,0x08,
	0xce,0x60,0xac,0xa0,0x7e,0x6c,0xbf,0xcf,0xc2,0x75,0x62,0xac,0x7b,0xb6,0x3e,0xba,
	0x3e,0x1a,0x72,0xc8,0x32,0x28,0x8a,0x83,0x0f,0x43,0x3c,0x04,0xc2,0x35,0x9a,0x88,
	0xb2,0x93,0x94,0xb3,0x45,0xba,0x04,0x90,0x97,0x60,0xc6,0x14,0xcf,0x05,0x05,0xce,
	0x60,0x58,0xcd,0x5a,0x02,0x8a,0x48,0xc2,0xa5,0x49,0x6c,0x36,0x1d,0x65,0x8e,0x4b,
	0xa1,0xa5,0x6a,0x0c,0x97,0xbe,0xc8,0x58,0x7e,0x19,0x0f,0xc0,0x32,0x5c,0xaf,0x1b,
	0x62,0x0a,0x29,0x6b,0x9b,0x04,0x85,0xbe,0x31,0xc4,0xc3,0x00,0x89,0x4c,0x44,0x1d,
	0x92,0xc0,0x1f,0x96,0xb2,0x86,0x28,0xc4,0x0b,0xbe,0xca,0x45,0x62,0xab,0xb6,0xad,
	0xc6,0xcb,0xb2,0x5a,0x83,0xc8,0x08,0x4a,0xa8,0x84,0xc6,0x67,0x1d,0x1b,0x56,0x91,
	0x12,0x71,0xc3,0x5c,0x34,0x99,0x37,0x0b,0x0f,0xcf,0xcc,0x18,0x12,0x65,0x17,0xb2,
	0xc2,0x37,0x53,0x72,0xc7,0xae,0x13,0xab,0x33,0x0e,0x45,0xa7,0x78,0xbf,0x93,0x71,
	0x05,0x7b,0x98,0x49,0x90,0x59,0x12,0x08,0x72,0x77,0xba,0x2a,0x05,0x80,0x7e,0x34,
	0x94,0xc4,0x99,0xc9,0x2b,0x5a,0x5f,0x7b,0x50,0xc8,0x24,0xb6,0x5f,0xb0,0xb0,0xae,
	0x39,0xb0,0x0e,0x10,0x63,0xc5,0x57,0x4e,0x50,0x6b,0x0c,0x01,0xc9,0xcf,0x04,0x99,
	0x29,0xbd,0x71,0x90,0x2f,0x38,0xcc,0x07,0x0e,0x2e,0x0b,0x4a,0xb7,0x9c,0x07,0x77,
	0x23,0x09,0xab,0x12,0x0d,0x7b,0x6a,0x27,0x81,0xa1,0x58,0x9a,0x17,0xae,0x1a,0x5e,
	0xc5,0xc7,0xcd,0x41,0x93,0x17,0x5c,0x45,0x00,0xce,0xce,0x10,0x9e,0x40,0xc8,0xb7,
	0xb1,0xac,0xbd,0xb8,0x2f,0x4b,0x17,0xcb,0xcd,0x8f,0x1a,0x17,0x1e,0xba,0x07,0x17,
	0x5b,0x0c,0xc3,0xc9,0x46,0x3f,0xa9,0x0b,0xce,0x88,0x19,0x53,0x3d,0x2f,0x36,0xc7,
	0x55,0x49,0xc3,0x2d,0x57,0x8b,0x10,0x3d,0xb2,0x5a,0x06,0x19,0x8d,0xb4,0x0e,0x11,
	0x53,0x08,0x87,0x38,0x53,0x49,0x86,0xcb,0x7a,0x00,0x0d,0xa6,0x5a,0x5d,0x54,0x48,
	0xb3,0x92,0x06,0x05,0x03,0xcc,0x38,0xc0,0xb3,0x69,0x5d,0xa8,0xc3,0x14,0x74,0x35,
	0xc1,0x86,0xb4,0x1e,0xc6,0x0d,0x77,0x04,0x07,0x57,0x76,0x1f,0x95,0x7a,0x59,0x69,
	0x33,0x1a,0xb1,0x6e,0x4c,0x33,0x05,0xc8,0x84,0x6d,0x0d,0xc5,0xc1,0xc1,0x7d,0x12,
	0x39,0xc1,0x56,0x1d,0x1a,0xbe,0x57,0x08,0x25,0xc5,0x0e,0x8a,0xb1,0x5c,0x11,0x65,
	0x2b,0xc2,0xae,0xc5,0x0b,0x1a,0xb4,0xc9,0x0a,0x40,0x4f,0x31,0x01,0x99,0x0a,0x37,
	0x96,0x69,0x7f,0x1f,0x6d,0x01,0xcd,0x79,0x2c,0x4f,0x28,0x78,0x8a,0x72,0x77,0x40,
	0x01,/*0x05,*/0x68,/*0xc6,*/0x38,0x23,/*0x53,*/0x24,/*0x41,*/0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x55,/*0x7d,*/0x4c,0x20,/*0x46,0x47,0xa9,0xaf,*/0x25,/*0xc9,*/0x80,/*0x82,0x8c,0x8e,0xcd,*/0x9c,0x5b,0x46,0x48,/*0xa5,*/0x35,/*0x37,*/
	0x54,/*0xc6,*/0xb9,0x33,0x18,/*0x19,0x1c,0x1d,*/0x21,/*0x26,0x27,0xcf,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x81,/*0x83,0x8d,0x8f,0xaa,*/0x02,/*0x06,*/0x59,0x11,/*0x15,*/0x18,/*0x19,0x1c,0x1d,*/0xb2,0xb8,0x07,0x68,/*0xc6,*/0x79,
	0x02,/*0x1d,*/0x20,/*0x46,0x47,0xa9,0xaf,*/0xa8,0x80,/*0x82,0x8c,0x8e,0xcd,*/0x1b,/*0x1f,0x6d,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x43,/*0xa4,*/0x00,/*0x04,0x2a,0x6e,*/0x18,/*0x19,0x1c,0x1d,*/0xc3,0x6f,0x4c,0x02,/*0x06,*/0x45,0xa4,0x12,
	0x6f,/*0xc2,*/0x20,/*0x46,0x47,0xa9,0xaf,*/0x4b,/*0x95,0x97,0x99,0x9b,*/0xc1,0x21,/*0x26,0x27,0xcf,*/0x80,/*0x82,0x8c,0x8e,0xcd,*/0xc5,0x1e,/*0x55,*/0x18,/*0x19,0x1c,0x1d,*/0x0a,/*0x0e,0x47,*/0xc1,0x7c,0x75,0x43,/*0xa4,*/0x50,0x25,/*0xc9,*/
	0x90,/*0x92,0x9c,0x9e,*/0x69,/*0xc9,*/0x81,/*0x83,0x8d,0x8f,0xaa,*/0x44,0x4b,/*0x95,0x97,0x99,0x9b,*/0xc5,0x19,/*0x49,0x4e,*/0x1e,/*0x55,*/0x56,0x91,/*0x93,0x9d,0x9f,*/0x69,/*0xc9,*/0xb1,0x08,/*0x09,0x0c,0x0d,*/0xb7,/*0xca,*/0x08,/*0x09,0x0c,0x0d,*/0x6e,
	0x1b,/*0x1f,0x6d,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x74,0xc2,0x58,/*0x76,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x60,0x03,/*0x07,0x26,0x6c,*/0x1e,/*0x55,*/0xb7,/*0xca,*/0x18,/*0x19,0x1c,0x1d,*/0x72,0x20,/*0x46,0x47,0xa9,0xaf,*/0x90,/*0x92,0x9c,0x9e,*/0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0xa5,
	0x55,/*0x7d,*/0x5d,0xb7,/*0xca,*/0x1a,/*0x28,0xce,*/0x25,/*0xc9,*/0x76,0x77,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0xc8,0x18,/*0x4f,*/0x7a,0xb6,0xa7,0x7b,0x40,0x38,/*0x3a,0x6c,*/
	0x5a,/*0x65,*/0x1f,/*0x35,0x37,0x54,*/0x16,0xc8,0x0b,/*0x0f,*/0x1f,/*0x35,0x37,0x54,*/0x19,/*0x49,0x4e,*/0x31,/*0x33,0x41,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0xc9,0xc1,0x19,0x31,/*0x33,0x41,*/0xad,0xc2,0xb2,
	0xb7,/*0xca,*/0x68,/*0xc6,*/0x66,/*0xa3,*/0x58,/*0xa1,*/0x1c,0x2d,0x94,/*0x96,0x98,0x9a,*/0x21,/*0x49,*/0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x66,/*0xa3,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0xce,0x0b,/*0x0f,0xca,*/0x2c,/*0xcd,*/0x23,/*0x53,*/0x2e,
	0xbb,0x52,0x16,0x45,0xaf,0x68,/*0xc6,*/0x11,/*0x15,*/0x67,0x1a,/*0x28,0xce,*/0x26,0x18,/*0x19,0x1c,0x1d,*/0x49,0x1b,0x13,0xa8,/*0xae,*/0xc4,
	0x8f,0x01,/*0x05,*/0xb5,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0xa1,/*0xc7,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x85,/*0x87,0x89,0x8b,*/0xc7,0x64,/*0x7c,*/0x11,/*0x15,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x82,0x08,/*0x09,0x0c,0x0d,*/0xb0,0x34,/*0x36,0x7b,*/0x11,/*0x15,*/
	0x10,/*0x14,0x4a,*/0x2e,0xb4,0xc3,0x06,0x73,0x5e,0x7f,0x00,/*0x04,0x2a,0x6e,*/0x19,/*0x49,0x4e,*/0x02,/*0x06,*/0x4d,0x10,/*0x11,*/0x68,/*0xa9,*/0x1f,/*0x35,0x37,0x54,*/0x42,
	0xb6,0x02,/*0x06,*/0x52,/*0x63,*/0xc3,0xbc,0x50,0x02,/*0x1d,*/0x2c,0x1f,/*0x35,0x37,0x54,*/0xbb,0x21,/*0x49,*/0x2a,0x20,/*0x46,0x47,0xa9,0xaf,*/0xb0,0x48,/*0xa5,*/0xc5,
	0x1a,/*0x1e,*/0x80,/*0x82,0x8c,0x8e,0xcd,*/0x02,/*0x1d,*/0xc5,0x13,0x20,/*0x46,0x47,0xa9,0xaf,*/0x95,/*0x97,0x99,0x9b,*/0x1e,/*0x55,*/0xb3,0x09,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x1e,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x85,/*0x87,0x89,0x8b,*/0x7f,0x00,/*0x04,*/
	0x46,0xc1,0x43,/*0xa4,*/0x10,/*0x11,*/0x38,/*0x3a,0x6c,*/0x31,/*0x33,0x41,*/0x7a,/*0xb8,*/0x01,/*0x05,*/0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x3c,/*0x3e,*/0x18,/*0x19,0x1c,0x1d,*/0x24,0xbc,0x18,/*0x4f,*/0xbf,0x31,/*0x33,*/
	0xcb,0x27,0x39,/*0x3b,*/0x58,/*0xa1,*/0x95,0x4e,0x58,/*0x76,*/0xbf,0x03,/*0x07,0x26,0x6c,*/0x57,0x50,0x51,0x18,/*0x19,0x1c,0x1d,*/0x01,/*0x05,*/0x18,/*0x19,0x1c,0x1d,*/0xa1,/*0xc7,*/
	0x60,0x44,0xc5,0xc0,0x35,/*0x37,*/0x20,/*0x46,0x47,0xa9,0xaf,*/0x2e,0x1f,/*0x35,0x37,0x54,*/0x00,/*0x04,0x2a,0x6e,*/0xb7,/*0xca,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x43,/*0xa4,*/0x2e,0xa1,/*0xc7,*/0x0a,/*0x0e,0x5e,0xb4,*/0x18,/*0x4f,*/
	0x75,0x02,/*0x06,*/0x11,/*0x15,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0xcc,0x7f,0x90,/*0x92,0x9c,0x9e,*/0x2b,0xa4,0x71,/*0x80,0x82,0x8c,0x8e,0xbd,*/0x5c,0x2e,0x31,/*0x33,0x41,*/0x20,/*0xc1,0xc8,0xce,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x35,/*0x37,*/
	0xc4,0x00,/*0x04,0x2a,0x6e,*/0x11,/*0x15,*/0x31,/*0x33,0x41,*/0x4b,/*0x95,0x97,0x99,0x9b,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x71,/*0x80,0x82,0x8c,0x8e,0xbd,*/0x12,/*0x16,*/0x12,/*0x16,*/0x94,/*0x96,0x98,0x9a,*/0x18,/*0x19,0x1c,0x1d,*/0xb3,0x08,/*0x09,0x0c,0x0d,*/0x38,/*0x3a,0x6c,*/0x21,/*0x49,*/0xb7,/*0xca,*/
	0xa7,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x2b,0xb3,0x0d,0x10,/*0x11,*/0x84,/*0x86,0x88,0x8a,*/0x18,/*0x19,0x1c,0x1d,*/0x24,/*0xb0,*/0x4b,/*0x95,0x97,0x99,0x9b,*/0x6a,/*0x78,*/0x81,/*0x83,0x8d,0x8f,0xaa,*/0x69,/*0xc9,*/0x44,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x81,/*0x83,0x8d,0x8f,0xaa,*/
	0x0a,/*0x0e,0x47,*/0xc3,0x71,/*0x80,0x82,0x8c,0x8e,0xbd,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x0a,/*0x0e,0x47,*/0x51,/*0x84,0x86,0x88,0x8a,0xcc,*/0x03,0x10,/*0x11,*/0x20,/*0x46,0x47,0xa9,0xaf,*/0xac,0x57,/*0xab,0xad,*/0x71,/*0x80,0x82,0x8c,0x8e,0xbd,*/0x4b,/*0xc4,*/0x35,/*0x37,*/0xc3,0xa1,/*0xc7,*/
	0x01,/*0x05,*/0x58,/*0xa1,*/0x02,0xc3,0x18,/*0x4f,*/0x18,/*0x19,0x1c,0x1d,*/0x54,/*0xc6,*/0x02,/*0x06,*/0x1a,/*0x28,0xce,*/0x6a,/*0xbe,*/0x24,/*0xb0,*/0x85,/*0x87,0x89,0x8b,*/0x13,/*0x17,0xcb,*/0x31,/*0x33,*/0x12,/*0x16,*/0x38,/*0x3a,0x6c,*/
	0x73,0xc5,0x0b,/*0x0f,*/0x5f,/*0xbe,*/0x45,0x77,0x38,/*0x3a,0x6c,*/0x62,0x1f,/*0x35,0x37,0x54,*/0x18,/*0x4f,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x1c,0xbc,0x2c,/*0xcd,*/0x08,/*0x09,0x0c,0x0d,*/0xa7,
	0x34,/*0x36,*/0x18,/*0x19,0x1c,0x1d,*/0x34,/*0x36,*/0x30,/*0x32,*/0x2a,0x03,/*0x07,0x26,0x6c,*/0xb6,0x00,/*0x04,0x2a,0x6e,*/0x21,/*0x49,*/0x91,/*0x93,0x9d,0x9f,*/0xc0,0x2c,/*0xcd,*/0x29,/*0x61,0xa6,*/0xac,0x2c,0x2f,
	0xb6,0x00,/*0x04,0x2a,0x6e,*/0x94,/*0x96,0x98,0x9a,*/0x58,/*0xa1,*/0x3d,/*0x3f,*/0x5d,0x7a,/*0xb8,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x6b,0x91,/*0x93,0x9d,0x9f,*/0x57,/*0xab,0xad,*/0xb7,/*0xca,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x81,/*0x83,0x8d,0x8f,0xaa,*/0xbc,0x20,/*0xc1,0xc8,0xce,*/
	0x28,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x22,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x43,/*0xa4,*/0x5d,0x07,0x08,/*0x09,0x0c,0x0d,*/0x13,/*0x17,0xcb,*/0x94,/*0x96,0x98,0x9a,*/0x4f,0x54,/*0xc6,*/0x71,0x2f,0x68,/*0xc6,*/0xbc,
	0xba,0x51,/*0x84,0x86,0x88,0x8a,0xcc,*/0x5e,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x0d,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x84,/*0x86,0x88,0x8a,*/0x1a,/*0x28,0xce,*/0x13,/*0x17,0xcb,*/0x2a,0x18,/*0x19,0x1c,0x1d,*/0x19,/*0x49,0x4e,*/0x77,0x81,/*0x83,0x8d,0x8f,0xaa,*/0x43,0x71,/*0x80,0x82,0x8c,0x8e,0xbd,*/
	0xaf,0x12,/*0x16,*/0x68,/*0xa9,*/0x20,/*0x46,0x47,0xa9,0xaf,*/0x94,/*0x96,0x98,0x9a,*/0x02,/*0x06,*/0x81,/*0x83,0x8d,0x8f,0xaa,*/0x44,0x01,/*0x05,*/0x6e,0x68,/*0xc6,*/0x02,/*0x1d,*/0x57,/*0xab,0xad,*/0x22,0x51,/*0x84,0x86,0x88,0x8a,0xcc,*/0x7f,
	0x0a,/*0x0e,0x47,*/0x2d,0x24,/*0x41,*/0x44,0x67,0x03,/*0x07,0x26,0x6c,*/0x0a,/*0x0e,0x47,*/0x08,/*0x09,0x0c,0x0d,*/0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x0a,/*0x0e,0x47,*/0x75,0x5c,/*0x78,*/0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x39,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x21,/*0x26,0x27,0xcf,*/
	0x59,0x75,0x91,/*0x93,0x9d,0x9f,*/0x44,0x3c,/*0x3e,*/0xc7,0x71,/*0x80,0x82,0x8c,0x8e,0xbd,*/0x03,/*0x07,0x26,0x6c,*/0x4a,0x38,/*0x3a,0x6c,*/0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x20,/*0xc1,0xc8,0xce,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x5a,/*0x65,*/0x10,/*0x11,*/0xa7,
	0x2c,/*0xcd,*/0x02,/*0x06,*/0x19,/*0x49,0x4e,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x91,/*0x93,0x9d,0x9f,*/0x73,0x71,/*0x80,0x82,0x8c,0x8e,0xbd,*/0x60,/*0xcf,*/0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x02,/*0x1d,*/0x48,/*0xa5,*/0xa8,0x91,/*0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x17,0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x6b,
	0x34,/*0x36,*/0xa2,0x0b,/*0x0f,*/0x77,0x10,/*0x14,0x4a,*/0x57,/*0xab,0xad,*/0xa8,0x70,0x15,0x25,/*0xc9,*/0x21,/*0x49,*/0x52,/*0x63,*/0x0a,/*0x0e,0x5e,0xb4,*/0x6b,0x18,/*0x19,0x1c,0x1d,*/0x7c,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,

	/* opcodes */
	0xad,0x43,0xcf,0x1c,0xaa,0x12,0x5f,0x52,0x82,0x72,0x94,0x67,0x3e,0x7f,0x15,0x60,
	0xce,0x1e,0x93,0x07,0x36,0x1c,0x07,0x38,0xbb,0x0e,0x22,0x57,0x34,0x99,0x3a,0xb6,
	0x82,0xbe,0xca,0xa2,0x66,0x72,0x1b,0xb2,0xc7,0xb4,0x06,0x98,0x10,0xc5,0xa5,0x70,
	0x94,0x06,0x19,0x1e,0xc2,0x01,0xab,0x3f,0xac,0x1f,0x4a,0x2f,0x43,0x81,0x3c,0x77,
	0xcf,0x2b,0x74,0x14,0x5d,0xa9,0x2f,0x00,0xb3,0x3c,0xc0,0x0e,0x3d,0x18,0x34,0x02,
	0x0b,0xc0,0xcc,0x17,0xcc,0xa3,0xbf,0x37,0x91,0x51,0x09,0x4e,0x81,0x09,0xb7,0x1b,
	0x2c,0xbe,0x00,0x36,0x78,0x07,0xc4,0x01,0x15,0x18,0x2c,0x04,0xbc,0xc8,0x5b,0xc9,
	0xb6,0x96,0xce,0xbc,0xae,0xcf,0x74,0x01,0x9f,0x3e,0x09,0x17,0x05,0x0e,0x70,0xa3,
	0x0e,0x3e,0x0c,0x5b,0x37,0x3c,0x60,0x91,0xc6,0xa5,0xca,0x4f,0x0d,0x4e,0x12,0x27,
	0x72,0x3c,0xa8,0xb0,0x31,0xc3,0xbd,0x2d,0x16,0xa9,0x4a,0x4a,0xc6,0x5c,0x0f,0x1e,
	0x1f,0x26,0xc1,0x80,0x14,0x1a,0x6f,0x47,0x00,0xb3,0x44,0x95,0x2c,0xb2,0x28,0xaf,
	0x83,0x1f,0x06,0x6d,0x0a,0x09,0xad,0x82,0xcf,0x0d,0x22,0x15,0xc7,0x6f,0xc1,0x04,
	0xba,0x3d,0x05,0x93,0x14,0x5a,0x17,0x02,0x27,0x86,0x86,0xad,0x29,0x08,0x25,0x91,
	0xb7,0x19,0x38,0x30,0x33,0x1e,0x27,0xa5,0x18,0xba,0xaf,0x76,0xcd,0x94,0x44,0x34,
	0x0b,0x4a,0x78,0x86,0x37,0x4c,0x0a,0xc3,0x0b,0x97,0x7c,0x7c,0x2e,0xca,0x43,0x56,
	0x09,0x14,0x71,0x21,0xa8,0x4a,0x49,0xbf,0x0e,0xbc,0x1c,0x26,0xc8,0xac,0x06,0x90,
	0x5a,0xb1,0x16,0x79,0x78,0x57,0xc8,0xa8,0x96,0xa6,0x65,0x76,0x05,0x96,0x19,0xbe,
	0xa8,0x15,0x8c,0x05,0xc6,0x63,0x8c,0x0f,0x2c,0x1e,0x17,0xaf,0x44,0xc3,0xc1,0xcf,
	0x23,0x05,0x9b,0x5d,0x58,0x2a,0x85,0x8e,0x66,0xc2,0xc3,0x6a,0x0f,0x9c,0x4d,0x10,
	0x13,0x4c,0x17,0x6e,0x96,0x48,0xa8,0xbb,0xc7,0x8a,0x48,0xc0,0x3c,0xa7,0xcd,0xbe,
	0xa4,0xcd,0x77,0x66,0x40,0xb1,0xbb,0x91,0xaa,0x44,0x5c,0x73,0x57,0x06,0x0d,0x22,
	0x55,0xaa,0xc9,0x06,0xae,0xc8,0x0d,0xa5,0x07,0x97,0x47,0x14,0x6f,0x66,0x7c,0x97,
	0x9c,0x6a,0xb8,0xaa,0x3d,0x4f,0x40,0x21,0x50,0x74,0x99,0x20,0xc4,0xb7,0x0a,0x44,
	0xc5,0x6a,0x9a,0x6f,0xc9,0x43,0x12,0x7b,0x1f,0x2c,0x92,0x1a,0xcf,0xb7,0x19,0x0a,
	0x0a,0x5b,0x19,0xc6,0xb0,0xaa,0xac,0x07,0x29,0xc8,0x46,0x1c,0x72,0x1a,0x10,0x2d,
	0x96,0x11,0x57,0x39,0xca,0x6a,0x0e,0x02,0x23,0x2a,0x0b,0x0c,0x98,0x0e,0x68,0xbf,
	0x5a,0x56,0x65,0x23,0x34,0x23,0xc3,0x6b,0x1c,0xc7,0x07,0x74,0x95,0x82,0xa4,0x0a,
	0x11,0xa2,0x6a,0x81,0xa3,0xab,0x00,0x1f,0x5d,0x37,0x07,0xbf,0x82,0x70,0x99,0x19,
	0x1d,0x27,0x5e,0x0c,0x11,/*0x15,*/0x0d,0xbd,0x1c,0x36,0xc5,0x8b,0x0f,0x1a,0x06,0x58,0x9a,
	0x28,0x7d,0x8a,0x21,0x69,0x34,0x07,0x7d,0x86,0x9c,0x03,0x93,0xc5,0x5b,0x6e,0x7d,
	0x66,0x6a,0xc2,0x4a,0x8a,0x6b,0x45,0x60,0x3d,0x0e,0x8b,0x51,0x53,0x85,0x7e,0x74,
	0x6f,0x21,0x5b,0x5f,0x3e,0x60,0xc7,0xc0,0x58,0x22,0xc8,0x1c,0xbe,0xc0,0x8b,0x6a,
	0x9b,0x7b,0x0a,0x73,0x63,0x0d,0x2a,0x25,0x58,0x9c,0xb4,0xca,0x5d,0x9d,0xb7,0x8d,
	0xbc,0x0c,0x74,0x3b,0x0c,0x92,0x54,0x38,0x1d,0x21,0x45,0x88,0x92,0xc9,0x57,0x12,
	0x68,0xcd,0xca,0x3d,0xce,0x9d,0x49,0xad,0x58,0xab,0xa7,0x8b,0xc9,0x90,0x6b,0xcf,
	0xa6,0x7e,0x10,0x26,0x12,0xa6,0xbc,0x40,0x17,0xb8,0x8a,0x1e,0x59,0x07,0x02,/*0x06,*/0x00,/*0x04,*/
	0x52,0x60,0x24,0x59,0x17,0x0b,0x0b,0x58,0x05,0x83,0xc9,0xbf,0xc9,0x43,0x98,0x9e,
	0x70,0x1a,0xb4,0xb3,0xc6,0x5f,0x7a,0xcf,0xa2,0x8d,0xa9,0xbe,0xcb,0x09,0x7c,0xce,
	0xc6,0xa3,0xcf,0x43,0x50,0x75,0x0a,0x3c,0x5a,0xb9,0x17,0x84,0x61,0x25,0xb9,0x00,
	0x98,0x00,0x48,0x8c,0x0a,0x30,0x0d,0x4f,0x0b,0x00,0x53,0x0c,0x36,0x5c,0xc5,0x08,
	0x77,0x4b,0xcd,0x65,0x7f,0x54,0xb9,0xb5,0x31,0x4f,0x19,0x23,0x46,0x1d,0x10,0x09,
	0xb7,0x42,0x0f,0x1b,0xce,0xce,0x16,0x84,0x05,0x86,0x3b,0x36,0x53,0x54,0x6d,0xcc,
	0xbc,0xcb,0x96,0xc6,0x48,0xaf,0x3b,0x63,0xc8,0x18,0xb3,0x99,0xba,0xab,0x03,0x51,
	0x86,0x34,0x0f,0xbb,0x3e,0x07,0x74,0x00,0xcf,0xc5,0xc9,0x2a,0x0a,0x0d,0x59,0xad,
	0x60,0x0d,0xc0,0x9e,0x6d,0x96,0x03,0x09,0xb3,0xbd,0xa1,0x1f,0x13,0xcd,0x9e,0x0f,
	0xcf,0x1f,0x7d,0x38,0x07,0x0f,0xb7,0x68,0x19,0xa6,0x47,0x39,0x09,0x43,0x11,0x0b,
	0xac,0x4b,0x41,0x64,0x6c,0x16,0x33,0x59,0xc0,0x43,0xcb,0x5e,0xaf,0x21,0x0c,0x57,
	0x0e,0x82,0x9c,0x1f,0xa4,0x10,0x7d,0x94,0x57,0x15,0x12,0x4f,0x5c,0x13,0xc4,0x39,
	0x3c,0x1c,0x08,0x43,0x8c,0x1d,0x19,0x87,0x0a,0xab,0x1b,0xcb,0x50,0xaa,0xc4,0x6a,
	0xc7,0x07,0x32,0x42,0x0d,0x15,0x9f,0xc4,0x5f,0x14,0x47,0x14,0x5b,0xac,0xa1,0x9f,
	0xce,0x01,0x14,0x32,0x0c,0x87,0x10,0x1c,0x7c,0xbd,0x48,0xca,0x5d,0xcf,0xaa,0xae,
	0x85,0x35,0x63,0x5b,0x8c,0x23,0x55,0xc4,0x11,0x30,0x2a,0x81,0x83,0x2d,0x00,0xc5,
	0xc5,0x77,0x9e,0xba,0x08,0x2f,0xa9,0x95,0xc7,0xa4,0x03,0x91,0x64,0x3c,0x72,0xbf,
	0x36,0x17,0x4f,0x89,0x4a,0xa7,0xbb,0xa6,0x88,0x49,0x49,0x0a,0x43,0x6f,0x71,0x72,
	0x5a,0x15,0x8f,0x44,0x74,0x0d,0x26,0x51,0x4d,0x40,0xbf,0x78,0xa0,0x61,0x65,0xcb,
	0x00,0x1f,0xb5,0x6c,0xca,0xca,0x32,0xcc,0x8b,0xb4,0x5c,0x9a,0x0f,0xa5,0xc5,0x90,
	0x58,0x43,0xcb,0x92,0xab,0xb0,0xac,0x33,0x63,0x85,0x98,0x1e,0x76,0x87,0x07,0xc1,
	0x0f,0x9c,0x50,0x06,0x0a,0x1f,0x12,0x97,0x13,0x0f,0x05,0x61,0xcf,0x1e,0xa0,0x62,
	0x5d,0x2c,0x0e,0x33,0xc8,0x45,0xcb,0xbd,0x99,0x0f,0x9e,0xcc,0xc5,0x24,0xbe,0xcc,
	0x07,0x39,0x26,0x24,0x26,0x01,0x0d,0x09,0x9f,0x17,0x07,0x4f,0xc6,0x0b,0xb0,0x6d,
	0x00,0x79,0x89,0xc8,0x11,0x29,0xc8,0x06,0x8e,0x04,0x53,0x3d,0xa3,0xc5,0xa2,0x72,
	0x53,0x40,0x99,0x9b,0x05,0x4c,0x1f,0x04,0x7f,0x25,0x93,0x05,0x4e,0x93,0x0d,0xc9,
	0x26,0x70,0xce,0x1f,0x70,0x30,0xbe,0x1f,0x9c,0x17,0x05,0xc1,0x47,0x3f,0x2d,0xc4,
	0x29,0x0c,0x76,0xac,0x32,0x03,0x5c,0x81,0x8d,0x17,0xc0,0x0e,0xc7,0x05,0x3f,0x0f,
	0x6a,0x8d,0x03,0x01,0x57,0x9c,0x22,0xb5,0x8b,0x6d,0x07,0x92,0x51,0xc5,0x83,0x1a,
	0x76,0x12,0xa5,0x4a,0xc9,0xb1,0xc4,0x61,0x1b,0xa8,0x30,0x20,0x58,0x85,0x2b,0x13,
	0xcb,0x3b,0x2a,0x05,0x5a,0x11,0xcf,0x72,0x7f,0x35,0xb8,0x2e,0xa1,0x24,0x92,0x17,
	0xa4,0xc5,0x28,0xcb,0x58,0x95,0xc0,0xc5,0x17,0x99,0x23,0xa5,0x6a,0xb5,0x96,0x08,
	0x0e,0x1b,0x0d,0x3a,0xc3,0xc0,0xcc,0x04,0x89,0x0a,0x14,0x89,0x87,0x17,0xc1,0x55,
	0x19,0xb9,0x3a,0x14,0x0d,0x16,0x99,0x6d,0x25,0xb6,0x4d,0x88,0x1c,0xc0,0xce,0x8a,
	0x88,0xb7,0x75,0x2c,0x7d,0x15,0x67,0xb1,0x80,0x23,0x9c,0xb5,0x61,0xb2,0xc3,0xaa,
	0x10,0x86,0x74,0x80,0xc9,0x37,0x1a,0x31,0x37,0x4a,0x5f,0x03,0x69,0x36,0x45,0x02,
	0x4a,0x3c,0x02,0x2a,0x3f,0xaf,0x7b,0xad,0xc5,0xc0,0x85,0x67,0x04,0x11,0x03,0x5d,
	0x13,0x4b,0xce,0xb9,0x0c,0xc3,0x6b,0x24,0x0f,0xcf,0x1d,0x10,0xc9,0x47,0xbb,0xc8,
	0xa5,0x1b,0xc2,0x5a,0x0a,0x62,0x89,0x26,0x03,0xc8,0x01,0x8c,0x97,0x06,0xae,0x86,
	0xce,0x3d,0x8f,0x19,0x02,0xcf,0x00,0x16,0x8e,0xce,0x25,0x7c,0x14,0xba,0x3b,0x0b,
	0xcd,0x80,0x6d,0x5f,0x99,0x2e,0x9c,0x83,0x95,0x74,0xc5,0x5b,0x82,0x19,0x0c,0x7a,
	0x15,0x42,0x66,0x62,0x90,0x27,0xc2,0xc9,0x1a,0x8b,0xa5,0x0f,0x97,0xca,0x96,0x28,
	0xca,0x52,0x60,0x70,0x50,0xc6,0x8a,0x7a,0x0e,0x22,0x16,0x0c,0x75,0xb2,0x92,0x0b,
	0x87,0x06,0x47,0xcd,0x0e,0x01,0x1d,0xa3,0x0c,0x36,0x97,0x2d,0x78,0x28,0x13,0xa0,
	0xc6,0x23,0xc5,0x15,0x31,0xaf,0x52,0x2c,0x04,0x83,0x89,0x96,0x7f,0x1f,0x56,0x0c,
	0x65,0x35,0x5c,0x7c,0x38,0x8a,0x13,0xca,0x45,0x07,0x16,0x34,0x9a,0x0e,0x90,0x68,
	0x0c,0x7b,0x8b,0xc8,0x3b,0x3f,0x45,0x2a,0x12,0x1d,0x3d,0x2a,0x0f,0x92,0xb3,0x35,
	0xb1,0x0f,0x0d,0x86,0xbb,0xba,0x6a,0x14,0x44,0xc0,0x11,0x41,0x03,0x86,0x16,0x80,
	0x60,0xb3,0x1d,0x91,0xab,0xa0,0xa9,0xab,0x9c,0x44,0xcb,0x6b,0x54,0xc3,0xa6,0x35,
	0xc2,0x8f,0x13,0x06,0x18,0x36,0x78,0xcb,0x7b,0x13,0x0a,0x83,0x0a,0xc9,0x16,0x78,
	0x9e,0x6a,0xca,0x74,0x14,0x7d,0x71,0x96,0x2e,0x48,0xa8,0x22,0x7d,0x01,0x0c,0x5b,
	0x43,0x53,0x87,0xb3,0x49,0x4a,0xab,0x27,0x65,0x19,0x5e,0x06,0x46,0x7f,0x62,0x60,
	0xbd,0xab,0x12,0x30,0x5b,0x56,0xa6,0x01,0x00,0x03,0xca,0x7c,0x5a,0x1e,0xc0,0x14,
	0x02,0x0c,0x10,0x75,0x0a,0xa7,0x13,0x5d,0x44,0x99,0x91,0x2e,0xce,0x16,0x3a,0x6d,
	0x12,0x0c,0x70,0xb3,0xcf,0x7f,0x93,0x6d,0x00,0x87,0xc4,0x71,0xc1,0x8a,0x74,0x6a,
	0x95,0x8d,0xb0,0xc6,0x25,0x4e,0x1a,0xbf,0x1f,0x37,0x14,0x24,0x51,0x41,0x8f,0x7d,
	0xca,0xb4,0x50,0xb0,0x49,0x0b,0x51,0x0b,0xa1,0x9e,0x88,0x0f,0xc6,0xa9,0xc1,0x18,
	0x17,0x57,0x01,0xca,0x17,0x09,0x9a,0x72,0x4e,0x15,0x1b,0x39,0x10,0x9c,0x20,0xc7,
	0x1d,0x6c,0x34,0xc6,0x94,0x34,0x5a,0x49,0x76,0x16,0x11,0x8f,0x60,0x17,0x02,0x43,
	0x19,0x99,0xc7,0x96,0x12,0xbd,0x0a,0x6c,0xab,0x0d,0x59,0x03,0x4e,0x08,0x3a,0xc1,
	0x8b,0x53,/*0x7c,0xc2,0xc5,*/0x32,0x87,0x13,/*0x17,0x19,0x1d,*/0x8c,/*0xc3,*/0x63,/*0x74,*/0x2b,/*0xb4,*/0xca,0x86,/*0x97,0x9f,*/0x06,0xac,0xc9,0x55,/*0xb4,0xb9,0xbf,*/0x7a,0xbe,
	0x1b,/*0x1f,0x5c,0x68,*/0xcb,0x61,0x84,/*0x8c,0x95,0x9d,*/0x19,/*0x1d,0xa7,*/0xc6,/*0xc7,*/0x5b,/*0x5c,0x62,*/0x4c,0x01,/*0x05,0x28,0x2a,*/0x69,0xb2,/*0xb3,*/0xb4,0x0b,/*0x0f,*/0x62,0x13,0x61,
	0x6f,/*0xc5,*/0xaf,0xaa,0x29,0xaa,/*0xac,*/0x1e,/*0x1f,*/0x4a,/*0x90,*/0x09,0xae,/*0xaf,*/0x23,/*0x26,*/0x57,/*0xca,*/0x8e,0xc3,0x3b,0x37,/*0x5d,0x72,*/0x51,/*0xc7,0xc8,*/
	0x52,0x6f,0x5d,0x6c,/*0x73,*/0x67,/*0xcf,*/0xb3,0x47,/*0x4e,*/0xb7,0x12,/*0x13,0x16,0x17,*/0x6e,0x20,/*0x21,0x22,0x23,0xcf,*/0x0b,0x10,0x02,/*0x4b,*/0x81,/*0xa8,*/0x60,/*0x61,0x78,*/
	0x83,/*0x8b,*/0xc0,0x86,0x28,/*0x2a,0x77,*/0xcb,0x3c,/*0x4f,0xa3,*/0x75,0x45,/*0x6c,0x6d,0xc3,*/0xa6,0x02,/*0x06,0x2c,0x2e,0x73,*/0x37,/*0x8a,*/0xaf,0x6f,0x03,/*0x7e,0xc7,*/0x00,/*0x01,0x04,0x05,*/0x6a,
	0x32,/*0x67,0x98,*/0x0c,/*0x0d,*/0xc6,0x01,/*0x05,0x80,*/0x38,/*0x84,0x86,0x88,0x8a,0x95,0x97,0x99,0x9b,*/0x80,/*0x82,0x84,0x86,0x88,0x8a,0x8c,0x8e,0x91,0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x00,/*0x04,0x6a,0x75,*/0xc3,0x46,0xce,0x08,/*0x09,0x0c,0x0d,*/0x3c,0x93,/*0x9e,*/0x56,/*0x60,*/0x96,/*0x9a,*/0x6d,
	0x3c,/*0x4a,*/0x99,0x53,/*0x62,*/0x49,0x32,0x02,/*0x06,0x89,*/0x34,/*0x67,*/0x18,/*0x19,0xcc,0xcd,*/0x96,0xcb,/*0xcc,*/0x2c,/*0x65,*/0x77,0x7e,0x4a,/*0x4d,*/0xc4,0x9f,
	0x29,/*0x92,*/0xcc,0x85,/*0x91,*/0x5e,/*0xc7,*/0xb0,0x6b,/*0x72,*/0x09,/*0x77,*/0x81,/*0x83,0x8d,0x8f,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x27,0xca,0x56,0x4d,/*0x6e,*/0x6a,0x87,/*0x96,0xba,*/0x11,/*0xc2,*/
	0x03,/*0x7e,0xc7,*/0x11,/*0x15,0x74,*/0x24,0xcb,0xcb,0x08,0x9d,0xcb,0x06,0xb1,0x18,/*0x19,0xcc,0xcd,*/0x92,0x21,/*0x23,0x30,*/0x69,0x22,/*0x95,*/0x5d,/*0x64,0xce,*/
	0xc8,0x5d,0x42,/*0xaf,*/0x18,/*0x1c,0x91,*/0x13,/*0x17,*/0x96,/*0xab,*/0xcb,0x02,/*0x40,*/0x24,/*0x26,*/0x21,0x18,/*0x19,0x1c,0x1d,*/0x8d,0x09,/*0x0d,0x4f,*/0xa0,/*0xb2,*/0x02,/*0x06,*/0xad,
	0xb6,0x34,/*0x90,*/0x7b,0xce,/*0xcf,*/0x04,/*0x8e,0x8f,*/0xb2,/*0xb3,0xb4,0xb5,*/0x01,/*0x05,0x17,*/0xb3,/*0xce,*/0x30,/*0x48,0xa3,*/0x35,/*0x37,*/0x06,/*0x07,0x1a,0x1b,0x42,0x43,0xc4,0xc5,*/0x29,0x63,0x5e,0xa5,/*0xb6,*/0xbe,
	0x52,0x3d,/*0x46,0x79,0xc3,*/0x83,0x32,0x2b,/*0x63,*/0x0e,0x08,/*0x0c,*/0x1b,/*0x1f,*/0x1a,0x00,/*0x04,*/0x8b,0x44,0x24,/*0xca,*/0x9f,/*0xa8,*/0x75,0x07,/*0xc2,*/
	0x64,0x00,/*0x04,*/0x09,/*0x0d,*/0x29,/*0x2b,0x62,*/0xc3,0xa2,0xaa,/*0xc9,*/0x2e,0x11,/*0x21,0x23,0x44,*/0x00,0x1b,/*0x1f,0x89,0xa3,*/0x00,0xa5,0x19,/*0x1d,*/0x28,/*0x2a,*/0x3a,/*0xbd,*/
	0x50,0xa8,/*0xcb,*/0x7f,/*0xb8,0xce,*/0x7c,/*0xb1,*/0x02,/*0x1e,0x67,*/0xce,0xcd,0x6a,/*0x6b,*/0x5b,/*0x5c,*/0x46,/*0xb4,0xb9,0xbe,*/0xb8,/*0xb9,*/0x16,0xc0,/*0xc1,*/0x75,0x09,/*0x0d,0x6a,*/0xc4,
	0x65,0x70,0x03,0x1c,/*0x1d,*/0x21,/*0x7e,*/0xce,0x54,/*0xb6,*/0x66,0xc7,0x3e,0x28,/*0x29,0x2a,0x2b,0x62,0x63,*/0x46,0x7a,/*0x7b,0xb9,*/0x34,/*0x51,0x67,*/0x55,/*0x99,*/0x03,
	0x94,/*0x96,0x98,0x9a,*/0x10,/*0x14,0x88,0x99,*/0x18,/*0x80,*/0xc3,0xc0,0x26,/*0x7e,*/0x5b,0x56,/*0xaa,0xad,*/0x02,/*0x06,0x34,*/0x19,/*0x45,*/0x10,/*0x14,*/0x31,0x1a,/*0x1b,0x1e,0x1f,*/0x64,0x00,/*0x01,0x04,0x05,*/0x43,/*0xb4,0xc0,*/
	0xab,0x49,/*0x74,*/0x45,0x22,/*0x7a,*/0x51,/*0x87,0x9e,*/0xce,0x05,0x4c,0x18,/*0x1c,*/0x12,0x80,/*0x82,0x84,0x86,0x88,0x8a,0x8c,0x8e,0x91,0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x16,0xa3,0x03,/*0x07,0x1a,0x1e,*/0x4a,0x68,/*0xb8,0xbe,*/
	0x1b,0x9d,/*0xc6,*/0x01,/*0x05,0x80,*/0x04,/*0x05,*/0x18,/*0x1c,0x5a,0xa6,*/0x4a,/*0x96,0xc0,0xc6,*/0xce,0x20,/*0x4b,0xb8,*/0x69,0x01,/*0x48,*/0x00,/*0x04,0x93,*/0x6d,0x4a,/*0x70,0x71,0x88,0x99,*/0x34,0x00,/*0x01,*/0x19,/*0x1d,0x32,0xb9,*/
	0x1e,/*0x67,*/0x72,/*0xb9,*/0x10,/*0x5e,*/0x00,0x19,0x48,/*0x49,0xa2,0xa3,0xa4,0xa5,*/0x44,0x84,/*0xa6,*/0xa1,0xb2,0x48,/*0x49,*/0x19,/*0x5e,0xbe,*/0x88,/*0xc7,*/0x0b,0x62,0x20,/*0x22,0x3f,0xba,*/
	0x13,/*0x93,0xb4,*/0xce,/*0xcf,*/0x00,/*0x01,0x04,0x05,0x67,0x85,0xb1,*/0x0d,0xa5,0x0a,/*0x4a,0x4b,0xc7,*/0x28,/*0xb6,0xb9,*/0x80,/*0x88,0x91,0x99,*/0x3c,0xc9,0x6a,0xb7,/*0xcb,*/0xad,0x39,0x1e,/*0x1f,*/0x7e,
	0xa2,0x6b,0x4d,0x06,/*0x07,0x1a,0x1b,0x42,0x43,0xc4,0xc5,*/0x75,/*0x88,*/0x7e,/*0xb5,*/0x50,0xc4,0xbd,0x00,/*0x04,0x22,0x4b,*/0x18,/*0x1c,0x6f,0x80,*/0x31,/*0x67,*/0x19,/*0x1d,0x88,*/0x81,/*0x83,0x8d,0x8f,*/0x90,/*0x99,*/0xc8,
	0x9b,0x6f,0xc6,0x35,/*0x37,0xa7,*/0x1f,/*0x49,*/0x84,/*0x8c,0x95,0x9d,*/0x63,/*0xa3,*/0x87,0xb0,0x26,0x03,/*0x07,*/0x07,0x4a,0x60,/*0x61,*/0xc8,0x71,/*0xc6,*/
	0x4c,0x93,/*0x9b,*/0x0b,/*0x0f,0x39,0x48,*/0x53,/*0x86,0x9f,0xc4,*/0x41,0x9d,0x30,/*0x32,0x85,*/0x8d,0xc1,/*0xce,*/0x00,/*0x66,0xb5,*/0x80,/*0x82,0x84,0x86,0x88,0x8a,0x8c,0x8e,0x91,0x93,0x95,0x97,0x99,0x9b,0x9d,0x9f,*/0x0c,0x5e,0xa4,0x40,/*0x41,*/0x16,/*0x56,0xbc,0xc4,*/
	0x39,/*0x53,*/0x38,/*0x39,0x3a,0x3b,0x87,0x8f,0x96,0x9e,*/0x0c,/*0x47,*/0x34,/*0x36,*/0x10,/*0x14,0x55,0x76,*/0x09,/*0x15,0x30,0xa3,*/0x82,0x54,/*0xa6,*/0x81,/*0x83,0x8d,0x8f,*/0x25,/*0x44,0x7c,*/0x97,0xc9,0x3b,0x10,/*0x14,*/0x1a,/*0x33,0x50,*/0x11,/*0x15,*/
	0xa1,0x7d,0x0d,0xa3,/*0xc6,*/0x1a,/*0x49,0x6f,0x72,*/0xcb,0x44,0xa8,/*0xa9,*/0x0c,/*0xa6,*/0xa3,0x19,/*0x1d,0x8e,*/0x61,/*0xbc,0xcb,*/0x38,/*0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x81,0x83,0x85,0x87,0x89,0x8b,0x8d,0x8f,0x90,0x92,0x94,0x96,0x98,0x9a,0x9c,0x9e,*/0x1b,/*0x1f,0x2d,0x2f,*/0x02,/*0x06,0x38,0x3a,*/0x25,/*0x27,*/
	0x10,/*0x14,0xa2,*/0xb8,/*0xb9,*/0x10,/*0x11,0x14,0x15,*/0x44,/*0x45,*/0x58,0x0f,/*0x65,*/0x00,/*0x01,0x04,0x05,0x4f,0x7d,*/0xc1,0x35,/*0x37,0xcc,*/0x07,/*0xa6,*/0x20,/*0x23,0x4d,0x8a,0x93,0x9b,*/0x7b,/*0xc1,*/0x5e,0x92,/*0xa5,0xa8,*/0x6e,0x57,/*0x89,0x98,*/
	0x5c,/*0x5d,*/0x6a,/*0x94,*/0x03,/*0x57,*/0xc0,/*0xc1,*/0x21,/*0x2e,0x83,0x8f,0xb6,*/0xa6,/*0xa7,*/0x09,0x12,0xcb,0x31,0x50,/*0x51,*/0x30,/*0x32,*/0xb6,0xa4,/*0xa8,*/0x18,/*0x84,0x95,*/0x4f,/*0xbe,*/
	0x03,/*0x07,0x4a,0x4b,0x51,*/0xb2,0x51,/*0x96,*/0xbe,0x00,/*0x04,0xb3,0xb5,*/0x4d,/*0xcb,*/0x76,0x4c,/*0xa0,*/0x0a,/*0x83,0xa0,*/0xce,0x33,0x66,/*0xa3,*/0x33,/*0x44,*/0x44,/*0x45,*/0x95,/*0xa4,0xcd,*/0x4b,/*0x73,*/
	0xae,0x51,/*0x97,0xc3,*/0x6a,0x25,/*0x81,0x83,0x8d,0x8f,*/0x05,0x39,/*0x3b,*/0xbd,0x1e,/*0x1f,*/0x52,0xa4,/*0xa9,*/0x85,0x08,/*0x09,0x14,0x15,0xc9,*/0xcc,0x8b,0xb2,/*0xb3,0xb4,0xb5,*/0x62,/*0xbe,*/
	0x75,/*0xb3,*/0x4a,/*0x5f,*/0x1a,/*0x1e,*/0x51,0x5a,0x6d,0x62,0x80,0x43,0xab,/*0xb4,*/0x1a,/*0x30,0x47,*/0x4a,/*0x79,*/0x10,/*0x11,*/0xba,0xce,0x94,/*0x96,0x98,0x9a,*/
	0x3d,/*0xca,*/0x36,/*0x4f,0x8f,*/0x38,0x58,/*0x59,0xb6,0xb7,*/0x8e,0x6c,/*0xc9,*/0x82,/*0xca,*/0x11,/*0x15,0x74,*/0x4a,/*0xa8,*/0x04,0x50,/*0x67,0x81,*/0x19,/*0x1d,*/0x01,/*0x05,0xad,0xb9,*/0x32,/*0x78,0x96,*/0x04,/*0x05,*/0xb0,/*0xcc,*/
	0x62,/*0xae,0xba,*/0x43,0x02,/*0x06,*/0x64,0x3e,/*0xa0,*/0x75,0x26,0x82,/*0xb2,*/0x10,/*0x14,*/0x2e,/*0x47,*/0x78,0x96,/*0x9a,*/0xab,0x1c,/*0xbd,*/0xa2,/*0xa3,*/0x87,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
	 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  , 0  ,
};



static int decrypt_type0(int val,int param,int opcode)
{
	int a = val;
	int swap = param & 0x03;

	if (swap == 1) a = BITSWAP8(a,5,4,3,6,7,1,2,0);
	if (swap == 2) a = BITSWAP8(a,0,5,3,2,4,7,6,1);
	if (swap == 3) a = BITSWAP8(a,0,1,7,5,3,6,2,4);

	val = 0x3d;
	if (BIT(a,0)) val ^= 0x23;
	if (BIT(a,1)) val ^= 0x08;
	if (BIT(a,2)) val ^= 0x04;
	if (BIT(a,3)) val ^= 0x20;
	if (BIT(a,4)) val ^= 0x01;
	if (BIT(a,6)) val ^= 0x10;
	if (BIT(a,5))
	{
		val ^= 0xc0;

		if (param & 0x10)
		{
			val ^= 0xb3;
		}
		else
		{
			if (param & 0x02) val ^= 0x40;
			if (param & 0x08) val ^= 0x40;
			if (opcode) val ^= 0x40;
		}
	}
	if (BIT(a,7))
	{
		val ^= 0x80;

		if (param & 0x04) val ^= 0x25;

		if (param & 0x02) val ^= 0x40;
		if (param & 0x08) val ^= 0x40;
		if (opcode) val ^= 0x40;
	}

	if (param & 0x02) val ^= 0xc0;
	if (param & 0x08) val ^= 0xe1;
	if (param & 0x10) val ^= 0x23;
	if (opcode) val ^= 0xc0;

	if (param & 0x08) val = BITSWAP8(val,7,6,5,3,2,1,4,0);

	return val;
}

static int decrypt_type1(int val,int param,int opcode)
{
	int a = val;
	int swap;

	param = (param << 1) | opcode;
	if (param & 0x02) param ^= 1;
	if (param & 0x08) param ^= 1;
	if (param & 0x10) param ^= 1;

	swap = param & 0x03;

	if (swap == 1) a = BITSWAP8(a,2,6,0,7,4,3,5,1);
	if (swap == 2) a = BITSWAP8(a,7,4,2,0,5,3,6,1);
	if (swap == 3) a = BITSWAP8(a,0,2,3,5,1,4,6,7);
	if (param & 0x04) a = BITSWAP8(a,7,6,5,4,0,2,3,1);

	val = 0x21;
	if (BIT(a,0)) val ^= 0x02;
	if (BIT(a,1)) val ^= 0xd4;
	if (BIT(a,2)) val ^= 0x08;
	if (BIT(a,3)) val ^= 0x21;
	if (BIT(a,4)) val ^= 0x01;
	if (BIT(a,5)) val ^= 0x48;
	if (BIT(a,6)) val ^= 0xd8;
	if (BIT(a,7)) val ^= 0x7c;

	if (param & 0x10) val ^= 0x48;
	if (param & 0x20) val ^= 0x84;

	if (param & 0x08) val = BITSWAP8(val,7,6,1,4,3,2,5,0);

	return val;
}

static int decrypt_type2(int val,int param,int opcode)
{
	int a = val;
	int swap;

	param = (param << 1) | opcode;
	if (param & 0x04) param ^= 1;
	if (param & 0x08) param ^= 1;

	swap = param & 0x03;

	if (swap == 1) a = BITSWAP8(a,3,5,0,1,7,2,6,4);
	if (swap == 2) a = BITSWAP8(a,3,4,7,6,5,1,2,0);
	if (swap == 3) a = BITSWAP8(a,4,2,0,5,3,6,7,1);

	val = 0x04;
	if (BIT(a,0)) val ^= 0x42;
	if (BIT(a,1)) val ^= 0x84;
	if (BIT(a,2)) val ^= 0x70;
	if (BIT(a,3)) val ^= 0xa4;
	if (BIT(a,4)) val ^= 0x22;
	if (BIT(a,5)) val ^= 0xcf;
	if (BIT(a,6)) val ^= 0x04;
	if (BIT(a,7)) val ^= 0x47;

	if (BIT(a,7) && BIT(a,6))
		val ^= 0xf4;

	if (param & 0x04) val ^= 0x0a;
	if (param & 0x08) val ^= 0x47;
	if (param & 0x10) val ^= 0x12;
	if (param & 0x20) val ^= 0x84;

	return val;
}

static int decrypt_type3(int val,int param,int opcode)
{
	int a = val;
	int swap;

	param = (param << 1) | opcode;
	if (param & 0x04) param ^= 1;
	if (param & 0x08) param ^= 1;
	if (param & 0x10) param ^= 1;

	swap = param & 0x03;

	if (swap == 1) a = BITSWAP8(a,2,3,0,4,5,7,6,1);
	if (swap == 2) a = BITSWAP8(a,2,4,7,0,5,1,3,6);
	if (swap == 3) a = BITSWAP8(a,0,3,2,7,5,1,6,4);

	if (param & 0x08)	a = BITSWAP8(a,7,6,5,4,2,3,1,0);
	if (BIT(a,4))		a = BITSWAP8(a,7,6,5,4,2,3,1,0);

	val = 0x68;
	if (BIT(a,2)) val ^= 0x26;
	if (BIT(a,3)) val ^= 0x80;
	if (BIT(a,5)) val ^= 0x10;
	if (param & 0x04)
	{
		if (BIT(a,0)) val ^= 0x04;
		if (BIT(a,1)) val ^= 0x01;
		if (BIT(a,4)) val ^= 0x5d;
		if (BIT(a,6)) val ^= 0x43;
		if (BIT(a,7)) val ^= 0xf6;
	}
	else
	{
		if (BIT(a,0)) val ^= 0x43;
		if (BIT(a,1)) val ^= 0x40;
		if (BIT(a,4)) val ^= 0x1f;
		if (BIT(a,6)) val ^= 0x01;
		if (BIT(a,7)) val ^= 0xb2;
	}

	if (param & 0x04) val ^= 0x01;
	if (param & 0x08) val ^= 0x08;
	if (param & 0x10) val ^= 0x10;
	if (param & 0x20) val ^= 0x80;

	return val;
}

static int decrypt_type4(int val,int param,int opcode)
{
	int a = val;
	int j;
	int swap;
	int v;

	param = (param << 1) | opcode;
	param ^= 1;
	if (param & 0x08) param ^= 1;
	if (param & 0x10) param ^= 1;

	swap = param & 0x03;
	v = (param & 0x0c) >> 2;

	if (swap == 1) a = BITSWAP8(a,2,3,0,4,5,7,6,1);
	if (swap == 2) a = BITSWAP8(a,2,4,7,0,5,1,3,6);
	if (swap == 3) a = BITSWAP8(a,0,3,2,7,5,1,6,4);

	val = 0xea;
	if (BIT(a,4)) val ^= 0x41;
	if (BIT(a,3)) val ^= 0x07;
	if (BIT(a,5)) val ^= 0x04;
	if (BIT(a,2)) val ^= 0x18;
	if (BIT(a,6)) val ^= 0x10;
	if (BIT(a,0)) val ^= 0xa0;
	if (BIT(a,7)) val ^= 0x60;
	if (BIT(a,1)) val ^= 0x80;

	if (param & 0x02) val ^= 0x1f;
	if (param & 0x20) val ^= 0x1f;

	if (!BIT(a,5) && ((param & 0x10) || !BIT(a,6)))
	{
		if (BIT(a,0)) val ^= 0x20;
		if (BIT(a,7)) val ^= 0x21;
		if (BIT(a,1)) val ^= 0xe0;
		if (BIT(a,4)) val ^= 0xe1;
	}

	for (j = 0;j < v;j++)
	{
		val = BITSWAP8(val,7,6,5,2,1,3,4,0);
	}

	return val;
}

static int decrypt_type5(int val,int param,int opcode)
{
	int a = val;
	int swap;

	param = (param << 1) | opcode;
	if (param & 0x02) param ^= 1;
	if (param & 0x10) param ^= 1;
	if (param & 0x20) param ^= 1;

	swap = param & 0x03;

	if (swap == 1) a = BITSWAP8(a,2,1,6,4,5,7,3,0);
	if (swap == 2) a = BITSWAP8(a,7,0,4,3,2,1,5,6);
	if (swap == 3) a = BITSWAP8(a,3,4,7,5,1,6,0,2);

	if (param & 0x04)
	{
		val = 0x22;
		if (param & 0x08) val ^= 0x04;
		if (param & 0x10) val ^= 0x08;
		if (param & 0x20) val ^= 0x80;

		if (BIT(a,0)) val ^= 0xd6;
		if (BIT(a,1)) val ^= 0x30;
		if (BIT(a,2)) val ^= 0x11;
		if (BIT(a,3)) val ^= 0x04;
		if (BIT(a,4)) val ^= 0x08;
		if (BIT(a,5)) val ^= 0x42;
		if (BIT(a,6)) val ^= 0xc6;
		if (BIT(a,7)) val ^= 0x03;
	}
	else
	{
		val = 0x24;
		if (param & 0x08) val ^= 0x01;
		if (param & 0x10) val ^= 0x08;
		if (param & 0x20) val ^= 0x80;

		if (BIT(a,0)) val ^= 0x54;
		if (BIT(a,1)) val ^= 0x08;
		if (BIT(a,2)) val ^= 0x82;
		if (BIT(a,3)) val ^= 0x01;
		if (BIT(a,4)) val ^= 0x30;
		if (BIT(a,5)) val ^= 0x40;
		if (BIT(a,6)) val ^= 0x52;
		if (BIT(a,7)) val ^= 0x86;
	}

	if (BIT(a,2) ^ BIT(a,7))
	{
		if (BIT(a,4) ^ BIT(a,1))
			val ^= 0x38;

		if (BIT(a,0) ^ BIT(a,5) ^ BIT(a,6))
			val ^= 0x02;
	}

	return val;
}

/* only 0x20 possible encryptions for this method - all others have 0x40 */
static int decrypt_type6(int val,int param,int opcode)
{
	int a = val;
	int swap;

	param = (param << 1) | opcode;
	if (param & 0x02) param ^= 1;
	if (param & 0x04) param ^= 1;
	if (param & 0x10) param ^= 1;

	swap = param & 0x03;

	if (swap == 1) a = BITSWAP8(a,2,1,6,4,5,7,3,0);
	if (swap == 2) a = BITSWAP8(a,7,0,4,3,2,1,5,6);
	if (swap == 3) a = BITSWAP8(a,3,4,7,5,1,6,0,2);

	val = 0x0e;
	if (BIT(a,0)) val ^= 0x33;
	if (BIT(a,1)) val ^= 0x8e;
	if (BIT(a,2)) val ^= 0x01;
	if (BIT(a,3)) val ^= 0xe0;
	if (BIT(a,4)) val ^= 0x5b;
	if (BIT(a,5)) val ^= 0xea;
	if (BIT(a,6)) val ^= 0x58;
	if (BIT(a,7)) val ^= 0x0e;

	if (BIT(a,1) && BIT(a,5))
		val ^= 0xa8;

	if (BIT(a,6))
	{
		if (BIT(a,0) ^ BIT(a,4) ^ BIT(a,1))
			val ^= 0x11;
	}
	if (BIT(a,1))
	{
		if (!BIT(a,5))
		{
			if (BIT(a,0) ^ BIT(a,4))
				val ^= 0x11;
		}
	}

	if (param & 0x04) val ^= 0x24;
	if (param & 0x08) val ^= 0x4a;
	if (param & 0x10) val ^= 0x80;

	return val;
}

static int decrypt(int val,int param,int opcode)
{
	switch ((param & 0xe0) >> 5)
	{
		case 0: return decrypt_type0(val,param & 0x1f,opcode);
		case 1: return decrypt_type1(val,param & 0x1f,opcode);
		case 2: return decrypt_type2(val,param & 0x1f,opcode);
		case 3: return decrypt_type3(val,param & 0x1f,opcode);
		case 4: return decrypt_type4(val,param & 0x1f,opcode);
		case 5: return decrypt_type5(val,param & 0x1f,opcode);
		case 6: return decrypt_type6(val,param & 0x1f,opcode);
		default:
		case 7: return val;	/* 0xff in our tables */
	}
}

static data8_t mc8123_decrypt(offs_t addr,data8_t val,UINT8 *key,int opcode)
{
	int tbl_num;

	/* pick the translation table from bits fd57 of the address */
	tbl_num = (addr & 7) + ((addr & 0x10)>>1) + ((addr & 0x40)>>2) + ((addr & 0x100)>>3) + ((addr & 0xc00)>>4) + ((addr & 0xf000)>>4) ;

	return decrypt(val,key[tbl_num + opcode * 0x1000],opcode);
}

static void sys16_decrypt(UINT8 *key)
{
	UINT8 *decrypted;
	UINT8 *rom = memory_region(REGION_CPU2);
	int A;
	decrypted = auto_malloc(0x50000);

	memory_set_opcode_base(1,decrypted);

	for (A = 0x0000;A < 0x8000;A++)
	{
		UINT8 src = rom[A];

		/* decode the opcodes */
		decrypted[A] = mc8123_decrypt(A,src,key,1);

		/* decode the data */
		rom[A] = mc8123_decrypt(A,src,key,0);
	}

	/* copy the opcodes from the not encrypted part of the ROMs */
	for (A = 0x8000;A < 0x50000;A++)
		decrypted[A] = rom[A];
}


void mc8123_decrypt_0054(void)
{
	sys16_decrypt(key_0054);
}

void mc8123_decrypt_0066(void)
{
	sys16_decrypt(key_0066);
}

void mc8123_decrypt_0043(void)
{
	UINT8 *rom = memory_region(REGION_CPU1);
	int diff = memory_region_length(REGION_CPU1) / 2;
	int A;
	UINT8 *key = key_0043;


	memory_set_opcode_base(0,rom+diff);

	for (A = 0x0000;A < 0x8000;A++)
	{
		UINT8 src = rom[A];

		/* decode the opcodes */
		rom[A+diff] = mc8123_decrypt(A,src,key,1);

		/* decode the data */
		rom[A] = mc8123_decrypt(A,src,key,0);
	}
	for (A = 0x10000;A < 0x20000;A++)
	{
		UINT8 src = rom[A];

		/* decode the opcodes */
		rom[A+diff] = mc8123_decrypt((A & 0x3fff) + 0x8000,src,key,1);

		/* decode the data */
		rom[A] = mc8123_decrypt((A & 0x3fff) + 0x8000,src,key,0);
	}

if (0)
{
	FILE *f;

	f = fopen("a","wb");
	if (!f) return;
	fwrite(rom,1,0x40000,f);
	fclose(f);
}
}

void mc8123_decrypt_ninjakid2(void)
{
	UINT8 *rom = memory_region(REGION_CPU2);
	int diff = memory_region_length(REGION_CPU2) / 2;
	int A;
	UINT8 *key = key_ninjakid2;


	memory_set_opcode_base(1,rom+diff);

	for (A = 0x0000;A < 0x8000;A++)
	{
		UINT8 src = rom[A];

		/* decode the opcodes */
		rom[A+diff] = mc8123_decrypt(A,src,key,1);

		/* decode the data */
		rom[A] = mc8123_decrypt(A,src,key,0);
	}
}
