/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.*;
import xnap.gui.text.EmacsKeyBindings;
import xnap.net.IChannel;
import xnap.util.Notifier;
import xnap.util.Preferences;
import xnap.util.Updater;
import xnap.util.event.ChatListener;
import xnap.util.event.StatusListener;

import java.awt.*;
import java.beans.*;
import javax.swing.*;

public class GUINotifier extends Notifier
    implements PropertyChangeListener
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    protected ChatListener chatListener;
    protected Object chatListenerLock = new Object();
    protected StatusListener listener;
    protected Object listenerLock = new Object();
    protected boolean guiVisible = false;
    protected Object guiVisibleLock = new Object();

    protected Preferences prefs = Preferences.getInstance();


    /**
     * Used to center dialogs.
     */
    private Component mainComponent;
    
    //--- Constructor(s) ---
    
    protected GUINotifier()
    {
	updateEmacsKeyBindings();
	updateToolTips();

	prefs.addPropertyChangeListener(this);
    }

    //--- Method(s) ---

    public static Notifier getInstance() 
    {
	if (singleton == null) {
	    synchronized (GUINotifier.class) {
		if (singleton == null) {
		    singleton = new GUINotifier();
		}
	    }
	}

	return singleton;
    }

    /**
     * Convenience access for gui.
     */
    public static GUINotifier getGUIInstance() 
    {
	return (GUINotifier)getInstance();
    }

    /**
     * Used for the blinking tux.
     */
    public void setChatListener(ChatListener newValue) 
    {
	synchronized (chatListenerLock) {
	    chatListener = newValue;
	}
    }

    public void channelAdded(IChannel c)
    {
	synchronized (chatListenerLock) {
	    if (chatListener != null) {
		chatListener.channelAdded(c);
	    }
	}
    }

    private boolean canPurge = true;

    public boolean canPurgeResumeRepository()
    {
	Runnable r = new Runnable()
	    {
		public void run()
		{
		    canPurge = JOptionPane.showConfirmDialog
			(mainComponent, XNap.tr("The resume repository format has changed.\nPress Yes to save your repository. Then quit\nand launch the previous XNap version to resume\nyour downloads.\nIf you hit No, your resumes will be lost."), 
			 XNap.tr("Keep Resume Repository"), 
			 JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE)
			== JOptionPane.NO_OPTION;
		}
	    };
	
	synchronized (guiVisibleLock) {
	    while (!guiVisible) {
		try {
		    guiVisibleLock.wait();
		}
		catch (InterruptedException e) {
		}
	    }
	}

	try {
	    SwingUtilities.invokeAndWait(r);
	}
	catch (Exception e) {
	}

	return canPurge;
    } 

    public void setGUIVisible(Component c, boolean newValue)
    {
	mainComponent = c;

	if (Updater.hasMacOSXModalDialogBug()) {
	    Dialogs.showNotification(mainComponent, "MacOSXModalDialogBug", 
				     XNap.tr("Mac OS Java Bug"),
				     XNap.tr("Your operating system might have a bug that affects modal dialogs. If any dialogs come up blank, please update to Mac OS X 10.2 (Jaguar)"));
	}

	if (Updater.hasMacOSXToolTipsBug() && !prefs.getShowToolTips()) {
	    Dialogs.showNotification(mainComponent, "MacOSXModalTooltipsBug", 
				     XNap.tr("Mac OS Java Bug"),
				     XNap.tr("Your operating system might have a bug that causes freezes when showing tooltips. Therefore tooltips have been disabled. You can reenable them in the preferences dialog."));
	}
	
//  	if (Updater.isMacOSX()) {
//  	    Dialogs.showNotification(mainComponent, "MacOSXApplicationMenu", 
//  				     XNap.tr("Mac OS"),
//  				     XNap.tr("Please do not exit XNap through the Mac OS application menu. XNap will not be able to cleanly shutdown if you do. Please use the Exit item in the File menu instead."));
//  	}

	if (prefs.getOldVersion() >= 0) {
	    if (prefs.getOldVersion() < 8) {
		Dialogs.showFullPathSharing(mainComponent);
	    }

	    // this dialog is too complicated
//  	    if (prefs.getOldVersion() < 10) {
//  		Dialogs.showPortRange(mainComponent);
//  	    }
	}

	synchronized (guiVisibleLock) {
	    guiVisible = true;
	    guiVisibleLock.notifyAll();
	}
    }

    public void setStatus(String newValue)
    {
	synchronized (listenerLock) {
	    if (listener != null) {
		listener.setStatus(newValue);
	    }
	}
    }

    public void setStatusListener(StatusListener newValue) 
    {
	synchronized (listenerLock) {
	    listener = newValue;
	}
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	String p = e.getPropertyName();

	if (p.equals("useEmacsKeyBindings")) {
	    updateEmacsKeyBindings();
	}
	else if (p.equals("showToolTips")) {
	    updateToolTips();
	}
    }

    private void updateEmacsKeyBindings()
    {
	if (prefs.getUseEmacsKeyBindings()) {
	    EmacsKeyBindings.loadEmacsKeyBindings();
	}
    }

    private void updateToolTips()
    {
	ToolTipManager.sharedInstance().setEnabled(prefs.getShowToolTips());
    }

}
