/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.util;

import xnap.util.EventList;
import xnap.util.Preferences;
import xnap.util.event.ListEvent;
import xnap.util.event.ListListener;

import java.awt.*;
import java.util.*;
import javax.swing.*;
import org.apache.log4j.Logger;

/**
 */
public class SwingSynchronizedCache implements ListListener
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    private LinkedList insertCache = new LinkedList();
    private ListListener listener;
    private boolean insertPending = false;
    private volatile int maxInsertsPerRun = 100;

    //--- Constructor(s) ---

    public SwingSynchronizedCache(EventList list, ListListener listener)
    {
	this.listener = listener;

	list.addListListener(this);
    }

    public SwingSynchronizedCache(ListListener listener)
    {
	this.listener = listener;
    }

    //--- Method(s) ---

    public void addToCache(ListEvent event) 
    {
	synchronized (insertCache) {
	    insertCache.addLast(event);
	    if (insertPending) {
		return;
	    }
	    insertPending = true;
	}
	SwingUtilities.invokeLater(new InsertRunnable());
    }

    public void elementAdded(ListEvent event)
    {
	addToCache(event);
    }

    public void elementRemoved(ListEvent event)
    {
	if (event.getID() == ListEvent.LIST_CLEARED) {
	    synchronized (insertCache) {
		insertCache.clear();
		insertPending = false;
		addToCache(event);
	    }
	}
	else {
	    addToCache(event);
	}
    }

    public void setMaxInsertsPerRun(int newValue)
    {
	maxInsertsPerRun = newValue;
    }
    
    private class InsertRunnable implements Runnable
    {
	public void run() 
	{
	    synchronized (insertCache) {
		int j = 0;
		for (Iterator i = insertCache.iterator();
		     i.hasNext() && j < maxInsertsPerRun; j++) {
		    ListEvent e = (ListEvent)i.next();
		    if (e.getID() == ListEvent.ELEMENT_ADDED) {
			listener.elementAdded(e);
		    }
		    else {
			listener.elementRemoved(e);
		    }

		    i.remove();
		}

		if (insertCache.size() > 0) {
		    SwingUtilities.invokeLater(this);
		}    
		else {
		    insertPending = false;
		}
	    }
	}
    }

}
