/*
 *  Java Napster version x.yz (for current version number as well as for
 *  additional information see version.txt)
 *
 *  Previous versions of this program were written by Florian Student
 *  and Michael Ransburg available at www.weblicity.de/jnapster and
 *  http://www.tux.org/~daneel/content/projects/10.shtml respectively.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.net;

import xnap.net.IBrowse;
import xnap.net.IChannel;
import xnap.net.IUser;
import xnap.plugin.nap.net.msg.MessageHandler;
import xnap.plugin.nap.net.msg.client.PrivateMessage;
import xnap.plugin.nap.net.msg.client.WhoisRequestMessage;
import xnap.plugin.nap.util.NapPreferences;
import xnap.user.UserData2;
import xnap.user.UserManager;
import xnap.util.Preferences;

public class User extends AbstractNapUser {

    //--- Constant(s) ---

    public static final int WHOIS_REQUERY_INTERVALL = 2 * 60 * 1000;
    public static final int WHOIS_RESENT_INTERVALL = 10 * 60 * 1000;

    //--- Data field(s) ---

    /**
     * Milli seconds when last whois query for this user was sent.
     */
    private long lastWhoisSent = 0;

    /**
     * Milli seconds when last whois query for this user was received.
     */
    private long lastWhoisResponse = 0;

    private Server server;
    private GlobalUser parent;

    //--- Constructor(s) ---

    public User(String name, Server server)
    {
	this.server = server;

	if (name.equals(server.getUsername())) {
	    fakeLocalFileCount = true;
	}

	this.parent = getParentByName(name);
	this.parent.addServer(server);
    }

    //--- Method(s) ---

    public static GlobalUser getParentByName(String name)
    {
	GlobalUser parent = (GlobalUser)UserManager.getInstance().get(name);
	if (parent == null) {
	    parent = new GlobalUser(name, true);
	    UserManager.getInstance().add(parent);
	}

	return parent;
    }	

    public IBrowse getBrowse()
    {
	return new Browse(this);
    }

    public UserData2 getData()
    {
	return parent.getData();
    }

    public IChannel getPrivateChannel()
    {
	return MessageHandler.getPrivateChannel(this);
    }

    public int getQueuedCount()
    {
	return parent.getQueuedCount();
    }

    public Server getServer()
    {
	return server;
    }

    public synchronized void incQueuedCount(int newValue)
    {
	parent.incQueuedCount(newValue);
    }

    public boolean isActionSupported(Class actionClass)
    {
	return true;
    }

    /**
     * Checks if peer shares enough to be allowed to download. Serves to
     * block leechers. Automatically  calls <code>update()</code> if 
     * needed. 
     */
    public boolean isAllowedToDownload()
    {
	if (getMaxUploads() == TRANSFER_UNLIMITED) {
	    return true;
	}

	NapPreferences prefs = NapPreferences.getInstance();

	if (prefs.getUseMinimumShares()) {
	    if (update() && getFileCount() < prefs.getMinimumShares()) {
		// send message to peer
		if (prefs.getSendMinimumSharesMessage()) {
		    PrivateMessage msg = new PrivateMessage
			(getName(), prefs.getMinimumSharesMessage());
		    MessageHandler.send(server, msg);
		}
		return false;
	    }
	}
	return true;
    } 

    public boolean isAllowedToRequestDownload()
    {
	Preferences prefs = Preferences.getInstance();

	if (prefs.getLimitDownloadsPerUser()) {
	    if (getMaxDownloads() == IUser.TRANSFER_NEVER) {
		return false;
	    }
	    else if (getMaxDownloads() == IUser.TRANSFER_DEFAULT) {
		return (getLocalDownloadCount() + getQueuedCount() 
			< prefs.getMaxDownloadsPerUser());
	    }
	}
	
	return true;
    }

    public boolean isUpToDate()
    {
	return (System.currentTimeMillis() - lastWhoisResponse
		< WHOIS_RESENT_INTERVALL);
    }

    /**
     * Called by message class when whois query was received.
     */
    public void notifyWhoisReceived()
    {
	lastWhoisResponse = System.currentTimeMillis();
	lastWhoisSent = 0;
    }

    public void setStatus(int newValue)
    {
	parent.setStatus(newValue);
	super.setStatus(newValue);
    }

    public void setStatus(String newValue)
    {
	String s = newValue.toLowerCase();
	if (s.equals("active")) {
	    setStatus(IUser.STATUS_ONLINE);
	}
	else if (s.equals("remote")) {
	    setStatus(IUser.STATUS_AWAY);
	}
	else { // "inactive"
	    setStatus(IUser.STATUS_OFFLINE);
	}
    }

    public void setTemporary(boolean newValue)
    {
	super.setTemporary(newValue);
	parent.setTemporary(newValue);
    }

    public boolean shouldRequery()
    {
	return (System.currentTimeMillis() - lastWhoisSent
		> WHOIS_REQUERY_INTERVALL);
    }

    public String toString() 
    {
	StringBuffer sb = new StringBuffer();
	sb.append(getName());
	if (server != null) {
	    sb.append("@");
	    sb.append(server.getHost());
	}

	return sb.toString();
    }

    /**
     * Sends a whois query to the server if not up to date.
     *
     * @return true, if up to date; false, if not.
     */
    public boolean update(boolean force)
    {
	if (force || !isUpToDate()) {
	    if (shouldRequery()) {
		MessageHandler.send(server, 
				    new WhoisRequestMessage(getName()));
		lastWhoisSent = System.currentTimeMillis();
	    }
	    return false;
	}
	    
	return true;
    }

    public boolean update()
    {
	return update(false);
    }

}
