/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.net.msg;

import xnap.plugin.nap.Plugin;
import xnap.plugin.nap.net.Server;
import xnap.plugin.nap.util.Connector;
import xnap.plugin.nap.net.msg.client.ClientMessage;

import java.io.*;
import java.util.*;
import org.apache.log4j.Logger;

public class SendQueue {

    //--- Constant(s) ---

    //--- Data field(s) ---

    private static Logger logger = Logger.getLogger(SendQueue.class);

    /**
     * Queued messages.
     */
    private LinkedList queue = new LinkedList();

    /**
     * Associated server.
     */
    private Server server;

    private boolean locked = false;

    //--- Constructor(s) ---

    public SendQueue(Server server)
    {
	this.server = server;
    }
    
    //--- Method(s) ---

    /**
     * Returns true, if a send worker needs to be notified.
     */
    public synchronized boolean add(ClientMessage msg, boolean lowPriority)
    {
	if (lowPriority) {
	    queue.addLast(msg);
	}
	else {
	    queue.addFirst(msg);
	}

	//logger.debug(server + ": " + queue.size() + " queued");

	return !isLocked();
    }

    /**
     * Worker could not send message, because server has died. 
     * Remove all pending messages.
     */
    public synchronized void clear()
    {
	logger.debug("removing all " + queue.size() 
		     + " messages for: " + server);
	for (Iterator i = queue.iterator(); i.hasNext();) {
	    ClientMessage msg = (ClientMessage)i.next();
	    if (msg.listener != null) {
		msg.listener.exceptionThrown
		    (new IOException(Plugin.tr("Server disconnected")));
	    }
	}
	queue.clear();
	setLocked(false);
    }

    public Server getServer()
    {
	return server;
    }

    /**
     * Returns true, if a send worker is currently sending messages from this queue.
     */
    public synchronized boolean isLocked()
    {
	return locked;
    }

    /**
     * Removes the top message from the queue.
     *
     * @return if queue is not empty, top message; null otherwise
     */
    public synchronized ClientMessage pop()
    {
	//logger.debug("popping from + " + server ": " + queue.size() + " queued");

	if (queue.isEmpty()) {
	    setLocked(false);
	    return null;
	}

	return (ClientMessage)queue.removeFirst();
    }

    public synchronized void setLocked(boolean newValue)
    {
	locked = newValue;
    }

}
