/*
 * sound.c, v0.5
 *
 * Copyright (C) 1998 Rasca, Berlin
 * EMail: thron@gmx.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include "sound.h"

#ifdef HasOSS
/* definitions for M$ WAVE format
 */
#define RIFF        0x46464952
#define WAVE        0x45564157
#define FMT         0x20746D66
#define DATA        0x61746164
#define PCM_CODE    1
#define WAVE_MONO   1
#define WAVE_STEREO 2

typedef struct {
  unsigned long    main_chunk; /* 'RIFF' */
  unsigned long    length;     /* filelen */
  unsigned long    chunk_type; /* 'WAVE' */
  unsigned long    sub_chunk;	/* 'fmt' */
  unsigned long    sub_c_len;	/* length of sub_chunk, =16 */
  unsigned short   format;     /* should be 1 for PCM-code */
  unsigned short   modus;      /* 1 mono, 2 stereo */
  unsigned long    sample_fq;  /* frequence of sample */
  unsigned long    byte_p_sec;
  unsigned short   byte_p_spl;	/* samplesize; 1 or 2 bytes */
  unsigned short   bits_p_spl;	/* 8, 12 or 16 bit */
} WAVheader;

typedef struct {
	unsigned long    data_chunk;	/* data */
	unsigned long    data_length;  /* samplecount */
} WAVdaHead;

#define VERBOSE		if(verbose)
#define VERBOSE2	if(verbose>1)

static char *dev = NULL;	/* name of the device */
static int audio = 0;		/* file descriptor of the device */
#ifdef DEBUG
static int verbose =1;
#else
static int verbose =0;
#endif

/*
 * check the device and remember the device name
 */
int snd_init (char *devname) {
	if (!devname)
		return (0);
	dev = (char *) malloc (strlen (devname) +1);
	if (!dev)
		return (0);
	strcpy (dev, devname);
	audio = open (dev, O_WRONLY);
	if (audio == -1) {
		perror (dev);
		return (0);
	}
	close (audio);
	return (1);
}

/*
 */
int
snd_open_play (void)
{
	audio = open (dev, O_WRONLY);
	if (audio == -1) {
		perror (dev);
		return (0);
	}
	return (audio);
}

/*
 */
int snd_close_play (void) {
	if (close (audio) == -1) {
		perror (dev);
		return (0);
	}
	audio =0;
	return (1);
}

/*
 */
int snd_open_rec (void) {
	audio = open (dev, O_RDONLY);
	if (audio == -1) {
		perror (dev);
		return (0);
	}
	return (1);
}

/*
 */
int snd_close_rec (void) {
	if (close (audio) == -1) {
		perror (dev);
		return (0);
	}
	audio =0;
	return (1);
}

/*
 */
void snd_verbose (int val) {
	verbose = val;
}

/*
 */
int snd_sync (void) {
#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_SYNC, 0) == 0) {
		return (1);
	}
#endif
	perror (dev);
	return (0);
}

/*
 */
int
snd_post (void) {
#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_POST, 0) == 0) {
		return (1);
	}
#endif
	perror (dev);
	return (0);
}

/*
 * must be called if sound was played and now
 * device options should be changed!
 */
int snd_reset (void) {

#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_RESET, 0) == 0) {
		return (1);
	}
#endif
	perror (dev);
	return (0);
}

/*
 */
int snd_get_fmts (void) {
#ifdef HasOSS
	int mask = 0;
	if (ioctl (audio, SNDCTL_DSP_GETFMTS, &mask) == 0) {
		return (mask);
	}
#endif
	perror (dev);
	return (0);
}

/*
 */
int
snd_set_fmt (int fmt)
{
	if (ioctl (audio, SNDCTL_DSP_SETFMT, &fmt) == 0) {
		return (fmt);
	}
	perror (dev);
	return(0);
}

/*
 * set the sample rate
 */
int snd_set_speed (int speed) {
	int tmp = speed;

#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_SPEED, &tmp) < 0) {
		perror (dev);
		return (0);
	}
#endif
	VERBOSE fprintf (stderr, "snd_set_speed (%d) ->%d\n", speed, tmp);
	return (tmp);
}

/*
 */
int snd_set_stereo (int boolval) {
	int tmp = boolval;
#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_STEREO, &tmp) < 0) {
		perror (dev);
		return (0);
	}
#endif
	VERBOSE fprintf (stderr, "snd_set_stereo (%s) ->%s\n",
					boolval ? "on" : "off", boolval != tmp? "failed" : "ok");
	if (boolval != tmp) {
		return (0);
	}
	return (1);
}

/*
 */
int
snd_set_channels (int num)
{
#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_CHANNELS, &num) < 0) {
		perror (dev);
		return (0);
	}
#endif
	return (num);
}

/*
 */
int snd_set_samplesize (int samplesize) {
	int tmp = samplesize;

#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_SAMPLESIZE, &tmp) < 0) {
		perror (dev);
		return (0);
	}
#endif
	VERBOSE fprintf (stderr, "snd_set_samplesize (%d) ->%s\n",
						samplesize, tmp == samplesize? "ok" : "failed");
	if (samplesize != tmp) {
		return (0);
	}
	return (tmp);
}

/*
 */
int snd_get_blksize (void) {
	int blksize = 0;

#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_GETBLKSIZE, &blksize) < 0) {
		perror (dev);
		return (0);
	}
#endif
	VERBOSE fprintf (stderr, "snd_get_buffer () ->%d\n", blksize);
	return (blksize);
}


/*
 * convert buffer from 16 to 8 bit
 */
int snd_16to8_bits (char *buf, int size) {
	unsigned long c;
	char *p1 = buf;
	char *p2 = buf;

	VERBOSE2 fprintf (stderr, "snd_16to8_bits(.., %d) ->%d\n", size, (size>>1));
	c = size >> 1;
	p2++;
	while (c--) {
		*p1++ = *p2 + 128;
		p2 += 2;
	}
	return (size >> 1);
}

/*
 * buff should be at least 36 bytes long ..
 * checks if it is a wave file and returns the sample size
 */
int snd_is_wave (unsigned char *buff) {
	WAVheader *wh;

	if (!buff)
		return (0);
	wh = (WAVheader *) buff;
	if (!(wh->main_chunk == RIFF && wh->chunk_type == WAVE &&
			wh->sub_chunk == FMT)) {
		return (0);
	}
	if (wh->modus > 2) {
		return (0);
	}
	return (wh->bits_p_spl);
}

/*
 */
int snd_wave_speed (unsigned char *buff) {
	WAVheader *wh;

	if (!buff)
		return (0);
	wh = (WAVheader *) buff;
	return (wh->sample_fq);
}

/*
 * returns the length of the sound data in the wave file
 */
int snd_wave_length (unsigned char *buff) {
	WAVdaHead *wdh;

	if (!buff)
		return (0);
	wdh = (WAVdaHead *) (buff + sizeof (WAVheader));
	return (wdh->data_length);
}

/*
 */
int snd_wave_stereo (unsigned char *buff) {
	WAVheader *wh;

	if (!buff)
		return (0);
	wh = (WAVheader *) buff;
	if (wh->modus == WAVE_STEREO)
		return (1);
	return (0);
}

/*
 * 'header' must be at least 44 bytes long
 */
int snd_wave_info (unsigned char *header, int *mode) {
	int samplesize;

	*mode = 0;
	samplesize = snd_is_wave(header);
	if (!samplesize)
		return (0);
	if (samplesize == 16)
		*mode |= SND_16BIT;
	else
		*mode |= SND_8BIT;
	if (snd_wave_stereo (header))
		*mode |= SND_STEREO;
	else
		*mode |= SND_MONO;
	return(snd_wave_speed (header));
}

/*
 * prepare for playing wave files, returns the internal buffer
 * fragmentation size which we need to build one buffer,
 */
int snd_wave_prepare (int *speed, int *mode) {
	int samplesize, blksize;

	samplesize = snd_set_samplesize (*mode & (SND_16BIT | SND_8BIT));
	if (!samplesize) {
		samplesize = snd_set_samplesize (8);
		*mode |= SND_16TO8;
		*mode  = *mode & ~SND_16BIT;
		*mode  = *mode | SND_8BIT;
	}
	if (*mode & SND_STEREO) {
		snd_set_stereo(1);
	} else
		snd_set_stereo(0);
	*speed = snd_set_speed (*speed);

	blksize = snd_get_blksize();
	if (*mode & SND_16TO8)
		blksize *= 2;
	return (blksize);
}

/*
 */
int snd_wave_prepare_header (int speed, int mode, char **dst) {
	static char header[sizeof(WAVheader) + sizeof (WAVdaHead)];
	WAVheader *wav;
	WAVdaHead *da;
	wav = (WAVheader *) header;
	wav->main_chunk = RIFF;
	wav->length     = 0;
	wav->chunk_type = WAVE;
	wav->sub_chunk  = FMT;
	wav->sub_c_len	= 16;
	wav->format		= PCM_CODE;
	wav->modus		= (mode & SND_MONO) ? WAVE_MONO : WAVE_STEREO;
	wav->sample_fq	= speed;
	wav->byte_p_spl	= (mode & SND_16BIT) ? 2 : 1;
	wav->byte_p_sec	= wav->sample_fq * wav->modus * wav->byte_p_spl;
	wav->bits_p_spl	= (mode & SND_16BIT) ? 16 : 8;
	da = (WAVdaHead*) (header + sizeof (WAVheader));
	da->data_chunk = DATA;
	da->data_length= 0;
	*dst = header;
	return (sizeof (WAVheader) + sizeof (WAVdaHead));
}

/*
 */
int snd_get_interval (bufsize, speed, mode) {
	int kb_per_sec, milisec;

	if (mode & SND_16TO8)
		bufsize /= 2;
	kb_per_sec = speed * ((mode & (SND_16BIT | SND_8BIT)) / 8) *
				((mode & SND_STEREO) ? 2 : 1);

	milisec = 1000 * ((double)1 / ((double)kb_per_sec / bufsize));

	VERBOSE fprintf (stderr, "snd_get_interval (%d, %d, %d) %d ->%d mili sec\n",
						bufsize, speed, mode, kb_per_sec, milisec);
	return (milisec);
}

/*
 * play sound data from buffer
 */
int snd_play_data (unsigned char *buf, int size, int mode) {
	unsigned char *p;
	int written;

	if (mode & SND_16TO8) {
		p = (unsigned char *) malloc (sizeof (unsigned  char) * size);
		if (!p)
			return (0);
		memcpy (p, buf, size);
		size = snd_16to8_bits (p, size);
	} else {
		p = buf;
	}

	VERBOSE2 fprintf (stderr, "snd_play_data (.., %d, ..)\n", size);
	written = write (audio, p, size);
	if (mode & SND_16TO8)
		free (p);
	return (written);
}

/*
 */
int snd_rec_data (unsigned char *buf, int max_size) {
	int num;
	num = read (audio, buf, max_size);
	return (num);
}

/*
 */
unsigned int snd_get_caps (void) {
	unsigned int caps = 0;

#ifdef HasOSS
	if (ioctl (audio, SNDCTL_DSP_GETCAPS, &caps) < 0) {
		perror (dev);
		return (0);
	}
	VERBOSE fprintf (stderr, "snd_get_caps () ->0x%x\n", caps);
#endif
	return (caps);
}

/*
 */
int snd_has_feature (unsigned int feature) {
	int caps;
	if ((caps = snd_get_caps()) == 0)
		return (-1);
	return (caps & feature);
}

/*
 */
int
snd_set_trigger_out (void)
{
	int trigger = ~PCM_ENABLE_OUTPUT;
	ioctl (audio, SNDCTL_DSP_SETTRIGGER, &trigger);
	trigger = PCM_ENABLE_OUTPUT;
	ioctl (audio, SNDCTL_DSP_SETTRIGGER, &trigger);
	return (trigger);
}

#endif
