/*
 * xt.c
 * gnutime/gnuavi test program - Xt frontend
 *
 * Copyright (C) 1998 Rasca, Berlin
 * EMail: thron@gmx.de
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <sys/time.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#ifdef HasSharedMemory
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#endif
#include "video.h"
#include "sound.h"
#include "xt.h"

static void x_fini(void);

static XtAppContext gAppCont;
static Widget gTop, gVid;
static String fallback[] = {
	"*background: blue",
	"*translations: #override \\n\\"
	" <Key>Q: quit() \\n\\"
	" <Key>P: play() \\n\\"
	" <Key>S: stop() \\n",
	NULL
};

typedef struct {
	int loop;		/* flag if looping is on */
	int depth;		/* display bits per pixel */
	video *vid;		/* */
	long time_ms;	/* */
	unsigned char *image;
	int use_snd;
	int use_shm;
	Widget widget;
	XImage *ximage;
	GC gc;
} app;

/*
 * called if "q" was pressed
 */
static void
action_quit (Widget w, XEvent *ev, String *parms, Cardinal *num_parms)
{
	x_fini();
}

/*
 */
static void
action_stop (Widget w, XEvent *ev, String *parms, Cardinal *num_parms)
{
	printf ("stop() not done ..\n");
}

/*
 */
static void
action_play (Widget w, XEvent *ev, String *parms, Cardinal *num_parms)
{
	printf ("play() not done..\n");
}

static XtActionsRec action_tab[] = {
	{"stop",	action_stop},
	{"play",	action_play},
	{"quit",	action_quit},
	};

/*
 * exit event handler
 */
static void
evh_exit (Widget w, XtPointer xtp, XEvent *ev, Boolean *boo)
{
	if ((ev->type == ClientMessage) &&                      
		(ev->xclient.format == 32) &&
		(ev->xclient.data.l[0] == *((Atom*)xtp)))
	x_fini();
}

void
x_play_sound (XtPointer xtp, XtIntervalId *id)
{
}

/*
 * returns time in mili secs
 */
long
time_ms (void)
{
	struct timeval curr_time;
	gettimeofday (&curr_time, NULL);
	return ((curr_time.tv_sec * 1000L + curr_time.tv_usec / 1000L)&0x7FFFFF);
}

/*
 * time callback procedure to display a frame
 */
void
x_display_frame (XtPointer xtp, XtIntervalId *id)
{
	app *ap = (app *)xtp;
	video *vid = ap->vid;
	XImage *ximage = ap->ximage;
	char *buf = ap->image;
	unsigned char *snd_buf;
	long time;
	int todo = 1, snd_size =0;

	time = ap->time_ms;
	ap->time_ms = time_ms();
	/* printf (" -t %ld %ld\n", ap->time_ms, time); */

	while (todo) {
		switch (vid_next_in(vid)) {
			case VNI_VIDEO:
				if (!vid_get_frame (vid, buf)) {
					if (ap->loop != 1)
						return;
					vid_reset (vid);
					XtAppAddTimeOut (gAppCont, vid->tpf, x_display_frame, ap);
					return;
				}
				/* printf (" -f\n"); */
				todo = 0;
				break;
			case VNI_SOUND:
				if (ap->use_snd) {
					if (vid_get_sound (vid, &snd_buf, &snd_size)) {
						snd_play_data (snd_buf, snd_size, 0);
						/* printf (" -s snd_size=%d\n", snd_size); */
					} else {
						vid_skip_obj(vid);
						todo = 0;
					}
				} else
					vid_skip_obj (vid);
				break;
			default:
				if (ap->loop) {
					vid_reset (vid);
					XtAppAddTimeOut (gAppCont, vid->tpf, x_display_frame, ap);
				}
				return;
				break;
		}
	}
#ifdef HasSharedMemory
	if (ap->use_shm)
		XShmPutImage (XtDisplay(gTop), XtWindow(ap->widget), ap->gc, ximage,
				0, 0, 0, 0, vid->width, vid->height, True);
	else
#endif
		XPutImage (XtDisplay(gTop), XtWindow(ap->widget), ap->gc, ximage,
				0, 0, 0, 0, vid->width, vid->height);

	time = vid->tpf - (time_ms() - time) ;
	/* printf (" -t %ld\n", time); */
	if (time < 0) {
		if (vid->tpf + time < 0)
			vid_skip_frame (vid);
		time = 0;
	}
#ifdef DEBUG
	printf ("x_display_frame() time=%ld tpf=%d\n", time, vid->tpf);
#endif
	XtAppAddTimeOut (gAppCont, time, x_display_frame, ap);
}

/*
 */
void
x_init (int *argc, char **argv)
{
	gTop = XtVaAppInitialize (&gAppCont, "XMplay", NULL, 0,
				argc, argv, fallback, XtNinput, True, NULL);
	XtAppAddActions (gAppCont, 
				action_tab, sizeof(action_tab)/sizeof(XtActionsRec));
}

/*
 */
void
x_main (int width, int height)
{
	static Atom wm_del;

	gVid = XtVaCreateManagedWidget ("xvid", coreWidgetClass, gTop,
				XtNwidth, width, XtNheight, height, NULL);
	XtRealizeWidget (gTop);
	wm_del = XInternAtom (XtDisplay(gTop), "WM_DELETE_WINDOW", False);
	XSetWMProtocols (XtDisplay(gTop), XtWindow(gTop), &wm_del, 1);
	XtAddEventHandler (gTop, NoEventMask, True, evh_exit, &wm_del);
}

/*
 */
static app ap;
static XShmSegmentInfo shminfo;

/*
 */
void
x_loop (video *vid, int use_shm, int loop, int use_snd)
{
	int pad, codec;
	Screen *screen = XtScreen (gTop);
	Display *dpy = XtDisplay(gTop);
	Visual *vis = DefaultVisualOfScreen (screen);
#ifdef HasSharedMemory

	if (use_shm) {
		use_shm = XShmQueryExtension(dpy);
	}
#else
	use_shm = 0;
#endif

	ap.loop = loop;
	ap.vid = vid;
	ap.depth = DefaultDepthOfScreen (screen);
	ap.widget = gVid;
	ap.gc = XCreateGC (dpy, XtWindow(ap.widget), 0, NULL);
#ifdef HasSharedMemory
	if (use_shm) {
		ap.ximage =XShmCreateImage (dpy, vis, ap.depth, ZPixmap, NULL,
					&shminfo, vid->width, vid->height);
	} else
#endif
			{
		ap.image = XtMalloc (vid->width * vid->height * 4 /* ap.bpp / 8*/);
		ap.ximage =XCreateImage (dpy, vis,
						ap.depth, ZPixmap, 0, ap.image, vid->width, vid->height,
						8, 0);
	}

	if (!ap.ximage) {
		fprintf (stderr, "Can't create XImage!\n");
		exit (2);
	}
	pad = ap.ximage->bytes_per_line -
				ap.ximage->width * ap.ximage->bits_per_pixel / 8 ;
#ifdef DEBUG
	printf (" ximage bytes /p line = %d\n", ap.ximage->bytes_per_line);
	printf ("                  bpp = %d\n", ap.ximage->bits_per_pixel);
	printf ("                depth = %d\n", ap.ximage->depth);
	printf ("                  pad = %d\n", pad);
#endif
	switch (ap.ximage->depth) {
		case 15:
			codec = VV_COD_RGB555;
			break;
		case 16:
			codec = VV_COD_RGB565;
			break;
		case 24:
			codec = VV_COD_RGB888;
			break;
		default:
			printf ("visual not supported\n");
			exit (2);
	}
	vid_set_video_parms (vid, codec, pad);

#ifdef HasSharedMemory
	if (use_shm) {
		shminfo.shmid = shmget (IPC_PRIVATE,
							vid->width * vid->height * 4 /* (ap.bpp/8)*/,
							IPC_CREAT|0777);
		if (shminfo.shmid == -1) {
			perror ("shmget()");
			exit (2);
		}
		shminfo.shmaddr = ap.image = shmat (shminfo.shmid, 0, 0);
		shminfo.readOnly = False;
		ap.ximage->data = shminfo.shmaddr;
		if (XShmAttach (dpy, &shminfo) == 0) {
			perror ("XShmAttach()");
			exit (2);
		}
	}
#endif
	ap.use_shm = use_shm;
	ap.use_snd = use_snd;
	ap.time_ms = time_ms();

	XtAppAddTimeOut (gAppCont, 1, x_play_sound, &ap);
	XtAppAddTimeOut (gAppCont, 2, x_display_frame, &ap);
	XtAppMainLoop(gAppCont);
}

/*
 */
void
x_fini (void)
{
#ifdef HasSharedMemory
	if (ap.use_shm) {
		XShmDetach (XtDisplay(gTop), &shminfo);
		shmdt (shminfo.shmaddr);
		shmctl (shminfo.shmid, IPC_RMID, 0);
	} else
#endif
	{
		XtFree (ap.image);
	}
	exit (0);
}

