/*
 * main.c,
 *
 * Copyright (C) 1997,98 Rasca, Berlin
 * Copyright (C) 2003,04 Karl, Frankfurt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../config.h"  /* autoconf output */

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <signal.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#ifdef HAVE_SHMAT
#include <X11/extensions/XShm.h>
#endif // HAVE_SHMAT
#ifdef HasDGA
#include <X11/extensions/xf86dga.h>
#endif //HasDGA
#ifdef HasVideo4Linux
#include <video.h>
#endif //HasVideo4Linux
#include "main.h"
#include "control.h"
#include "frame.h"
#include "codecs.h"
#include "job.h"

static AppData app;

void
usage(char *prog) {
    printf("Usage: %s, ver %s, (c) rasca, berlin 1997,98,99, khb (c) 2003\n", prog, VERSION);
    printf("[--fps #.#]         frames per second (float)\n");
    printf("[--verbose #]       verbose level, '-v' is --verbose 1\n");
    printf("[--time #.#]        time to record in seconds (float)\n");
    printf("[--frames #]        frames to record, don't use it with --time\n");
    printf("[--continue [yes|no]]        autocontinue after maximum frames/time\n");
    printf("[--cap_geometry #x#[+#+#]]    size of the capture window (WIDTHxHEIGHT+X+Y)\n");
    printf("[--start_no #]      start number for the file names\n");
    #ifdef HAVE_LIBZ
    printf("[--compress <0-9>]  save the frames compressed, '.gz' is added\n");
    #endif
    printf("[--use_clone]       use the CLON chunk for MNGs if needed\n");
    printf("[--expand_to_24bit] expand 15/16bpp to 24bpp (only for PPM)\n");
    #ifdef HAVE_LIBJPEG
    printf("[--quality #]       quality value for JPEG files (100 - 0)\n");
    #endif
    printf("[--source <src>]    select input source: x11, shm\n");
    printf("[--file <file>]     file pattern, e.g. out%%03d.xwd\n");
    printf("[--gui [yes|no]]    turn on/off gui\n");
#ifdef HAVE_FFMPEG_AUDIO
    printf("[--audio [yes|no]]  turn on/off audio capture\n");
    printf("[--audio_in <src>]  specify audio input device or '-' for pipe input\n");
    printf("[--audio_rate #]    sample rate for audio capture\n");
    printf("[--audio_bits #]    bit rate for audio capture\n");
    printf("[--audio_channels #]    number of audio channels\n");
#endif // HAVE_FFMPEG_AUDIO
    printf("Supported output formats:\n");
    printf(" X window dump              (.xwd)\n");
    printf(" Portable anymap            (.pnm, .pgm, .ppm)\n");
    #ifdef HAVE_LIBPNG
    printf(" Portable network graphics  (.png)\n");
    printf(" Multiple network graphics  (.mng)\n");
    #endif
    #ifdef HAVE_LIBJPEG
    printf(" Joint picture expert group (.jpg, .jpeg)\n");
    #endif
    printf(" MPEG                       (.mpeg)\n");
    exit(1);
}


void
cleanup_when_interrupted (int signal) {
    Job *job = job_ptr();
    
    XVC_StopCapture (job);
}


/*
 * main()
 */
int
main(int argc, char **argv) {

    struct option options[] = {
        {"fps",      required_argument, NULL, 0},
        {"file",     required_argument, NULL, 0},
        {"verbose",  required_argument, NULL, 0},
        {"time",     required_argument, NULL, 0},
        {"frames",   required_argument, NULL, 0},
        {"cap_geometry", required_argument, NULL, 0},
        {"start_no", required_argument, NULL, 0},
        {"compress", required_argument, NULL, 0},
        {"use_clone", no_argument,		NULL, 0},
        {"expand_to_24bit", no_argument, NULL, 0},
        {"quality",  required_argument, NULL, 0},
        {"bpp", 	 required_argument, NULL, 0},
        {"source", 	 required_argument, NULL, 0},
        {"sync", 	 no_argument,		NULL, 0},
        {"step", 	 required_argument,	NULL, 0},
        {"gui",     optional_argument, NULL, 0},
        {"audio",     optional_argument, NULL, 0},
        {"audio_in", required_argument, NULL, 0},
        {"audio_rate", required_argument, NULL, 0},
        {"audio_bits", required_argument, NULL, 0},
        {"audio_channels", required_argument, NULL, 0},
        {"continue", optional_argument, NULL, 0},
        {NULL, 0, NULL, 0},
    };

    int opt_index = 0, c, resultCode; //, flags =0;
    #ifdef HasDGA
    int dga_evb, dga_errb;
    #endif //HasDGA
    Job main_job;
    
    app.cap_pos_x = app.cap_pos_y = -1;
    
    app.flags = 0;
    app.use_clone = 0;
    app.device = "/dev/video0";
    
    app.quality = 99;
    app.source = NULL;
    app.step = 1;
    app.bpp = 0;
    app.targetCodec = CODEC_MPEG1;
    app.mouseWanted = 1;
    
    app.snddev = "/dev/dsp";
    app.sndrate = 22050;
    app.sndsize = 32000;
    app.sndchannels = 1; 
//    app.flags |= FLG_AUDIO_WANTED;
    
    // this is a hook for a GUI to do some pre-init functions ...
    // possibly to set some fallback options read from a rc file or Xdefaults
    if (! XVC_PreInit( argc, argv, &app ) ) {
        fprintf(stderr, "Can't do GUI pre_init ... aborting\n");
        exit(2);
    }
    
    /* read options file now */
    XVC_ReadOptionsFile( &app );
    
    if (app.source) {
        if (strstr(app.source, "v4l") != NULL) {
            app.flags |= FLG_USE_V4L;
            if (strchr(app.source, ':') != NULL) {
                app.device = strchr(app.source, ':') + 1;
            }
        } else if (strstr(app.source, "dga") != NULL)
            app.flags |= FLG_USE_DGA;
        else if (strstr(app.source, "shm") != NULL)
            app.flags |= FLG_USE_SHM;
    }
    while ((c = getopt_long(argc, argv, "v", options, &opt_index)) != -1) {
        switch (c) {
            case 0: /* is a long option */
                switch (opt_index) {
                    case 0: /* frames per second */
                        app.fps = atof(optarg);
                        break;
                    case 1: /* file for saving */
                        app.file = strdup(optarg);
                        break;
                    case 2:
                        app.verbose = atoi(optarg);
                        break;
                    case 3: /* time to record */
                        app.time = atof(optarg);
                        if ( app.time != 0 ) app.frames = 0;
                        break;
                    case 4:
                        app.frames = atoi(optarg);
                        if ( app.frames != 0 ) app.time = 0;
                        break;
                    case 5: /* cap_geometry */
                        sscanf(optarg, "%dx%d+%d+%d", &app.cap_width, &app.cap_height, &app.cap_pos_x, &app.cap_pos_y);
                        break;
                    case 6:
                        app.start_no = atoi(optarg);
                        break;
                    case 7:
                        app.compress = atoi(optarg);
                        break;
                    case 8:
                        app.use_clone = 1;
                        break;
                    case 9:
                        app.flags |= FLG_EXPAND_TO_24BIT;
                        break;
                    case 10:
                        app.quality = atoi(optarg);
                        if ((app.quality < 0) || (app.quality > 100))
                            app.quality = 75; /* reset to default */
                        break;
                    case 11:
                        app.bpp = atoi(optarg);
                        break;
                    case 12:
                        app.source = strdup(optarg);
                        if (strstr(optarg, "v4l") != NULL) {
                            app.flags &= ~FLG_SOURCE;
                            app.flags |= FLG_USE_V4L;
                            if (strchr(optarg, ':') != NULL) {
                                app.device = strchr(optarg, ':')+1;
                            }
                        } else if (strstr(optarg, "dga") != NULL) {
                            app.flags &= ~FLG_SOURCE;
                            app.flags |= FLG_USE_DGA;
                        } else if (strstr(optarg, "shm") != NULL) {
                            app.flags &= ~FLG_SOURCE;
                            app.flags |= FLG_USE_SHM;
                        } else {
                            app.flags &= ~FLG_SOURCE;
                            printf("using normal x server access ..\n");
                        }
                        break;
                    case 13:
                        app.flags |= FLG_SYNC;
                        break;
                    case 14:
                        app.step = atoi(optarg);
                        break;
                    case 15: // gui
                        {
                            char *tmp;
                            if (! optarg) {
                                if (optind < argc) {
                                    tmp = (argv[optind][0] == '-')?"yes":argv[optind++];
                                } else {
                                    tmp = "yes";
                                }
                            } else {
                                tmp = strdup(optarg);
                            }
                            if (strstr(tmp, "no") != NULL) {
                                app.flags |= FLG_NOGUI;
                            } else {
                                app.flags &= ~FLG_NOGUI;
                            }
                        }
                        break;
                    case 16: //audio
                        {
                            char *tmp;
                            if (! optarg) {
                                if (optind < argc) {
                                    tmp = (argv[optind][0] == '-')?"yes":argv[optind++];
                                } else {
                                    tmp = "yes";
                                }
                            } else {
                                tmp = strdup(optarg);
                            }
                            if (strstr(tmp, "no") != NULL) {
                                app.flags &= ~FLG_AUDIO_WANTED;
                            } else {
                                app.flags |= FLG_AUDIO_WANTED;
                            }
                        }
                        break;
                    case 17: //audio_in
                        app.flags |= FLG_AUDIO_WANTED;
                        app.snddev = strdup(optarg);
                        break;
                    case 18: //audio_rate
                        app.flags |= FLG_AUDIO_WANTED;
                        app.sndrate = atoi(optarg);
                        break;
                    case 19: //audio_bits
                        app.flags |= FLG_AUDIO_WANTED;
                        app.sndsize = atoi(optarg);
                        break;
                    case 20: //audio_channels
                        app.flags |= FLG_AUDIO_WANTED;
                        app.sndchannels = atoi(optarg);
                        break;
                    case 21: //continue
                        {
                            char *tmp;
                            if (! optarg) {
                                if (optind < argc) {
                                    tmp = (argv[optind][0] == '-')?"yes":argv[optind++];
                                } else {
                                    tmp = "yes";
                                }
                            } else {
                                tmp = strdup(optarg);
                            }
                            if (strstr(tmp, "no") != NULL) {
                                app.flags &= ~FLG_AUTO_CONTINUE;
                            } else {
                                app.flags |= FLG_AUTO_CONTINUE;
                            }
                        }
                        break;
                    default:
                        usage(argv[0]);
                        break;
                }
                break;
                
            case 'v':
                app.verbose++;
                break;
                
            default:
                usage(argv[0]);
                break;
        }
    }
    if (argc != optind)
        usage(argv[0]);
    
    if (app.verbose)
        app.flags |= FLG_RUN_VERBOSE;
    if (app.use_clone)
        app.flags |= FLG_MNG_CLONE;
    
    if (app.flags & FLG_USE_V4L) {
        #ifndef HasVideo4Linux
        app.flags &= ~FLG_USE_V4L;
        #else //HasVideo4Linux
        VIDEO *video;
        
        video = video_open(app.device, O_RDWR);
        if (!video)
            exit(3);
        if (app.verbose) {
            printf("%s:\n cctm=%d channels=%d audios=%d maxwidth=%d "
            "maxheight=%d minwidth=%d minheight=%d\n",
            video->name, video->type&VID_TYPE_CAPTURE,
            video->channels, video->audios, video->maxwidth,
            video->maxheight, video->minwidth, video->minheight);
        }
        if (!(video->type & VID_TYPE_CAPTURE)) {
            fprintf(stderr, "video device can't capture to memory\n");
            exit(3);
        }
        app.cap_width = min(app.cap_width, video->maxwidth);
        app.cap_height = min(app.cap_height, video->maxheight);
        app.cap_width = max(app.cap_width, video->minwidth);
        app.cap_height = max(app.cap_height, video->minheight);
        video_close(video);
        #endif //HasVideo4Linux
    }
    
    #ifndef HAVE_LIBZ
    app.compress = 0;
    #else //HAVE_LIBZ
    if (app.compress > 9)
        app.compress = 9;
    #endif //HAVE_LIBZ
    

    // these are the hooks for a GUI to create the GUI,
    // the selection frame, and do some initialization ...
    if ( ! XVC_CreateGUI(&app) ) {
        fprintf(stderr, "Can't create GUI ... aborting\n");
        exit(2);
    }
    if ( ! XVC_CreateFrame(&app) ) {
        fprintf(stderr, "Can't create selection Frame ... aborting\n");
        exit(2);
    }
    if ( ! XVC_InitGUI(&app) ) {
        fprintf(stderr, "Can't initialize GUI ... aborting\n");
        exit(2);
    }
    if ( app.cap_pos_x >= 0 || app.cap_pos_y >= 0 ) {
        XVC_ChangeFrame ( app.cap_pos_x, app.cap_pos_y, app.cap_width, app.cap_height, TRUE );
    }
    
    
    if (app.verbose) {
        char *mp;
        switch (app.mouseWanted) {
            case 2:
                mp = "black";
                break;
            case 1:
                mp = "white";
                break;
            default:
                mp = "none";
        }
        
        printf("Current settings:\n");
        printf("  frames per second = %f\n", app.fps);
        printf("  video encoding    = %s\n", tCodecNames[app.targetCodec]);
        printf("  file pattern      = %s\n", app.file);
        printf("  verbose level     = %d\n", app.verbose);
        printf("  frames to store   = %d\n", app.frames);
        printf("  time to capture   = %f sec\n", app.time);
        printf("  autocontinue      = %s\n", ((app.flags & FLG_AUTO_CONTINUE)?"yes":"no"));
        #ifdef HAVE_LIBZ
        printf("  compression level = %d\n", app.compress);
        #endif
        #ifdef HAVE_LIBJPEG
        printf("  quality (jpeg)    = %d\n", app.quality);
        #endif
        printf("  input source      = %s (%d)\n", app.source, app.flags & FLG_SOURCE);
        printf("  capture pointer   = %s\n", mp);
#ifdef HAVE_FFMPEG_AUDIO
        printf("  capture audio     = %s\n", ((app.flags & FLG_AUDIO_WANTED)?"yes":"no"));
	printf("   - input          = %s\n", app.snddev);
	printf("   - sample rate    = %i\n", app.sndrate);
	printf("   - bit rate       = %i\n", app.sndsize);
	printf("   - channels       = %i\n", app.sndchannels);
#endif // HAVE_FFMPEG_AUDIO
        printf("  animate command   = %s\n", app.play_cmd);
        printf("  make video command= %s\n", app.video_cmd);
        printf("  edit frame command= %s\n", app.edit_cmd);
        printf("  help command      = %s\n", app.help_cmd);
    }
    
    signal(SIGINT, cleanup_when_interrupted);
    
    // this is a hook for the GUI's main loop
    resultCode = XVC_RunGUI();
    return (resultCode);
}



