/*
 * xt_options.c
 *
 * Copyright (C) 1997,98 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * This file contains the Xt GUI's options dialog
 *
 */

#include "../config.h"	/* autoconf output */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include "Xw/Label.h"
#include "Xw/Box.h"
#include "Xw/Field.h"
#include "Xw/Button.h"
#include "Xw/Toggle.h"
#include "xt_options.h"
#include "job.h"
#include "xutil.h"
#include "app_data.h"

#define DLG_CANCEL	0
#define DLG_OK		1
#define DLG_SAVE	2

static Widget wShell, wFps, wFile, wMaxFrames, wMaxTime, wQuality;
static Widget wCompress, wCodec, wShm, wMouse, wMouseType;
extern Widget next, prev;
static Job *sJob;

void ChangeLabel (int pic_no);
static void CbOk (Widget w, XtPointer client_data, XtPointer call_data);
static void CbSave (Widget w, XtPointer client_data, XtPointer call_data);
static void CbCancel (Widget w, XtPointer client_data, XtPointer call_data);
static void CbActivate (Widget w, XtPointer client_data, XtPointer call_data);


/*
 * read the new values and close the window
 */
static void
CbOk (Widget w, XtPointer client_data, XtPointer call_data)
{
#include "codecs.h"

	char *str;
	int val_int;
	float val_float;
	Job *job = (Job *) sJob;

	str = XwFieldGetString (wQuality);
	if ( str ) {
		if ( *str ) {
			val_int = job_quality ();
			sscanf (str, "%d", &val_int);
			job_set_quality (val_int);
		}
		free (str);
	}

	str = XwFieldGetString (wCompress);
	if ( str ) {
		if ( *str ) {
			val_int = job_compression ();
			sscanf (str, "%d", &val_int);
			job_set_compression (val_int);
		}
		free (str);
	}

	str = XwFieldGetString (wFps);
	if (str) {
		if (*str) {
			val_float = job_fps ();
			sscanf (str, "%f", &val_float);
			job_set_fps (val_float);
		}
		free (str);
	}

	str = XwFieldGetString (wFile);
	if (str) {
		if (*str) {
			job_set_file (str);
		}
		free (str);
	}

	str = XwFieldGetString (wMaxFrames);
	if (str) {
		if (*str)
			sscanf (str, "%d", &job->max_frames);
		free (str);
	}

	str = XwFieldGetString (wMaxTime);
	if (str) {
		if (*str)
			sscanf (str, "%f", &job->max_time);
		free (str);
	}

	str = XwFieldGetString (wCodec);
	if (str) {
		if (*str) {
			int i, a = -1;
			Boolean found = FALSE;

			for  (i = 0; i < NUMCODECS; i ++) {
				if (strcmp(tCodecNames[i], str) == 0) a = i;
			}
			if (a < CODEC_MPEG1) {
				a = CODEC_MPEG1;
				fprintf(stderr, "Unsupported Codec selected, resetting to MPEG1!\n");
			}
			job->targetCodec = a;
		}
		free (str);
	}

#ifdef HAVE_SHMAT
	XtVaGetValues (wShm, XtNstate, &val_int, NULL);
	if (val_int) {
		job->flags |= FLG_USE_SHM;
		job_set_capture ();
	} else {
		job->flags &= ~FLG_USE_SHM;
		job_set_capture ();
	}
#endif
	XtVaGetValues (wMouse, XtNstate, &val_int, NULL);
	if (val_int) {
		XtVaGetValues (wMouseType, XtNstate, &val_int, NULL);
		if (val_int) {
			job->mouseWanted = 2;
		} else {
			job->mouseWanted = 1;
		}
	} else {
		job->mouseWanted = 0;
	}

	XtDestroyWidget (wShell);
	ChangeLabel (job->pic_no);

        // unset autocontinue unless we capture to movie and file is mutable
        if ( job->flags & FLG_AUTO_CONTINUE && ( (! is_filename_mutable(job->file)) ||
                (! job->flags & FLG_MULTI_IMAGE) ) ) {
            job->flags &= ~FLG_AUTO_CONTINUE;
            fprintf(stderr, "Output not a video file or no counter in filename\nDisabling autocontinue!\n");
        }
        
    /* previous and next buttons have different meanings for on-the-fly encoding
     * and individual frame capture */
    if (( job->flags & FLG_MULTI_IMAGE ) == 0 ) {
        if (job->pic_no > job->step ) XtSetSensitive(prev, True);
        else XtSetSensitive(prev, False);
    } else {
        if ( is_filename_mutable(job->file) ) {
            XtSetSensitive(next, True);    
            XtSetSensitive(prev, True);    
        } else {
            XtSetSensitive(next, False);    
            if (job->movie_no > 0 ) XtSetSensitive(prev, True);    
            else XtSetSensitive(prev, False);    
        }
    }
        
}

/*
 * save the values
 */
static void
CbSave (Widget w, XtPointer client_data, XtPointer call_data)
{

#include "codecs.h"

	char *str;
	int val_int;
	float val_float;
	char *home;
	char file[PATH_MAX+1];
	FILE *fp;
	Job *job = (Job *) sJob ;

	str = XwFieldGetString (wQuality);
	if ( str ) {
		if ( *str ) {
			val_int = job_quality ();
			sscanf (str, "%d", &val_int);
			job_set_quality (val_int);
		}
		free (str);
	}

	str = XwFieldGetString (wCompress);
	if ( str ) {
		if ( *str ) {
			val_int = job_compression ();
			sscanf (str, "%d", &val_int);
			job_set_compression (val_int);
		}
		free (str);
	}

	str = XwFieldGetString (wFps);
	if (str) {
		if (*str) {
			val_float = job_fps ();
			sscanf (str, "%f", &val_float);
			job_set_fps (val_float);
		}
		free (str);
	}

	str = XwFieldGetString (wFile);
	if (str) {
		if (*str) {
			job_set_file (str);
		}
		free (str);
	}

	str = XwFieldGetString (wMaxFrames);
	if (str) {
		if (*str)
			sscanf (str, "%d", &job->max_frames);
		free (str);
	}

	str = XwFieldGetString (wMaxTime);
	if (str) {
		if (*str)
			sscanf (str, "%f", &job->max_time);
		free (str);
	}

	str = XwFieldGetString (wCodec);
	if (str) {
		if (*str) {
			int i, a = -1;
			Boolean found = FALSE;

			for  (i = 0; i < NUMCODECS; i ++) {
				if (strcmp(tCodecNames[i], str) == 0) a = i;
			}
			if (a < CODEC_MPEG1) {
				a = CODEC_MPEG1;
				fprintf(stderr, "Unsupported Codec selected, resetting to MPEG1!\n");
			}
			job->targetCodec = a;
		}
		free (str);
	}

#ifdef HAVE_SHMAT
	XtVaGetValues (wShm, XtNstate, &val_int, NULL);
	if (val_int) {
		job->flags |= FLG_USE_SHM;
		job_set_capture ();
	} else {
		job->flags &= ~FLG_USE_SHM;
		job_set_capture ();
	}
#endif
	XtVaGetValues (wMouse, XtNstate, &val_int, NULL);
	if (val_int) {
		XtVaGetValues (wMouseType, XtNstate, &val_int, NULL);
		if (val_int) {
			job->mouseWanted = 2;
		} else {
			job->mouseWanted = 1;
		}
	} else {
		job->mouseWanted = 0;
	}

        if ( ! XVC_WriteOptionsFile(job) ) {
            fprintf(stderr, "Unable to write personal options file!\n");
        }
}

/*
 */
static void
CbCancel (Widget w, XtPointer client_data, XtPointer call_data)
{
	XtDestroyWidget (wShell);
}

/*
 */
static void
CbActivate (Widget w, XtPointer client_data, XtPointer call_data)
{
	static Time t = CurrentTime;
	Widget next = (Widget) client_data;

#ifdef DEBUG
	printf ("%s: CbActivate()\n", __FILE__);
#endif
	if (XtCallAcceptFocus (next, &t))
		XtSetKeyboardFocus (XtParent(w), next);
}


/*
 * create the options dialog window
 */
void
CbOptions (Widget w, XtPointer client_data, XtPointer call_data)
{
#include "codecs.h"

	Job *job = (Job *) client_data;
	Widget frame, box, info;
	Widget fps_lbl, file_lbl, max_frames_lbl, max_time_lbl, quality_lbl;
	Widget compress_lbl, codec_lbl, mouse_lbl, ok, cancel, save;
	Position x, y;
	char str[PATH_MAX+1];

	sJob = job;
#ifdef DEBUG
	printf ("CbOptions() fps=%f\n", job->fps);
#endif
	wShell = XtVaCreatePopupShell ("options", transientShellWidgetClass,
				GetToplevel(w), XtNinput, TRUE, NULL);
	frame = XtVaCreateManagedWidget ("frame", xwBoxWidgetClass, wShell, NULL);

	info = XtVaCreateManagedWidget ("info", xwLabelWidgetClass, frame,
					XtNshadowType, XtShadowNone, NULL);

	sprintf (str, "%f", job->fps);
	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	wFps = XtVaCreateManagedWidget ("fps_val", xwFieldWidgetClass, box,
					XtNstring, str, XtNonlyNumber, True, NULL);
	fps_lbl =XtVaCreateManagedWidget ("fps", xwLabelWidgetClass, box, NULL);

	sprintf (str, "%s", job->file);
	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	wFile = XtVaCreateManagedWidget ("file_val", xwFieldWidgetClass, box,
					XtNstring, str, XtNprev, wFps, NULL);
	file_lbl =XtVaCreateManagedWidget ("file", xwLabelWidgetClass, box, NULL);

	sprintf (str, "%d", job->max_frames);
	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	wMaxFrames = XtVaCreateManagedWidget ("max_frames_val", xwFieldWidgetClass,
					box, XtNstring, str, XtNonlyNumber, True,
					XtNprev, wFile, NULL);
	max_frames_lbl = XtVaCreateManagedWidget ("max_frames", xwLabelWidgetClass,
					box, NULL);

	sprintf (str, "%f", job->max_time);
	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	wMaxTime = XtVaCreateManagedWidget ("max_time_val", xwFieldWidgetClass,
					box, XtNstring, str, XtNonlyNumber, True,
					XtNprev, wMaxFrames, NULL);
	max_time_lbl =XtVaCreateManagedWidget ("max_time", xwLabelWidgetClass,
					box, NULL);

	sprintf (str, "%d", job->quality);
	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	wQuality = XtVaCreateManagedWidget ("quality_val", xwFieldWidgetClass,
					box, XtNstring, str, XtNonlyNumber, True,
					XtNlength, 3, XtNprev, wMaxTime, NULL);
	quality_lbl = XtVaCreateManagedWidget ("quality", xwLabelWidgetClass, box,
					NULL);


	sprintf (str, "%d", job->compress);
	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	wCompress = XtVaCreateManagedWidget ("compress_val", xwFieldWidgetClass,
					box, XtNstring, str, XtNlength, 1,
					XtNonlyNumber, True, XtNprev, wQuality, NULL);
	compress_lbl = XtVaCreateManagedWidget ("compress", xwLabelWidgetClass,
					box, NULL);

	sprintf (str, "%s", tCodecNames[(job->targetCodec)]);
	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	wCodec = XtVaCreateManagedWidget ("codec_val", xwFieldWidgetClass,
					box, XtNstring, str, XtNprev, wCompress, NULL);
	codec_lbl = XtVaCreateManagedWidget ("codec (e.g. MPEG4)", xwLabelWidgetClass,
					box, NULL);

	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	wShm = XtVaCreateManagedWidget ("shm", xwToggleWidgetClass, box,
					XtNstate, (Boolean)(job->flags & FLG_USE_SHM ? 1:0),
					XtNprev, wCodec, NULL);

	wMouse = XtVaCreateManagedWidget ("mouse", xwToggleWidgetClass, box,
					XtNstate, (Boolean)((job->mouseWanted > 0) ? 1:0),
					XtNprev, wShm, NULL);

	wMouseType = XtVaCreateManagedWidget ("black", xwToggleWidgetClass, box,
					XtNstate, (Boolean)((job->mouseWanted > 1) ? 1:0),
					XtNprev, wMouse, NULL);


	/*
	 */
	box = XtVaCreateManagedWidget ("box", xwBoxWidgetClass, frame,
			XtNorientation, XtorientHorizontal, NULL);
	ok = XtVaCreateManagedWidget ("ok", xwButtonWidgetClass, box,
				XtNprev, wShm, NULL);
	cancel = XtVaCreateManagedWidget ("cancel", xwButtonWidgetClass, box,
				XtNprev, ok, NULL);
	save = XtVaCreateManagedWidget ("save", xwButtonWidgetClass, box,
				XtNprev, cancel, XtNnext, wFps, NULL);

	XtVaSetValues (cancel, XtNnext, save, NULL);
	XtVaSetValues (ok, XtNnext, cancel, NULL);
	XtVaSetValues (wShm, XtNnext, ok, NULL);
	XtVaSetValues (wCompress, XtNnext, wShm, NULL);
	XtVaSetValues (wQuality, XtNnext, wCompress, NULL);
	XtVaSetValues (wMaxTime, XtNnext, wQuality, NULL);
	XtVaSetValues (wMaxFrames, XtNnext, wMaxTime, NULL);
	XtVaSetValues (wFile, XtNnext, wMaxFrames, NULL);
	XtVaSetValues (wFps, XtNnext, wFile, XtNprev, save, NULL);

	XtAddCallback (wFps, XtNactivateCallback, CbActivate, wFile);
	XtAddCallback (wFile, XtNactivateCallback, CbActivate, wMaxFrames);
	XtAddCallback (wMaxFrames, XtNactivateCallback, CbActivate, wMaxTime);
	XtAddCallback (wMaxTime, XtNactivateCallback, CbActivate, wQuality);
	XtAddCallback (wQuality, XtNactivateCallback, CbActivate, wCompress);
	XtAddCallback (wCompress, XtNactivateCallback, CbActivate, wShm);

	XtAddCallback (ok, XtNcallback, CbOk, job);
	XtAddCallback (cancel, XtNcallback, CbCancel, NULL);
	XtAddCallback (save, XtNcallback, CbSave, job);

	XtTranslateCoords (w, 0, 0, &x, &y);
	XtVaSetValues (wShell, XtNx, x, XtNy, y, NULL);

	XtPopup (wShell, XtGrabNonexclusive);
}

/*
 */
void
OptionDoneAction (Widget w, XEvent *ev, String *parms, Cardinal *num_parms)
{
	if (*num_parms) {
		if (*parms[0] == '1') {
			CbOk (w, (XtPointer)DLG_OK, NULL);
		} else if (*parms[0] == '2') {
			CbSave (w, (XtPointer)DLG_SAVE, NULL);
		} else {
			CbCancel (w, NULL, NULL);
		}
	}
}

/*
 */
void
OptionOpenAction (Widget w, XEvent *ev, String *parms, Cardinal *num_parms)
{
	Job *job = job_ptr();
	CbOptions (w, (XtPointer) job, (XtPointer) NULL);
}
