/*
 * xtojpg.c,
 *
 * Copyright (C) 1997,98 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../config.h"  /* autoconf output */

#ifdef HAVE_LIBJPEG
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/XWDFile.h>
#include "job.h"
#include "xtojpg.h"
#include "colors.h"
#include "main.h"
#include "jpeglib.h"

typedef struct {
	unsigned char red;
	unsigned char green;
	unsigned char blue;
} rgb;


/* global */
static unsigned char *jpg_image = NULL;
static JSAMPROW *row_pointer = NULL;
static int job_active = 0;
static struct jpeg_compress_struct cjpeg;


/*
 */
void *
JPGcolorTable (XColor *colors, int ncolors)
{
	rgb *color_tab;
	int i;

	color_tab = (rgb *) malloc (ncolors * sizeof (rgb));
	if (!color_tab)
		return (NULL);
	for (i =0; i < ncolors; i++) {
		color_tab[i].red = colors[i].red;
		color_tab[i].green = colors[i].green;
		color_tab[i].blue = colors[i].blue;
	}
	return (color_tab);
}

/*
 * for TrueColor and DirectColor
 * write a jpeg out to the named file
 */
void
XImageToJPGC (FILE *fp, XImage *image, Job *job)
{
	static struct jpeg_error_mgr jerr;
	static ColorInfo c_info;
	static unsigned long max_val;
	unsigned char *line_ptr, *p8, *p24;
	unsigned short *p16;
	int row, col;

	if ( job->state & VC_START ) {
		/* it's the first call, prepare the header and some statics
		 */
#ifdef DEBUG
		dump_ximage_info (image);
#endif
		/* get all the masks and max vals .. */
		GetColorInfo (image, &c_info);
		max_val = 255;

		jpeg_create_compress (&cjpeg);
		job_active = 1;
		cjpeg.image_width = image->width;
		cjpeg.image_height= image->height;
		cjpeg.err = jpeg_std_error (&jerr);
		cjpeg.input_components = 3;
		cjpeg.in_color_space = JCS_RGB;
		jpeg_set_defaults (&cjpeg);
		cjpeg.dct_method = JDCT_FASTEST;
		jpeg_set_quality (&cjpeg, job->quality, TRUE);

		/* we need an area for the jpeg image
		 */
		jpg_image = (unsigned char *) malloc (image->width * image->height * 3);
		row_pointer = (JSAMPROW *) malloc (sizeof(void*) * image->height);
	}
	jpeg_stdio_dest (&cjpeg, fp);
	jpeg_start_compress (&cjpeg, TRUE);

	/* for ZPixmap bits_per_pixel could be 1,4,8,16,24,32
	 * but for Direct- and TrueColor it could only be 8,16,24,32 (?)
	 */
	switch (image->bits_per_pixel) {
		case 8:
			/* for 8bpp x server
			 */
			p8 = (unsigned char *) image->data;
			line_ptr = jpg_image;
			for (row = 0; row < image->height; row++) {
				row_pointer[row] = jpg_image+(row * image->width * 3);
				for (col = 0; col < image->width; col++) {
					*line_ptr++ =
							((*p8 & image->red_mask) >> c_info.red_shift)
							* max_val / c_info.red_max_val;
					*line_ptr++ =
							((*p8 & image->green_mask) >> c_info.green_shift)
							* max_val / c_info.green_max_val;
					*line_ptr++ =
							((*p8 & image->blue_mask) >> c_info.blue_shift)
							* max_val / c_info.blue_max_val;
					p8++;
				}
				/* eat paded bytes .. */
				p8 += image->bytes_per_line -
						 	image->bits_per_pixel / 8 * image->width;
			}
			break;

		case 16: {
			/* for 16bpp and 15bpp x server
			 *
			 * there is no need for exact calculation, so
			 * we do some approximations to speed up ..
			 */
			register unsigned long r_shift1, r_shift2, g_shift1, g_shift2,
					b_shift1, b_shift2, left_shift;
			r_shift1 = c_info.red_bit_depth;
			r_shift2 = c_info.red_bit_depth * 2;
			g_shift1 = c_info.green_bit_depth;
			g_shift2 = c_info.green_bit_depth * 2;
			b_shift1 = c_info.blue_bit_depth;
			b_shift2 = c_info.blue_bit_depth * 2;
			left_shift=8; /* we expand to 24bit */
			p16 = (unsigned short *) image->data;
			line_ptr = jpg_image;

			for (row = 0; row < image->height; row++) {
				row_pointer[row] = jpg_image+(row * image->width * 3);
				for (col = 0; col < image->width; col++, p16++) {
					*line_ptr = (*p16 & image->red_mask) >> c_info.red_shift;
					if (*line_ptr > 0)
					*line_ptr = (((*line_ptr << left_shift)-1) >> r_shift1) +
								(((*line_ptr++<<left_shift)-1) >> r_shift2);
					else line_ptr++;

					*line_ptr = (*p16 & image->green_mask) >>c_info.green_shift;
					if (*line_ptr > 0)
					*line_ptr = (((*line_ptr << left_shift)-1) >> g_shift1) +
								(((*line_ptr++<<left_shift)-1) >> g_shift2);
					else line_ptr++;

					*line_ptr = (*p16 & image->blue_mask) >> c_info.blue_shift;
					if (*line_ptr > 0)
					*line_ptr = (((*line_ptr << left_shift)-1) >> b_shift1) +
								(((*line_ptr++<<left_shift)-1) >> b_shift2);
					else line_ptr++;
				}
				/* eat paded bytes .. we have to devide by 2 because
				 * the p16 pointer is unsigned short *
				 */
				p16 += (image->bytes_per_line -
						(image->bits_per_pixel >> 3) * image->width) >> 1;
			}
			}
			break;

		case 24:
			if (image->byte_order == LSBFirst) {
				p24 = (unsigned char *) image->data;
				line_ptr = jpg_image;
				for (row = 0; row < image->height; row++) {
					row_pointer[row] = jpg_image+(row * image->width * 3);
					for (col = 0; col < image->width; col++) {
						/* we have to swap */
						*line_ptr++ = p24[2];
						*line_ptr++ = p24[1];
						*line_ptr++ = p24[0];
						p24 += 3;
					}
					/* eat paded bytes .. */
					p24 += image->bytes_per_line -
						 	(image->bits_per_pixel >> 3) * image->width;
				}
			} else {
				for (row = 0; row < image->height; row++) {
					row_pointer[row] = image->data+(row * image->width * 3);
				}
			}
			break;

		case 32: {
			register unsigned long rm = image->red_mask,
						gm = image->green_mask,
						bm = image->blue_mask,
						r_shift = c_info.red_shift,
						g_shift = c_info.green_shift,
						b_shift = c_info.blue_shift,
						*p32 = (unsigned long *) image->data;

			line_ptr = jpg_image;

			for (row = 0; row < image->height; row++) {
				row_pointer[row] = jpg_image+(row * image->width * 3);

				for (col = 0; col < image->width; col++, p32++) {
					*line_ptr++ = (*p32 & rm) >> r_shift;
					*line_ptr++ = (*p32 & gm) >> g_shift;
					*line_ptr++ = (*p32 & bm) >> b_shift;
				}
				p32 += (image->bytes_per_line -
					 	(image->bits_per_pixel >> 3) * image->width) >> 2;
			}
			}
			break;

		default:
			printf ("bits_per_pixel not supported: %d\n",image->bits_per_pixel);
			break;
	}
	/* write out the image
	 */
	jpeg_write_scanlines (&cjpeg, row_pointer, image->height);
	jpeg_finish_compress (&cjpeg);
}

/*
 * for PseudoColor/8bpp
 * write a JPEG out to the named file, created by a palette
 */
void
XImageToJPG8 (FILE *fp, XImage *image, Job *job)
{
	static struct jpeg_error_mgr jerr;
	register unsigned char *line_ptr, *col_ptr;
	register int row, col;

	if ( job->state & VC_START ) {
#ifdef DEBUG
		dump_ximage_info (image);
#endif
		jpg_image = (unsigned char *) malloc (image->width * image->height * 3);
		row_pointer = (JSAMPROW *) malloc (sizeof(void*) * image->height);

		jpeg_create_compress (&cjpeg);
		job_active = TRUE;

		cjpeg.image_width = image->width;
		cjpeg.image_height= image->height;
		cjpeg.err = jpeg_std_error (&jerr);
		cjpeg.input_components = 3;
		cjpeg.in_color_space = JCS_RGB;
		jpeg_set_defaults (&cjpeg);
		if (job->quality != 75)
			jpeg_set_quality (&cjpeg, job->quality, TRUE);
	}
	jpeg_stdio_dest (&cjpeg, fp);
	jpeg_start_compress (&cjpeg, TRUE);

	switch (image->bits_per_pixel) {
		case 8:
			line_ptr = jpg_image;
			for (row = 0; row < image->height; row++) {
				col_ptr = image->data + (row * image->bytes_per_line);
				row_pointer[row] = jpg_image+(row * image->width * 3);
				for (col = 0; col < image->width; col++) {
					*line_ptr++ = ((rgb*)job->color_table)[*col_ptr].red;
					*line_ptr++ = ((rgb*)job->color_table)[*col_ptr].green;
					*line_ptr++ = ((rgb*)job->color_table)[*col_ptr].blue;
					col_ptr++;
				}
			}
			jpeg_write_scanlines (&cjpeg, row_pointer, image->height);
			jpeg_finish_compress (&cjpeg);
			break;
		default:
			printf ("Visual not supported!\n");
			break;
	}
}

/*
 * for GrayScale and StaticGray images
 * GrayScale seems not to work :(
 */
void
XImageToJPGG (FILE *fp, XImage *image, Job *job)
{
	static struct jpeg_error_mgr jerr;
	register int row, pad;

	if ( job->state & VC_START ) {
#ifdef DEBUG
		dump_ximage_info (image);
#endif
		row_pointer = (JSAMPROW *) malloc (sizeof(void*) * image->height);

		jpeg_create_compress (&cjpeg);
		job_active = TRUE;

		cjpeg.image_width = image->width;
		cjpeg.image_height= image->height;
		cjpeg.err = jpeg_std_error (&jerr);
		cjpeg.input_components = 1;
		cjpeg.in_color_space = JCS_GRAYSCALE;
		jpeg_set_defaults (&cjpeg);
		if (job->quality != 75)
			jpeg_set_quality (&cjpeg, job->quality, TRUE);
	}
	jpeg_stdio_dest (&cjpeg, fp);
	jpeg_start_compress (&cjpeg, TRUE);

	switch (image->bits_per_pixel) {
		case 8:
			for (row = 0; row < image->height; row++) {
				pad = image->bytes_per_line - image->width;
				row_pointer[row] = image->data+(row * (image->width+pad));
			}
			jpeg_write_scanlines (&cjpeg, row_pointer, image->height);
			jpeg_finish_compress (&cjpeg);
			break;
		default:
			printf ("Visual not supported!\n");
			break;
	}
}

/*
 * clean up some previous allocated areas
 */
void
JpgClean (Job *job)
{
	if (job_active != TRUE)
		return;
	if (jpg_image) {
		free (jpg_image);
		jpg_image = NULL;
	}
	if (row_pointer) {
		free (row_pointer);
		row_pointer = NULL;
	}
	jpeg_destroy_compress (&cjpeg);
	job_active = 0;
}
#endif

