/*
 * xtomng.c,
 * XImge to MNG (multiple network graphic) module
 *
 * Copyright (C) 1997-98 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../config.h"  /* autoconf output */

#ifdef HAVE_LIBPNG
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/XWDFile.h>
#ifndef HAVE_LIBZ
#error "You need zlib for MNG support"
#endif
#include <zlib.h>
#include <png.h>
#include "job.h"
#include "xtomng.h"
#include "mngutil.h"
#include "app_data.h"
#include "colors.h"

/* globals */
static png_structp png_ptr = NULL;
static png_infop info_ptr = NULL;
static unsigned char **row_pointers = NULL;
static int job_active = 0;


/*
 * for TrueColor and DirectColor
 * write a MNG out to the named file pointer
 */
void
XImageToMNGC (FILE *fp, XImage *image, Job *job)
{
	int row, col;
	unsigned char *col_ptr, *row_ptr, *p8;
	unsigned short *p16;
	static png_color_8 sig_bit;
	static unsigned long crc = 0;
	static ColorInfo c_info;

#ifdef DEBUG
	printf ("XImageToMNGC()\n");
#endif
	/* destroy the old one, we need this static pointer to could
	 * access it also in MngClean()
	 */
	if (png_ptr)
		png_destroy_write_struct (&png_ptr, (png_infopp)NULL);
	png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
				(void *)NULL, NULL, NULL);
	if (!png_ptr) {
		printf ("Fatal error in XImageToMNGC()\n");
		exit (99);
	}
	png_init_io (png_ptr, fp);
	png_set_compression_level (png_ptr, job->compress);

	if ( job->state & VC_START ) {
		/* it's the first call, prepare some stuff
		 */
#ifdef DEBUG2
		dump_ximage_info (image);
#endif
		job_active = 1;
		job->crc = 0;
		if (image->bits_per_pixel != 24) {
			/* in 24bpp mode we don't need the buffer
			 */
			row_pointers =(unsigned char **) malloc (
				image->height*sizeof(void*) + image->width * image->height * 3);
			row_ptr = (unsigned char *) (row_pointers+image->height);
			for (row = 0; row < image->height; row++) {
				row_pointers[row] = row_ptr + (row * image->width * 3);
			}

			GetColorInfo (image, &c_info);

			/* on my 16bpp server these values are 5,6,5
			 */
			sig_bit.red = c_info.red_bit_depth;
			sig_bit.green = c_info.green_bit_depth;
			sig_bit.blue = c_info.blue_bit_depth;
		}
		/* we need the info_ptr only once
		 */
		info_ptr = png_create_info_struct (png_ptr);
		if (!info_ptr) {
			png_destroy_write_struct (&png_ptr, (png_infopp)NULL);
			return;
		}
		if (setjmp (png_ptr->jmpbuf)) {
			png_destroy_write_struct (&png_ptr, &info_ptr);
			(*job->close) (fp);
			exit(99);
		}

		info_ptr->width = image->width;
		info_ptr->height= image->height;
		info_ptr->bit_depth = 8;
		info_ptr->color_type = PNG_COLOR_TYPE_RGB;
		if (image->bits_per_pixel != 24) {
			info_ptr->color_type |= PNG_COLOR_MASK_COLOR;
		}
		mng_write_sig (png_ptr);
		mng_write_MHDR (png_ptr, image->width, image->height, 1000 /* msec */);
		mng_write_nEED (png_ptr, "draft 41");
		mng_write_FRAM (png_ptr, 4, NULL, 2, 0, 0, 0, job->time_per_frame);
	}

	png_set_sig_bytes (png_ptr, 8);

	if (image->bits_per_pixel < 24) {
		/* may be we need this? seems to be so
		 */
		png_set_sBIT (png_ptr, info_ptr, &sig_bit);
		png_set_shift (png_ptr, &sig_bit);
	}


	/* for ZPixmap bits_per_pixel could be 1,4,8,16,24,32
	 * but for Direct- and TrueColor it could only be 8,16,24,32
	 */
	switch (image->bits_per_pixel) {
		case 8:
			/* for 8bpp x server
			 * may be it should be possible to rework for usage
			 * of a palette ..?!
			 */
			crc = 0;
			p8 = (unsigned char *) image->data;
			for (row = 0; row < image->height; row++) {
				row_ptr = row_pointers[row];
				for (col = 0; col < image->width; col++, p8++) {
					*row_ptr++ =(*p8 & image->red_mask)   >> c_info.red_shift;
					*row_ptr++ =(*p8 & image->green_mask) >>c_info.green_shift;
					*row_ptr++ =(*p8 & image->blue_mask)  >> c_info.blue_shift;
				}
				/* eat paded bytes in the row ..
				 */
				p8 += image->bytes_per_line -
						(image->bits_per_pixel >> 3) * image->width;
				if (job->flags & FLG_MNG_CLONE)
					crc = crc32 (crc, row_pointers[row], image->width * 3);
			}
			/* write out the row
			 */
			if ((job->flags & FLG_MNG_CLONE) && (crc == job->crc)) {
				/* CRC is the same so we just clone it */
#				ifdef DEBUG
				printf ("dupe found! (%d)\n", job->pic_no);
#				endif
				mng_write_CLON (png_ptr, job->pic_no-1, job->pic_no, 1);
			} else {
				job->crc = crc;
				mng_write_DEFI (png_ptr, job->pic_no);
				png_write_info (png_ptr, info_ptr);
				png_write_image (png_ptr, (png_bytepp)row_pointers);
				png_write_end (png_ptr, info_ptr);
			}
			break;

		case 16:
			/* for 16bpp and 15bpp x server
			 */
			crc = 0;
			p16 = (unsigned short *) image->data;
			for (row = 0; row < image->height; row++) {
				row_ptr = row_pointers[row];
				for (col = 0; col < image->width; col++, p16++) {
					*row_ptr++ =(*p16 & image->red_mask)  >> c_info.red_shift;
					*row_ptr++ =(*p16 & image->green_mask)>>c_info.green_shift;
					*row_ptr++ =(*p16 & image->blue_mask) >> c_info.blue_shift;
				}
				/* eat paded bytes in the row ..
				 * ( bytes_per_line - bits_per_pixel / 8 * width ) / 2
				 */
				p16 += (image->bytes_per_line -
						(image->bits_per_pixel >> 3) * image->width) >> 1;

				if (job->flags & FLG_MNG_CLONE)
					crc = crc32 (crc, row_pointers[row], image->width * 3);
			}
			/* write out the row
			 */
			if ((job->flags & FLG_MNG_CLONE) && (crc == job->crc)) {
				/* CRC is the same so we just clone it */
#				ifdef DEBUG
				printf ("dupe found! (%d)\n", job->pic_no);
#				endif
				mng_write_CLON (png_ptr, job->pic_no-1, job->pic_no, 1);
			} else {
				job->crc = crc;
				mng_write_DEFI (png_ptr, job->pic_no);
				png_write_info (png_ptr, info_ptr);
				png_write_image (png_ptr, (png_bytepp)row_pointers);
				png_write_end (png_ptr, info_ptr);
			}
			break;
		case 24:
			if (job->flags & FLG_MNG_CLONE) {
				crc = 0;
				if (!row_pointers) {
					row_pointers =
						(unsigned char **) malloc (image->height*sizeof(void*));
				}
				for (row = 0; row < image->height; row++) {
					/* allign the row  pointer
					 */
					row_pointers[row]=image->data+(row * image->bytes_per_line);
					/* swap the row and calculate crc
					 */
					if (image->byte_order == LSBFirst) {
						/* we have to swap all RGBs :( */
						unsigned char t;
						col_ptr = row_pointers[row];
						for (col = 0; col < image->width; col++) {
							t = col_ptr[0];
							col_ptr[0] = col_ptr[2];
							col_ptr[2] = t;
							col_ptr +=3;
						}
					}
					crc = crc32(crc, row_pointers[row], image->width * 3);
				}
				if (crc == job->crc) {
					mng_write_CLON (png_ptr, job->pic_no-1, job->pic_no, 1);
				} else {
					job->crc = crc;
					mng_write_DEFI (png_ptr, job->pic_no);
					png_write_info (png_ptr, info_ptr);
					png_write_image (png_ptr, (png_bytepp)row_pointers);
					png_write_end (png_ptr, info_ptr);
				}
				free (row_pointers);
				row_pointers = NULL;
			} else {
				/* don't check crc */
				mng_write_DEFI (png_ptr, job->pic_no);
				png_write_info (png_ptr, info_ptr);
				for (row = 0; row < image->height; row++) {
					/* allign the row  pointer
					 */
					row_ptr = image->data+(row * image->bytes_per_line);
					/* write out the row
					 */
					if (image->byte_order == LSBFirst) {
						/* we have to swap all RGBs :( */
						unsigned char t;
						col_ptr = row_ptr;
						for (col = 0; col < image->width; col++) {
							t = col_ptr[0];
							col_ptr[0] = col_ptr[2];
							col_ptr[2] = t;
							col_ptr +=3;
						}
					}
					png_write_row (png_ptr, (png_bytep)row_ptr);
				}
				png_write_end (png_ptr, info_ptr);
			}
			break;
		case 32: /* alpha + RGB */ {
			register unsigned int
				rmask = image->red_mask,
				gmask = image->green_mask,
				bmask = image->blue_mask,
				rshift= c_info.red_shift,
				gshift= c_info.green_shift,
				bshift= c_info.blue_shift,
				*p32 = (unsigned int *)image->data;

			for (row = 0; row < image->height; row++) {
				row_ptr = row_pointers[row];
				for (col = 0; col < image->width; col++) {
					*row_ptr++ = (*p32 & rmask) >> rshift;
					*row_ptr++ = (*p32 & gmask) >> gshift;
					*row_ptr++ = (*p32 & bmask) >> bshift;
					p32++;
				}
			}
			if (job->flags & FLG_MNG_CLONE)
				crc = crc32(0,row_pointers[0],image->width*image->height*3);
			if ((job->flags & FLG_MNG_CLONE) && (job->crc == crc)) {
				/* CRC is the same so we just clone it */
#				ifdef DEBUG
				printf ("dupe found! (%d)\n", job->pic_no);
#				endif
				mng_write_CLON (png_ptr, job->pic_no-1, job->pic_no, 1);
			} else {
				job->crc = crc;
				mng_write_DEFI (png_ptr, job->pic_no);
				png_write_info (png_ptr, info_ptr);
				png_write_image (png_ptr, (png_bytepp)row_pointers);
				png_write_end (png_ptr, info_ptr);
			}
			} break;
		default:
			printf ("bits_per_pixel not supported: %d\n",image->bits_per_pixel);
			break;
	}
}

/*
 * for indexed color and gray scale MNG files
 */
void
XImageToMNG8 (FILE *fp, XImage *image, Job *job)
{
	int row;
	unsigned long crc;

	if (png_ptr)
		png_destroy_write_struct (&png_ptr, (png_infopp)NULL);
	png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
				(void *)NULL, NULL, NULL);
	if (!png_ptr) {
		printf ("Fatal error in XImageToMNG8()\n");
		exit (99);
	}
	png_init_io (png_ptr, fp);
	png_set_compression_level (png_ptr, job->compress);

	if ( job->state & VC_START ) {
		/* it's the first call, prepare some stuff
		 */
#ifdef DEBUG2
		dump_ximage_info (image);
#endif
		job_active = 1;
		info_ptr = png_create_info_struct (png_ptr);
		if (!info_ptr) {
			png_destroy_write_struct (&png_ptr, (png_infopp)NULL);
			return;
		}
		if (setjmp (png_ptr->jmpbuf)) {
			png_destroy_write_struct (&png_ptr, &info_ptr);
			(*job->close)(fp);
			exit (99);
		}
		info_ptr->width = image->width;
		info_ptr->height= image->height;
		info_ptr->bit_depth = 8;
		if (job->win_attr.visual->class == StaticGray) {
			info_ptr->color_type = PNG_COLOR_TYPE_GRAY;
		} else {
			info_ptr->color_type = PNG_COLOR_TYPE_PALETTE;
		}
		mng_write_sig (png_ptr),
		mng_write_MHDR (png_ptr, image->width, image->height, 1000 /* msec */);
		mng_write_nEED (png_ptr, "draft 41");
		mng_write_FRAM (png_ptr, 4, NULL, 2, 0, 0, 0, job->time_per_frame);
		row_pointers = (unsigned char **) malloc (image->height *sizeof(void*));
	}

	if (job->win_attr.visual->class != StaticGray) {
		png_set_PLTE (png_ptr, info_ptr, job->color_table, job->ncolors);
	}
	png_set_sig_bytes (png_ptr, 8);

	/* for ZPixmap bits_per_pixel could be 1,4,8,16,24,32
	 * but for Direct- and TrueColor it could only be 16,24,32
	 */
	switch (image->bits_per_pixel) {
		case 8:
			/* for 8bpp server
			 */
			crc = 0;
			for (row = 0; row < image->height; row++) {
				row_pointers[row] = image->data+(row * image->bytes_per_line);
				/* calc crc to check if the previous image was the same */
				if (job->flags & FLG_MNG_CLONE)
					crc = crc32(crc, row_pointers[row], image->width);
			}
			if ((job->flags & FLG_MNG_CLONE) && (crc == job->crc)) {
				/* the previous image was the same, so we just
				 * write a CLON chunk to the MNG file
				 */
				mng_write_CLON (png_ptr, job->pic_no-1, job->pic_no, 1);
			} else {
				/* save the last CRC32 */
				job->crc = crc;
				mng_write_DEFI (png_ptr, job->pic_no);
				png_write_info (png_ptr, info_ptr);
				png_write_image (png_ptr, (png_bytepp)row_pointers);
				png_write_end (png_ptr, info_ptr);
			}
			break;
		default:
			printf ("bits_per_pixel not supported: %d\n",image->bits_per_pixel);
			break;
	}
}


/*
 * clean up some allocated stuff
 */
void
MngClean (Job *job)
{
#ifdef DEBUG
	printf ("MngClean()\n");
#endif
	if (!job_active) {
		return;
	}
	if (row_pointers) {
		free (row_pointers);
		row_pointers = NULL;
	}
	if (png_ptr && info_ptr) {
		mng_write_MEND (png_ptr);
		png_destroy_write_struct (&png_ptr, &info_ptr);
		png_ptr = NULL;
		info_ptr = NULL;
	}
}
#endif /* HAVE_LIBPNG */

