/*
 * xtoqtf.c,
 * XImge to QT file module, todo: sound support :)
 *
 * Copyright (C) 1998 Rasca, Berlin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "../config.h"  /* autoconf output */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#ifdef HAVE_LIBPNG
#include <zlib.h>
#include <png.h>
#endif
#ifdef HAVE_LIBJPEG
#include <jpeglib.h>
#endif
#include "job.h"
#include "xtoqtf.h"
#include "gt.h"
#include "colors.h"

/* reserved space for the moov atom
 */
#define HEADER_SIZE 1024

/* globals */
static unsigned char *line = NULL;
static FILE *qtfp = NULL;
static int bpp;	/* bits per pixel in the qt file */
static gt_atom *space_atom, *data_atom;
static int codec = GT_VID_FMT_RAW, pic_no = 0;
static int *offset = NULL;
#ifdef HAVE_LIBJPEG
static struct jpeg_compress_struct jinfo;
#endif
#ifdef HAVE_LIBPNG
static png_structp p_ptr= NULL;
static png_infop pinfo = NULL;
#endif

/*
 * remember offset of every frame during capturing
 */
static void
save_offset (int picno, int file_offset)
{
	static int size = 1024;
	extern int *offset;

	if (!offset) {
		/* alloc offsets for 1024 frames */
		offset = (int *) malloc (sizeof(int) * size);
	} else if (picno > size - 1) {
		/* realloc */
		size += 1024;
		offset = realloc (offset, sizeof (int) * size);
	}
	if (offset) {
		offset[picno] = file_offset;
#ifdef DEBUG2
		printf ("offset: %d -> %d\n", picno, file_offset);
#endif
	} else {
		perror ("save_offset()");
		exit (99);
	}
}

/*
 * for TrueColor and DirectColor
 * write a Qt out to the named file pointer
 */
void
XImageToQTFC (FILE *fp, XImage *image, Job *job)
{
	int row, col, i, j, pad;
	unsigned char *p8;
	unsigned short *p16;
	static int line_size = 0;
	static ColorInfo ci;
#ifdef HAVE_LIBJPEG
	static struct jpeg_error_mgr jerr;
#endif
#ifdef HAVE_LIBPNG
	static png_color_8 sig_bit;
#endif

#ifdef DEBUG2
	printf ("XImageToQTFC() flags=%d\n", job->flags);
#endif
	/* destroy the old one, we need this static pointer to could
	 * access it also in MngClean()
	 */

	if ( job->state & VC_START ) {
		/* it's the first call, prepare some stuff
		 */
#ifdef DEBUG
		dump_ximage_info (image);
#endif
		/* xanim don't know about free or skip, so we use a dummy mdat
		 * atom here */
		qtfp = fp;
		space_atom = gt_alloc_atom (GTA_skip, HEADER_SIZE-8);
		data_atom = gt_alloc_atom (GTA_movie_data, 0);
		gt_write_atom (space_atom, qtfp);
		gt_write_atom (data_atom, qtfp);
		pic_no = 0;
		save_offset (pic_no++, ftell(qtfp));
		GetColorInfo (image, &ci);

		if (job->target == CAP_QTJ) {
			/* we use jpeg codec */
#ifdef HAVE_LIBJPEG
			jinfo.err = jpeg_std_error (&jerr);
			jpeg_create_compress (&jinfo);
			jpeg_stdio_dest (&jinfo, qtfp);
			jinfo.image_width = image->width;
			jinfo.image_height = image->height;
			jinfo.input_components = 3;	/* only rgb24 is supported */
			jinfo.in_color_space = JCS_RGB;
			jpeg_set_defaults(&jinfo);
			/* optional parameters */
			jpeg_set_quality (&jinfo, job->quality, TRUE);
			jinfo.dct_method = JDCT_FASTEST;
			jpeg_start_compress (&jinfo, TRUE);
			codec = GT_VID_FMT_JPEG;
			bpp = 24;
#else
			fprintf (stderr, "Warning: JPEG not compiled in!\n");
			codec = GT_VID_FMT_RAW;
			bpp = image->bits_per_pixel;
#endif
		} else if (job->target == CAP_QTP) {
#ifdef  HAVE_LIBPNG
			/* to do */
			p_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
						(void *)NULL, NULL, NULL);
			pinfo = png_create_info_struct (p_ptr);
			if (image->bits_per_pixel == 16) {
				sig_bit.red   = ci.red_bit_depth;
				sig_bit.green = ci.green_bit_depth;
				sig_bit.blue  = ci.blue_bit_depth;
			}
			if (0/*setjmp (p_ptr->jmpbuf)*/) {
				printf ("fatal error!\n");
				job->state = VC_STOP;
				return;
			}
			codec = GT_VID_FMT_PNG;
			if (image->bits_per_pixel == 32)
				bpp = 32;	/* store with alpha */
			else
				bpp = 24;
#else
			fprintf (stderr, "Warning: PNG not compiled in!\n");
			codec = GT_VID_FMT_RAW;
			bpp = image->bits_per_pixel;
#endif
		} else /* CAP_QTR */ {
			codec = GT_VID_FMT_RAW;
			bpp = image->bits_per_pixel;
		}
		line_size = image->width * bpp / 8;
		line = malloc (line_size);
	} else {
		save_offset (pic_no++, ftell(qtfp));
#ifdef HAVE_LIBJPEG
		if (codec == GT_VID_FMT_JPEG)
			jpeg_start_compress (&jinfo, TRUE);
#endif
#ifdef HAVE_LIBPNG
		if (codec == GT_VID_FMT_PNG)
			p_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
						(void *)NULL, NULL, NULL);
#endif
	}

#ifdef HAVE_LIBJPEG
	/* what a dirty hack, just to pass the right pointer to ->write() */
	if (codec == GT_VID_FMT_JPEG)
		fp = (void *)&jinfo;
#endif
#ifdef HAVE_LIBPNG
	if (codec == GT_VID_FMT_PNG) {
		png_init_io (p_ptr, qtfp);
		png_set_compression_level (p_ptr, job->compress);
		if (image->bits_per_pixel == 32)
			png_set_IHDR (p_ptr, pinfo, image->width, image->height, 8,
				PNG_COLOR_TYPE_RGB_ALPHA, PNG_INTERLACE_NONE,
				PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);
		else
			png_set_IHDR (p_ptr, pinfo, image->width, image->height, 8,
				PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
				PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);
		if (image->bits_per_pixel == 16) {
			png_set_sBIT (p_ptr, pinfo, &sig_bit);
			png_set_shift(p_ptr, &sig_bit);
		}
		png_write_info (p_ptr, pinfo);
		fp = (void *)p_ptr;
	}
#endif

	/* for ZPixmap bits_per_pixel could be 1,4,8,16,24,32
	 * but for Direct- and TrueColor it could only be 8,16,24,32
	 */
	switch (image->bits_per_pixel) {
		case 16:
			/* for 16bpp and 15bpp x server
			 */
			if (codec == GT_VID_FMT_JPEG) {
				/* this should work for 16 and 15 bpp */
				register unsigned long
					r_shift0 = ci.red_shift,
					r_shift1 = ci.red_bit_depth,
					r_shift2 = ci.red_bit_depth << 1,
					g_shift0 = ci.green_shift,
					g_shift1 = ci.green_bit_depth,
					g_shift2 = ci.green_bit_depth << 1,
					b_shift0 = ci.blue_shift,
					b_shift1 = ci.blue_bit_depth,
					b_shift2 = ci.blue_bit_depth << 1;

				p16 = (unsigned short *) image->data;
				pad = (image->bytes_per_line - (image->width << 1)) >> 1;
				for (row = 0; row < image->height; row++) {
					p8 = line;
					for (col = 0; col < image->width; col++) {
						*p8 = ((*p16 & image->red_mask)  >> r_shift0);
						if (*p8)
							*p8 = (((*p8 << 8) - 1) >> r_shift1) +
									(((*p8 << 8) - 1) >> r_shift2);
						p8++;
						*p8 = ((*p16 & image->green_mask) >> g_shift0);
						if (*p8)
							*p8 = (((*p8 << 8) - 1) >> g_shift1) +
									(((*p8 << 8) - 1) >> g_shift2);
						p8++;
						*p8 = ((*p16++ & image->blue_mask) >> b_shift0);
						if (*p8)
							*p8 = (((*p8 << 8) - 1) >> b_shift1) +
									(((*p8 << 8) - 1) >> b_shift2);
						p8++;
					}
					(*job->write) (line, 1, line_size, fp);
					p16 += pad;
				}
			} else if (codec == GT_VID_FMT_PNG) {
				p16 = (unsigned short *) image->data;
				for (row = 0; row < image->height; row++) {
					p8 = line;
					for (col = 0; col < image->width; col++) {
						*p8++ = (p16[col] & image->red_mask) >> ci.red_shift;
						*p8++ = (p16[col] & image->green_mask) >>ci.green_shift;
						*p8++ = (p16[col] & image->blue_mask) >>ci.blue_shift;
					}
					(*job->write) (line, 1, line_size, fp);
					p16 += image->bytes_per_line >> 1;
				}
			} else 
			if (image->depth == 16) {
				/* 16bpp, convert 565 to 555 which is
				   needed by qt in raw rgb mode .. */
				p16 = (unsigned short *) image->data;
				if (image->byte_order == LSBFirst) {
					unsigned short t16;
					for (row = 0; row < image->height; row++) {
						for (i = 0, j=0; i < image->width; i++) {
							t16 = ((*p16>>1) & (unsigned short)
								(image->red_mask | image->green_mask)) |
								(*p16 & (unsigned short)image->blue_mask);

							line[j++] = ((unsigned char *)&t16)[1];
							line[j++] = ((unsigned char *)&t16)[0];
							p16++;
						}
						(*job->write) (line, 1, line_size, fp);
						/* eat paded bytes in the row ..
						 * ( bytes_per_line - bits_per_pixel / 8 * width ) / 2
						 */
						p16 += (image->bytes_per_line -
							(image->bits_per_pixel >> 3) * image->width) >> 1;
					}
				} else /* MSBFirst */ {
					unsigned short * t16;
					for (row = 0; row < image->height; row++) {
						t16 = (unsigned short *) line;
						for (i = 0; i < image->width; i++) {
							t16[i] = ((*p16>>1) & (unsigned short)
								(image->red_mask | image->green_mask)) |
								(*p16 & (unsigned short)image->blue_mask);
							p16++;
						}
						(*job->write) (line, 1, line_size, fp);
						/* eat paded bytes in the row ..
						 * ( bytes_per_line - bits_per_pixel / 8 * width ) / 2
						 */
						p16 += (image->bytes_per_line -
							(image->bits_per_pixel >> 3) * image->width) >> 1;
					}
				}
			} else {
				/* 15bpp */
				p16 = (unsigned short *) image->data;
				if (image->byte_order == LSBFirst) {
					for (row = 0; row < image->height; row++) {
						p8 = (unsigned char *) p16;
						for (i=0, j=0; i < image->width; i++) {
							line[j] = p8[j+1]; j++;
							line[j] = p8[j-1]; j++;
						}
						(*job->write) (line, 1, line_size, fp);
						p16 += image->bytes_per_line >> 1;
					}
				} else /* MSBFirst */ {
					for (row = 0; row < image->height; row++) {
						(*job->write) (p16, 1, line_size, fp);
						p16 += image->bytes_per_line >> 1;
					}
				}
			}
			break;

		case 24:
			p8 = (unsigned char *) image->data;

			if (image->byte_order == LSBFirst) {
				unsigned char *t8;
				for (row = 0; row < image->height; row++) {
					t8 = line;
					for (col = 0, i=0; col < image->width; col++) {
						/* we have to swap all RGBs :( */
						t8[i]   = p8[i+2];
						t8[i+1] = p8[i+1];
						t8[i+2] = p8[i];
						i += 3;
					}
					(*job->write) (line, 1, line_size, fp);
					p8 += image->bytes_per_line;
				}
			} else /* MSBFirst */ {
				for (row = 0; row < image->height; row++) {
					/* write out the row
					 */
					(*job->write) (p8, 1, line_size, fp);
					p8 += image->bytes_per_line;
				}
			}
			break;

		case 32: /* alpha + RGB */
			if (codec == GT_VID_FMT_JPEG) {
				/* here we ignore the alpha channel,
				 * cause jpeg can't save alpha values */
				register unsigned int
					rmask = image->red_mask,
					gmask = image->green_mask,
					bmask = image->blue_mask,
					rshift= ci.red_shift,
					gshift= ci.green_shift,
					bshift= ci.blue_shift,
					*p32 = (unsigned int *) image->data;
				for (row = 0; row < image->height; row++) {
					p8 = line;
					for (col = 0; col < image->width; col++) {
						*p8++ = ((*p32 & rmask) >> rshift);
						*p8++ = ((*p32 & gmask) >> gshift);
						*p8++ = ((*p32 & bmask) >> bshift);
						p32++;
					}
					(*job->write) (line, 1, line_size, fp);
					p32 += (image->bytes_per_line - (image->bits_per_pixel >>3)
							* image->width) >> 2;
				}
			} else {
				/* honor alpha channel */
				register unsigned int
					rmask = image->red_mask,
					gmask = image->green_mask,
					bmask = image->blue_mask,
					amask = ci.alpha_mask,
					rshift= ci.red_shift,
					gshift= ci.green_shift,
					bshift= ci.blue_shift,
					ashift= ci.alpha_shift,
					*p32 = (unsigned int *) image->data;
				for (row = 0; row < image->height; row++) {
					p8 = line;
					for (col = 0; col < image->width; col++) {
						*p8++ = ((*p32 & amask) >> ashift);
						*p8++ = ((*p32 & rmask) >> rshift);
						*p8++ = ((*p32 & gmask) >> gshift);
						*p8++ = ((*p32 & bmask) >> bshift);
						p32++;
					}
					(*job->write) (line, 1, line_size, fp);
					p32 += (image->bytes_per_line - (image->bits_per_pixel >>3)
							* image->width) >> 2;
				}
			}
			break;

		case 8:
		default:
			printf ("bits_per_pixel not supported: %d\n",image->bits_per_pixel);
			return;
			break;
	}
#ifdef HAVE_LIBJPEG
	if (codec == GT_VID_FMT_JPEG) {
		jpeg_finish_compress (&jinfo);
	}
#endif
#ifdef HAVE_LIBPNG
	if (codec == GT_VID_FMT_PNG) {
		png_write_end (p_ptr, pinfo);
		/* release the sturctures, but don't free memory */
		png_destroy_write_struct (&p_ptr, NULL);
	}
#endif
}

/*
 * for indexed color source
 */
void
XImageToQTF8 (FILE *fp, XImage *image, Job *job)
{
	/* not done until now */
	printf ("Visual not supported: %d\n",image->bits_per_pixel);
}

/*
 * for GrayScale and StaticGray visual
 */
void
XImageToQTFG (FILE *fp, XImage *image, Job *job)
{
	int row, col;
	unsigned char *p8;
#ifdef HAVE_LIBJPEG
	static struct jpeg_error_mgr jerr;
#endif

#ifdef DEBUG
	printf ("XImageToQTFG() flags=%d\n", job->flags);
#endif
	/* destroy the old one, we need this static pointer to could
	 * access it also in MngClean()
	 */

	if ( job->state & VC_START ) {
		/* it's the first call, prepare some stuff
		 */
#ifdef DEBUG2
		dump_ximage_info (image);
#endif
		qtfp = fp;
		space_atom = gt_alloc_atom (GTA_skip, HEADER_SIZE-8);
		data_atom = gt_alloc_atom (GTA_movie_data, 0);
		gt_write_atom (space_atom, qtfp);
		gt_write_atom (data_atom, qtfp);
		pic_no = 0;
		bpp = 8;
		save_offset (pic_no++, ftell(qtfp));

		if (job->target == CAP_QTJ) {
			/* we use jpeg codec */
#ifdef HAVE_LIBJPEG
			jinfo.err = jpeg_std_error (&jerr);
			jpeg_create_compress (&jinfo);
			jpeg_stdio_dest (&jinfo, qtfp);
			jinfo.image_width = image->width;
			jinfo.image_height = image->height;
			jinfo.input_components = 1;	/* 1 byte per pixel */
			jinfo.in_color_space = JCS_GRAYSCALE;
			jpeg_set_defaults(&jinfo);
			/* optional parameters */
			jpeg_set_quality (&jinfo, job->quality, TRUE);
			jinfo.dct_method = JDCT_IFAST;
			jpeg_start_compress (&jinfo, TRUE);
			codec = GT_VID_FMT_JPEG;
#else
			fprintf (stderr, "Warning: JPEG not compiled in!\n");
			codec = GT_VID_FMT_RAW;
#endif
		} else if (job->target == CAP_QTP) {
#ifdef  HAVE_LIBPNG
			/* to do */
			p_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
						(void *)NULL, NULL, NULL);
			pinfo = png_create_info_struct (p_ptr);
			if (0/*setjmp (p_ptr->jmpbuf)*/) {
				printf ("fatal error!\n");
				job->state = VC_STOP;
				return;
			}
			codec = GT_VID_FMT_PNG;
#else
			fprintf (stderr, "Warning: PNG not compiled in!\n");
			codec = GT_VID_FMT_RAW;
#endif
		} else /* CAP_QTR */ {
			codec = GT_VID_FMT_RAW;
			line = malloc (image->width * image->height);
		}
	} else {
		save_offset (pic_no++, ftell(qtfp));
#ifdef HAVE_LIBJPEG
		if (codec == GT_VID_FMT_JPEG)
			jpeg_start_compress (&jinfo, TRUE);
#endif
#ifdef HAVE_LIBPNG
		if (codec == GT_VID_FMT_PNG)
			p_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING,
						(void *)NULL, NULL, NULL);
#endif
	}

#ifdef HAVE_LIBJPEG
	/* what a dirty hack, just to pass the right pointer to ->write() */
	if (codec == GT_VID_FMT_JPEG)
		fp = (void *)&jinfo;
#endif
#ifdef HAVE_LIBPNG
	if (codec == GT_VID_FMT_PNG) {
		png_init_io (p_ptr, qtfp);
		png_set_compression_level (p_ptr, job->compress);
		png_set_IHDR (p_ptr, pinfo, image->width, image->height, 8,
				PNG_COLOR_TYPE_GRAY, PNG_INTERLACE_NONE,
				PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);
		png_write_info (p_ptr, pinfo);
		fp = (void *)p_ptr;
	}
#endif

	/* for ZPixmap bits_per_pixel could be 1,4,8,16,24,32
	 * but for Direct- and TrueColor it could only be 8,16,24,32
	 */
	switch (image->bits_per_pixel) {
		case 8:
			p8 = (unsigned char *) image->data;
			if (codec == GT_VID_FMT_RAW) {
				unsigned char *t8;
				for (row = 0; row < image->height; row++) {
					t8 = line;
					for (col = 0; col < image->height; col++) {
						*t8++ = ~p8[col];
					}
					(*job->write) (line, 1, image->width, fp);
					p8 += image->bytes_per_line;
				}
			} else {
				for (row = 0; row < image->height; row++) {
					/* write out the row
					 */
					(*job->write) (p8, 1, image->width, fp);
					p8 += image->bytes_per_line;
				}
			}
			break;

		default:
			printf ("bits_per_pixel not supported: %d\n",image->bits_per_pixel);
			break;
	}
#ifdef HAVE_LIBJPEG
	if (codec == GT_VID_FMT_JPEG) {
		jpeg_finish_compress (&jinfo);
	}
#endif
#ifdef HAVE_LIBPNG
	if (codec == GT_VID_FMT_PNG) {
		png_write_end (p_ptr, pinfo);
		/* release the sturctures, but don't free memory */
		png_destroy_write_struct (&p_ptr, NULL);
	}
#endif
}

/*
 * clean up some allocated stuff and write some
 * control data in the qt file
 */
void
QtClean (Job *job)
{
	unsigned long mdat_size, moov_pos;
	gt_stsd_atom *stsd;
	gt_stts_atom *stts;
	gt_stsz_atom *stsz;
	gt_stsc_atom *stsc;
	gt_stco_atom *stco;
	gt_vmhd_atom *vmhd;
	gt_mdhd_atom *mdhd;
	gt_hdlr_atom *minf_hdlr, *mdia_hdlr;
	gt_dref_atom *dref;
	gt_tkhd_atom *tkhd;
	gt_mvhd_atom *mvhd;
	gt_udta_atom *udta;
	gt_atom *stbl, *minf, *dinf, *mdia, *trak, *moov;
	int no_of_frames = job->pic_no - job->start_no, i;
	
#ifdef DEBUG
	printf ("QtClean() state=%d\n", job->state);
#endif
	if (!qtfp) {
		return;
	}
	moov_pos = ftell (qtfp);
	save_offset (pic_no, moov_pos);
	mdat_size= moov_pos - HEADER_SIZE;
	fseek (qtfp, HEADER_SIZE, SEEK_SET);
	/* write length of mdat atom */
	gt_write4byte (mdat_size, qtfp);

	/* sample description atom */
	stsd = gt_alloc_atom (GTA_sample_desc, 1);
	stsd->count = 1;
	stsd->tab->size = 86;
	stsd->tab->format = codec;
	stsd->tab->index = 1;	/* points to an entry in 'dref' */
	stsd->tab->width = job->area->width;
	stsd->tab->height = job->area->height;
	stsd->tab->hres.high = 72;
	stsd->tab->vres.high = 72;
	stsd->tab->frame_count = 1;
	/* comp_name: max 1 + 31 chars */
	if (codec == GT_VID_FMT_JPEG) {
		strcpy (stsd->tab->comp_name+1, GT_COMP_JPEG);
		stsd->tab->comp_name[0] = strlen (GT_COMP_JPEG);
	} else if (codec == GT_VID_FMT_PNG) {
		strcpy (stsd->tab->comp_name+1, GT_COMP_PNG);
		stsd->tab->comp_name[0] = strlen (GT_COMP_PNG);
	} else {
		strcpy (stsd->tab->comp_name+1, GT_COMP_RAW);
		stsd->tab->comp_name[0] = strlen (GT_COMP_RAW);
	}
	if ((bpp == 8) && (job->win_attr.visual->class == StaticGray)) {
		stsd->tab->depth = 40; /* 8 bit gray scale */
	} else {
		stsd->tab->depth = bpp;
	}
	stsd->tab->ctab_id = -1;

	/* time to sample atom */
	stts = gt_alloc_atom (GTA_time_to_sample, 1);
	stts->count = 1;
	stts->tab->num_samples = no_of_frames;
	stts->tab->duration = job->time_per_frame;

	/* sample to chunk  */
	stsc = gt_alloc_atom (GTA_sample_to_chunk, 1);
	stsc->count = 1;
	stsc->tab->first_chunk = 1;
	stsc->tab->samples_per_chunk = stts->tab->num_samples;
	stsc->tab->sample_id = 1;	/* point to the first entry in the stsd tab!? */

	/* sample size atom .. */
	if (codec == 0xff /*GT_VID_FMT_RAW*/) {
		stsz = gt_alloc_atom (GTA_sample_size, 0);
		stsz->sample_size = job->area->width * job->area->height * bpp/ 8;
		stsz->count = 0;
	} else {
		stsz = gt_alloc_atom (GTA_sample_size, no_of_frames);
		stsz->sample_size = 0;
		stsz->count = no_of_frames;
		for (i = 0; i < no_of_frames; i++ ) {
			stsz->tab[i].size = offset[i+1] - offset[i];
			/* printf ("QtClean() size %d\n", stsz->tab[i].size); */
		}
	}

	/* sample chunk offset atom */
	stco = gt_alloc_atom (GTA_chunk_offset, 1);
	stco->count = 1;
	stco->tab[0].offset = HEADER_SIZE+8; 	/*  offset of mdat */
	stco->size = sizeof (gt_stco_atom) - sizeof (stco->tab) +
			stco->count * sizeof (gt_stco_entry);

	/* sample table */
	stbl = gt_alloc_atom (GTA_sample_table, 5);
	stbl->memb = 5;
	stbl->suba[0] = (gt_atom*)stsd;
	stbl->suba[1] = (gt_atom*)stts;
	stbl->suba[2] = (gt_atom*)stsc;
	stbl->suba[3] = (gt_atom*)stsz;
	stbl->suba[4] = (gt_atom*)stco;
	stbl->size = 8 +
			stsd->size + stts->size + stsc->size + stsz->size + stco->size;

	/* video media information header */
	vmhd = gt_alloc_atom (GTA_video_media_header, 0);
	vmhd->flags[2] = 1;
	/* grmode, opcolor ? */

	/* data handler reference atom ??? */
	minf_hdlr = gt_alloc_atom (GTA_handler_ref, 0);
	minf_hdlr->comp_type = GT_DATA_REF;
	minf_hdlr->comp_subtype = GT_ALIAS;
	minf_hdlr->comp_man = GT_COMP_MAN;

	/* */
	dref = gt_alloc_atom (GTA_data_ref, 1);
	dref->count = 1;
	dref->tab->size = 12;
	dref->tab->type = GT_ALIAS;
	dref->tab->flags[2]= 0x01;	/* self reference */

	/* */
	dinf = gt_alloc_atom (GTA_data_info, 1);
	dinf->memb = 1;
	dinf->suba[0] = (gt_atom *)dref;
	dinf->size = 8 + dref->size;

	/* media information atom */
	minf = gt_alloc_atom (GTA_media_info, 4);
	minf->memb = 4;
	minf->suba[0] = (gt_atom *)vmhd;
	minf->suba[1] = (gt_atom *)minf_hdlr;
	minf->suba[2] = (gt_atom *)dinf;
	minf->suba[3] = (gt_atom *)stbl;
	minf->size += vmhd->size + minf_hdlr->size + dinf->size + stbl->size;


	/* media header */
	mdhd = gt_alloc_atom (GTA_media_header, 0);
	mdhd->ctime = gt_time();
	mdhd->mtime = gt_time();
	mdhd->time_scale = 1000;
	mdhd->duration = job->time_per_frame * no_of_frames;

	/* media handler reference atom */
	mdia_hdlr = gt_alloc_atom (GTA_handler_ref, strlen(GT_MID_HDLR_NAME));
	mdia_hdlr->comp_type = GT_MEDIA_REF;
	mdia_hdlr->comp_subtype = GT_VIDEO;
	mdia_hdlr->comp_man = GT_COMP_MAN;
	mdia_hdlr->comp_flags = 0x40000000;		/* ? */
	mdia_hdlr->comp_flags_mask = 0x00010047;	/* ? */
	strcpy (mdia_hdlr->comp_name+1, GT_MID_HDLR_NAME);
	mdia_hdlr->comp_name[0] = strlen (GT_MID_HDLR_NAME);
	mdia_hdlr->size = 32 + 1 +strlen (GT_MID_HDLR_NAME);

	/* media atom */
	mdia = gt_alloc_atom (GTA_media, 3);
	mdia->memb = 3;
	mdia->suba[0] = (gt_atom *)mdhd;
	mdia->suba[1] = (gt_atom *)mdia_hdlr;
	mdia->suba[2] = (gt_atom *)minf;
	mdia->size += mdhd->size + mdia_hdlr->size + minf->size;

	/* track header */
	tkhd = gt_alloc_atom (GTA_track_header,0);
	tkhd->flags[2] = (char)(0x01 | 0x02);
	tkhd->ctime = gt_time();
	tkhd->mtime = gt_time();
	tkhd->track_id = 1;
	tkhd->duration = job->time_per_frame * no_of_frames;
	tkhd->matrix[1]  = 1;	/* 1/1/1 */
	tkhd->matrix[17] = 1;
	tkhd->matrix[32] = 64;
	tkhd->width.high = job->area->width;
	tkhd->height.high= job->area->height;

	/* track */
	trak = gt_alloc_atom (GTA_track, 2);
	trak->memb = 2;
	trak->suba[0] = (gt_atom *)tkhd;
	trak->suba[1] = (gt_atom *)mdia;
	trak->size += tkhd->size + mdia->size;

	/* movie header */
	mvhd = gt_alloc_atom (GTA_movie_header, 0);
	mvhd->ctime = gt_time();
	mvhd->mtime = gt_time();
	mvhd->time_scale = 1000;
	mvhd->duration = tkhd->duration;
	mvhd->pref_rate.high = 1;
	mvhd->matrix[1] =  1;	/* 1/1/1 */
	mvhd->matrix[17] = 1;
	mvhd->matrix[32] = 64;
	mvhd->next_track_id = 2;

	/* user data
		no space for the text pointers is provided! */
	udta = gt_alloc_atom (GTA_user_data, 3);
	udta->tab[0]->type = GT_UD_INFO;
	udta->tab[0]->tlen = strlen ("Made by 'xvidcap'");
	udta->tab[0]->size = 8 + 4 + udta->tab[0]->tlen;
	udta->tab[0]->text = "Made by 'xvidcap'";
	udta->tab[1]->type = GT_UD_PRODUCER;
	udta->tab[1]->tlen = strlen ("XVidCap User");
	udta->tab[1]->size = 8 + 4 + udta->tab[1]->tlen;
	udta->tab[1]->text = "XVidCap User";
	udta->tab[2]->type = GT_UD_LOOP;
	udta->tab[2]->size = 8;
	udta->size = 8 +udta->tab[0]->size +udta->tab[1]->size +udta->tab[2]->size;

	/* movie atom, with 3 subatoms */
	moov = gt_alloc_atom (GTA_movie, 3);
	moov->memb = 3;
	moov->suba[0] = (gt_atom *)mvhd;
	moov->suba[1] = (gt_atom *)udta;
	moov->suba[2] = (gt_atom *)trak;
	moov->size += mvhd->size + udta->size + trak->size;
	
#ifdef DEBUG
	printf ("size of moov atom: %d bytes\n", moov->size);
#endif
	if (moov->size > HEADER_SIZE-8) {
		/* header size is too large, so add it at the end */
		fseek (qtfp, moov_pos, SEEK_SET);
		gt_write_atom (moov, qtfp);
	} else {
		fseek (qtfp, 0, SEEK_SET);
		gt_write_atom (moov, qtfp);
		space_atom->size = HEADER_SIZE - moov->size;
		gt_write_atom (space_atom, qtfp);
	}
	gt_atom_free (moov);
	qtfp = NULL;
#ifdef HAVE_LIBJPEG
	if (codec == GT_VID_FMT_JPEG)
		jpeg_destroy_compress (&jinfo);
#endif
	if (line) {
		free (line);
		line = NULL;
	}
}

