# -*- coding: utf-8 -*-

import os, os.path, subprocess
from fcntl import flock, LOCK_SH, LOCK_EX

from RestrictedManager.core import DriverHandler, MANAGER_ROOT, notify_reboot_required

from gettext import gettext as _

LRMC_DISABLED_FILE = "/etc/default/linux-restricted-modules-common"

class LRMDriverHandler(DriverHandler):
    """Abstract class for a restricted driver
    ie a driver wich ships with linux-restricted-modules-$(uname -r)."""

    def __init__(self, module):
        DriverHandler.__init__(self, module)

    # Returns true if the module is not listed in
    # /etc/default/linux-restricted-modules-common
    def is_enabled(self):
        return (self.name not in self._lrmc_disabled) and DriverHandler.is_enabled(self)

    def description(self):
        """One-line description of the module (human name)."""
        if (self.name not in self._lrmc_disabled):
            return DriverHandler.description(self)
        else:
            # FIXME: can't we find a better one?
            return self.name

    def enable(self):
        """Enable the module."""

        if self.name in self._lrmc_disabled:
            self._lrmc_disabled.remove(self.name)
            self.save_lrmc_disabled()

            # create fglrx.ko and such
            proc = subprocess.Popen((["/sbin/lrm-manager", "--quick"]), stdout=subprocess.PIPE)


        DriverHandler.enable(self)

    def disable(self):
        """Disable the module."""

        DriverHandler.disable(self)

    @classmethod
    def load_lrmc_disabled(klass):
        """Load the list of disabled modules
        in /etc/default/linux-restricted-modules-common."""
        klass._lrmc_disabled = []

        try:
            disabled_file = open(LRMC_DISABLED_FILE, "r")
        except IOError:
            return

        modules = ""
        try:
            flock(disabled_file.fileno(), LOCK_SH)
            for line in disabled_file:
                try:
                    line = line[:line.index("#")]
                except ValueError:
                    pass

                # example:
                # # Use a space-separated list of modules you wish to not have linked
                # # on boot.  The following example shows a (condensed) list of all
                # # modules shipped in the linux-restricted-modules packages:
                #
                # DISABLED_MODULES="fglrx nv"

                line = line.strip()
                if not line.startswith('DISABLED_MODULES="'):
                    continue
                else:
                    modules = line[len('DISABLED_MODULES="'):].strip()
                    modules = modules[0:len(modules)-1].strip()

        finally:
            disabled_file.close()

        if len(modules):
            klass._lrmc_disabled = modules.split(" ")

        # "nv" spans to nvidia, nvidia_legacy
        # "fc" spans to fcdsl fcdsl2 fcdslsl fcdslslusb fcdslusb fcdslusb2 fcdslusba fcpci fcusb fxusb
        # "ltm" spans to ltmodem ltserial

        spanning = {'nv': ['nvidia', 'nvidia_legacy'],
            'fc': ['fcdsl', 'fcdsl2', 'fcdslsl', 'fcdslslusb', 'fcdslusb', 'fcdslusb2', 'fcdslusba', 'fcpci', 'fcusb', 'fxusb'],
            'ltm': ['ltmodem', 'ltserial']}
        for keyword in spanning.keys():
            if (klass._lrmc_disabled.count(keyword) > 0):
                for i in range(klass._lrmc_disabled.count(keyword)):
                    klass._lrmc_disabled.remove(keyword)
                for spanned in spanning[keyword]:
                    klass._lrmc_disabled.append(spanned)

    @classmethod
    def save_lrmc_disabled(klass):
        """Save the list of disabled in /etc/default/linxu-restricted-modules-common"""

        # here we save the current file as it is, except DISABLED line
        lrmc = []

        try:
            disabled_file = open(LRMC_DISABLED_FILE, "r")
        except IOError:
            return

        try:
            flock(disabled_file.fileno(), LOCK_SH)
            for line in disabled_file:
                if not line.strip().startswith('DISABLED_MODULES="'):
                    lrmc.append(line)
                else:
                    line = line.strip()
                    continue

        finally:
            disabled_file.close()

        modules_list = (klass._lrmc_disabled and reduce(lambda x,y: x + " " + y, klass._lrmc_disabled)) or ""
        modules_line = 'DISABLED_MODULES="' + modules_list + '"'
        lrmc.append(modules_line)

        os.umask(022)
        disabled_file = open(LRMC_DISABLED_FILE, "w")
        try:
            flock(disabled_file.fileno(), LOCK_EX)

            for line in lrmc:
                disabled_file.write(line)
        finally:
            disabled_file.write('\n')
            disabled_file.close()

LRMDriverHandler.load_lrmc_disabled()
