/* ------------------------------------------------------------------------
 * client.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-19 by Steve Houston.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000, 2001 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <iostream>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

// -- 3Dwm Includes
#include <Nobel++/Nobel.hh>
#include <Nobel++/ImageLoader.hh>
#include <Nobel/Shape.hh>
#include <Nobel/Image.hh>
#include <Nobel/NodeKit.hh>
#include <Nobel/Appearance.hh>
#include <Nobel/GeometryKit.hh>
#include <Nobel/TextureKit.hh>
#include <Nobel/TriangleGeometry.hh>

using namespace Nobel;

// -- Local Function Prototypes

static void usage(const char *binary, const char *msg);
static void createTexture(const char* filename, Node_ptr root,
			  NodeKit_ptr node_kit, GeometryKit_ptr geo_kit,
			  TextureKit_ptr tex_kit);

// -- Code Segment

int main(int argc, char *argv[]) {
    
    try {

	// Get the filename from the command line
	if (argc < 2)
	    usage(argv[0], "texture file missing");

	// Initialize the connection
	Nobelxx::Client client("Texture Client", argc, argv);
	       
	// Resolve the necessary kits
	NodeKit_var node_kit =
	    client.resolve<NodeKit>(Nobelxx::name<NodeKit>());
	GeometryKit_var geo_kit =
	    client.resolve<GeometryKit>(Nobelxx::name<GeometryKit>());
	TextureKit_var tex_kit = 
	    client.resolve<TextureKit>(Nobelxx::name<TextureKit>());

	// And here's the scene graph root (local to the client)
	Node_var root = client.root();

	// Build the texture
	createTexture(argv[1], root, node_kit, geo_kit, tex_kit);
    }
    catch (const CORBA::Exception &e) {

	// The ORB threw an exception
	std::cerr << "Uncaught CORBA exception: " << e << std::endl;
	return EXIT_FAILURE;
    }
    catch (...) {
	
	// Another exception was raised
	std::cerr << "Exception thrown. Exiting." << std::endl;
	return EXIT_FAILURE;
    }
    
    return EXIT_SUCCESS;
}

void usage(const char *binary, const char *msg)
{
    std::cerr << "Error: " << msg << std::endl;
    std::cerr << "USAGE: " << binary << " <texture file>" << std::endl;
    throw 0;
}

void createTexture(const char *filename, Node_ptr root,
		   NodeKit_ptr node_kit, GeometryKit_ptr geo_kit,
		   TextureKit_ptr tex_kit)
{
    Nobelxx::ImageLoader img;

    // Create some CORBA objects
    Shape_var shape = node_kit->createShape();
    Appearance_var appear = geo_kit->createAppearance();
    Texture_var tex = tex_kit->createTexture();
    TriangleGeometry_var geo = geo_kit->createTriangleGeometry();

    // Use the ImageLoader utility class to load and send the
    // texture to the server.
    img.loadPNG(filename);
    tex->loadRaw(img.rawData(), img.width(), img.height(), img.pixelType());

    // Link this texture to the appearance object
    appear->setTexture(tex);

    // Set the appearance into the shape
    shape->setAppearance(appear);

    // Initialise the vertex list
    Vertex3D vertex;
    geo->setVertexNumber(4);

    vertex.x = -0.5f;
    vertex.y = -0.5f;
    vertex.z = 0.0f;
    geo->setVertex(0, vertex);

    vertex.x = 0.5f;
    vertex.y = -0.5f;
    vertex.z = 0.0f;
    geo->setVertex(1, vertex);

    vertex.x = 0.5f;
    vertex.y = 0.5f;
    vertex.z = 0.0f;
    geo->setVertex(2, vertex);

    vertex.x = -0.5f;
    vertex.y = 0.5f;
    vertex.z = 0.0f;
    geo->setVertex(3, vertex);

    // Initialise the tex coordinate list
    TexCoord tc;
    geo->setTexCoordNumber(4);

    tc.u = 0.0f;
    tc.v = 0.0f;
    geo->setTexCoord(0, tc);

    tc.u = 1.0f;
    tc.v = 0.0f;
    geo->setTexCoord(1, tc);

    tc.u = 1.0f;
    tc.v = 1.0f;
    geo->setTexCoord(2, tc);

    tc.u = 0.0f;
    tc.v = 1.0f;
    geo->setTexCoord(3, tc);

    // Set the indices
    geo->setFaceNumber(2);
    Triangle face;
    Triangle texCoordFace;

    face.a = 0;
    face.b = 1;
    face.c = 3;
    texCoordFace.a = 0;
    texCoordFace.b = 1;
    texCoordFace.c = 3;
    geo->setVertexIndex(0, face);
    geo->setTexCoordIndex(0, texCoordFace);

    face.a = 1;
    face.b = 2;
    face.c = 3;
    texCoordFace.a = 1;
    texCoordFace.b = 2;
    texCoordFace.c = 3;
    geo->setVertexIndex(1, face);
    geo->setTexCoordIndex(1, texCoordFace);

    // Add geometry to shape and shape to root
    shape->addGeometry(geo);
    root->insert(shape);
}
