/* ------------------------------------------------------------------------
 * $Id: Vector2D.hh,v 1.1 2001/06/15 14:42:04 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-15 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Vector2D_hh_
#define _Vector2D_hh_

// -- System Includes
#include <iostream>

// -- Class Declarations

/**
 * Two-dimensional vector class.
 **/
class Vector2D {
public:
    
    /**
     * Default constructor.
     **/
    Vector2D(float x = 0., float y = 0.);

    /**
     * Constructor.
     **/
    Vector2D(const float c[2]);

    /**
     * Copy constructor.
     **/
    Vector2D(const Vector2D &v);

    /**
     * Destructor.
     **/
    ~Vector2D() { }

    /**
     * Access operator (indexing).
     **/
    float &operator () (int n) { return _coord[n]; }

    /**
     * Access operator (indexing). Read-only.
     **/
    float operator () (int n) const { return _coord[n]; }
    
    float x() const { return _coord[0]; }
    float y() const { return _coord[1]; }

    float &x() { return _coord[0]; }
    float &y() { return _coord[1]; }
    
    /**
     * Vector addition assignment.
     **/
    Vector2D &operator += (const Vector2D &v);
    
    /**
     * Vector subtraction assignment.
     **/
    Vector2D &operator -= (const Vector2D &v);

    /**
     * Vector element-wise addition.
     **/    
    Vector2D operator + (const Vector2D &v) const;

    /**
     * Vector element-wise subtraction.
     **/
    Vector2D operator - (const Vector2D &v) const;

    /**
     * Vector element-wise division.
     **/
    Vector2D operator / (const Vector2D &v) const;

    /**
     * Vector element-wise multiplication.
     **/
    Vector2D operator * (const Vector2D &v) const;
    
    /**
     * Scalar addition assignment.
     **/
    Vector2D &operator += (float f);

    /**
     * Scalar addition assignment.
     **/
    Vector2D &operator -= (float f);

    /**
     * Scalar subtraction assignment.
     **/
    Vector2D &operator /= (float f);

    /**
     * Scalar multiplication assignment.
     **/
    Vector2D &operator *= (float f);

    /**
     * Scalar addition.
     **/
    Vector2D operator + (const float f) const;

    /**
     * Scalar subtraction.
     **/
    Vector2D operator - (const float f) const;

    /**
     * Scalar division.
     **/
    Vector2D operator / (const float f) const;

    /**
     * Scalar multiplication.
     **/
    Vector2D operator * (const float f) const;

    /**
     * Unary negation.
     **/
    Vector2D operator - () const;

    /**
     * Assignment operator.
     **/
    Vector2D &operator = (const Vector2D &v);
    
    /**
     * Compute magnitude of vector.
     **/
    float magnitude();

    /**
     * Normalize vector (make the vector unit-length).
     **/
    void normalize();

    /**
     * Clear the vector (set it to [x y] = [0 0]).
     **/
    void clear();
    
    /**
     * Zero the vector (same as clear, set it to [0 0]).
     **/
    void zero();

    /**
     * Vector output operator.
     **/
    friend std::ostream & operator << (std::ostream &f, const Vector2D &v);

    /**
     * Compute the (vector) dot product.
     * 
     * @param a first vector.
     * @param b second vector.
     * @return the dot product of the two vectors.
     **/
    static float dot(const Vector2D &a, const Vector2D &b);
    
private:
    
    /// Vector coordinates array
    float _coord[2];
};

/**
 * Vector equality operator.
 * 
 * @param a first vector.
 * @param b second vector.
 * @return true if the vectors are equal.
 **/
bool operator == (const Vector2D &a, const Vector2D &b);

/**
 * Scalar vector multiplication.
 **/
inline Vector2D operator * (float f, Vector2D v) {
    return v * f;
}

#endif /* Vector2D.hh */
