/* ------------------------------------------------------------------------
 * VNCInterface.hh
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _VNCInterface_hh_
#define _VNCInterface_hh_

// -- System Includes
#include <string>

// -- 3Dwm Includes
#include "Celsius/Socket.hh"
#include "Garbo/WSInterface.hh"

/**
 * VNC implementation of a windowing system interface.
 **/
class VNCInterface : public WindowingSystemInterface {
    
public:
    
    /**
     * Constructor. This will connect to the specified VNC server and
     * setup the session. 
     * 
     * @param host the name of the host to connect to.
     * @param port the VNC server port (begins at 5900).
     * @param password the password to use for connecting.
     **/
    VNCInterface(const char *host, int port, const char *password);

    /**
     * Destructor.
     **/
    virtual ~VNCInterface();

    /**
     * Run the VNC interface (called by the Celsius thread).
     **/
    virtual void run();

    // Inherited methods
    virtual void updateFramebuffer(int x, int y, int w, int h);
    virtual void pointerEvent(int x, int y, int button_mask);
    virtual void keyEvent(unsigned int key, bool pressed = true);
    
private:

    // Read data from the VNC server
    void readData(void *data, int len);

    // Write data to the VNC server
    void writeData(const void *data, int len);

    // Handle version exchange
    void handleVNCVersion();

    // Authenticate VNC session
    bool handleVNCAuthentication(const char *pwd);
    
    // Initialize VNC session
    void handleVNCInitialization();
    
    // Handle a VNC server message (called by main loop)
    void handleVNCServerMessage();

    // Handle server cut text
    void handleVNCServerCutText();

    // Handle frame buffer update
    void handleVNCFramebufferUpdate();

    // Connect to a VNC server
    void connectToVNCServer(const char *password);

    // Set pixel format
    void setVNCPixelFormat();

    // Set encodings
    void setVNCEncodings();

    // Private member variables
    std::string _host;
    int _port;
    Socket _sock;
    int _serverMajor, _serverMinor;
    bool _incremental;
    PixelFormat _native_pf;
    
    // Read buffer (used in framebuffer updates)
    static const int BUFFER_SIZE = 640 * 480;
    char buffer[BUFFER_SIZE];
};

#endif /* VNCInterface.hh */
