/* ------------------------------------------------------------------------
 * $Id: EvalFunctors.hh,v 1.1 2001/06/19 22:48:00 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-18 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _EvalFunctors_hh_
#define _EvalFunctors_hh_

// -- Zorn Includes
#include "EvalVisitor.hh"

// -- Class Definitions

namespace Zorn {

    class AddFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_double ||
		right.getType() == Value::type_double) {
		double result = 
		    (left.getType() == Value::type_double ?
		     left.getDouble() : left.getInt())
		    +
		    (right.getType() == Value::type_double ?
		     right.getDouble() : right.getInt());
		return Value(result);
	    }
	    else {
		int result = left.getInt() + right.getInt();
		return Value(result);
	    }
	}
    };

    class SubFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_double ||
		right.getType() == Value::type_double) {
		double result =
		    (left.getType() == Value::type_double ?
		     left.getDouble() : left.getInt())
		    -
		    (right.getType() == Value::type_double ?
		     right.getDouble() : right.getInt());
		return Value(result);
	    }
	    else {
		int result = left.getInt() - right.getInt();
		return Value(result);
	    }
	}
    };

    class DivFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {	
	    if (left.getType() == Value::type_double ||
		right.getType() == Value::type_double) {
		double result = 
		    (left.getType() == Value::type_double ?
		     left.getDouble() : left.getInt())
		    /
		    (right.getType() == Value::type_double ?
		     right.getDouble() : right.getInt());
		return Value(result);
	    }
	    else {
		int result = left.getInt() / right.getInt();
		return Value(result);
	    }
	}
    };

    class MulFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_double ||
		right.getType() == Value::type_double) {
		double result = 
		    (left.getType() == Value::type_double ?
		     left.getDouble() : left.getInt())
		    *
		    (right.getType() == Value::type_double ?
		     right.getDouble() : right.getInt());
		return Value(result);
	    }
	    else {
		int result = left.getInt() * right.getInt();
		return Value(result);
	    }
	}
    };

    class AndFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    return Value(left.getBool() && right.getBool());
	}
    };

    class OrFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    return Value(left.getBool() || right.getBool());
	}
    };

    class EqFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_string ||
		right.getType() == Value::type_string) {
	    
		if (left.getType() != Value::type_string ||
		    right.getType() != Value::type_string) 
		    return Value(false);
		else
		    return Value(left.getString() == right.getString());
	    }
	    else {
		bool result = 
		    (left.getType() == Value::type_double ? left.getDouble() : 
		     left.getType() == Value::type_int ? left.getInt() :
		     left.getBool())
		    ==
		    (right.getType() == Value::type_double ? right.getDouble():
		     right.getType() == Value::type_int ? right.getInt() :
		     right.getBool());
		return Value(result);
	    }
	}
    };

    class NeqFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_string ||
		right.getType() == Value::type_string) {
	    
		if (left.getType() != Value::type_string ||
		    right.getType() != Value::type_string) 
		    return Value(false);
		else
		    return Value(left.getString() !=  right.getString());
	    }
	    else {
		bool result = 
		    (left.getType() == Value::type_double ? left.getDouble() : 
		     left.getType() == Value::type_int ? left.getInt() :
		     left.getBool())
		    ==
		    (right.getType() == Value::type_double ? right.getDouble():
		     right.getType() == Value::type_int ? right.getInt() :
		     right.getBool());
		return Value(result);
	    }
	}
    };

    class LeqFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_string ||
		right.getType() == Value::type_string) {
		return Value(false);
	    }
	    else {
		bool result = 
		    (left.getType() == Value::type_double ?
		     left.getDouble() : left.getInt())
		    <=
		    (right.getType() == Value::type_double ?
		     right.getDouble() : right.getInt());
		return Value(result);
	    }
	}
    };

    class GeqFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_string ||
		right.getType() == Value::type_string) {
		return Value(false);
	    }
	    else {
		bool result = 
		    (left.getType() == Value::type_double ?
		     left.getDouble() : left.getInt())
		    >=
		    (right.getType() == Value::type_double ?
		     right.getDouble() : right.getInt());
		return Value(result);
	    }
	}
    };

    class GreatFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_string ||
		right.getType() == Value::type_string) {
		return Value(false);
	    }
	    else {
		bool result = 
		    (left.getType() == Value::type_double ?
		     left.getDouble() : left.getInt())
		    >
		    (right.getType() == Value::type_double ?
		     right.getDouble() : right.getInt());
		return Value(result);
	    }
	}
    };

    class LessFunctor : public EvalVisitor::BinaryFunctor {
    public:
	Value execute(const Value &left, const Value &right) {
	    if (left.getType() == Value::type_string ||
		right.getType() == Value::type_string) {
		return Value(false);
	    }
	    else {
		bool result = 
		    (left.getType() == Value::type_double ?
		     left.getDouble() : left.getInt())
		    <
		    (right.getType() == Value::type_double ?
		     right.getDouble() : right.getInt());
		return Value(result);
	    }
	}
    };

    class NegFunctor : public EvalVisitor::UnaryFunctor {
    public:
	Value execute(const Value &operand) {
	    if (operand.getType() == Value::type_int)
		return Value(-operand.getInt());
	    else if (operand.getType() == Value::type_double)
		return Value(-operand.getDouble());
	    else 
		return operand;
	}
    };

    class NotFunctor : public EvalVisitor::UnaryFunctor {
    public:
	Value execute(const Value &operand) {
	    if (operand.getType() == Value::type_bool)
		return Value(!operand.getBool());
	    else
		return operand;
	}
    };

};

#endif /* EvalFunctors.hh */
