/* ------------------------------------------------------------------------
 * $Id: Value.hh,v 1.2 2001/06/20 12:32:11 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-12 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Value_hh_
#define _Value_hh_

// -- System Includes
#include <iostream>
#include <string>

namespace Zorn {

    // -- Class Declarations

    /**
     * Polymorphic value class.
     **/
    class Value {
    public:
    
	/**
	 * Value type enumeration.
	 **/
	typedef enum { type_double, type_string, type_int, type_bool } type_t;

	/**
	 * Constructor.
	 **/
	Value() { set(0); }

	/**
	 * Copy constructor.
	 **/
	Value(const Value &t) :
	    _type(t._type), _double_val(t._double_val), _int_val(t._int_val),
	    _string_val(t._string_val), _bool_val(t._bool_val) { }

	/**
	 * Constructor (double initialization).
	 *
	 * @param d double value.
	 **/
	Value(double d) { set(d); }

	/**
	 * Constructor (integer initialization).
	 *
	 * @param i integer value.
	 **/
	Value(int i) { set(i); }

	/**
	 * Constructor (string initialization).
	 *
	 * @param s string value.
	 **/
	Value(const std::string &s) { set(s); }

	/**
	 * Constructor (boolean initialization).
	 *
	 * @param b boolean value.
	 **/
	Value(bool b) { set(b); }
    
	/**
	 * Set the value to a double.
	 *
	 * @param d double value.
	 **/
	void set(double d) {
	    _type = type_double;
	    _double_val = d;
	    convert();
	}

	/**
	 * Set the value to an integer.
	 *
	 * @param i integer value.
	 **/
	void set(int i) {
	    _type = type_int;
	    _int_val = i;
	    convert();
	}

	/**
	 * Set the value to a string.
	 *
	 * @param s string value.
	 **/
	void set(const std::string &s) {
	    _type = type_string;
	    _string_val = s;
	}

	/**
	 * Set the value to a boolean.
	 *
	 * @param b boolean value.
	 **/
	void set(bool b) {
	    _type = type_bool;
	    _bool_val = b;
	    convert();
	}
    
	type_t getType() const { return _type; }

	double getDouble() const {
	    return _double_val;
	}
	int getInt() const {
	    return _int_val; 
	}
	const std::string &getString() const {
	    return _string_val; 
	}
	bool getBool() const {
	    return _bool_val; 
	}

	friend std::ostream &operator << (std::ostream &os,
					  const Value &value);
    
    private:

	void convert() {
	    switch (_type) {
	    case type_double: 
		_int_val = (int) _double_val;
		_bool_val = _double_val != 0;
		break;
	    case type_int:
		_double_val = (double) _int_val;
		_bool_val = _int_val != 0;
		break;
	    case type_bool:
		_double_val = _bool_val ? 1.0 : 0.0;
		_int_val = _bool_val ? 1 : 0;
		break;
	    default:
		break;
	    };
	}

	type_t _type;
	double _double_val;
	int _int_val;
	std::string _string_val;
	bool _bool_val;
    };

    /**
     * Named value class.
     **/
    struct NamedValue : public std::pair<std::string, Value> {
	NamedValue(const std::string &name, const Value &value)
	    : std::pair<std::string, Value>(name, value) { }
	const std::string &name() const { return first; }
	const Value &value() const { return second; }
    };

    /**
     * Stream output operator.
     *
     * @param os output stream.
     * @param value value to output.
     * @return the output stream.
     **/
    inline std::ostream &operator << (std::ostream &os, const Value &value)
    {
	// And dump the value as well;
	switch (value.getType()) {
	case Value::type_bool:
	    os << (value.getBool() ? "true" : "false");
	    break;
	case Value::type_int:
	    os << value.getInt();
	    break;
	case Value::type_double:
	    os << value.getDouble();
	    break;
	case Value::type_string:
	    os << value.getString();
	    break;
	default: break;
	}
	return os;
    }

};

#endif /* Value.hh */
