/* ------------------------------------------------------------------------
 * $Id: Matrix3D.cc,v 1.8 2001/07/30 15:19:00 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-08-11 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- Local Includes
#include "Exception.hh"
#include "Matrix3D.hh"

// -- Code Segment

void Matrix3D::clear(float f)
{
    for (int i = 0; i < _dim; i++) {
	val(i, 0) = f; val(i, 1) = f; val(i, 2) = f; val(i, 3) = f;
    }
}

void Matrix3D::identity()
{
    clear();
    val(0, 0) = 1; val(1, 1) = 1; val(2, 2) = 1; val(3, 3) = 1;
}

void Matrix3D::transpose()
{
    float temp[_dim * _dim];
    for (int i = 0; i < _dim; i++) {
	val(temp, 0, i) = val(i, 0); val(temp, 1, i) = val(i, 1);
	val(temp, 2, i) = val(i, 2); val(temp, 3, i) = val(i, 3);
    }
    memcpy(_data, temp, sizeof(_data));
}

float Matrix3D::determinant() const
{
    float d;
    d  =  val(0, 0) * val(1, 1) * val(2, 2);
    d +=  val(0, 1) * val(1, 2) * val(2, 0);
    d +=  val(0, 2) * val(1, 0) * val(2, 1);
    d += -val(0, 2) * val(1, 1) * val(2, 0);
    d += -val(0, 1) * val(1, 0) * val(2, 2);
    d += -val(0, 0) * val(1, 2) * val(2, 1);
    return d;
}

void Matrix3D::invert()
{
    // First, compute the determinant and check for singular matrix
    float d = determinant();
    if (Math::equal(d, 0.) == true) throw Exception("singular matrix");

    // We need to make a temporary copy of the matrix
    Matrix3D m = *this;
    
    // Perform matrix computations
    val(0, 0) =  (m(1, 1) * m(2, 2) - m(1, 2) * m(2, 1)) / d;
    val(0, 1) = -(m(0, 1) * m(2, 2) - m(0, 2) * m(2, 1)) / d;
    val(0, 2) =  (m(0, 1) * m(1, 2) - m(0, 2) * m(1, 1)) / d;
    val(1, 0) = -(m(1, 0) * m(2, 2) - m(1, 2) * m(2, 0)) / d;
    val(1, 1) =  (m(0, 0) * m(2, 2) - m(0, 2) * m(2, 0)) / d;
    val(1, 2) = -(m(0, 0) * m(1, 2) - m(0, 2) * m(1, 0)) / d;
    val(2, 0) =  (m(1, 0) * m(2, 1) - m(1, 1) * m(2, 0)) / d;
    val(2, 1) = -(m(0, 0) * m(2, 1) - m(0, 1) * m(2, 0)) / d;
    val(2, 2) =  (m(0, 0) * m(1, 1) - m(0, 1) * m(1, 0)) / d;
    
    val(0, 3) = -(m(0,3) * val(0,0) + m(1,3) * val(0,1) + m(2,3) * val(0,2));
    val(1, 3) = -(m(0,3) * val(1,0) + m(1,3) * val(1,1) + m(2,3) * val(1,2));
    val(2, 3) = -(m(0,3) * val(2,0) + m(1,3) * val(2,1) + m(2,3) * val(2,2));
}

Matrix3D Matrix3D::inverse() const
{
    // Copy source matrix, invert and return it
    Matrix3D result(*this);
    result.invert();
    return result;
}

Matrix3D Matrix3D::operator * (const Matrix3D &m) const
{
    Matrix3D res;
    for (int i = 0; i < _dim; i++)
	for (int j = 0; j < _dim; j++) 
	    for (int k = 0; k < _dim; k++)
		res.val(i, j) += val(i, k) * m.val(k, j);
    return res;
}

Matrix3D Matrix3D::operator + (const Matrix3D &m) const
{
    Matrix3D res;
    for (int i = 0; i < _dim; i++) {
	res.val(i, 0) = val(i, 0) + m.val(i, 0);
	res.val(i, 1) = val(i, 1) + m.val(i, 1);
	res.val(i, 2) = val(i, 2) + m.val(i, 2);
	res.val(i, 3) = val(i, 3) + m.val(i, 3);
    }
    return res;
}

Matrix3D Matrix3D::operator - (const Matrix3D &m) const
{
    Matrix3D res;
    for (int i = 0; i < _dim; i++) {
	res.val(i, 0) = val(i, 0) - m.val(i, 0);
	res.val(i, 1) = val(i, 1) - m.val(i, 1);
	res.val(i, 2) = val(i, 2) - m.val(i, 2);
	res.val(i, 3) = val(i, 3) - m.val(i, 3);
    }
    return res;
}

Matrix3D Matrix3D::operator * (float f) const
{
    Matrix3D res;
    for (int i = 0; i < _dim; i++) {
	res.val(i, 0) = f * val(i, 0); res.val(i, 1) = f * val(i, 1);
	res.val(i, 2) = f * val(i, 2); res.val(i, 3) = f * val(i, 3);
    }
    return res;
}

Matrix3D Matrix3D::operator + (float f) const 
{
    Matrix3D res;
    for (int i = 0; i < _dim; i++) {
	res.val(i, 0) = f + val(i, 0); res.val(i, 1) = f + val(i, 1);
	res.val(i, 2) = f + val(i, 2); res.val(i, 3) = f + val(i, 3);
    }
    return res;
}

Matrix3D Matrix3D::operator - (float f) const
{
    Matrix3D res;
    for (int i = 0; i < _dim; i++) {
	res.val(i, 0) = f - val(i, 0); res.val(i, 1) = f - val(i, 1);
	res.val(i, 2) = f - val(i, 2); res.val(i, 3) = f - val(i, 3);
    }
    return res;
}

Matrix3D Matrix3D::operator / (float f) const
{
    Matrix3D res;
    for (int i = 0; i < _dim; i++) {
	res.val(i, 0) = f / val(i, 0); res.val(i, 1) = f / val(i, 1);
	res.val(i, 2) = f / val(i, 2); res.val(i, 3) = f / val(i, 3);
    }
    return res;
}

std::ostream & operator << (std::ostream &f, const Matrix3D &m)
{
    for (int i = 0; i < m.rows(); i++) {
	f << "| ";
	for (int j = 0; j < m.columns(); j++)
	    f << m(i, j) << " ";
	f << "|" << std::endl;
    }
    return f;
}
