/* ------------------------------------------------------------------------
 * ServerImpl.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-10-24 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <iostream>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Thread.hh"
#include "Celsius/Guard.hh"
#include "Celsius/Logger.hh"
#include "Celsius/Time.hh"
#include "Polhem/LogGroup.hh"
#include "Polhem/NodeImpl.hh"
#include "Polhem/ServerHandleImpl.hh"
#include "Polhem/SceneManager.hh"
#include "Polhem/ServerImpl.hh"

using namespace Nobel;
    
// -- Code Segment

ServerImpl::ServerImpl(SceneManager *smanager)
    : _smanager(smanager), _quit(false)
{
    // empty
}

ServerImpl::~ServerImpl()
{
    // Get rid of all clients
    for (clist_t::iterator i = _clients.begin(); i != _clients.end(); i++) {
	
	// @@@ Call scene manager, too!
	destroyClient(i->handle);
    }
}

ServerHandle_ptr ServerImpl::connect(const char *name, ClientHandle_ptr c)
{
    Guard<Mutex> guard(_mutex);

    // Create a new local root and assign it to the new server handle
    NodeImpl *root = _smanager->createClientRoot();
    ServerHandleImpl *handle = new ServerHandleImpl(root);
    _clients.push_back(client_t(name, handle, ClientHandle::_duplicate(c)));
    
    Logger::log(LogGroup::Server) << "Client '" << name << "' connected." 
				  << std::endl;
    return handle->_this();
}

void ServerImpl::run()
{
    // Sleep for a while, then ping clients. 
    while (_quit == false) {
	Time::sleep(1.0);
	ping();
    }
}

void ServerImpl::ping()
{
    Guard<Mutex> guard(_mutex);
    clist_t temp;
    
    // Iterate through all clients 
    for (clist_t::iterator i = _clients.begin();i != _clients.end();i++) {
	
	try {
	    // Ping the object and, upon success, add it to the temp list
	    i->client->ping();
	    temp.push_back(*i);
	}
	catch (const CORBA::COMM_FAILURE &) {
	    Logger::log(LogGroup::Server) << "Lost client '" << i->name 
					  << "'." << std::endl;
	    destroyClient(i->handle);
	}
    }
    _clients = temp;
}

void ServerImpl::destroyClient(ServerHandleImpl *handle)
{
    // @@@ Need to perform GC of client-constructed objects here
    deactivate(handle);
    delete handle;
}
