/* ------------------------------------------------------------------------
 * $Id: View.cc,v 1.12 2001/07/27 14:50:18 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-11-24 by Niklas Elmqvist.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Exception.hh"
#include "Polhem/NodeImpl.hh"
#include "Polhem/ViewSlotImpl.hh"
#include "Polhem/RendererImpl.hh"
#include "Polhem/View.hh"

using namespace Nobel;

// -- Code Segment

View::View(RendererImpl *r)
    : _currentSlot(ViewSlot::_nil()),
      _renderer(r),
      _fov(60.0f),
      _fov_changed(true)
{
    _rotation.fromAngleAxis(0.0f, Vector3D(0.0f, 1.0f, 0.0f));
}

View::~View()
{
    // Deactivate and deallocate renderer
    deactivate(_renderer);
    delete _renderer;    
}
    
void View::attachTo(ViewSlot_ptr slot)
{
    _currentSlot = slot;
    // @@@Send attachment event to client?
}

void View::render(NodeImpl *root)
{
    // Sanity check
    if (_currentSlot == 0) throw Exception();

    // Do we need to change the project parameters?
    if (_fov_changed) {
	float old_fov, old_aspect, near, far;
	
	// Retrieve old values and update the field-of-view
	_renderer->getProjection(old_fov, old_aspect, near, far);
	_renderer->setProjection(_fov, old_aspect, near, far);

	_fov_changed = false;
    }

    // Set the viewing transform into the renderer
    _renderer->setViewTransform(computeViewTransform());
    
    // Render the scene
    root->traverse(Nobel::Renderer_var(_renderer->_this()));
}

Matrix3D View::computeViewTransform()
{
    Matrix3D trans_matrix, rot_matrix;
    
    // Turn the rotation quaternion into a matrix
    rot_matrix = _rotation.toMatrix();
    
    // Compute the translation matrix (it's just a distance)
    trans_matrix.identity();
    trans_matrix(0, 3) = _translation.x();
    trans_matrix(1, 3) = _translation.y();
    trans_matrix(2, 3) = _translation.z();
    
    // Retrieve the view slot transformation
    Nobel::Matrix m;
    Nobel::Transform_var transform = _currentSlot->transformation();
    transform->store(m);
    Matrix3D slot_matrix(m);

    // Compute the resulting matrix and invert it
    Matrix3D view_matrix = slot_matrix * (rot_matrix * trans_matrix);
    view_matrix.invert();

    return view_matrix;
}

void View::rotate(float angle, const Nobel::Vertex3D &axis)
{
    if (_currentSlot == 0) throw Exception();
    Nobel::Transform_var transform = _currentSlot->transformation();
    transform->rotate(angle, axis);
}
    
void View::translate(const Nobel::Vertex3D &v)
{
    if (_currentSlot == 0) throw Exception();
    Nobel::Transform_var transform = _currentSlot->transformation();
    transform->translate(v);
}

void View::orbit(float angle, const Vector3D &axis)
{
    Quaternion orbit_rotation;
    
    // Construct the rotation quaternion from the specified values
    orbit_rotation.fromAngleAxis(angle, axis);

    // Apply it to the existing rotation quaternion
    _rotation = _rotation * orbit_rotation;
    _rotation.normalize();
}

void View::dolly(float delta)
{
    // Modify the view distance (z value) and clamp it to legal values
    // @@@ If negative distance, the rotation should invert!
    _translation.z() += delta;
    if (_translation.z() < 0.0f) _translation.z() = 0.0f;
}

void View::zoom(float delta)
{
    float old_fov = _fov;

    // Modify field of view and clamp it to legal values
    // @@@ Define what the legal values are!
    _fov += delta;
    if (_fov < 0.0f) _fov = 0.0f;
    else if (_fov > 180.0f) _fov = 180.0f;

    if (old_fov != _fov) _fov_changed = true;
}

void View::track(const Vector2D &v)
{
    // Modify the view position in the X-Y plane, which is
    // perpendicular to the view direction.
    _translation.x() += v.x();
    _translation.y() += v.y();
}

View::ViewParameters View::getViewParameters() const
{
    ViewParameters params;

    // Retrieve the parameters from the renderer
    _renderer->getViewportDimensions(params.width, params.height);
    _renderer->getProjection(params.fov, params.aspect,
			     params.near, params.far);
    
    return params;
}
