/* ------------------------------------------------------------------------
 * $Id: BSPTreeBuilder.cc,v 1.1 2001/07/27 14:50:20 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-17 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <stack>

// -- 3Dwm Includes
#include "Celsius/Math.hh"
#include "Solid/BSP.hh"
#include "Solid/BSPTree.hh"
#include "Solid/BSPTreeBuilder.hh"

// Partitioning plane thickness
float BSPTreeBuilder::_thickness = Math::epsilon;

// -- Code Segment

BSPTree *BSPTreeBuilder::build(const std::vector<TriangleFace> &faces,
			       const std::vector<Vector3D> &vertices,
			       const std::vector<Vector3D> &normals,
			       const std::vector<Vector2D> &texCoords)
{
    // Create a new BSP container
    BSPTree *bsp = new BSPTree();

    // Add geometry data to the new BSP tree
    bsp->addVertices(vertices);
    bsp->addNormals(normals);
    bsp->addTexCoords(texCoords);
    
    // Call the recursive tree building function
    BSP *tree = buildTree(*bsp, faces, true);
    bsp->setTree(tree);
    
    return bsp;
}

BSPTree *BSPTreeBuilder::combine(BSPTree *t1, BSPTree *t2, SetOperation op)
{
    // Clone the first operand
    BSPTree *bsp = new BSPTree(*t1);
    
    // Retrieve the offset for the new vertices
    int v_offset = bsp->getVertexCount();
    int n_offset = bsp->getNormalCount();
    int tc_offset = bsp->getTexCoordCount();
    
    // Now, insert all the vertices, normals and texture coordinates
    // from the second tree into the new tree.
    bsp->addVertices(t2->getVertices());
    bsp->addNormals(t2->getNormals());
    bsp->addTexCoords(t2->getTexCoords());
    
    // Build a list of triangle faces from the second BSP tree
    std::vector<TriangleFace> face_list;
    t2->getTree()->buildFaceList(face_list);
    
    // Step through the faces and add the index offsets. This is
    // because the new faces will be added to the existing tree, and
    // their vertex, normal and texture coordinate offsets will have
    // changed.
    for (std::vector<TriangleFace>::iterator i = face_list.begin();
	 i != face_list.end(); i++) {
	i->v += v_offset;
	i->n += n_offset;
	i->tc += tc_offset;
    }
    
    // Now, perform incremental CSG building
    bsp->combine(face_list, op);
    
    return bsp;
}

BSPTree *BSPTreeBuilder::complement(BSPTree *tree)
{
    // Clone the tree
    BSPTree *bsp = new BSPTree(*tree);

    // Now, complement the clone
    bsp->complement();
    
    return bsp;
}

BSPTreeBuilder::Classification
BSPTreeBuilder::classifyPoint(const Plane3D &bp, const Vector3D &p)
{
    // Classify point and decide on a classification value
    float value = bp.classify(p);
    return Math::equal(value, 0.0, _thickness) ? Coincident :
	(value < 0 ? InBackOf : InFrontOf);
}

std::ostream &operator << (std::ostream &f,
			   const BSPTreeBuilder::Classification &c)
{
    f << (c == BSPTreeBuilder::Coincident ? "coincident" : 
	  (c == BSPTreeBuilder::InFrontOf ? "in front of" : "in back of"));
    return f;
}

/**
 * TriangleFace index struct. Contains all the indices (vertex, normal and
 * texture coordinate index) for one point in the triangle.
 **/
struct index_t {
    int v, n, tc;
    index_t(int _v, int _n, int _tc) : v(_v), n(_n), tc(_tc) { }
};

void BSPTreeBuilder::partitionTriangleFace(BSPTree &t,
				       const Plane3D &bp,
				       const TriangleFace &f,
				       std::vector<TriangleFace> &on_list,
				       std::vector<TriangleFace> &front_list,
				       std::vector<TriangleFace> &back_list)
{
    bool coincident = true;
    std::vector<index_t> front_indices, back_indices;
    Vector2D tc1, tc2;
    Vector3D pt1, pt2, n1, n2;
    Classification c1, c2;
    
    // Start with the last vertex
    pt1 = t.getVertex(f.v[2]);
    n1 = t.getNormal(f.n[2]);
    tc1 = t.getTexCoord(f.tc[2]);
    
    c1 = classifyPoint(bp, pt1);
    
    // Cycle through all vertices
    for (int i = 0; i < 3; i++) {
    
	// Classify this vertex
	pt2 = t.getVertex(f.v[i]);
	n2 = t.getNormal(f.n[i]);
	tc2 = t.getTexCoord(f.tc[i]);
	
	c2 = classifyPoint(bp, pt2);
    
	// Are the two points on different sides of the plane? 
	if (c2 == InFrontOf) {
	    
	    // Mark that not all vertices are coincident with plane
	    coincident = false;
	    
	    if (c1 == InBackOf) {

		// Compute lines for all entities
		Line<Vector3D> v_line(pt1, pt2);
		Line<Vector3D> n_line(n1, n2);
		Line<Vector2D> tc_line(tc1, tc2);

		// Now, find the intersection with the plane
		float param = bp.intersect(v_line);
	    
		// Compute and add intersection to vertex list
		int v_index = t.addVertex(v_line * param);
		int n_index = t.addNormal(n_line * param);
		int tc_index = t.addTexCoord(tc_line * param);
		
		// Add index to both index lists
		front_indices.push_back(index_t(v_index, n_index, tc_index));
		back_indices.push_back(index_t(v_index, n_index, tc_index));
	    }

	    // Add this vertex index, too
	    front_indices.push_back(index_t(f.v[i], f.n[i], f.tc[i]));
	}
	else if (c2 == InBackOf) {

	    // Mark that not all vertices are coincident with plane
	    coincident = false;

	    if (c1 == InFrontOf) {

		// Compute lines for all entities
		Line<Vector3D> v_line(pt1, pt2);
		Line<Vector3D> n_line(n1, n2);
		Line<Vector2D> tc_line(tc1, tc2);

		// Now, find the intersection with the plane
		float param = bp.intersect(v_line);
	    
		// Compute and add intersection to vertex list
		int v_index = t.addVertex(v_line * param);
		int n_index = t.addNormal(n_line * param);
		int tc_index = t.addTexCoord(tc_line * param);
		
		// Add index to both index lists
		front_indices.push_back(index_t(v_index, n_index, tc_index));
		back_indices.push_back(index_t(v_index, n_index, tc_index));
	    }

	    // Add this vertex index, too
	    back_indices.push_back(index_t(f.v[i], f.n[i], f.tc[i]));
	}
	else {
	    
	    // Point belongs in both front and back
	    front_indices.push_back(index_t(f.v[i], f.n[i], f.tc[i]));
	    back_indices.push_back(index_t(f.v[i], f.n[i], f.tc[i]));
	}
	
	// Move on to the next vertex
	pt1 = pt2;
	n1 = n2; 
	tc1 = tc2;
	c1 = c2;
    }
    
    // If the face is coincident with the plane, just add the face
    if (coincident == true) {
	on_list.push_back(f);
    }
    else {

	// Generate triangles for the front face list
	if (front_indices.size() > 2) {

	    TriangleFace t;

	    // Use the same material as the original
	    t.material = f.material;
	    
	    // Build the triangle indices
	    t.v = TriangleIndex(front_indices[0].v,
			       front_indices[1].v,
			       front_indices[2].v);
	    t.n = TriangleIndex(front_indices[0].n,
			       front_indices[1].n,
			       front_indices[2].n);
	    t.tc = TriangleIndex(front_indices[0].tc,
				front_indices[1].tc,
				front_indices[2].tc);
	    
	    // Add the first triangle
	    front_list.push_back(t);
	    
	    // If there are more indices, generate a second
	    if (front_indices.size() > 3) {
		
		// Build the triangle indices
		t.v = TriangleIndex(front_indices[2].v,
				   front_indices[3].v,
				   front_indices[0].v);
		t.n = TriangleIndex(front_indices[2].n,
				   front_indices[3].n,
				   front_indices[0].n);
		t.tc = TriangleIndex(front_indices[2].tc,
				    front_indices[3].tc,
				    front_indices[0].tc);

		// Add the second triangle
		front_list.push_back(t);
	    }
	}
	
	// Now, generate triangles for the back face list instead
	if (back_indices.size() > 2) {

	    TriangleFace t;
	    
	    // Use the same material as the original
	    t.material = f.material;
	    
	    // Build the triangle indices
	    t.v = TriangleIndex(back_indices[0].v,
			       back_indices[1].v,
			       back_indices[2].v);
	    t.n = TriangleIndex(back_indices[0].n,
			       back_indices[1].n,
			       back_indices[2].n);
	    t.tc = TriangleIndex(back_indices[0].tc,
				back_indices[1].tc,
				back_indices[2].tc);
	    
	    // Add the first triangle
	    back_list.push_back(t);
	    
	    // If there are more indices, generate a second
	    if (back_indices.size() > 3) {
		
		// Build the triangle indices
		t.v = TriangleIndex(back_indices[2].v,
				   back_indices[3].v,
				   back_indices[0].v);
		t.n = TriangleIndex(back_indices[2].n,
				   back_indices[3].n,
				   back_indices[0].n);
		t.tc = TriangleIndex(back_indices[2].tc,
				    back_indices[3].tc,
				    back_indices[0].tc);
		
		// Add the second triangle
		back_list.push_back(t);
	    }
	}
    }
}

void BSPTreeBuilder::partition(BSPTree &t, const Plane3D &bp,
			       const std::vector<TriangleFace> &faces,
			       std::vector<TriangleFace> &on_list, 
			       std::vector<TriangleFace> &front_list, 
			       std::vector<TriangleFace> &back_list)
{
    // Now, partition all faces into the front and back halfspaces
    for (std::vector<TriangleFace>::const_iterator i = faces.begin();
	 i != faces.end(); i++)

	// Partition the triangle face
	partitionTriangleFace(t, bp, *i, on_list, front_list, back_list);
}

BSP *BSPTreeBuilder::buildTree(BSPTree &t, const std::vector<TriangleFace> &faces,
			       bool front)
{
    // Base case: Are we all out of faces in this half-space? 
    if (faces.size() == 0)
	
	// Yes, we are. Create a BSP leaf that represents a single cell!
	return new BSPLeaf(front);
    
    // Select a partition plane (assume counter-clock-wise vertex
    // winding). We may want a better plane selection scheme later!
    TriangleIndex pf = faces[0].v;
    Plane3D bp(t.getVertex(pf.a()), t.getVertex(pf.b()), t.getVertex(pf.c()));
    
    // Create a new node, passing the binary partition
    BSPNode *node = new BSPNode(bp);
    
    // Set up the front and back polygon lists
    std::vector<TriangleFace> front_list, back_list;
    
    // Now, partition all faces into the front and back halfspaces.
    // Add all coincident faces to the current face list for the node.
    partition(t, bp, faces, node->getFaces(), front_list, back_list);
    
    // We have partitioned the faces into two sets, now it is time to
    // call the tree building method recursively on each set.
    
    // First, we set the front subtree
    node->setFront(buildTree(t, front_list, true));
    
    // And then, the back subtree
    node->setBack(buildTree(t, back_list, false));
    
    return node;
}
