;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;David M. Russinoff
;;;Advanced Micro Devices, Inc.
;;;June, 2001
;;;***************************************************************

(in-package "ACL2")

(local (include-book "../support/top"))

(include-book "basic")


;;;**********************************************************************
;;;                             BVECP
;;;**********************************************************************

(defun bvecp (x k)
  (and (integerp x)
       (>= x 0)
       (< x (expt 2 k))))

(in-theory (disable bvecp))

(defthm bvecp-forward
  (implies (bvecp x k)
           (and (integerp x)
                (<= 0 x)
                (< x (expt 2 k))))
  :rule-classes :forward-chaining)

(defthm natp-bvecp
    (implies (bvecp x n)
	     (natp x)))

(defthm bvecp<=
    (implies (and (natp n)
		  (bvecp x n))
	     (<= x (1- (expt 2 n))))
  :rule-classes ())

(defthm bvecp-fl
    (implies (and (not (zp n))
		  (bvecp x n))
	     (bvecp (fl (* 1/2 x)) (1- n))))

(in-theory (disable bvecp-fl))

(defthm bvecp-plus
    (implies (and (bvecp x m)
		  (bvecp y n)
		  (natp m)
		  (natp n))
	     (bvecp (* x y) (+ m n)))
  :rule-classes ())

(defthm bvecp+1
    (implies (and (natp n)
		  (bvecp x n))
	     (bvecp x (+ 1 n))))

(in-theory (disable bvecp+1))

(defthm bvecp-shift
    (implies (and (bvecp x n)
		  (natp n)
		  (natp k)
		  (>= n k))
	     (bvecp (fl (/ x (expt 2 k))) (- n k))))

(in-theory (disable bvecp-shift))

;;This lemma may be enabled to induce case-splitting on bit vectors of
;;length 1:

(defthm bvecp-1-rewrite
    (iff (bvecp x 1)
	 (member x '(0 1))))

(in-theory (disable bvecp-1-rewrite))


;;;**********************************************************************
;;;                             BITN
;;;**********************************************************************

(defun bitn (x n)
  (if (logbitp n x) 1 0))

(in-theory (disable bitn))

(defthm bitn-rewrite
    (implies (and (natp x)
		  (natp k))
	     (equal (bitn x k)
		    (rem (fl (/ x (expt 2 k)))
			 2))))

(in-theory (disable bitn-rewrite))

(defthm natp-bitn
    (natp (bitn x n))
    :rule-classes (:type-prescription :rewrite))

(defthm bvecp-bitn
    (bvecp (bitn x n) 1))

(defthm bitn-bvecp-1
    (implies (bvecp x 1)
	     (equal (bitn x 0) x)))

(defthm bitn-bitn-0
    (equal (bitn (bitn x n) 0)
	   (bitn x n)))

(defthm bitn-0-0
  (equal (bitn 0 k) 0))

(defthm bitn-0-1
    (or (= (bitn x n) 0) (= (bitn x n) 1))
  :rule-classes ())

(defthm bitn-rec-0
    (implies (natp x)
	     (equal (bitn x 0)
		    (rem x 2))))

(in-theory (disable bitn-rec-0))

(defthm bitn-rec-pos-def
    (implies (and (natp x)
		  (natp k)
		  (> k 0))
	     (equal (bitn x k)
		    (bitn (fl (/ x 2)) (1- k))))
  :rule-classes ((:definition :controller-alist ((bitn t t)))))

(in-theory (disable bitn-rec-pos-def))

(defthm bitn-rem-bitn
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (> n k))
	     (equal (bitn (rem x (expt 2 n)) k)
		    (bitn x k))))

(in-theory (disable bitn-rem-bitn))

(defthm bitn-bvecp
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (bitn x n) 0)))

(defthm bitn-bvecp-0
    (implies (and (bvecp x n)
		  (natp m)
		  (natp n))
	     (equal (bitn x (+ m n)) 0)))

(in-theory (disable bitn-bvecp-0))

(defthm bitn-force-1
    (implies (and (natp n)
		  (bvecp x (1+ n))
		  (<= (expt 2 n) x))
	     (equal (bitn x n) 1)))

(in-theory (disable bitn-force-1))

(defthm bitn-force-2
    (implies (and (bvecp x n)
		  (natp n)
		  (natp k)
		  (< k n)
		  (<= (- (expt 2 n) (expt 2 k)) x))
	     (equal (bitn x k) 1)))

(in-theory (disable bitn-force-2))

(defthm bitn-expt
    (implies (natp n)
	     (equal (bitn (expt 2 n) n) 1)))

(in-theory (disable bitn-expt))

(defthm bit+expt
    (implies (and (natp x)
		  (natp n))
	     (not (equal (bitn (+ x (expt 2 n)) n)
			 (bitn x n))))
  :rule-classes ())

(defthm bit+expt-2
    (implies (and (natp x)
		  (natp n)
		  (natp m)
		  (> m n))
	     (equal (bitn (+ x (expt 2 m)) n)
		    (bitn x n))))

(in-theory (disable bit+expt-2))

(defthm bitn+mult
    (implies (and (natp x)
		  (natp k)
		  (natp n)
		  (natp m)
		  (> m n))
	     (equal (bitn (+ x (* k (expt 2 m))) n)
		    (bitn x n))))

(in-theory (disable bitn+mult))

(defthm rem+bitn
    (implies (and (natp a)
		  (natp n))
	     (= (rem a (expt 2 (1+ n)))
		(+ (* (bitn a n) (expt 2 n))
		   (rem a (expt 2 n)))))
  :rule-classes ())

(defthm rem-bitn-0
    (implies (and (natp a)
		  (natp n))
	     (iff (= (rem a (expt 2 (1+ n))) 0)
		  (and (= (rem a (expt 2 n)) 0)
		       (= (bitn a n) 0))))
  :rule-classes ())

(defthm bitn-shift
    (implies (and (natp x)
		  (natp n)
		  (natp k))
	     (= (bitn (* x (expt 2 k)) (+ n k))
		(bitn x n)))
  :rule-classes ())

(defthm bitn-shift-gen
    (implies (and (natp x)
		  (natp (* x (expt 2 k)))
		  (natp n)
		  (natp (+ n k))
		  (integerp k))
	     (= (bitn (* x (expt 2 k)) (+ n k))
		(bitn x n)))
  :rule-classes ())

(defthm bitn-shift-2
    (implies (and (natp x)
		  (natp k)
		  (natp r))
	     (equal (bitn (fl (/ x (expt 2 r))) k)
		    (bitn x (+ k r)))))

(in-theory (disable bitn-shift-2))

(defthm bitn-shift-3
    (implies (and (natp n)
		  (natp m)
		  (natp k)
		  (bvecp x m)
		  (<= m n))
	     (equal (bitn (+ x (* k (expt 2 m))) n)
		    (bitn k (- n m)))))

(in-theory (disable bitn-shift-3))


;;;**********************************************************************
;;;                         BITS
;;;**********************************************************************

(defun bits (x i j)
  (fl (/ (rem x (expt 2 (1+ i))) (expt 2 j))))

(in-theory (disable bits))

(defthm natp-bits
    (implies (and (natp x)
		  (natp i)
		  (natp j))
	     (natp (bits x i j)))
  :rule-classes (:type-prescription :rewrite))

(defthm bvecp-bits
  (implies (and ;(natp x)
            (rationalp x) ;new
            (>= x 0)      ;new
            (natp i)
            (natp j)
            (= n (- (1+ i) j)))
           (bvecp (bits x i j) n)))

(defthm bits-rem
    (implies (and (integerp x)
		  (natp n))
	     (equal (bits x n 0)
		    (rem x (expt 2 (1+ n))))))

(in-theory (disable bits-rem))

(defthm rem-bits-equal
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (= (rem x (expt 2 (1+ i))) (rem y (expt 2 (1+ i)))))
	     (= (bits x i j) (bits y i j)))
  :rule-classes ())

(defthm bits-0-0
    (implies (and (integerp i)
		  (integerp j)
		  (>= i 0))
	     (equal (bits 0 i j) 0)))

(defthm bits-
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (< i j))
	     (equal (bits x i j) 0)))

(defthm bits-n-n-rewrite
    (implies (and (natp x)
		  (natp n))
	     (equal (bits x n n)
		    (bitn x n))))

(defthm bits-tail
    (implies (and (natp n)
		  (bvecp x (1+ n)))
	     (equal (bits x n 0)
		    x)))

(defthm bvecp-bits-0
    (implies (and (natp i)
		  (natp j)
		  (>= i j)
		  (bvecp x j))
	     (equal (bits x i j) 0)))

(in-theory (disable bvecp-bits-0))

(defun sumbits (x n)
  (if (zp n)
      0
    (+ (* (expt 2 (1- n)) (bitn x (1- n)))
       (sumbits x (1- n)))))

(defthm sumbits-bits
    (implies (and (natp x)
		  (natp n)
		  (> n 0))
	     (equal (sumbits x n)
		    (bits x (1- n) 0))))

(in-theory (disable sumbits-bits))

(defthm sumbits-thm
    (implies (and (bvecp x n)
		  (natp n)
		  (> n 0))
	     (equal (sumbits x n)
		    x)))

(in-theory (disable sumbits-thm))

(defthm bits-shift-1
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp k))
	     (equal (bits (fl (/ x (expt 2 k)))
			  i
			  j)
		    (bits x (+ i k) (+ j k)))))

(in-theory (disable bits-shift-1))

(defthm bits-shift-2
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  (>= i (+ j k)))
	     (equal (bitn (bits x i j) k)
		    (bitn x (+ j k)))))

(in-theory (disable bits-shift-2))

(defthm bits-shift-3
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (>= i (+ j k)))
	     (equal (bits (bits x i j) k l)
		    (bits x (+ k j) (+ l j)))))

(in-theory (disable bits-shift-3))

(defthm bits-shift-4
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (>= i (+ j l))
		  (< i (+ j k)))
	     (equal (bits (bits x i j) k l)
		    (bits x i (+ l j)))))

(in-theory (disable bits-shift-4))

(defthm bits-shift-5
    (implies (and (natp x)
		  (natp k)
		  (natp i))
	     (equal (* (expt 2 k) (bits x i 0))
		    (bits (* (expt 2 k) x) (+ i k) 0)))
  :rule-classes ())

(defthm bits-0-rem-0
    (implies (and (natp x)
		  (natp m)
		  (natp n))
	     (iff (= (rem x (expt 2 (+ m n 1))) 0)
		  (and (= (bits x (+ m n) n) 0)
		       (= (rem x (expt 2 n)) 0))))
  :rule-classes ())

(defthm bits-0-bitn-0
    (implies (and (natp x)
		  (natp n)
		  (not (= n 0)))
	     (iff (= (bits x n 0) 0)
		  (and (= (bitn x n) 0)
		       (= (bits x (1- n) 0) 0))))
  :rule-classes ())

(defthm bits-plus-bits
    (implies (and (natp x)
		  (natp r)
		  (natp n)
		  (natp m)
		  (> n r)
		  (> m n))
	     (= (bits x (1- m) r)
		(+ (bits x (1- n) r)
		   (* (expt 2 (- n r)) (bits x (1- m) n)))))
  :rule-classes ())

(defthm bits-plus-bitn
    (implies (and (natp x)
		  (natp m)
		  (natp n)
		  (> n m))
	     (= (bits x n m)
		(+ (* (bitn x n) (expt 2 (- n m)))
		   (bits x (1- n) m))))
  :rule-classes ())

(defthm bitn-plus-bits
    (implies (and (natp x)
		  (natp n)
		  (natp m)
		  (> n m))
	     (= (bits x n m)
		(+ (bitn x m)
		   (* 2 (bits x n (1+ m))))))
  :rule-classes ())

(defthm bits-plus-mult
    (implies (and (natp m)
		  (natp n)
		  (>= n m)
		  (natp k)
		  (<= k m)
		  (natp y)
		  (bvecp x k))
	     (= (bits (+ x (* y (expt 2 k))) n m)
		(bits y (- n k) (- m k))))
  :rule-classes ())

(defthm bits-plus-mult-2
    (implies (and (natp m)
		  (natp n)
		  (>= n m)
		  (natp k)
		  (> k n)
		  (natp y)
		  (natp x))
	     (= (bits (+ x (* y (expt 2 k))) n m)
		(bits x n m)))
  :rule-classes ())

(defthm bits-sum-0
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j)
		  (> j 0)
		  (= (bits (+ x y) (1- j) 0) 0))
	     (equal (bits (+ x y) i j)
		    (bits (+ (bits x i j) (bits y i j) (logior (bitn x (1- j)) (bitn y (1- j))))
			  (- i j) 0)))
  :rule-classes ())

(defthm bits-sum
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j)
		  (> j 0))
	     (equal (bits (+ x y) i j)
		    (bits (+ (bits x i j)
			     (bits y i j)
			     (bitn (+ (bits x (1- j) 0)
				      (bits y (1- j) 0))
				   j))
			  (- i j) 0)))
  :rule-classes ())


;;;**********************************************************************
;;;                          CAT
;;;**********************************************************************

(defun cat (x y n)
  (+ (* (expt 2 n) x) y))

(in-theory (disable cat))

(defthm cat-nat
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (natp (cat x y n)))
  :rule-classes (:type-prescription :rewrite))

(defthm bvecp-cat
    (implies (and (natp n)
		  (natp p)
		  (>= p n)
		  (bvecp x (- p n))
		  (bvecp y n))
	     (bvecp (cat x y n) p)))

(defthm cat-0-rewrite
    (implies (acl2-numberp x)		  
	     (equal (cat 0 x n) x)))

(defthm bitn-cat-1
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp n)
		  (>= n (1+ i)))
	     (equal (bitn (cat x y n) i)
		    (bitn y i))))

(defthm bitn-cat-2
    (implies (and (natp n)
		  (natp i)
		  (>= i n)
		  (natp x)
		  (bvecp y n))
	     (equal (bitn (cat x y n) i)
		    (bitn x (- i n)))))

(defthm bits-cat-1
    (implies (and (natp x)
		  (natp y)
		  (natp j)
		  (natp i)
		  (natp n)
		  (>= n (1+ i)))
	     (equal (bits (cat x y n) i j)
		    (bits y i j))))

(defthm bits-cat-2
    (implies (and (natp n)
		  (natp j)
		  (>= j n)
		  (natp i)
		  (>= i j)
		  (natp x)
		  (bvecp y n))
	     (equal (bits (cat x y n) i j)
		    (bits x (- i n) (- j n)))))

(defthm bits-cat-3
    (implies (and (natp n)
		  (natp j)
		  (natp i)
		  (>= i n)
		  (> n j)
		  (natp x)
		  (bvecp y n))
	     (equal (bits (cat x y n) i j)
		    (cat (bits x (- i n) 0)
			 (bits y (1- n) j)
			 (- n j)))))

(defthm cat-assoc
    (implies (and ;(natp x)
		  ;(natp y)
		  ;(natp z)
		  (integerp m) ; (natp m)
                  (integerp n) ; (natp n)
)
	     (equal (cat (cat x y m) z n)
		    (cat x (cat y z n) (+ m n)))))


(in-theory (disable cat-assoc))



(defthm cat-bits-bits
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (>= i j)
		  (= j (1+ k))
		  (>= k l)
		  (= n (1+ (- k l))))
	     (equal (cat (bits x i j) (bits x k l) n)
		    (bits x i l))))

(defthm cat-bitn-bits
    (implies (and (natp x)
		  (natp j)
		  (natp k)
		  (natp l)
		  (= j (1+ k))
		  (>= k l)
		  (= n (1+ (- k l))))
	     (equal (cat (bitn x j) (bits x k l) n)
		    (bits x j l))))

(defthm cat-bits-bitn
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp k)
		  (>= i j)
		  (= j (1+ k)))
	     (equal (cat (bits x i j) (bitn x k) 1)
		    (bits x i k))))

(defun mulcat (l n x)
  (if (and (integerp n) (> n 0))
      (cat (mulcat l (1- n) x)
	   x
	   l)
    0))

(defthm bvecp-mulcat
    (implies (and (natp n)
		  (natp l)
		  (bvecp x l)
		  (= p (* l n)))
	     (bvecp (mulcat l n x) p)))

(defthm mulcat-1
    (implies (and (natp x)
		  (natp l))
	     (equal (mulcat l 1 x) x)))

(defthm mulcat-0
    (implies (and (natp l) (natp n))
	     (equal (mulcat l n 0) 0)))

(defthm mulcat-n-1
    (implies (and (integerp n) (> n 0))
	     (equal (mulcat 1 n 1)
		    (1- (expt 2 n)))))


;;;**********************************************************************
;;;                       SETBITS
;;;**********************************************************************

(defun setbits (x i j y)
  (cat (cat (ash x (- (1+ i)))
	    y
	    (1+ (- i j)))
       (bits x (1- j) 0)
       j))

(in-theory (disable setbits))

(defmacro setbitn (x n y)
  `(setbits ,x ,n ,n ,y))

(defthm natp-setbits
    (implies (and (natp x)
		  (natp i)
		  (natp j)
		  (natp y))
	     (natp (setbits x i j y))))

(defthm bvecp-setbits
    (implies (and (bvecp x n)
		  (natp n)
		  (bvecp y (1+ (- i j)))
		  (natp i)
		  (natp j)
		  (< i n)
		  (<= j i))
	     (bvecp (setbits x i j y) n)))

(defthm setbits-rewrite-1
    (implies (and (bvecp x n)
		  (natp n)
		  (> n 0)
		  (natp i)
		  (natp j)
		  (<= j i)
		  (bvecp y (1+ (- i j))))
	     (equal (setbits x i j y)
		    (cat (cat (bits x (1- n) (1+ i))
			      y
			      (1+ (- i j)))
			 (bits x (1- j) 0)
			 j))))

(defthm bits-neg-1
    (implies (natp x)
	     (equal (bits x -1 0) 0)))

(defthm setbits-rewrite-2
    (implies (and (bvecp x (1+ i))
		  (natp i)
		  (natp j)
		  (<= j i)
		  (bvecp y (1+ (- i j))))
	     (equal (setbits x i j y)
		    (cat y
			 (bits x (1- j) 0)
			 j))))

(defthm bitn-setbits-1
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (<= j i)
		  (< k j))
	     (equal (bitn (setbits x i j y) k)
		    (bitn x k))))

(defthm bitn-setbits-2
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (<= k i)
		  (<= j k))
	     (equal (bitn (setbits x i j y) k)
		    (bitn y (- k j)))))

(defthm bitn-setbits-3
    (implies (and (natp x)
		  (bvecp y (1+ (- i j)))
		  (natp i)
		  (natp j)
		  (natp k)
		  (< i k)
		  (<= j i))
	     (equal (bitn (setbits x i j y) k)
		    (bitn x k))))

(defthm bits-setbits-1
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (<= j i)
		  (< k j)
		  (<= l k))
	     (equal (bits (setbits x i j y) k l)
		    (bits x k l))))

(defthm bits-setbits-2
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (<= k i)
		  (<= l k)
		  (<= j l))
	     (equal (bits (setbits x i j y) k l)
		    (bits y (- k j) (- l j)))))

(defthm bits-setbits-3
    (implies (and (natp x)
		  (bvecp y (1+ (- i j)))
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (<= l k)
		  (< i l)
		  (<= j i))
	     (equal (bits (setbits x i j y) k l)
		    (bits x k l))))

(defthm bits-setbits-4
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (<= k i)
		  (<= j k)
		  (< l j)
		  (bvecp y (1+ (- i j))))
	     (equal (bits (setbits x i j y) k l)
		    (cat (bits y (- k j) 0)
			 (bits x (1- j) l)
			 (- j l)))))

(defthm bits-setbits-5
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (< i k)
		  (<= l i)
		  (<= j l)
		  (bvecp y (1+ (- i j))))
	     (equal (bits (setbits x i j y) k l)
		    (cat (bits x k (1+ i))
			 (bits y (- i j) (- l j))
			 (1+ (- i l))))))

(defthm bits-setbits-6
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (natp k)
		  (natp l)
		  (< i k)
		  (<= j i)
		  (< l j)
		  (bvecp y (1+ (- i j))))
	     (equal (bits (setbits x i j y) k l)
		    (cat (cat (bits x k (1+ i))
			      y
			      (1+ (- i j)))
			 (bits x (1- j) l)
			 (- j l)))))

 
;;;**********************************************************************
;;;                       SHFT
;;;**********************************************************************

(defun shft (x s l)
  (rem (fl (* (expt 2 s) x)) (expt 2 l)))

(in-theory (disable shft))

(defthm natp-shft
    (implies (and (natp x)
		  (natp n)
		  (integerp s))
	     (natp (shft x s n)))
  :rule-classes (:type-prescription :rewrite))

(defthm bvecp-shft
    (implies (and (natp x)
		  (natp n)
		  (integerp s))
	     (bvecp (shft x s n) n)))

(in-theory (disable bvecp-shft))


;;;**********************************************************************
;;;                       COMP1
;;;**********************************************************************

(defun comp1 (x n)
  (1- (- (expt 2 n) x)))

(in-theory (disable comp1))

(defthm integerp-comp1-nat
    (implies (and (integerp x)
		  (natp n))
	     (integerp (comp1 x n)))
  :rule-classes (:type-prescription :rewrite))

(defthm natp-comp1
    (implies (and (natp n)
		  (bvecp x n))
	     (natp (comp1 x n)))
    :rule-classes (:type-prescription :rewrite))

(defthm bvecp-comp1
    (implies (and (natp n)
		  (bvecp x n))
	     (bvecp (comp1 x n) n)))

(defthm comp1-comp1-rewrite
    (implies (and (integerp n)
		  (integerp x))
	     (equal (comp1 (comp1 x n) n)
                    x)))

(defthm comp1-2+1
    (implies (and (natp x)
		  (natp n))
	     (equal (+ 1 (* 2 (comp1 x n)))
		    (comp1 (* 2 x) (1+ n)))))

(defthm comp1-fl-rewrite
    (implies (and (not (zp n))
		  (bvecp x n))
	     (equal (fl (* 1/2 (comp1 x n)))
		    (comp1 (fl (* 1/2 x)) (1- n)))))

(in-theory (disable comp1-fl-rewrite))

(defthm comp1-rem-2
    (implies (and (not (zp n))
		  (bvecp x n))
	     (equal (rem (comp1 x n) 2)
		    (comp1 (rem x 2) 1))))

(in-theory (disable comp1-rem-2))

(defthm rem-comp1-2
    (implies (and (natp n)
		  (not (= n 0))
		  (bvecp x n))
	     (not (= (rem (comp1 x n) 2)
		     (rem x 2))))
  :rule-classes ())

(defthm bitn-comp1-not-equal
    (implies (and (natp n)
		  (bvecp x n)
		  (natp k)
		  (< k n))
	     (not (= (bitn (comp1 x n) k)
		     (bitn x k))))
  :rule-classes ())

(defthm bits-comp1
    (implies (and (natp m) 
		  (natp i) 
		  (natp j)
		  (> m i)
		  (>= i j)
		  (bvecp x m))
	     (equal (bits (comp1 x m) i j)
		    (comp1 (bits x i j) (1+ (- i j))))))

(in-theory (disable bits-comp1))

(defthm rem-comp1-rewrite
    (implies (and (natp n)
		  (natp m)
		  (bvecp x m)
		  (not (= n 0))
		  (>= m n))
	     (equal (rem (comp1 x m) (expt 2 n))
		    (comp1 (rem x (expt 2 n)) n))))

(in-theory (disable rem-comp1-rewrite))

(defthm comp1-bitn
    (implies (and (natp m) 
		  (natp n) 
		  (> m n)
		  (bvecp x m))
	     (equal (bitn (comp1 x m) n)
		    (comp1 (bitn x n) 1))))

(in-theory (disable comp1-bitn))


;;;**********************************************************************
;;;                       LOGAND, LOGIOR, and LOGXOR
;;;**********************************************************************

(in-theory (disable logand logior logxor))

(defthm logand-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (logand x y)
		    (+ (* 2 (logand (fl (/ x 2)) (fl (/ y 2))))
		       (logand (rem x 2) (rem y 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logand t t)))))

(in-theory (disable logand-rewrite))

(defthm logior-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (logior i j)
		    (+ (* 2 (logior (fl (/ i 2)) (fl (/ j 2))))
		       (logior (rem i 2) (rem j 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logior t t)))))

(in-theory (disable logior-rewrite))

(defthm logxor-def-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (logxor x y)
		    (+ (* 2 (logxor (fl (/ x 2)) (fl (/ y 2))))
		       (logxor (rem x 2) (rem y 2)))))
  :rule-classes ((:definition :controller-alist ((binary-logxor t t)))))

(in-theory (disable logxor-def-rewrite))

(defthm natp-logand
    (implies (and (natp i)
		  (natp j))
	     (natp (logand i j)))
  :rule-classes (:type-prescription :rewrite))

(defthm bvecp-logand
    (implies (and (natp n)
		  (bvecp x n)
		  (natp y))
	     (bvecp (logand x y) n)))

(defthm natp-logior
    (implies (and (natp i)
		  (natp j))
	     (natp (logior i j)))
  :rule-classes (:type-prescription :rewrite))

(defthm bvecp-logior
    (implies (and (natp n)
		  (bvecp x n)
		  (bvecp y n))
	     (bvecp (logior x y) n)))

(defthm natp-logxor
    (implies (and (natp i)
		  (natp j))
	     (natp (logxor i j)))
  :rule-classes (:type-prescription :rewrite))

(defthm bvecp-logxor
    (implies (and (natp n)
		  (bvecp x n)
		  (bvecp y n))
	     (bvecp (logxor x y) n)))

(defun logop-2-induct (x y)
  (if (or (zp x) (zp y))
      ()
    (logop-2-induct (fl (/ x 2)) (fl (/ y 2)))))

(defun logop-2-n-induct (x y n)
  (if (zp n)
      (cons x y)
    (logop-2-n-induct (fl (/ x 2)) (fl (/ y 2)) (1- n))))

(defun logop-3-induct (x y z)
  (if (and (natp x) (natp y) (natp z))
      (if (and (zp x) (zp y) (zp z))
	  t
	(logop-3-induct (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2))))
    t))

(defthm logand-rem-2
    (implies (and (natp x)
		  (natp y))
	     (equal (rem (logand x y) 2)
		    (logand (rem x 2) (rem y 2)))))

(in-theory (disable logand-rem-2))

(defthm logand-fl-2-rewrite
    (implies (and (natp x)
		  (natp y))
	     (equal (fl (* 1/2 (logand x y)))
		    (logand (fl (* 1/2 x)) (fl (* 1/2 y))))))

(in-theory (disable logand-fl-2-rewrite))

(defthm logior-rem-2
    (implies (and (natp i)
		  (natp j))
	     (equal (rem (logior i j) 2)
		    (logior (rem i 2) (rem j 2)))))

(in-theory (disable logior-rem-2))

(defthm logior-fl-2-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (fl (* 1/2 (logior i j)))
		    (logior (fl (* 1/2 i)) (fl (* 1/2 j))))))

(in-theory (disable logior-fl-2-rewrite))

(defthm logxor-rem-2
    (implies (and (natp i)
		  (natp j))
	     (equal (rem (logxor i j) 2)
		    (logxor (rem i 2) (rem j 2)))))

(in-theory (disable logxor-rem-2))

(defthm logxor-fl-2-rewrite
    (implies (and (natp i)
		  (natp j))
	     (equal (fl (* 1/2 (logxor i j)))
		    (logxor (fl (* 1/2 i)) (fl (* 1/2 j))))))

(in-theory (disable logxor-fl-2-rewrite))

(defthm logand-x-0
    (equal (logand x 0) 0))

(defthm logand-0-y
    (equal (logand 0 y) 0))

(defthm logior-x-0
    (implies (natp x)
	     (equal (logior x 0) x)))

(defthm logior-0-y
    (implies (natp y)
	     (equal (logior 0 y) y)))

(defthm logxor-x-0
    (implies (integerp x)
	     (equal (logxor x 0) x)))

(defthm logxor-0-y
    (implies (integerp y)
	     (equal (logxor 0 y) y)))

(defthm logand-self
    (implies (natp x)
	     (equal (logand x x) x)))

(defthm logior-self
    (implies (natp x)
	     (equal (logior x x) x)))

(defthm logxor-self
    (implies (natp x)
	     (equal (logxor x x) 0)))

(defthm logior-not-0
    (implies (and (natp x)
		  (natp y)
		  (= (logior x y) 0))
	     (and (= x 0) (= y 0)))
  :rule-classes ())

(defthm logand-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logand x (1- (expt 2 n)))
		    x)))

(in-theory (disable logand-ones))

(defthm logand-x-1
    (implies (bvecp x 1)
	     (equal (logand x 1) x)))

(defthm logand-1-x
    (implies (bvecp x 1)
	     (equal (logand 1 x) x)))

(defthm logior-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logior x (1- (expt 2 n)))
		    (1- (expt 2 n))))
  :rule-classes ())

(defthm logior-x-1
    (implies (bvecp x 1)
	     (equal (logior x 1) 1)))

(defthm logior-1-x
    (implies (bvecp x 1)
	     (equal (logior 1 x) 1)))

(defthm logxor-ones
    (implies (and (natp n)
		  (bvecp x n))
	     (equal (logxor x (1- (expt 2 n)))
		    (comp1 x n)))
  :rule-classes ())

(defthm logand-commutative
    (implies (and (integerp x)
		  (integerp y))
	     (equal (logand x y) (logand y x))))

(defthm logior-commutative
    (implies (and (integerp x)
		  (integerp y))
	     (equal (logior x y) (logior y x))))

(defthm logxor-commutative
    (implies (and (integerp x)
		  (integerp y))
	     (equal (logxor x y) (logxor y x))))

(defthm logand-associative
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logand (logand x y) z)
		    (logand x (logand y z)))))

(in-theory (disable logand-associative))

(defthm logior-associative
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logior (logior x y) z)
		    (logior x (logior y z)))))

(in-theory (disable logior-associative))

(defthm logxor-associative
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logxor (logxor x y) z)
		    (logxor x (logxor y z)))))

(in-theory (disable logxor-associative))

(defthm comp1-logxor
    (implies (and (natp n)
		  (bvecp x n)
		  (bvecp y n))
	     (equal (comp1 (logxor x y) n)
		    (logxor (comp1 x n) y))))

(in-theory (disable comp1-logxor))

(defthm logior-logand
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logior x (logand y z))
		    (logand (logior x y) (logior x z)))))

(in-theory (disable logior-logand))

(defthm logand-logior
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logand x (logior y z))
		    (logior (logand x y) (logand x z)))))

(in-theory (disable logand-logior))

(defthm logior-logand-2
    (implies (and (natp x)
		  (natp y)
		  (natp z))
	     (equal (logand  (logior y z) x)
		    (logior (logand y x) (logand z x)))))

(in-theory (disable logior-logand-2))

(defthm log3
    (implies (and (natp x) (natp y) (natp z))
	     (equal (logior (logand x y) (logior (logand x z) (logand y z)))
		    (logior (logand x y) (logand (logxor x y) z))))
  :rule-classes ())

(defthm logxor-rewrite-2
    (implies (and (bvecp x n)
		  (bvecp y n)
                  (natp n)
		  (not (= n 0)))
	     (equal (logxor x y)
		    (logior (logand x (comp1 y n))
			    (logand y (comp1 x n))))))

(in-theory (disable logxor-rewrite-2))

(defthm logior-expt
    (implies (and (natp n)
		  (natp x)
		  (bvecp y n))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x) y)))
  :rule-classes ())

(defthm logior-expt-2
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* (expt 2 n) (logior x y))))
  :rule-classes ())

(defthm rem-logior
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (rem (logior x y) (expt 2 n))
		    (logior (rem x (expt 2 n)) (rem y (expt 2 n))))))

(in-theory (disable rem-logior))

(defthm logand-bnd
    (implies (and (natp x)
		  (natp y))
	     (<= (logand x y) x))
  :rule-classes :linear)

(defthm logand-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (logand (* (expt 2 n) x) y)
		(* (expt 2 n) (logand x (fl (/ y (expt 2 n)))))))
  :rule-classes ())

(defthm rem-logand-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (= (rem (logand x y) (expt 2 n))
		(logand (rem x (expt 2 n)) y)))
  :rule-classes ())

(defthm rem-logand-rewrite
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (rem (logand x y) (expt 2 n))
		    (logand (rem x (expt 2 n)) (rem y (expt 2 n))))))

(in-theory (disable rem-logand-rewrite))

(defthm rem-logxor-rewrite
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (rem (logxor x y) (expt 2 n))
		    (logxor (rem x (expt 2 n))
			    (rem y (expt 2 n))))))

(in-theory (disable rem-logxor-rewrite))

(defthm logand-rem-expt
    (implies (and (natp x)
		  (natp y)
		  (natp n)
		  (< x (expt 2 n)))
	     (= (logand x y)
		(logand x (rem y (expt 2 n)))))
  :rule-classes ())

(defthm bitn-logand
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (bitn (logand x y) n)
		    (logand (bitn x n) (bitn y n)))))

(in-theory (disable bitn-logand))

(defthm bits-logand
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j))
	     (equal (bits (logand x y) i j)
		    (logand (bits x i j) (bits y i j)))))

(in-theory (disable bits-logand))

(defthm bitn-logior
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (bitn (logior x y) n)
		    (logior (bitn x n) (bitn y n)))))

(in-theory (disable bitn-logior))

(defthm bits-logior
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j))
	     (equal (bits (logior x y) i j)
		    (logior (bits x i j) (bits y i j)))))

(in-theory (disable bits-logior))

(defthm logxor-bitn
    (implies (and (natp x)
		  (natp y)
		  (natp n))
	     (equal (bitn (logxor x y) n)
		    (logxor (bitn x n) (bitn y n)))))

(in-theory (disable logxor-bitn))

(defthm bits-logxor-2
    (implies (and (natp x)
		  (natp y)
		  (natp i)
		  (natp j)
		  (>= i j))
	     (equal (bits (logxor x y) i j)
		    (logxor (bits x i j) (bits y i j)))))

(in-theory (disable bits-logxor-2))

(defthm bits-logxor-upper-slice
    (implies (and (equal n (+ 1 i))
                  (bvecp x n)
		  (bvecp y n)
		  (natp n)
		  (natp i)
		  (natp j)
		  (> n i)
		  (>= i j))
	     (equal (bits (logxor x y) i j)
		    (logxor (bits x i j) (bits y i j)))))

(defthm logand-expt-2
    (implies (and (natp x)
		  (natp k))
	     (= (logand x (expt 2 k))
		(* (expt 2 k) (bitn x k))))
  :rule-classes ())

(defthm logior-expt-3
    (implies (and (natp x)
		  (natp k))
	     (= (logior x (expt 2 k))
		(+ x
		   (* (expt 2 k) 
		      (- 1 (bitn x k))))))
  :rule-classes ())

(defthm logand-expt-3
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ())

(defthm logand-expt-4
    (implies (and (natp n)
		  (natp k)
		  (natp l)
		  (< l k)
		  (<= k n))
	     (= (logand (- (1- (expt 2 n)) (expt 2 l)) (- (expt 2 n) (expt 2 k)))
		(- (expt 2 n) (expt 2 k))))
  :rule-classes ())

(defthm bitn-logxor-0
    (implies (and (natp a)
		  (natp b))
	     (= (bitn (+ a b) 0)
		(bitn (logxor a b) 0)))
  :rule-classes ())


;like bits-tail
;for for-loop defuns in model.lisp
(defthm bits-reduce
  (implies (and (integerp x)
                (<= 0 x)
                (integerp j)
                (>= j 0)
                (<= x (expt 2 j)))
           (equal (bits x j 0) x))
  :hints (("Goal" :in-theory (enable natp)
           :use (:instance rem<n (m x) (n (* 2 (EXPT 2 J)))))))


