;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;David M. Russinoff
;;;Advanced Micro Devices, Inc.
;;;February, 1998
;;;***************************************************************
(in-package "ACL2")

;(include-book "floor")
(include-book "bits")

(in-theory (disable logand))

(local (include-book "comp1"))

(local (in-theory (enable logand logior logxor floor bits))) ;yuck

(defun COMP1 (x n)
  (if (natp n)
      (+ -1 (expt 2 n) (- (bits x (+ -1 n) 0)))
    0))

(in-theory (disable comp1))

 ;note that this isn't a rewrite rule b/c we believe it will never need to be
(defthm comp1-nonnegative-integer
  (and (integerp (comp1 x n))
       (<= 0 (comp1 x n)))
  :rule-classes ((:type-prescription :typed-term (comp1 x n))))

;comp1-nonnegative-integer is strictly better, and we don't need both
(in-theory (disable (:type-prescription comp1))) 

(defthm comp1-natp
  (natp (comp1 x n)))

(encapsulate
 ()
 (local (include-book "logand"))

;proved in logand.lisp
 (defthm logand-commutative
   (equal (logand j i)
          (logand i j)))

;proved in logand.lisp
 (defthm logand-commutativity-2
   (equal (logand j i k)
          (logand i j k)))

;proved in logand.lisp
 (defthm logand-combine-constants
   (implies (syntaxp (and (quotep i)
                          (quotep j)))
            (equal (logand i j k)
                   (logand (logand i j) k))))

;proved in logand.lisp
 (defthm logand-associative
   (equal (logand (logand i j) k)
          (logand i (logand j k))))

 (defthm LOGAND-self
   (implies (case-split (integerp i))
            (equal (logand i i) i))
   :hints (("Goal" :in-theory (enable logand))))

 )






;what is this?
(defun DEC1 (x n)
  (mod (1- (+ (expt 2 n) x)) (expt 2 n)))

;move to logand.lisp
(defthm LOGAND-0
  (equal (logand 0 y) 0)
  :hints (("Goal" :in-theory (enable logand)))
  )

(defthm LOGAND-DEF
  (implies (and (integerp x) (>= x 0)
                (integerp y) (>= y 0))
           (= (logand x y)
              (+ (* 2 (logand (fl (/ x 2)) (fl (/ y 2))))
                 (logand (mod x 2) (mod y 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable fl)
                              '(logand))

           :use ((:instance mod012)
                 (:instance mod012 (x y))
                 (:instance binary-logand (i x) (j y))
                 (:instance floor-fl (m x) (n 2))
                 (:instance floor-fl (m y) (n 2))
                 (:instance bitn-def-4)
                 (:instance bitn-def-4 (x y))))))

(defthm LOGAND-NAT
  (implies (and (integerp i) (>= i 0)
                (integerp j) (>= j 0))
           (and (integerp (logand i j))
                (>= (logand i j) 0)))
  :hints (("Goal" :in-theory (enable logand)))
  :rule-classes ())

(defthm LOGAND-MOD
  (implies (and (integerp x) (>= x 0)
                (integerp y) (>= y 0))
           (= (mod (logand x y) 2)
              (logand (mod x 2) (mod y 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand fl)
           :use ((:instance logand-def)
                 (:instance mod012)
                 (:instance mod012 (x y))
                 (:instance mod+-thm 
                            (m (logand (mod x 2) (mod y 2)))
                            (n 2)
                            (a (logand (fl (/ x 2)) (fl (/ y 2)))))
                 (:instance logand-nat (i (fl (/ x 2))) (j (fl (/ y 2))))
                 (:instance mod< (m (logand (mod x 2) (mod y 2))) (n 2))))))

(defthm LOGAND-FL
  (implies (and (integerp x) (>= x 0)
                (integerp y) (>= y 0))
           (= (fl (/ (logand x y) 2))
              (logand (fl (/ x 2)) (fl (/ y 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand fl)
           :use ((:instance logand-def)
                 (:instance logand-mod)
                 (:instance mod-fl (m (logand x y)) (n 2))
                 (:instance logand-nat (i x) (j y))))))

(local
 (defthm logand-1
   (implies (integerp x)
            (equal (logand -1 x) x))
   :hints (("Goal" :in-theory (enable logand))
           )))

(local
 (defthm logand-1-2
   (implies (integerp x)
            (equal (logand x -1) x))))

(local
 (defthm den-int
   (implies (integerp x)
            (equal (denominator x) 1))
   :hints (("Goal" :in-theory (disable rational-implies2)
            :use ((:instance lowest-terms
                             (n (denominator x))
                             (r x)
                             (q 1))
                  (:instance rational-implies1)
                  (:instance rational-implies2))))))

(local
 (defthm num-int
   (implies (integerp x)
            (equal (numerator x) x))
   :hints (("Goal" :in-theory (disable rational-implies2)
            :use ((:instance den-int)
                  (:instance rational-implies1)
                  (:instance rational-implies2))))))

(local
 (defthm floor*2
   (implies (integerp x)
            (equal (floor (* 2 x) 2) x))))

(local
 (defthm den-2x+1/2-1
   (implies (integerp x)
            (equal (denominator (/ (1+ (* 2 x)) 2))
                   (* 2 (- (numerator (/ (1+ (* 2 x)) 2))
                           (* x (denominator (/ (1+ (* 2 x)) 2)))))))
   :rule-classes ()
   :hints (("Goal" :in-theory (disable rational-implies2)
            :use ((:instance rational-implies1 (x (/ (1+ (* 2 x)) 2)))
                  (:instance rational-implies2 (x (/ (1+ (* 2 x)) 2))))))))

(local
 (defthm den-2x+1/2-2
    (implies (integerp x)
	     (equal (* 2 (numerator (/ (1+ (* 2 x)) 2)))
		    (* (denominator (/ (1+ (* 2 x)) 2))
		       (1+ (* 2 x)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable rational-implies2)
		  :use ((:instance rational-implies1 (x (/ (1+ (* 2 x)) 2)))
			(:instance rational-implies2 (x (/ (1+ (* 2 x)) 2)))
			(:instance den-2x+1/2-1))))))

(local
 (defthm worst-hack-yet
    (implies (and (rationalp n)
		  (rationalp d)
		  (rationalp s)
		  (rationalp q)
		  (= (* 2 n) (* d s))
		  (= d (* 2 q)))
	     (= n (* s q)))
  :rule-classes ()))

(local
 (defthm den-2x+1/2-3
    (implies (integerp x)
	     (equal (numerator (/ (1+ (* 2 x)) 2))
		    (* (1+ (* 2 x))
		       (- (numerator (/ (1+ (* 2 x)) 2))
			  (* x (denominator (/ (1+ (* 2 x)) 2)))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable rational-implies2)
		  :use ((:instance den-2x+1/2-1)
			(:instance den-2x+1/2-2)
			(:instance worst-hack-yet
				   (n (numerator (/ (1+ (* 2 x)) 2)))
				   (d (denominator (/ (1+ (* 2 x)) 2)))
				   (s (1+ (* 2 x)))
				   (q (- (numerator (/ (1+ (* 2 x)) 2))
					 (* x (denominator (/ (1+ (* 2 x)) 2)))))))))))

(local
 (defthm den-2x+1/2-4
    (implies (integerp x)
	     (equal (- (numerator (/ (1+ (* 2 x)) 2))
		       (* x (denominator (/ (1+ (* 2 x)) 2))))
		    1))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable rational-implies2)
		  :use ((:instance lowest-terms
				   (n (- (numerator (/ (1+ (* 2 x)) 2))
					 (* x (denominator (/ (1+ (* 2 x)) 2)))))
				   (r (1+ (* 2 x)))
				   (q 2)
				   (x (/ (1+ (* 2 x)) 2)))
			(:instance den-2x+1/2-1)
			(:instance den-2x+1/2-3)
			(:instance rational-implies1 (x (/ (1+ (* 2 x)) 2)))
			(:instance rational-implies2 (x (/ (1+ (* 2 x)) 2))))))))

(local
 (defthm den-2x+1/2
    (implies (integerp x)
	     (equal (denominator (+ 1/2 x)) 2))
  :hints (("Goal" :use ((:instance den-2x+1/2-1)
			(:instance den-2x+1/2-4))))))

(local
 (defthm num-2x+1/2
    (implies (integerp x)
	     (equal (numerator (+ 1/2 x)) (1+ (* 2 x))))
  :hints (("Goal" :in-theory (disable rational-implies2)
		  :use ((:instance rational-implies2 (x (/ (1+ (* 2 x)) 2)))
			(:instance den-2x+1/2))))))
;moved induct-nat to basic


(local
 (defthm floor*2+1-1
    (implies (and (integerp x) (>= x 0))
	     (equal (nonnegative-integer-quotient (1+ (* 2 x)) 2)
		    x))
  :rule-classes ()
  :hints (("Goal" :induct (induct-nat x)))))

(local
 (defthm floor*2+1-2
    (implies (and (integerp x) (> x 0))
	     (equal (nonnegative-integer-quotient (1- (* 2 x)) 2)
		    (1- x)))
  :rule-classes ()
  :hints (("Goal" :induct (induct-nat x)))))

(local
 (defthm floor*2+1
    (implies (integerp x)
	     (equal (floor (1+ (* 2 x)) 2) x))
  :rule-classes ()
  :hints (("Goal" :use ((:instance floor*2+1-2)
			(:instance floor*2+1-2 (x (- x))))))))


(defthm floor-logand
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logand i j) 2)
		(logand (floor i 2) (floor j 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand floor evenp floor*2)
		  :use ((:instance binary-logand)
			(:instance floor*2 (x (LOGAND (FLOOR I 2) (FLOOR J 2))))
			(:instance floor*2+1 (x (LOGAND (FLOOR I 2) (FLOOR J 2))))))))

;moved x-or-x/2 stuff to basic


(local
 (defthm floor-lognot-1
    (implies (integerp n)
	     (= (floor (lognot (* 2 n)) 2)
		(lognot (floor (* 2 n) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable floor)
		  :use ((:instance floor*2+1 (x (lognot n)))
			(:instance floor*2 (x n)))))))

(local
 (defthm floor-lognot-2
    (implies (integerp n)
	     (= (floor (lognot (1- (* 2 n))) 2)
		(lognot (floor (1- (* 2 n)) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable floor)
		  :use ((:instance floor*2+1 (x (1- n)))
			(:instance floor*2 (x (- n))))))))


(defthm floor-lognot
    (implies (integerp i)
	     (= (floor (lognot i) 2)
		(lognot (floor i 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable floor lognot)
		  :use ((:instance floor-lognot-1 (n (/ i 2)))
			(:instance floor-lognot-2 (n (/ (1+ i) 2)))
			(:instance x-or-x/2 (x i))))))


(defthm floor-logior
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logior i j) 2)
		(logior (floor i 2) (floor j 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand floor lognot)
		  :use ((:instance floor-lognot)
			(:instance floor-lognot (i j))
			(:instance floor-logand (i (lognot i)) (j (lognot j)))
			(:instance floor-lognot (i (logand (lognot i) (lognot j))))))))

(local
 (defthm fl-logior-1
    (implies (integerp i)
	     (iff (>= i 0) (< (lognot i) 0)))
  :rule-classes ()))

(local
 (defthm fl-logior-2
    (implies (and (integerp i) (< i 0)
		  (integerp j) (< j 0))
	     (and (integerp (logand i j))
		  (< (logand i j) 0)))
  :hints (("Goal" :in-theory (enable logand)))
  :rule-classes ()))

(defthm LOGIOR-NAT
  (implies (and (integerp i) (>= i 0)
                (integerp j) (>= j 0))
           (and (integerp (logior i j))
                (>= (logior i j) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable lognot logand)
           :use ((:instance fl-logior-1)
                 (:instance fl-logior-1 (i j))
                 (:instance fl-logior-1 (i (logand (lognot i) (lognot j))))
                 (:instance fl-logior-2 (i (lognot i)) (j (lognot j)))))))

(defthm LOGIOR-FL
  (implies (and (integerp i) (>= i 0)
                (integerp j) (>= j 0))
           (= (fl (/ (logior i j) 2))
              (logior (fl (/ i 2)) (fl (/ j 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior floor)
           :use ((:instance floor-logior)
                 (:instance floor-fl (m (logior i j)) (n 2))
                 (:instance floor-fl (m i) (n 2))
                 (:instance floor-fl (m j) (n 2))
                 (:instance logior-nat)))))

(defthm MOD-2*I
  (implies (integerp i)
           (equal (mod (* 2 i) 2) 0))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod (x (* 2 i)) (y 2))
			(:instance mod012 (x (* 2 i)))))))

(local
 (defthm mod-logior-2
   (implies (integerp i)
            (equal (mod (1+ (* 2 i)) 2) (- 1 (* 2 (- (floor (1+ (* 2 i)) 2) i)))))
   :rule-classes ()
   :hints (("Goal" :use ((:instance mod (x (1+ (* 2 i))) (y 2)))))))

(defthm MOD-2*I+1
  (implies (integerp i)
           (not (equal (mod (1+ (* 2 i)) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bit+-4 (x (- (floor (1+ (* 2 i)) 2) i)))
			(:instance mod-logior-2)))))

(local
 (defthm mod-logior-4
    (implies (integerp n)
	     (not (= (mod (lognot (* 2 n)) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-2*i+1 (i (1- (- n)))))))))

(local
 (defthm mod-logior-5
    (implies (integerp n)
	     (= (mod (lognot (1+ (* 2 n))) 2) 0))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-2*i (i (1- (- n)))))))))

(local
 (defthm mod-logior-6
    (implies (integerp n) (integerp (1- n)))
  :rule-classes ()))

(local
 (defthm mod-logior-7
    (implies (rationalp x)
	     (= (1- (+ 1/2 x))
		(+ -1/2 x)))
  :rule-classes ()))

(local
 (defthm mod-logior-8
    (implies (and (rationalp x)
		  (integerp (+ 1/2 x)))
	     (integerp (+ -1/2 x)))
  :hints (("Goal" :use ((:instance mod-logior-6 (n (+ 1/2 x)))
			(:instance mod-logior-7))))))


(defthm mod-logior-9
    (implies (integerp i)
	     (iff (= (mod (lognot i) 2) 0)
		  (not (= (mod i 2) 0))))
  :hints (("Goal" :in-theory (disable lognot)
		  :use ((:instance x-or-x/2 (x i))
			(:instance mod-2*i (i (/ i 2)))
			(:instance mod-logior-4 (n (/ i 2)))
			(:instance mod-2*i+1 (i (1- (/ (1+ i) 2))))
			(:instance mod-logior-5 (n (1- (/ (1+ i) 2))))))))

(local
 (defthm evenp-logand-1
    (implies (and (integerp i)
		  (integerp j)
		  (or (evenp i) (evenp j)))
	     (= (logand i j)
		(* 2 (logand (floor i 2) (floor j 2)))))
     :hints (("Goal" :in-theory (enable logand)))
     :rule-classes ()))

(local
 (defthm evenp-logand-2
    (implies (integerp x)
	     (iff (evenp x)
		  (= (mod x 2) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance x-or-x/2)
			(:instance mod-2*i (i (/ x 2)))
			(:instance mod-2*i+1 (i (1- (/ (1+ x) 2)))))))))

(local
 (defthm evenp-logand-3
    (implies (and (integerp i)
		  (integerp j)
		  (or (= (mod i 2) 0) (= (mod j 2) 0)))
	     (= (mod (logand i j) 2) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand floor evenp)
		  :use ((:instance evenp-logand-1)
			(:instance evenp-logand-2 (x i))
			(:instance evenp-logand-2 (x j))
			(:instance mod-2*i (i (logand (floor i 2) (floor j 2)))))))))

(local
 (defthm evenp-logand-4
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (evenp i) (evenp j)))
		  (or (= i -1) (= j -1)))
	     (or (= (logand i j) i)
		 (= (logand i j) j)))
  :rule-classes ()))

(local
 (defthm evenp-logand-5
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (= (mod i 2) 0) (= (mod j 2) 0)))
		  (or (= i -1) (= j -1)))
	     (not (= (mod (logand i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand evenp)
		  :use ((:instance evenp-logand-4)
			(:instance evenp-logand-2 (x i))
			(:instance evenp-logand-2 (x j)))))))

(local
 (defthm evenp-logand-6
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (evenp i) (evenp j)))
		  (not (or (= i -1) (= j -1))))
	     (= (logand i j)
		(1+ (* 2 (logand (floor i 2) (floor j 2))))))
     :hints (("Goal" :in-theory (enable logand)))
     :rule-classes ()))

(local
 (defthm evenp-logand-7
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (= (mod i 2) 0) (= (mod j 2) 0)))
		  (not (or (= i -1) (= j -1))))
	     (not (= (mod (logand i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand evenp floor)
		  :use ((:instance evenp-logand-6)
			(:instance evenp-logand-2 (x i))
			(:instance evenp-logand-2 (x j))
			(:instance mod-2*i+1 (i (logand (floor i 2) (floor j 2)))))))))

(local
 (defthm evenp-logand-8
    (implies (and (integerp i)
		  (integerp j)
		  (not (or (= (mod i 2) 0) (= (mod j 2) 0))))
	     (not (= (mod (logand i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand evenp floor)
		  :use ((:instance evenp-logand-5)
			(:instance evenp-logand-7))))))


(defthm evenp-logand
    (implies (and (integerp i)
		  (integerp j))
	     (iff (or (= (mod i 2) 0) (= (mod j 2) 0))
		  (= (mod (logand i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand evenp floor)
		  :use ((:instance evenp-logand-3)
			(:instance evenp-logand-8)))))


(defthm mod-logior-10
    (implies (and (integerp i)
		  (integerp j))
	     (iff (and (= (mod i 2) 0) (= (mod j 2) 0))
		  (= (mod (logior i j) 2) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand lognot)
		  :use ((:instance mod-logior-9 (i (logand (lognot i) (lognot j))))
			(:instance evenp-logand (i (lognot i)) (j (lognot j)))
			(:instance mod-logior-9)
			(:instance mod-logior-9 (i j))))))

(defthm LOGIOR-MOD
  (implies (and (integerp i) (>= i 0)
                (integerp j) (>= j 0))
           (= (mod (logior i j) 2)
              (logior (mod i 2) (mod j 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior)
           :use ((:instance mod-logior-10)
                 (:instance mod012 (x i))
                 (:instance mod012 (x j))
                 (:instance logior-nat)
                 (:instance mod012 (x (logior i j)))))))

(defthm LOGIOR-DEF
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (= (logior i j)
		(+ (* 2 (logior (fl (/ i 2)) (fl (/ j 2))))
		   (logior (mod i 2) (mod j 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior)
		  :use ((:instance logior-nat)
			(:instance mod-fl (m (logior i j)) (n 2))
			(:instance logior-mod)
			(:instance logior-fl)))))



;from logxor-def.lisp

(local
(defthm logorc1-mod-1
    (implies (and (integerp i) (integerp j))
	     (iff (= (mod (logorc1 i j) 2) 0)
		  (and (= (mod (lognot i) 2) 0)
		       (= (mod j 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior lognot)
		  :use ((:instance mod-logior-10 (i (lognot i))))))))

(local(defthm logorc1-mod
    (implies (and (integerp i) (>= i 0)
		  (integerp j))
	     (iff (= (mod (logorc1 i j) 2) 0)
		  (and (= (mod i 2) 1)
		       (= (mod j 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logior lognot)
		  :use ((:instance mod-logior-9)
			(:instance logorc1-mod-1)
			(:instance mod012 (x i)))))))

(local(defthm logeqv-mod-1
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (iff (= (mod (logeqv i j) 2) 0)
		  (or (= (mod (logorc1 i j) 2) 0)
		      (= (mod (logorc1 j i) 2) 0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logorc1 logand)
		  :use ((:instance evenp-logand (i (logorc1 i j)) (j (logorc1 j i))))))))

(local(defthm logeqv-mod
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (iff (= (mod (logeqv i j) 2) 0)
		  (not (= (logxor (mod i 2) (mod j 2))
			  0))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logorc1 logeqv)
		  :use ((:instance logeqv-mod-1)
			(:instance logorc1-mod)
			(:instance logorc1-mod (i j) (j i))
			(:instance mod012 (x i))
			(:instance mod012 (x j)))))))

(local(defthm logxor-mod-1
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (iff (= (mod (logxor i j) 2)
		     0)
		  (= (logxor (mod i 2) (mod j 2))
		     0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logorc1 logeqv)
		  :use ((:instance logeqv-mod)
			(:instance mod-logior-9 (i (logeqv i j))))))))

(local(defthm logand>=0
    (implies (and (integerp i)
		  (>= i 0)
		  (integerp j))
	     (>= (logand i j) 0))
     :hints (("Goal" :in-theory (enable logand)))
  :rule-classes ()))

(local(defthm logior<0
    (implies (and (integerp i)
		  (integerp j)
		  (< i 0))
	     (< (logior i j) 0))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand)
		  :use ((:instance logand>=0 (i (lognot i)) (j (lognot j))))))))

;opens up into nonnegative-integer-quotient!
(local (defthm logand<0
    (implies (and (integerp i)
		  (integerp j)
		  (< i 0)
		  (< j 0))
	     (< (logand i j) 0))
     :hints (("Goal" :in-theory (set-difference-theories
                                 (enable logand)
                                 '(
                                   ))))
  :rule-classes ()))

(defthm LOGXOR-NAT
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (and (integerp (logxor i j))
		  (>= (logxor i j) 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand logior)
		  :use ((:instance logior<0 (i (lognot i)))
			(:instance logior<0 (i (lognot j)) (j i))
			(:instance logand<0 (i (logorc1 i j)) (j (logorc1 j i)))))))

(defthm LOGXOR-MOD
    (implies (and (integerp i) (>= i 0)
		  (integerp j) (>= j 0))
	     (= (mod (logxor i j) 2)
		(logxor (mod i 2) (mod j 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-mod-1)
			(:instance logxor-nat)
			(:instance mod012 (x (logxor i j)))
			(:instance mod012 (x i))
			(:instance mod012 (x j))))))

(defthm floor-logorc1
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logorc1 i j) 2)
		(logorc1 (floor i 2) (floor j 2))))
  :rule-classes()
  :hints (("Goal" :in-theory (disable lognot logior floor)
		  :use ((:instance floor-logior (i (lognot i)))
			(:instance floor-lognot)))))

(defthm floor-logeqv
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logeqv i j) 2)
		(logeqv (floor i 2) (floor j 2))))
  :rule-classes()
  :hints (("Goal" :in-theory (disable logand logorc1 floor)
		  :use ((:instance floor-logorc1)
			(:instance floor-logorc1 (i j) (j i))
			(:instance floor-logand (i (logorc1 i j)) (j (logorc1 j i)))))))

(defthm floor-logxor
    (implies (and (integerp i)
		  (integerp j))
	     (= (floor (logxor i j) 2)
		(logxor (floor i 2) (floor j 2))))
  :rule-classes()
  :hints (("Goal" :in-theory (disable lognot logeqv floor)
		  :use ((:instance floor-lognot (i (logeqv i j)))
			(:instance floor-logeqv)))))

(defthm LOGXOR-FL
    (implies (and (integerp i)
		  (integerp j)
		  (>= i 0)
		  (>= j 0))
	     (= (fl (/ (logxor i j) 2))
		(logxor (fl (/ i 2)) (fl (/ j 2)))))
  :rule-classes()
  :hints (("Goal" :in-theory (disable logxor floor)
		  :use ((:instance floor-logxor)
			(:instance floor-fl (m i) (n 2))
			(:instance floor-fl (m j) (n 2))
			(:instance logxor-nat)
			(:instance floor-fl (m (logxor i j)) (n 2))))))

(defthm LOGXOR-DEF
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (= (logxor x y)
		(+ (* 2 (logxor (fl (/ x 2)) (fl (/ y 2))))
		   (logxor (mod x 2) (mod y 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-fl (i x) (j y))
			(:instance logxor-mod (i x) (j y))
			(:instance logxor-nat (i x) (j y))
			(:instance mod-fl (m (logxor x y)) (n 2))))))



;from loglemmas

(local
(defun log-induct (x y)
  (if (and (integerp x) (>= x 0)
	   (integerp y) (>= y 0))
      (if (or (= x 0) (= y 0))
	  ()
	(log-induct (fl (/ x 2)) (fl (/ y 2))))
    ())))

(defthm BIT-BASIC-A
    (implies (and (integerp x) (>= x 0))
	     (= (logand x 0)
		0))
  :rule-classes ())

(defthm BIT-BASIC-B
    (implies (and (integerp x) (>= x 0))
	     (= (logior x 0)
		x))
  :rule-classes ())

(defthm BIT-BASIC-C
    (implies (and (integerp x)
		  (integerp y))
	     (= (logand x y) (logand y x)))
  :rule-classes ())

(defthm BIT-BASIC-D
    (implies (and (integerp x)
		  (integerp y))
	     (= (logior x y) (logior y x)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bit-basic-c (x (lognot x)) (y (lognot y)))))))

(local
(defun log-induct-3 (x y z)
  (if (and (integerp x) (>= x 0)
	   (integerp y) (>= y 0)
	   (integerp z) (>= z 0))
      (if (or (= x 0) (= y 0) (= z 0))
	  ()
	(log-induct-3 (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2))))
    ())))

(defthm LOGAND-FL-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (fl (* 1/2 (logand x y)))
		    (logand (fl (/ x 2)) (fl (/ y 2)))))
  :hints (("Goal" :use ((:instance logand-fl)))))

(defthm LOGAND-MOD-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (mod (logand x y) 2)
		    (logand (mod x 2) (mod y 2))))
  :hints (("Goal" :use ((:instance logand-mod)))))

(defthm LOGAND-NAT-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (>= (logand x y) 0))
  :hints (("Goal" :use ((:instance logand-nat (i x) (j y))))))

(local
 (defthm fl-mod-equal
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (= (fl (/ x 2)) (fl (/ y 2)))
		  (= (mod x 2) (mod y 2)))
	     (= x y))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-fl (m x) (n 2))
			(:instance mod-fl (m y) (n 2)))))))

(local
(defthm bit-basic-e-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (EQUAL (LOGAND (LOGAND (MOD X 2) (MOD Y 2))
			    (MOD Z 2))
		    (LOGAND (MOD X 2)
			    (LOGAND (MOD Y 2) (MOD Z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012)
			(:instance mod012 (x y))
			(:instance mod012 (x z)))))))

(defthm BIT-BASIC-E
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logand (logand x y) z)
		(logand x (logand y z))))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance bit-basic-e-1)
				  (:instance fl-mod-equal
					      (x (logand (logand x y) z))
					      (y (logand x (logand y z))))))))

(in-theory (disable logior))

(local
(defthm bit-basic-f-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (EQUAL (LOGIOR (LOGIOR (MOD X 2) (MOD Y 2))
			    (MOD Z 2))
		    (LOGIOR (MOD X 2)
			    (LOGIOR (MOD Y 2) (MOD Z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012)
			(:instance mod012 (x y))
			(:instance mod012 (x z)))))))

(local
(defthm logior-fl-rewrite
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (fl (* 1/2 (logior x y)))
		    (logior (fl (/ x 2)) (fl (/ y 2)))))
  :hints (("Goal" :use ((:instance logior-fl (i x) (j y)))))))

(defthm LOGIOR-MOD-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (mod (logior x y) 2)
		    (logior (mod x 2) (mod y 2))))
  :hints (("Goal" :use ((:instance logior-mod (i x) (j y))))))

(defthm LOGIOR-NAT-REWRITE
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (>= (logior x y) 0))
  :hints (("Goal" :use ((:instance logior-nat (i x) (j y))))))

(defthm LOGIOR-0
    (implies (integerp x)
	     (equal (logior x 0) x))
  :hints (("Goal" :in-theory (enable logior))))

(defthm LOGIOR-0-2
    (implies (integerp x)
	     (equal (logior 0 x) x))
  :hints (("Goal" :in-theory (enable logior))))

(defthm BIT-BASIC-F
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logior (logior x y) z)
		(logior x (logior y z))))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance bit-basic-f-1)
				  (:instance fl-mod-equal
					      (x (logior (logior x y) z))
					      (y (logior x (logior y z))))))))

(local
(defthm bit-basic-g-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (equal (logior (mod x 2)
			    (logand  (mod y 2) (mod z 2)))
		    (logand (logior (mod x 2) (mod y 2))
			    (logior (mod x 2) (mod z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012)
			(:instance mod012 (x y))
			(:instance mod012 (x z)))))))

(local
(defthm bit-basic-g-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (logand (logior (mod x 2) (mod y 2)) (mod x 2))
		    (mod x 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012)
			(:instance mod012 (x y)))))))

(local
(defthm logand-x-x-1
    (implies (and (integerp x) (>= x 0))
	     (equal (logand (mod x 2) (mod x 2)) (mod x 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012))))))

(defthm LOGAND-X-X
    (implies (and (integerp x) (>= x 0))
	     (equal (logand x x) x))
  :hints (("Goal" :induct (log-induct x x))
	  ("Subgoal *1/2.1" :use ((:instance logand-x-x-1)
				  (:instance fl-mod-equal (y (logand x x)))))))

(local
(defthm bit-basic-g-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (= (logand (logior x y) x)
		x))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct x y))
	  ("Subgoal *1/2.2" :use ((:instance bit-basic-g-2)
				  (:instance fl-mod-equal
					      (y (logand (logior x y) x))))))))

(defthm BIT-BASIC-G
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logior x (logand y z))
		(logand (logior x y) (logior x z))))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance bit-basic-g-1)
				  (:instance fl-mod-equal
					      (x (logior x (logand y z)))
					      (y (logand (logior x y) (logior x z))))))
	  ("Subgoal *1/1" :use ((:instance bit-basic-g-3)
				(:instance bit-basic-c (y (logior x z)))
				(:instance bit-basic-g-3 (y z))))))

(local
(defthm bit-basic-h-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (equal (logand (mod x 2)
			    (logior  (mod y 2) (mod z 2)))
		    (logior (logand (mod x 2) (mod y 2))
			    (logand (mod x 2) (mod z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012)
			(:instance mod012 (x y))
			(:instance mod012 (x z)))))))

(defthm BIT-BASIC-H
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logand x (logior y z))
		(logior (logand x y) (logand x z))))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct-3 x y z))
	  ("Subgoal *1/2.1" :use ((:instance bit-basic-h-1)
				  (:instance fl-mod-equal
					      (x (logand x (logior y z)))
					      (y (logior (logand x y) (logand x z))))))))

(local
(defun op-dist-induct (x y n)
  (if (and (integerp n) (>= n 0))
      (if  (= n 0)
	  (list x y)
	(op-dist-induct (fl (/ x 2)) (fl (/ y 2)) (1- n)))
    ())))

(defthm OR-DIST-A
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (< y (expt 2 n)))
	     (< (logior x y) (expt 2 n)))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/2" :use ((:instance logior-def (i x) (j y))
				(:instance mod012)
				(:instance mod012 (x y))))))

(local
(defthm or-dist-b-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (< y (expt 2 n)))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* 2 (logior (fl (* (expt 2 (1- n)) x))
				(fl (/ y 2))))
		   (logior (mod (* (expt 2 n) x) 2)
			   (mod y 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logior-def (i (* (expt 2 n) x)) (j y)))))))

(local
(defthm or-dist-b-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (< y (expt 2 n)))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* 2 (logior (* (expt 2 (1- n)) x)
				(fl (/ y 2))))
		   (mod y 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-b-1)
			(:instance fl-int (x (* (expt 2 (1- n)) x)))
			(:instance mod-2*i (i (* (expt 2 (1- n)) x))))))))

(local
(defthm or-dist-b-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (< y (expt 2 n))
		  (= (logior (* (expt 2 (1- n)) x)
			     (fl (/ y 2)))
		     (+ (* (expt 2 (1- n)) x)
			(fl (/ y 2)))))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x)
		   (* 2	(fl (/ y 2)))
		   (mod y 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-b-2))))))

(local
(defthm or-dist-b-4
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (< y (expt 2 n))
		  (= (logior (* (expt 2 (1- n)) x)
			     (fl (/ y 2)))
		     (+ (* (expt 2 (1- n)) x)
			(fl (/ y 2)))))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x) y)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-b-3)
			(:instance mod-fl (m y) (n 2)))))))

(local
(defun or-dist-induct (y n)
  (if (and (integerp n) (>= n 0))
      (if (= n 0)
	  y
	(or-dist-induct (fl (/ y 2)) (1- n)))
    ())))

(defthm OR-DIST-B
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< y (expt 2 n)))
	     (= (logior (* (expt 2 n) x) y)
		(+ (* (expt 2 n) x) y)))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct y n))
	  ("Subgoal *1/2" :use ((:instance or-dist-b-4)))))

(local
(defthm or-dist-c-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(+ (* 2 (logior (* (expt 2 (1- n)) x)
				(* (expt 2 (1- n)) y)))
		   (logior (mod (* (expt 2 n) x) 2)
			   (mod (* (expt 2 n) y) 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logior-def (i (* (expt 2 n) x)) (j (* (expt 2 n) y))))))))

(local
(defthm or-dist-c-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* 2 (logior (* (expt 2 (1- n)) x)
			     (* (expt 2 (1- n)) y)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-c-1)
			(:instance mod-2*i (i (* (expt 2 (1- n)) x)))
			(:instance mod-2*i (i (* (expt 2 (1- n)) y))))))))

(local
(defthm or-dist-c-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (= (logior (* (expt 2 (1- n)) x)
			     (* (expt 2 (1- n)) y))
		     (* (expt 2 (1- n)) (logior x y))))			     
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* (expt 2 n)
		   (logior x y))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-c-2))))))

(defthm OR-DIST-C
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logior (* (expt 2 n) x)
			(* (expt 2 n) y))
		(* (expt 2 n) (logior x y))))
  :rule-classes ()
  :hints (("Goal" :induct (induct-nat n))
	  ("Subgoal *1/1" :use ((:instance or-dist-c-3)))))

(local
(defthm or-dist-d-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior x y)
		(logior (logior (* (expt 2 n) (fl (/ x (expt 2 n))))
				(mod x (expt 2 n)))
			(logior (* (expt 2 n) (fl (/ y (expt 2 n))))
				(mod y (expt 2 n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-fl (m x) (n (expt 2 n)))
			(:instance mod-fl (m y) (n (expt 2 n)))
			(:instance mod<n (m x) (n (expt 2 n)))
			(:instance mod<n (m y) (n (expt 2 n)))
			(:instance mod>=0 (m x) (n (expt 2 n)))
			(:instance mod>=0 (m y) (n (expt 2 n)))
			(:instance or-dist-b (x (fl (/ x (expt 2 n)))) (y (mod x (expt 2 n))))
			(:instance or-dist-b (x (fl (/ y (expt 2 n)))) (y (mod y (expt 2 n)))))))))

(local
(defthm or-dist-d-2
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp c) (>= c 0)
		  (integerp d) (>= d 0))
	     (= (logior (logior a b) (logior c d))
		(logior (logior a c) (logior b d))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bit-basic-f (x (logior a b)) (y c) (z d))
			(:instance bit-basic-f (x a) (y b) (z c))
			(:instance bit-basic-d (x b) (y c))
			(:instance bit-basic-f (x a) (y c) (z b))
			(:instance bit-basic-f (x (logior a c)) (y b) (z d)))))))

(defthm FL-NONNEG
    (implies (and (integerp n)
		  (rationalp x)
		  (>= x 0))
	     (not (< (* (EXPT 2 N) (FL (* X (/ (EXPT 2 N))))) 0)))
  :hints (("Goal"
		  :use (
			(:instance n<=fl-linear (x (* X (/ (EXPT 2 N)))) (n 0))))))

(local
(defthm or-dist-d-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior x y)
		(logior (logior (* (expt 2 n) (fl (/ x (expt 2 n))))
				(* (expt 2 n) (fl (/ y (expt 2 n)))))
			(logior (mod x (expt 2 n))
				(mod y (expt 2 n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod>=0 (m x) (n (expt 2 n)))
			(:instance mod>=0 (m y) (n (expt 2 n)))
			(:instance or-dist-d-1)
			(:instance or-dist-d-2
				   (a (* (expt 2 n) (fl (/ x (expt 2 n)))))
				   (b (mod x (expt 2 n)))
				   (c (* (expt 2 n) (fl (/ y (expt 2 n)))))
				   (d (mod y (expt 2 n)))))))))

(local
(defthm or-dist-d-4
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logior x y)
		(+ (* (expt 2 n)
		      (logior (fl (/ x (expt 2 n)))
			      (fl (/ y (expt 2 n)))))
		   (logior (mod x (expt 2 n))
			   (mod y (expt 2 n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-d-3)
			(:instance or-dist-c (x (fl (/ x (expt 2 n)))) (y (fl (/ y (expt 2 n)))))
			(:instance or-dist-b
				   (x (logior (fl (/ x (expt 2 n)))
					      (fl (/ y (expt 2 n)))))
				   (y (logior (mod x (expt 2 n))
					      (mod y (expt 2 n)))))
			(:instance mod<n (m x) (n (expt 2 n)))
			(:instance mod<n (m y) (n (expt 2 n)))
			(:instance or-dist-a (x (mod x (expt 2 n))) (y (mod y (expt 2 n))))
			(:instance mod>=0 (m x) (n (expt 2 n)))
			(:instance mod>=0 (m y) (n (expt 2 n))))))))

(local
(defthm or-dist-d-5
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (mod (logior x y) (expt 2 n))
		(mod (logior (mod x (expt 2 n)) (mod y (expt 2 n)))
		     (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-d-4)
			(:instance mod+-thm 
				   (m (logior (mod x (expt 2 n)) (mod y (expt 2 n))))
				   (n (expt 2 n))
				   (a (logior (fl (/ x (expt 2 n)))
					      (fl (/ y (expt 2 n))))))
			(:instance n<=fl-linear (x (/ x (expt 2 n))) (n 0))
			(:instance n<=fl-linear (x (/ y (expt 2 n))) (n 0))
			(:instance logior-nat (i (fl (/ x (expt 2 n)))) (j (fl (/ y (expt 2 n)))))
			(:instance logior-nat (i (mod x (expt 2 n))) (j (mod y (expt 2 n))))
			(:instance mod>=0 (m x) (n (expt 2 n)))
			(:instance mod>=0 (m y) (n (expt 2 n))))))))

(defthm OR-DIST-D
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (mod (logior x y) (expt 2 n))
		(logior (mod x (expt 2 n)) (mod y (expt 2 n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance or-dist-d-5)
			(:instance mod< 
				   (m (logior (mod x (expt 2 n)) (mod y (expt 2 n))))
				   (n (expt 2 n)))
			(:instance mod>=0 (m x) (n (expt 2 n)))
			(:instance mod>=0 (m y) (n (expt 2 n)))
			(:instance logior-nat (i (mod x (expt 2 n))) (j (mod y (expt 2 n))))
			(:instance mod<n (m x) (n (expt 2 n)))
			(:instance mod<n (m y) (n (expt 2 n)))
			(:instance or-dist-a (x (mod x (expt 2 n))) (y (mod y (expt 2 n))))))))

(local
(defthm and-dist-a-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (<= (logand (mod x 2) (mod y 2)) (mod x 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012)
			(:instance mod012 (x y)))))))

(defthm AND-DIST-A
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (<= (logand x y) x))
  :rule-classes ()
  :hints (("Goal" :induct (log-induct x y))
	  ("Subgoal *1/2" :use ((:instance and-dist-a-1)
				(:instance mod-fl (m (logand x y)) (n 2))
				(:instance mod-fl (m x) (n 2))))))

(local
(defthm and-dist-b-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0))
	     (= (logand (* (expt 2 n) x) y)
		(* 2 (logand (* (expt 2 (1- n)) x)
			     (fl (/ y 2))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logand-def (x (* (expt 2 n) x)))
			(:instance mod-2*i (i (* (expt 2 (1- n)) x))))))))

(local
(defthm and-dist-b-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (= (logand (* (expt 2 (1- n)) x) (fl (/ y 2))) 
		     (* (expt 2 (1- n)) (logand x (fl (/ (fl (/ y 2)) (expt 2 (1- n))))))))
	     (= (logand (* (expt 2 n) x) y)
		(* 2 
		   (* (expt 2 (1- n))
		      (logand x
			      (fl (/ (fl (/ y 2)) (expt 2 (1- n)))))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-b-1))))))

(local
(defthm and-dist-b-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (> n 0)
		  (= (logand (* (expt 2 (1- n)) x) (fl (/ y 2))) 
		     (* (expt 2 (1- n)) (logand x (fl (/ (fl (/ y 2)) (expt 2 (1- n))))))))
	     (= (logand (* (expt 2 n) x) y)
		(* 2 
		   (* (expt 2 (1- n))
		      (logand x
			      (fl (/ y (expt 2 n))))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-b-2)
			(:instance fl/int-rewrite (x (/ y 2)) (n (expt 2 (1- n)))))))))

;things in this file were opening up to non-negative-integer-quotient (e.g., and-dist-b)
(in-theory (disable floor))

(defthm AND-DIST-B
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logand (* (expt 2 n) x) y)
		(* (expt 2 n) (logand x (fl (/ y (expt 2 n)))))))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct y n))
	  ("Subgoal *1/2" :use ((:instance and-dist-b-3)))))

(local
(defthm and-dist-c-1
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0))
	     (= x (logior (* (expt 2 n) (fl (/ x (expt 2 n))))
			  (mod x (expt 2 n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-fl (m x) (n (expt 2 n)))
			(:instance or-dist-b (x (fl (/ x (expt 2 n)))) (y (mod x (expt 2 n))))
			(:instance mod<n (m x) (n (expt 2 n)))
			(:instance mod>=0 (m x) (n (expt 2 n))))))))

(local
(defthm and-dist-c-2
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logand x y)
		(logior (logand (* (expt 2 n) (fl (/ x (expt 2 n))))
				y)
			(logand (mod x (expt 2 n))
				y))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c-1)
			(:instance mod>=0 (m x) (n (expt 2 n)))
			(:instance bit-basic-h 
				   (x y)
				   (y (* (expt 2 n) (fl (/ x (expt 2 n)))))
				   (z (mod x (expt 2 n))))
			(:instance bit-basic-c (x (* (expt 2 n) (fl (/ x (expt 2 n))))))
			(:instance bit-basic-c (x (mod x (expt 2 n))))
			(:instance bit-basic-c
				   (x (logior (* (expt 2 n) (fl (/ x (expt 2 n))))
					      (mod x (expt 2 n))))))))))

(local
 (defthm and-dist-c-3
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logand x y)
		(logior (* (expt 2 n) 
			   (logand (fl (/ x (expt 2 n)))
				   (fl (/ y (expt 2 n)))))
			(logand (mod x (expt 2 n))
				y))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c-2)
			(:instance and-dist-b (x (fl (/ x (expt 2 n))))))))))

(local
(defthm and-dist-c-4
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (logand x y)
		(+ (* (expt 2 n) 
		      (logand (fl (/ x (expt 2 n)))
			      (fl (/ y (expt 2 n)))))
		   (logand (mod x (expt 2 n))
			   y))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c-3)
			(:instance or-dist-b
				   (x (logand (fl (/ x (expt 2 n)))
					      (fl (/ y (expt 2 n)))))
				   (y (logand (mod x (expt 2 n))
					      y)))
			(:instance mod>=0 (m x) (n (expt 2 n)))			
			(:instance mod<n (m x) (n (expt 2 n)))
			(:instance and-dist-a (x (mod x (expt 2 n)))))))))

(defthm AND-DIST-C
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (mod (logand x y) (expt 2 n))
		(logand (mod x (expt 2 n)) y)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c-4)
			(:instance mod+-thm 
				   (m (logand (mod x (expt 2 n)) y)) 
				   (n (expt 2 n))
				   (a (logand (fl (/ x (expt 2 n))) (fl (/ y (expt 2 n))))))
			(:instance mod>=0 (m x) (n (expt 2 n)))			
			(:instance mod<n (m x) (n (expt 2 n)))
			(:instance and-dist-a (x (mod x (expt 2 n))))
			(:instance mod< (m (logand (mod x (expt 2 n)) y)) (n (expt 2 n)))))))

(defthm AND-DIST-D
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n)))
	     (= (logand x y)
		(logand x (mod y (expt 2 n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-c (x y) (y x))
			(:instance bit-basic-c)
			(:instance bit-basic-c (y (mod y (expt 2 n))))
			(:instance and-dist-a)
			(:instance mod< (m (logand x y)) (n (expt 2 n)))))))

(defthm BIT-DIST-A
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (bitn (logand x y) n)
		(logand (bitn x n) (bitn y n))))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/1" :use ((:instance logand-mod)
				(:instance bitn-alt-0)
				(:instance bitn-alt-0 (x y))
				(:instance bitn-alt-0 (x (logand x y)))))
	  ("Subgoal *1/2" :use ((:instance bitn-alt-pos (k n))
				(:instance bitn-alt-pos (k n) (x y))
				(:instance bitn-alt-pos (k n) (x (logand x y)))
				(:instance logand-fl)))))

(defthm BIT-DIST-B
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0))
	     (= (bitn (logior x y) n)
		(logior (bitn x n) (bitn y n))))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/1" :use ((:instance logior-mod)
				(:instance bitn-alt-0)
				(:instance bitn-alt-0 (x y))
				(:instance bitn-alt-0 (x (logior x y)))))
	  ("Subgoal *1/2" :use ((:instance bitn-alt-pos (k n))
				(:instance bitn-alt-pos (k n) (x y))
				(:instance bitn-alt-pos (k n) (x (logior x y)))
				(:instance logior-fl)))))

(defthm AND-BITS-A
    (implies (and (integerp x) (>= x 0)
		  (integerp k) (>= k 0))
	     (= (logand x (expt 2 k))
		(* (expt 2 k) (bitn x k))))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct x k))
	  ("Subgoal *1/1" :use ((:instance logand-def (y 1))
				(:instance mod012)
				(:instance bitn-alt-0)))
	  ("Subgoal *1/2" :use ((:instance logand-def (y (expt 2 k)))
				(:instance mod-2*i (i (expt 2 (1- k))))
				(:instance fl-int (x (expt 2 (1- k))))
				(:instance bitn-alt-pos)))))

(defthm AND-BITS-B
    (implies (and (integerp x) (>= x 0)
		  (integerp k) (>= k 0))
	     (= (logior x (expt 2 k))
		(+ x
		   (* (expt 2 k) 
		      (- 1 (bitn x k))))))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct x k))
	  ("Subgoal *1/1" :use ((:instance logior-def (i x) (j 1))
				(:instance mod012)
				(:instance mod-fl (m x) (n 2))
				(:instance bitn-alt-0)))
	  ("Subgoal *1/2" :use ((:instance logior-def (i x) (j (expt 2 k)))
				(:instance mod-2*i (i (expt 2 (1- k))))
				(:instance mod-fl (m x) (n 2))
				(:instance fl-int (x (expt 2 (1- k))))
				(:instance bitn-alt-pos)))))

(local
 (defthm fl-2**n-1/2
    (implies (and (integerp n) (> n 0))
	     (= (fl (/ (1- (expt 2 n)) 2))
		(1- (expt 2 (1- n)))))
  :rule-classes ()))

(local
(defthm mod-2**n-1/2
    (implies (and (integerp n) (> n 0))
	     (= (mod (1- (expt 2 n)) 2)
		1))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-2*i+1 (i (1- (expt 2 (1- n)))))
			(:instance mod012 (x (1- (expt 2 n)))))))))

(local
(defthm and-bits-c-<-0-1
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (> n 0)
		  (< x (expt 2 n)))
	     (= (logand x (- (expt 2 n) 1))
		x))
  :rule-classes ()
  :hints (("Goal" :induct (or-dist-induct x n))
	  ("Subgoal *1/2" :use ((:instance logand-def (y (1- (expt 2 n))))
				(:instance fl-2**n-1/2)
				(:instance mod-fl (m x) (n 2))
				(:instance mod012)
				(:instance mod-2**n-1/2))))))

(local
(defthm and-bits-c-<-0
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (> n 0)
		  (< x (expt 2 n)))
	     (= (logand x (- (expt 2 n) 1))
		(bits x (1- n) 0)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-0-1)
			(:instance mod< (m x) (n (expt 2 n))))))))

(local
(defthm and-bits-c-<-pos-1
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* 2 (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k)))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logand-def (y (- (expt 2 n) (expt 2 k))))
			(:instance expt-monotone (n k) (m n))
			(:instance mod-2*i (i (- (expt 2 (1- n)) (expt 2 (1- k))))))))))

(local
(defthm and-bits-c-<-pos-2
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits (fl (/ x 2)) (- n 2) (1- k)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-1))))))

(local
(defthm and-bits-c-<-pos-3
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (fl (/ (mod (fl (/ x 2)) (expt 2 (1- n))) (expt 2 (1- k)))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-2))))))

(local
(defthm and-bits-c-<-pos-4
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n)))
	     (= (mod (fl (/ x 2)) (expt 2 (1- n)))
		(fl (/ x 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod< (m (fl (/ x 2))) (n (expt 2 (1- n))))
			(:instance fl-def-linear (x (/ x 2))))
           :in-theory (disable mod-equal)))))

(local
(defthm and-bits-c-<-pos-5
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (fl (/ (fl (/ x 2)) (expt 2 (1- k)))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-3)
			(:instance and-bits-c-<-pos-4))))))

(local
(defthm and-bits-c-<-pos-6
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (fl (/ x (expt 2 k))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-5)
			(:instance fl/int-rewrite (x (/ x 2)) (n (expt 2 (1- k)))))))))

(local
(defthm and-bits-c-<-pos-7
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n))
	     (= (fl (/ x (expt 2 k)))
		(bits x (1- n) k)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod< (m x) (n (expt 2 n))))))))

(local
(defthm and-bits-c-<-pos
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (> k 0)
		  (< k n)
		  (= (logand (fl (/ x 2)) (- (expt 2 (1- n)) (expt 2 (1- k))))
		     (* (expt 2 (1- k)) (bits (fl (/ x 2)) (- n 2) (1- k)))))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-<-pos-6)
			(:instance and-bits-c-<-pos-7))))))

(local
(defun and-bits-induct (x n k)
  (if (and (integerp k) (>= k 0))
      (if (= k 0)
	  (list x n k)
	(and-bits-induct (fl (/ x 2)) (1- n) (1- k)))
    ())))

(local
(defthm and-bits-c-<
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (integerp k) (>= k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :induct (and-bits-induct x n k))
	  ("Subgoal *1/1" :use ((:instance and-bits-c-<-0)))
	  ("Subgoal *1/2" :use ((:instance and-bits-c-<-pos))))))

(local
(defthm and-bits-c-1
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(logand (mod x (expt 2 n)) (- (expt 2 n) (expt 2 k)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-dist-d (x (- (expt 2 n) (expt 2 k))) (y x))
			(:instance expt-monotone (n k) (m n))
			(:instance bit-basic-c (y (- (expt 2 n) (expt 2 k))))
			(:instance bit-basic-c (x (mod x (expt 2 n))) (y (- (expt 2 n) (expt 2 k)))))))))

(local
(defthm and-bits-c-2
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits (mod x (expt 2 n)) (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-1)
			(:instance and-bits-c-< (x (mod x (expt 2 n))))
			(:instance mod<n (m x) (n (expt 2 n)))
			(:instance mod>=0 (m x) (n (expt 2 n))))))))

(defthm AND-BITS-C
    (implies (and (integerp x) (>= x 0)
		  (integerp n) (>= n 0)
		  (integerp k) (>= k 0)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance and-bits-c-2)
			(:instance mod-bits (y (mod x (expt 2 n))) (i n) (j k))
			(:instance mod-mod (a n) (b n))))))



(defthm logand-expt-3
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (< k n))
	     (= (logand x (- (expt 2 n) (expt 2 k)))
		(* (expt 2 k) (bits x (1- n) k))))
  :rule-classes ()
  :hints (("Goal" :use (and-bits-c))))


;from logxor-lemmas.lisp

;proved in comp1.lisp
(defthm COMP1-COMP1
  (implies (and (case-split (natp n)); (integerp n)
                (case-split (bvecp x n)) ;added
                )
           (= (comp1 (comp1 x n) n)
              x)))

;(in-theory (enable comp1)) ;test removing this
;integerp-expt was here

(local (defthm fl-comp1-1
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (/ (comp1 x n) (expt 2 k))
		(+ (expt 2 (- n k))
		   (/ (- -1 x) (expt 2 k)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable comp1)
                              '( a10))
                              
		  :use ((:instance expo+ (m k) (n (- n k))))))))

(local (defthm fl=
    (implies (equal x y)
	     (equal (fl x) (fl y)))
  :rule-classes ()))

(local (defthm fl-comp1-2
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (fl (/ (comp1 x n) (expt 2 k)))
		(fl (+ (expt 2 (- n k))
		       (/ (- -1 x) (expt 2 k))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a10)
		  :use ((:instance fl-comp1-1)
			(:instance fl= 
				   (x (/ (comp1 x n) (expt 2 k))) 
				   (y (+ (expt 2 (- n k))
					 (/ (- -1 x) (expt 2 k))))))))))

(local (defthm fl-comp1-3
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (fl (/ (comp1 x n) (expt 2 k)))
		(+ (expt 2 (- n k))
		   (fl (/ (- -1 x) (expt 2 k))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a10)
		  :use ((:instance fl-comp1-2)
			(:instance fl+int-rewrite (n (expt 2 (- n k))) (x (/ (- (- x) 1) (expt 2 k)))))))))

(local (include-book "arith2"))
(local (in-theory (disable TWO-NATPS-ADD-TO-1)))

(local (include-book "fl2"))

(defthm FL-COMP1
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0)
                  (< x (expt 2 n)))
	     (= (fl (/ (comp1 x n) (expt 2 k)))
		(comp1 (fl (/ x (expt 2 k))) (- n k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable comp1)
                              '(a10))
		  :use ((:instance fl-comp1-3)
			(:instance floor-m+1 (m x) (n (expt 2 k)))))))

(local
(defthm mod=mod-1
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (= (mod a n) (mod b n)))
	     (= (- a (* n (fl (/ a n))))
		(- b (* n (fl (/ b n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-fl (m a))
			(:instance mod-fl (m b)))))))

(local
(defthm mod=mod-2
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (= (mod a n) (mod b n)))
	     (= (- a b) (* n (- (fl (/ a n)) (fl (/ b n))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod=mod-1))))))

(local
(defthm hack-m10
    (implies (and (rationalp a) (rationalp b) (rationalp c) (> b 0) (= a (* b c)))
	     (= (/ a b) c))
  :rule-classes ()))

(local
(defthm mod=mod-3
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (= (mod a n) (mod b n)))
	     (= (/ (- a b) n) (- (fl (/ a n)) (fl (/ b n)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod=mod-2)
			(:instance hack-m10 (a (- a b)) (b n) (c (- (fl (/ a n)) (fl (/ b n))))))))))

(local (defthm mod=mod
    (implies (and (integerp a) (>= a 0)
		  (integerp b) (>= b 0)
		  (integerp n) (> n 0)
		  (= (mod a n) (mod b n)))
	     (integerp (/ (- a b) n)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod=mod-3))))))

(local (defthm mod-comp1-1
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x))
	     (not (integerp (+ (expt 2 (1- n)) (- x) -1/2))))
  :rule-classes ()))

(local (in-theory (disable CANCEL-IN-PRODS-<)))

(local (defthm mod-comp1-2
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
                  (< x (expt 2 n)) ;new
                  (<= 0 x) ;new
                  )
	     (not (integerp (/ (- (comp1 x n) x) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (enable comp1)
           :use ((:instance expo+ (m (1- n)) (n 1))
                 (:instance mod-comp1-1))))))

(defthm MOD-COMP1
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n)))
	     (not (= (mod (comp1 x n) 2)
		     (mod x 2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-comp1-2)
			(:instance mod=mod (a (comp1 x n)) (b x) (n 2))))))

(local (defthm bitn-comp1-1
    (implies (= x y)
	     (= (mod x 2) (mod y 2)))
  :rule-classes ()))

(local (defthm bitn-comp1-2
    (implies (and (integerp n) (>= n k)
		  (integerp k) (>= k 0)
		  (integerp x) (>= x 0) (< x (expt 2 n)))
	     (= (mod (fl (/ (comp1 x n) (expt 2 k))) 2)
		(mod (comp1 (fl (/ x (expt 2 k))) (- n k)) 2)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance fl-comp1)
			(:instance bitn-comp1-1
				   (x (fl (/ (comp1 x n) (expt 2 k))))
				   (y (comp1 (fl (/ x (expt 2 k))) (- n k)))))))))

;(in-theory (disable CANCEL-TIMES-<-ERIC))

(local (include-book "expt"))
(local (include-book "expt2"))

(local (in-theory (disable EXPT-MONOTONE-LINEAR))) ;test


(local (defthm bitn-comp1-3
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n))
		  (integerp k)
		  (>= k 0)
		  (< k n))
	     (< (/ x (expt 2 k)) (expt 2 (- n k))))
  :rule-classes ()
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable expt-split
                                      expt-pull-negation-out-of-power)
                              '(a15))))))

(local (in-theory (disable CANCEL-TIMES-<-ERIC-1-BETTER-ALT
                           CANCEL-IN-PRODS-<-1-OF-2-WITH-3-OF-3
                           CANCEL-IN-PRODS-<-1-OF-2-WITH-1-OF-1)))


(local (defthm bitn-comp1-4
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n))
		  (integerp k)
		  (>= k 0)
		  (< k n))
	     (not (= (mod (fl (/ (comp1 x n) (expt 2 k))) 2)
		     (mod (fl (/ x (expt 2 k))) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable comp1)
		  :use ((:instance mod-comp1 (x (fl (/ x (expt 2 k)))) (n (- n k)))
			(:instance bitn-comp1-3)
			(:instance fl-def-linear (x (/ x (expt 2 k))))
			(:instance bitn-comp1-2))))))

(defthm BITN-COMP1-thm
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n))
		  (integerp k)
		  (>= k 0)
		  (< k n))
	     (not (= (bitn (comp1 x n) k)
		     (bitn x k))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bitn-comp1-4)
			(:instance bitn-def)
			(:instance bitn-def (x (comp1 x n)))))))

(local (defthm logxor-rewrite-1
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			(logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))
		(logior (logand (fl (/ x 2)) (fl (/ (comp1 y n) 2)))
			(logand (fl (/ y 2)) (fl (/ (comp1 x n) 2))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable comp1)
		  :use ((:instance fl-comp1 (k 1))
			(:instance fl-comp1 (k 1) (x y)))))))

(local (defthm logxor-rewrite-2
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (logand (fl (/ x 2)) (fl (/ (comp1 y n) 2)))
			(logand (fl (/ y 2)) (fl (/ (comp1 x n) 2))))
		(logior (fl (/ (logand x (comp1 y n)) 2))
			(fl (/ (logand y (comp1 x n)) 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logand-fl-rewrite comp1)
		  :use ((:instance logand-fl (y (comp1 y n)))
			(:instance logand-fl (x y) (y (comp1 x n))))))))

(local (defthm logxor-rewrite-3
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			(logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))
		(logior (fl (/ (logand x (comp1 y n)) 2))
			(fl (/ (logand y (comp1 x n)) 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logxor-rewrite-1)
			(:instance logxor-rewrite-2))))))

(local (defthm logxor-rewrite-4
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (fl (/ (logand x (comp1 y n)) 2))
			(fl (/ (logand y (comp1 x n)) 2)))
		(fl (/ (logior (logand x (comp1 y n))
			       (logand y (comp1 x n)))
		       2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logior-fl 
				   (i (logand x (comp1 y n)))
				   (j (logand y (comp1 x n)))))))))

(local (defthm logxor-rewrite-5
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			(logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))
		(fl (/ (logior (logand x (comp1 y n))
			       (logand y (comp1 x n)))
		       2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logxor-rewrite-3)
			(:instance logxor-rewrite-4))))))

(local (defthm logxor-rewrite-6
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n))
		  (= (logxor (fl (/ x 2)) (fl (/ y 2)))
		     (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			     (logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))))
	     (= (fl (/ (logxor x y) 2))
		(fl (/ (logior (logand x (comp1 y n))
			       (logand y (comp1 x n)))
		       2))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance logxor-rewrite-5)
			(:instance logxor-fl (i x) (j y)))))))

(local (defthm logxor-rewrite-7
    (implies (and (integerp n)
		  (> n 0)
		  (integerp x)
		  (>= x 0)
		  (< x (expt 2 n)))
	     (= (mod (comp1 x n) 2)
		(comp1 (mod x 2) 1)))
  :rule-classes ()
  :hints (("Goal" :use ((:instance bitn-comp1-4 (k 0))
			(:instance mod012)
			(:instance mod012 (x (comp1 x n))))))))

(local (defthm logxor-rewrite-8
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (mod (logxor x y) 2)
		(logior (logand (mod x 2) (comp1 (mod y 2) 1))
			(logand (mod y 2) (comp1 (mod x 2) 1)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-mod (i x) (j y))
			(:instance mod012)
			(:instance mod012 (x y)))))))

(local (defthm logxor-rewrite-9
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (mod (logxor x y) 2)
		(logior (logand (mod x 2) (mod (comp1 y n) 2))
			(logand (mod y 2) (mod (comp1 x n) 2)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable comp1 logxor)
		  :use ((:instance logxor-rewrite-8)
			(:instance logxor-rewrite-7)
			(:instance logxor-rewrite-7 (x y)))))))

(local (defthm logxor-rewrite-10
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (mod (logxor x y) 2)
		(logior (mod (logand x (comp1 y n)) 2)
			(mod (logand y (comp1 x n)) 2))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-rewrite-9)
			(:instance logand-mod (y (comp1 y n)))
			(:instance logand-mod (x y) (y (comp1 x n))))))))

(local (defthm logxor-rewrite-11
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (mod (logxor x y) 2)
		(mod (logior (logand x (comp1 y n))
			     (logand y (comp1 x n)))
		     2)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-rewrite-10)
			(:instance logior-mod (i (logand x (comp1 y n))) (j (logand y (comp1 x n)))))))))

(local (defthm logxor-rewrite-12
    (implies (and (integerp n) (> n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n))
		  (= (logxor (fl (/ x 2)) (fl (/ y 2)))
		     (logior (logand (fl (/ x 2)) (comp1 (fl (/ y 2)) (1- n)))
			     (logand (fl (/ y 2)) (comp1 (fl (/ x 2)) (1- n))))))
	     (= (logxor x y)
		(logior (logand x (comp1 y n))
			(logand y (comp1 x n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :use ((:instance logxor-rewrite-6)
			(:instance logxor-rewrite-11)
			(:instance mod-fl
				   (m (logxor x y))
				   (n 2))
			(:instance logxor-nat (i x) (j y))
			(:instance mod-fl
				   (m (logior (logand x (comp1 y n))
					      (logand y (comp1 x n))))
				   (n 2)))))))

(defthm LOGXOR-0
    (implies (integerp y)
	     (equal (logxor 0 y) y))
  :hints (("Goal" :in-theory (enable logior logand))))

(defthm LOGXOR-0-2
    (implies (integerp x)
	     (equal (logxor x 0) x))
  :hints (("Goal" :in-theory (enable logior logand))))

(local
(defun logxor-induct (x y n)
  (if (and (integerp n) (>= n 1))
      (if (= n 1)
	  (list x y)
	(logxor-induct (fl (/ x 2)) (fl (/ y 2)) (1- n)))
    ())))

(local (defthm x01
    (implies (and (integerp x)
		  (>= x 0)
		  (< x 2))
	     (or (= x 0) (= x 1)))
  :rule-classes ()))

(defthm LOGXOR-REWRITE
    (implies (and (integerp n) (>= n 1)
		  (integerp x) (>= x 0) (< x (expt 2 n))
		  (integerp y) (>= y 0) (< y (expt 2 n)))
	     (= (logxor x y)
		(logior (logand x (comp1 y n))
			(logand y (comp1 x n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable comp1 logxor)
		  :induct (logxor-induct x y n))
	  ("Subgoal *1/2" :in-theory (set-difference-theories
                                      (enable expt-split)
                                      '(a15))
           :use (logxor-rewrite-12))
	  ("Subgoal *1/1" :in-theory (enable comp1)
			  :use ((:instance x01)
				(:instance x01 (x y))
				(:instance fl-comp1 (x 0) (k 1))))))

(local
(defun log-induct-3 (x y z)
  (if (and (integerp x) (>= x 0)
	   (integerp y) (>= y 0)
	   (integerp z) (>= z 0))
      (if (or (= x 0) (= y 0) (= z 0))
	  ()
	(log-induct-3 (fl (/ x 2)) (fl (/ y 2)) (fl (/ z 2))))
    ())))

(local (defthm logxor-fl-rewrite
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (fl (* 1/2 (logxor x y)))
		    (logxor (fl (/ x 2)) (fl (/ y 2)))))
  :hints (("Goal" :use ((:instance logxor-fl (i x) (j y)))))))

(local (defthm logxor-mod-rewrite
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (equal (mod (logxor x y) 2)
		    (logxor (mod x 2) (mod y 2))))
  :hints (("Goal" :use ((:instance logxor-mod (i x) (j y)))))))

(local (defthm logxor-nat-rewrite
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0))
	     (>= (logxor x y) 0))
  :hints (("Goal" :use ((:instance logxor-nat (i x) (j y)))))))

(local (defthm fl-mod-equal
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (= (fl (/ x 2)) (fl (/ y 2)))
		  (= (mod x 2) (mod y 2)))
	     (= x y))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod-fl (m x) (n 2))
			(:instance mod-fl (m y) (n 2)))))))

(local
(defthm logxor-assoc-1
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (EQUAL (LOGXOR (LOGXOR (MOD X 2) (MOD Y 2))
			    (MOD Z 2))
		    (LOGXOR (MOD X 2)
			    (LOGXOR (MOD Y 2) (MOD Z 2)))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012)
			(:instance mod012 (x y))
			(:instance mod012 (x z)))))))

(defthm LOGXOR-ASSOC
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp z) (>= z 0))
	     (= (logxor (logxor x y) z)
		(logxor x (logxor y z))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable logxor)
		  :induct (log-induct-3 x y z))
	  ("Subgoal *1/2" :use ((:instance logxor-assoc-1)
				  (:instance fl-mod-equal
					      (x (logxor (logxor x y) z))
					      (y (logxor x (logxor y z))))))))

(local
(defthm logxor-x-x-1
    (implies (and (integerp x) (>= x 0))
	     (equal (logxor (mod x 2) (mod x 2)) 0))
  :rule-classes ()
  :hints (("Goal" :use ((:instance mod012))))))

(local
(defun log-induct (x y)
  (if (and (integerp x) (>= x 0)
	   (integerp y) (>= y 0))
      (if (or (= x 0) (= y 0))
	  ()
	(log-induct (fl (/ x 2)) (fl (/ y 2))))
    ())))

(defthm LOGXOR-X-X
    (implies (and (integerp x) (>= x 0))
	     (equal (logxor x x) 0))
  :hints (("Goal" :in-theory (disable logxor)
		  :induct (log-induct x x))
	  ("Subgoal *1/2" :use ((:instance logxor-x-x-1)
				  (:instance fl-mod-equal (x 0) (y (logxor x x)))))))

(local
(defun op-dist-induct (x y n)
  (if (and (integerp n) (>= n 0))
      (if  (= n 0)
	  (list x y)
	(op-dist-induct (fl (/ x 2)) (fl (/ y 2)) (1- n)))
    ())))

(in-theory (disable logxor))

(defthm BITN-LOGXOR
    (implies (and (case-split (integerp x))
                  (case-split (>= x 0))
                  (case-split (integerp y))
                  (case-split (>= y 0))
                  (case-split (integerp n))
                  (case-split (>= n 0)))
	     (= (bitn (logxor x y) n)
		(logxor (bitn x n) (bitn y n))))
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/1" :use ((:instance logxor-mod (i x) (j y))
				(:instance bitn-alt-0)
				(:instance bitn-alt-0 (x y))
				(:instance bitn-alt-0 (x (logxor x y)))))
	  ("Subgoal *1/2" :use ((:instance bitn-alt-pos (k n))
				(:instance bitn-alt-pos (k n) (x y))
				(:instance bitn-alt-pos (k n) (x (logxor x y)))
				(:instance logxor-fl (i x) (j y))))))

(in-theory (disable bitn-logxor)) ;why?

(defthm LOGXOR<2**N-better
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (< y (expt 2 n)))
	     (<= (logxor x y) (+ -1 (expt 2 n))))
  :rule-classes ()
  :hints (("Goal" :induct (op-dist-induct x y n))
	  ("Subgoal *1/2" :in-theory (set-difference-theories
                                      (enable expt-split)
                                      '(a15))
           :use ((:instance logxor-def)
                 (:instance mod012)
                 (:instance mod012 (x y))))))

(defthm LOGXOR<2**N
    (implies (and (integerp x) (>= x 0)
		  (integerp y) (>= y 0)
		  (integerp n) (>= n 0)
		  (< x (expt 2 n))
		  (< y (expt 2 n)))
	     (< (logxor x y) (expt 2 n)))
    :hints (("Goal" :use LOGXOR<2**N-better))
  :rule-classes ()
)

(local (include-book "mod2"))
(local (include-book "mod"))

(defthm COMP1-BNDS
  (and (< (comp1 x n) (expt 2 n))
       (>= (comp1 x n) 0))
  :rule-classes ())
