#|

   Fully Ordered Finite Sets, Version 0.81
   Copyright (C) 2003, 2004 by Jared Davis <jared@cs.utexas.edu>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of 
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public Lic-
   ense along with this program; if not, write to the Free Soft-
   ware Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.



 membership.lisp

  We are now getting into more interesting territory.  The primitive
  set functions nicely contain the potential messes that we would 
  have to face if we were implementing sets just using the primitive 
  list functions.  However, they are still plagued with order.

  The end user of the set library should not have to care that the 
  elements of the set are ordered or not.  (Well, with the exception
  that if they are trying to make a fast version of a function, they 
  might decide to exploit that order.)  

  Set reasoning should be done in terms of membership.  In tradition-
  al mathematics, subsets are defined through quantification over 
  membership, as is set equailty, and so forth.  But right now, the
  best we can do is induct over insert, which is somewhat messy.

  This file introduces the notions of set membership and subset (And,
  along the way, deletion).  The goal is to transform the insertion
  induction proofs into more traditional pick-a-point and double con-
  tainment proofs.  

  At the end of this file, we will disable all of the theorems that 
  pertain to the order of elements, providing an entirely membership-
  based reasoning environment for the outer level.

|#

(in-package "SETS")
(include-book "primitives")
(set-verify-guards-eagerness 2)
(set-state-ok t)



; First we introduce the set membership function, and show that a set
; can never be an element of itself.  We then show several easy prop-
; erties of membership, none of which should be surprising.
;   Some comments on efficiency.  We could go ahead and write another
; version of in, which could use the total order to stop early if it
; ever encountered an element too big.  I.e., looking for 1 in the 
; list '(2 3 4), it could say that since 1 << 2, we are done.  
;   Should we do so?  Really the only question is whether or not it 
; would be faster.  Certainly we can contrive situations in which it
; would be better, i.e. (in 1 '(2 3 4 .... 100000)), where we would 
; save 100,000 calls to in.  But we can also contrive situations that
; it would be slower, for example (in 100001 '(1 2 3 4 ... 100000)), 
; where we would incur the extra cost of 100,000 calls to <<.  
;   I have arbitrarily decided not to implement short-circuiting.  My
; reasoning is that (1) it is not clear which would be faster, (2) it
; is not clear what "typical" usage behavior of in would be, so even
; if we wanted to benchmark the two solutions, we could probably not
; come up with a good benchmarking suite, (3) both solutions are O(n) 
; anyway so I don't think there's much to be gained here, and (4) the
; current method is arguably "no less efficient" than an unordered 
; implementation.

(defun in (a X)
  (declare (xargs :guard (setp X)))
  (and (not (empty X))
       (or (equal a (head X))
	   (in a (tail X)))))

(defthm in-type
  (or (equal (in a X) t) 
      (equal (in a X) nil))
  :rule-classes :type-prescription)

(local (defthmd not-in-smaller
  (implies (> (acl2-count x) (acl2-count y))
           (not (in x y)))))

(defthm not-in-self 
  (not (in x x))
  :hints(("Goal" :in-theory (enable not-in-smaller))))

(defthm in-sfix-cancel
  (equal (in a (sfix X)) (in a X)))

(defthm never-in-empty
  (implies (empty X) (not (in a X))))

(defthm in-set
  (implies (in a X) (setp X)))

(defthm in-tail
  (implies (in a (tail X)) (in a X)))

(defthm in-tail-or-head
  (implies (and (in a X) 
                (not (in a (tail X))))
           (equal (head X) a)))



; In order to move away from set order, we need to be able to state
; theorems about membership instead.  Importantly, we will show that
;  - the head of the set is not an element of the tail,
;  - inserting an already-member does not change the set, and
;  - the very nice relationship in-insert.

(defthm head-minimal
  (implies (<< a (head X))
	   (not (in a X)))
  :hints(("Goal" :in-theory (enable primitive-order-theory))
         ("Subgoal *1/2" :cases ((head X)))))

(defthm head-minimal-2
  (implies (in a X)
	   (not (<< a (head X)))))

(defthm head-unique
  (not (in (head X) (tail X)))
  :hints(("Goal" 
          :in-theory (enable primitive-order-theory)
          :cases ((<< (head X) (head (tail X)))))))
         

(defthm insert-identity
  (implies (in a X)
           (equal (insert a X) X))
  :hints(("Goal" :in-theory (enable insert-induction-case))))

(defthm in-insert
  (equal (in a (insert b X))
         (or (in a X) 
             (equal a b)))
  :hints(("Goal" :in-theory (enable primitive-order-theory)
                 :induct (insert b X))
         ("Subgoal *1/1" :use (:instance tail-insert-empty
                                         (a b) (X X)))))



; The behavior of insert is determined by the set order.  Yet, we 
; often need to induct upon insert's definition to prove theorems.
; Still, we do not want to reason about the set order.  So, here I 
; am going to introduce a new induction scheme which hides the set
; order, yet still allows us to induct on insert very nicely.  We
; will finally disable the induction scheme that insert normally 
; uses, and set up an induction hint so that weak-insert-induction
; will automatically be tried.
 
(defthm weak-insert-induction-helper-1
  (implies (and (not (in a X))
                (not (equal (head (insert a X)) a)))
           (equal (head (insert a X)) (head X)))
  :hints(("Goal" :in-theory (enable primitive-order-theory))))

(defthm weak-insert-induction-helper-2
  (implies (and (not (in a X))
                (not (equal (head (insert a X)) a)))
           (equal (tail (insert a X)) (insert a (tail X)))) 
  :hints(("Goal" :in-theory (enable primitive-order-theory))))

(defthm weak-insert-induction-helper-3
  (implies (and (not (in a X))
                (equal (head (insert a X)) a))
           (equal (tail (insert a X)) (sfix X)))
  :hints(("Goal" :in-theory (enable primitive-order-theory))))

(defun weak-insert-induction (a X)
  (declare (xargs :guard (setp X)))
  (cond ((empty X) nil)
        ((in a X) nil)
        ((equal (head (insert a X)) a) nil)
        (t (list (weak-insert-induction a (tail X))))))

(in-theory (disable (:induction insert)))

(defthm use-weak-insert-induction t 
  :rule-classes ((:induction 
                  :pattern (insert a X)
                  :scheme (weak-insert-induction a X))))



; Now we introduce subset.  This is complicated because we want to 
; use MBE to make it fast.  The fast-subset function is a tail re-
; cursive, linear pass through both lists.  Subset, by comparison,
; is a nice to reason about definition and much simpler, but would
; require quadratic time if we didn't use MBE here.  

(defun fast-subset (X Y)
  (declare (xargs :guard (and (setp X) (setp Y))))
  (cond ((empty X) t)
        ((empty Y) nil)
        ((<< (head X) (head Y)) nil)
        ((equal (head X) (head Y)) (fast-subset (tail X) (tail Y)))
        (t (fast-subset X (tail Y)))))

(defun subset (X Y)
  (declare (xargs :guard (and (setp X) (setp Y))
                  :verify-guards nil))
  (mbe :logic (or (empty X)
                  (and (in (head X) Y)
                       (subset (tail X) Y)))
       :exec (fast-subset X Y)))

(defthm subset-type
  (or (equal (subset X Y) t)
      (equal (subset X Y) nil))
  :rule-classes :type-prescription)


(local (defthmd fse-lemma-1
  (implies (not (in (head Y) X))
           (equal (subset X Y) (subset X (tail Y))))))

(local (defthmd fast-subset-equivalence
  (implies (and (setp X) (setp Y))
           (equal (fast-subset X Y) (subset X Y)))
  :hints(("Goal" :in-theory (enable primitive-order-theory)
                 :induct (fast-subset X Y))
         ("Subgoal *1/5" :use (:instance fse-lemma-1 (X X) (Y Y)))
         ("Subgoal *1/4" :use (:instance fse-lemma-1 (X (tail X)) (Y Y))))))

(verify-guards subset 
  :hints(("Goal" :in-theory (enable fast-subset-equivalence))))




; We now show the basic properties of subset.  The first theorems are
; mundane but then we get more interesting, showing that subset is 
; reflexive and transitive.  The goal here is to build up sufficient
; theorems to do pick-a-point proofs, which come next.

(defthm subset-sfix-cancel-X
  (equal (subset (sfix X) Y) (subset X Y)))

(defthm subset-sfix-cancel-Y
  (equal (subset X (sfix Y)) (subset X Y)))

(defthm empty-subset
  (implies (empty X) (subset X Y)))

(defthm subset-in
  (implies (and (subset X Y) (in a X))
           (in a Y)))

(defthm subset-in-2
  (implies (and (subset X Y) (not (in a Y)))
           (not (in a X))))

(local (defthmd tail-empty-insert-empty 
  (implies (empty X)
           (empty (tail (insert a X))))))

(defthm subset-insert-X
  (equal (subset (insert a X) Y)
	 (and (subset X Y)
              (in a Y)))
  :hints(("Goal" 
          :in-theory (enable primitive-order-theory))
         ("Subgoal *1/2" 
	  :cases ((<< a (head X)) (in a X)))
	 ("Subgoal *1/2" 
	  :in-theory (disable subset-in)
	  :use (:instance subset-in (a a) (X (insert a X)) (Y Y)))
         ("Subgoal *1/1" 
          :use (:instance tail-empty-insert-empty (a a) (X X)))))
          
(defthmd subset-double-insert
  (implies (subset X Y) 
           (subset (insert a X) (insert a Y))))

(defthm subset-reflexive
  (subset X X)
  :hints(("Subgoal *1/2"
	  :use (:instance subset-double-insert
			  (a (head X)) 
			  (X (tail X)) 
			  (Y (tail X))))))

(defthm subset-transitive
  (implies (and (subset X Y) (subset Y Z))
           (subset X Z)))

(defthm subset-membership-tail
  (implies (and (subset X Y) (in a (tail X)))
           (in a (tail Y)))
  :hints(("Goal" :in-theory (enable primitive-order-theory)
          :use ((:instance subset-in (a a) (X X) (Y Y))
                (:instance in-tail-or-head (a a) (X Y))))))
          
(defthm subset-membership-tail-2
  (implies (and (subset X Y) (not (in a (tail Y))))
           (not (in a (tail X))))
  :hints(("Goal" :in-theory (disable subset-membership-tail)
                 :use (:instance subset-membership-tail))))


; We now turn our attention to enabling users to conduct proofs of 
; subset using a "pick a point" style.  In traditional mathematics, 
; subset is defined using quantification over members, as follows:
;
;    (subset X Y) iff "forall a : (in a X) implies (in a Y)"
; 
; Then, traditionally, one would prove the membership part of this 
; statement, and use it to conclude whether or not (subset X Y) is
; true.  But we cannot write, in ACL2, a theorem of the following
; form, because we do not really have quantification to work with:
;
;    "forall a : (in a ...) implies (in a ___)" => (subset ... ___)
;
; However, we *can* consider the contrapositive of this theorem, 
; which looks like the following:
;
;    ~(subset ... ___) => "exists a : (in a ...) ^ ~(in a ___)
; 
; We cannot do this directly, but we can write a function to find
; such an element if it exists.  This function is subset-witness,
; and we can define it without much trouble.  Its "signature" would
; be subset-witness : set * set -> bool * element.  The first value
; returned is a "status" -- a boolean value which answers if such an
; element exists.  The second value returned is the actual element, 
; if one can be found.
;
; Without further adieu, here is the subset-witness function and a
; few theorems which characterize its behavior.

(defun subset-witness (X Y)
  (declare (xargs :guard (and (setp X) (setp Y))))
  (cond ((empty X) (list nil nil))
        ((in (head X) Y) (subset-witness (tail X) Y))
        (t (list t (head X)))))

(defthm witness-means-subset
  (equal (car (subset-witness X Y))
         (not (subset X Y))))

(defthm witness-element-X
  (implies (car (subset-witness X Y))
           (in (cadr (subset-witness X Y)) X)))

(defthm witness-element-Y
  (implies (car (subset-witness X Y))
           (not (in (cadr (subset-witness X Y)) Y))))




; The point of introducing subset-witness is so that we can conclude 
; subset relationships just by having facts about membership.  Here 
; we prove the general theorem subset-by-membership, which shows the
; following:
;
;    LET:  (sbm-hyps) be a set of hypotheses,
;          (sbm-sub) be some object, and
;          (sbm-super) be some other object
;
;    IF:   under the hypotheses in (sbm-hyps), the objects satisfy
;          the following constraint:
;            (in a (sbm-sub)) => (in a (sbm-super)
;
;    THEN: (sbm-sub) is a subset of (sbm-super).
;
; This generic theorem can then be functionally instantiated to prove
; similar theorems for concrete expressions.  Ignore for the moment 
; the use of pick-a-point-trigger for the moment, and pretend it is 
; just "subset".

(encapsulate
  (((sbm-sub) => *)
   ((sbm-super) => *)
   ((sbm-hyps) => *))

  (local (defun sbm-sub () nil))
  (local (defun sbm-super () nil))
  (local (defun sbm-hyps () t))

  (defthm membership-constraint-sbm-subset
    (implies (sbm-hyps)
      (implies (in do-not-reuse-this-name-1 (sbm-sub)) 
               (in do-not-reuse-this-name-1 (sbm-super)))))
)

(local (defthmd witness
  (implies (sbm-hyps)
           (not (car (subset-witness (sbm-sub) (sbm-super)))))
  :hints(("Goal" :use ((:instance witness-element-X 
                                  (X (sbm-sub)) (Y (sbm-super)))
                       (:instance witness-element-Y 
                                  (X (sbm-sub)) (Y (sbm-super))))))))

(defun pick-a-point-trigger (X Y)
  (declare (xargs :guard (and (setp X) (setp Y))))
  (subset X Y))

(defthm subset-by-membership
  (implies (sbm-hyps)
           (pick-a-point-trigger (sbm-sub) (sbm-super)))
  :hints(("Goal" :in-theory (enable witness)
                 :use (:instance witness-means-subset 
                                 (X (sbm-sub)) (Y (sbm-super))))))



; So what is this pick-a-point-trigger thing?  We are going to go 
; one step further than just providing an encapsulate.  And, instead
; of users having to manually use functional-instance hints, we are 
; going to try to automatically supply the correct hints at the cor-
; rect times.
;
; Our strategy will be the following.  If we see (subset A B) as the
; conclusion of a rule then we will rewrite this subset into 
; (pick-a-point-trigger A B).  Call this process "tagging".
;
; Then we add a computed hint which looks for instances of pick-a-
; point-trigger, and if it finds them suggests a hint that will fun-
; ctionally instantiate the above theorem equal-by-membership with 
; the appropriate arguments.  The actual workings of this hint are 
; complicated and shown below.
;
; rewriting-goal-lit: Make sure that we only tag goals -- that is, 
; that we do not tag expressions generated while backchaining.
;
; rewriting-conc-lit: Make sure that we only tag parts of the con-
; clusion, i.e. not hypotheses.

(defun rewriting-goal-lit (x mfc state)
  (declare (ignore x state)
           (xargs :mode :program))
  (null (ACL2::mfc-ancestors mfc)))

(defun rewriting-conc-lit (fn x y mfc state)
  (declare (xargs :mode :program)
           (ignore state))
  (let ((clause (ACL2::mfc-clause mfc)))
    (member-equal `(,fn ,x ,y) (last clause))))

(defthm pick-a-point-subset-strategy
  (implies (and (syntaxp (rewriting-goal-lit x ACL2::mfc state))
                (syntaxp (rewriting-conc-lit 'subset x y ACL2::mfc state)))
           (equal (subset X Y) (pick-a-point-trigger X Y))))

(in-theory (disable pick-a-point-trigger))

; Now, what we are going to do next is create a computed hint that
; will look for instances of a trigger, and if it sees one, we will
; try to provide a functional instantiation hint.  This takes some 
; work.
;
; (harvest-function clause fn) -- extracts all terms from clause
; whose top level symbol is fn and returns them as a list.
;
; (remove-dupes harvested) -- removes any duplicate harvested clauses
; from the extracted clauses.  I'm not really sure this is necessary.
;
; (remove-functions clause harvested) -- removes all of the harvested
; functions from the original clause.  the remaining literals here
; will become hypotheses.
;
; The remaining literals are now in a disjunctive normal, i.e.:
;   (a ^ b ^ ...) => P  is  (~a v ~b v ... v P)
;   (a v b v ...) => P  is  subgoal1: (~a v P), sg2: (~b v P), ...
; In order to recover the hypotheses, we need to negate each of the
; literals as they occur here.  And, if there are more than one of
; them, we are going to AND their negations together.  This is done
; by the functions others-to-negated-list, and others-to-hyps.
;
; Finally we need to create the actual hint. We might have several
; harvested functions found.  Harvest is now a list of functions,
; of the form (trigger lhs rhs).  So, we want to extract all of the
; lhs and rhs'es, and build hints for each pair.

(defun harvest-function (clause fn)
  (declare (xargs :mode :program))
  (if (endp clause)
      nil
    (if (equal (caar clause) fn)
        (cons (car clause) (harvest-function (cdr clause) fn))
      (harvest-function (cdr clause) fn))))

(defun remove-dupes (harvested)
  (declare (xargs :mode :program))
  (if (endp harvested)
      nil
    (if (member-equal (car harvested) (cdr harvested))
        (cdr harvested)
      (cons (car harvested) (cdr harvested)))))

(defun remove-functions (clause harvested)
  (declare (xargs :mode :program))
  (if (endp clause)
      nil
    (if (member-equal (car clause) harvested)
        (remove-functions (cdr clause) harvested)
      (cons (car clause) (remove-functions (cdr clause) harvested)))))

(defun others-to-negated-list (others)
  (declare (xargs :mode :program))
  (if (endp others)
      nil
    (if (equal (caar others) 'not)  ; don't create double nots
        (cons (second (car others))
              (others-to-negated-list (cdr others)))
      (cons (list 'not (car others))
            (others-to-negated-list (cdr others))))))

(defun others-to-hyps (others)
  (declare (xargs :mode :program))
  (if (endp others)
      t
    (let ((negated (others-to-negated-list others)))
      (if (endp (cdr negated))  ; don't and singletons
          (car negated)
        (cons 'and (others-to-negated-list others))))))

(defun build-subset-hints (hyps harvest)
  (declare (xargs :mode :program))
  (if (endp harvest)
      nil
    (cons `(:functional-instance subset-by-membership
              (sbm-hyps  (lambda () ,hyps))
              (sbm-sub   (lambda () ,(second (first harvest))))
              (sbm-super (lambda () ,(third (first harvest)))))
          (build-subset-hints hyps (rest harvest)))))


(defconst *pick-a-point-docs*
  "~%NOTE:  Pick-a-Point Proof of Subset ~%~
  We suspect that this subset would best be proved by appealing to ~
  a membership argument.  That is, rather than try to directly show ~
  that this subset relationship holds, we will functionally ~
  instantiate the theorem subset-by-membership with the following ~
  hint: ~
  ~%~%~x0~%~
  Membership arguments are often a good way to prove that subsets ~
  hold.  If the proof fails but you think that you still want to ~
  use a membership argument, then you should try to prove that for ~
  each instance of subset-by-membership above, you can show: ~%     ~
    (implies (and (sbm-hyps)~%                  ~
                  (in a (sbm-sub)))~%             ~
             (in a (sbm-super)))~%~%~
  If it turns out that you do not want to use a membership argument, ~
  then you can explicitly disable this strategy with the following ~
  hint: (in-theory (disable SETS::pick-a-point-subset-strategy)). ~
  You can alternately turn off the subset strategy completely by ~
  invoking the macro, (SETS::disable-set-reasoning).~%")

(defun pick-a-point-subset-hint (id clause world stable)
  (declare (xargs :mode :program)
           (ignore world))
  (if (not stable)
      nil
    (let ((harvest (remove-dupes
                     (harvest-function clause 'pick-a-point-trigger))))
      (if (not harvest)
          nil 
        (let ((hints `(:use ,(build-subset-hints
                        (others-to-hyps (remove-functions clause harvest))
                        harvest))))
          (prog2$
             (ACL2::cw *pick-a-point-docs*
                       (cons (ACL2::string-for-tilde-@-clause-id-phrase id)
                             hints))
             hints))))))



(set-default-hints 
  '((pick-a-point-subset-hint id
                              clause
                              world
                              stable-under-simplificationp)))




; Proofs of equality by double containment are also very common.  So,
; to support these, we want to show that double containment is the 
; same as equality.  
;
; The general argument is the following:
;
;  Suppose that we have two sets which are subsets of one another, 
;  i.e. (subset X Y) and (subset Y X) are true.  First, we will show
;  that (head X) = (head Y).  Next we will show that (in a (tail X))
;  implies that (in a (tail Y)).  This fact is then used for a sub-
;  set by membership argument to show that (tail X) = (tail Y).
;  Now, (head X) = (head Y) and (tail X) = (tail Y) can be used 
;  together to show that X = Y (see primitives.lisp, head-tail-same)
;  so we are done.  
;
; Here are the details.  First we show that the heads are the same:

(local (defthmd double-containment-lemma-head
  (implies (and (subset X Y)
                (subset Y X))
           (equal (head X) (head Y)))
  :hints(("Goal" :in-theory (enable primitive-order-theory)))))


; Next we show that (tail X) is a subset of (tail Y), using a subset
; by membership argument:

(local (defthmd in-tail-expand
  (equal (in a (tail X))
         (and (in a X)
              (not (equal a (head X)))))))

(local (defthmd double-containment-lemma-in-tail
  (implies (and (subset X Y)
                (subset Y X))
           (implies (in a (tail X))     ; could be "equal" instead, 
                    (in a (tail Y))))   ; but that makes loops. :(
  :hints(("Goal"
         :in-theory (enable primitive-order-theory)
         :use ((:instance in-tail-expand (a a) (X X))
               (:instance in-tail-expand (a a) (X Y)))))))

(local (defthmd double-containment-lemma-tail 
  (implies (and (subset X Y)
                (subset Y X))
           (subset (tail X) (tail Y)))
  :hints(("Goal" :in-theory (e/d (double-containment-lemma-in-tail) 
                                 (subset))))))

                             

 
; Finally, we are ready to show that double containment is equality.
; To do this, we need to induct in such a way that we consider the 
; tails of X and Y.  Then, we will use our fact that about the tails
; being subsets of one another in the inductive case.

(local (defun double-tail-induction (X Y)
  (declare (xargs :guard (and (setp X) (setp Y))))
  (if (or (empty X) (empty Y))
      (list X Y)
    (double-tail-induction (tail X) (tail Y)))))

(local (defthm double-containment-is-equality-lemma
  (IMPLIES (AND (NOT (OR (EMPTY X) (EMPTY Y)))
                (IMPLIES (AND (SUBSET (TAIL X) (TAIL Y))
                              (SUBSET (TAIL Y) (TAIL X)))
                         (EQUAL (EQUAL (TAIL X) (TAIL Y)) T))
                (SETP X)
                (SETP Y)
                (SUBSET X Y)
                (SUBSET Y X))
           (EQUAL (EQUAL X Y) T))
  :hints(("Goal" 
          :use ((:instance double-containment-lemma-tail
                           (X X) (Y Y))
                (:instance double-containment-lemma-tail 
                           (X Y) (Y X))
                (:instance double-containment-lemma-head
                           (X X) (Y Y)))))))


(local (defthmd double-containment-is-equality
  (implies (and (setp X) 
		(setp Y)
                (subset X Y)
                (subset Y X))
           (equal (equal X Y) t))
  :hints(("Goal" :induct (double-tail-induction X Y)))))

(defthm double-containment 
  (implies (and (setp X)
                (setp Y))
           (equal (equal X Y)
                  (and (subset X Y)
                       (subset Y X))))
  :hints(("Goal" :use (:instance double-containment-is-equality))))


; Finally, we would like to add this strategy as a default hint, to 
; be tried on every theorem where it could possibly apply.  This is 
; unfortunately not an embeddable event, and so a user will have to 
; manually type "(enable-set-reasoning)" at the top of their files.
;
; Advanced users who are using other computed hints may want more 
; control than this, and will have to manually add the computed hint
; in their list.

(defmacro enable-set-reasoning ()
  `(progn
      (set-default-hints 
         '((pick-a-point-subset-hint id
                                     clause
                                     world
                                     stable-under-simplificationp)))
      (local (in-theory (enable pick-a-point-subset-strategy)))))

(defmacro disable-set-reasoning ()
  `(progn
      (set-default-hints nil)
      (local (in-theory (disable pick-a-point-subset-strategy)))))


(disable-set-reasoning)

; We are now done with the membership level.  We disable all of the 
; order based reasoning that we introduced here.  

(in-theory (disable head-minimal
                    head-minimal-2))


