/*****************************************************************
/
/ File   :   ifdhandler.c
/ Author :   David Corcoran <corcoran@linuxnet.com>
/ Date   :   June 15, 2000
/ Purpose:   This provides reader specific low-level calls.
/            Alot of the functionality listed in the specs is
/            not done.  I've done a minimum to get started.
/            See http://www.linuxnet.com for more information.
/ License:   See file LICENSE
/
******************************************************************/

#include "config.h"
#include "pcscdefines.h"
#include "ifdhandler.h"
#include "AdmHndlr.h"
#include "T0Hndlr.h"
#include "T1Hndlr.h"
#include <stdio.h>
#include <string.h>

#include "MCU_ATR.h"

static struct _IFDCard {
  UCHAR	 Atr[MAX_ATR_SIZE];
  DWORD  AtrLength;
} IFDCard;

RESPONSECODE IFDHCreateChannel ( DWORD Lun, DWORD Channel ) {

  ULONG rv = STATUS_UNSUCCESSFUL;
  static int acsinit = 0;

  if ( acsinit == 0 ) {
   acsinit = 1;
   rv = Adm_Initialize("usb", Lun, Channel);
  }
  
  if ( rv == STATUS_SUCCESS ) {
    return IFD_SUCCESS;
  } else {
    return IFD_COMMUNICATION_ERROR;
  }
  
}


RESPONSECODE IFDHCloseChannel ( DWORD Lun ) {
  
  ULONG rv;

  rv = Adm_UnInitialize(Lun);

  if ( rv == STATUS_SUCCESS ) {
    return IFD_SUCCESS;   
  } else {
    return IFD_COMMUNICATION_ERROR;
  }

}

RESPONSECODE IFDHGetCapabilities ( DWORD Lun, DWORD Tag, 
				   PDWORD Length, PUCHAR Value ) {
  
  switch ( Tag ) {

  case TAG_IFD_ATR:
    memcpy( Value, IFDCard.Atr, IFDCard.AtrLength );
    *Length = IFDCard.AtrLength;
    break;

  default:
    return IFD_ERROR_TAG;

  }

  return IFD_SUCCESS;
}

RESPONSECODE IFDHSetCapabilities ( DWORD Lun, DWORD Tag, 
				   DWORD Length, PUCHAR Value ) {

  return IFD_SUCCESS;
}

RESPONSECODE
IFDHSetProtocolParameters(	DWORD	Lun,
							DWORD	Protocol,
							UCHAR	Flags,
							UCHAR	PTS1,
							UCHAR	PTS2,
							UCHAR	PTS3)
{
	UCHAR			fl;
	UCHAR			dl;
	UCHAR			pps[100];
	ULONG			ppsLen;
	ULONG			rv;
	RESPONSECODE	result;

	/* Original code -- obsolete --
	pps[0]	= 0xFF;
	pps[1]	= (Flags << 4) | (0x0F & Protocol);
	ppsLen	= 2;
	*/

#ifdef PCSC_DEBUG
	printf("%s: IFDHSetProtocolParameters: Enter\n",__FILE__);
#endif

	/* ---- 1.4 fix (SLE4428) starts here ---- */
	// Fixed:	Simply return success when ATR 3B 00 is found.
	if(!memcmp(IFDCard.Atr,"\x3B\x00",2*sizeof(UCHAR))) {

#ifdef PCSC_DEBUG
		printf("IFDHSetProtocolParameters: Reader emulated ATR -- Memory Card Inserted\n");
#endif
		return IFD_SUCCESS;
	}
	/* ---- 1.4 fix (SLE4428) ends here ---- */

	/* ---- 1.3 fix (PPS) starts here ---- */
	// Fixed: 	Protocol error
	//			Flags error
	//			PTS1 error

#define SCARD_PROTOCOL_T0		0x0001
#define SCARD_PROTOCOL_T1		0x0002

	ULONG 		nProtocol = Protocol==SCARD_PROTOCOL_T0?0:1;

	Flags	= 0x10;		// PPS1 is always present for ACR38

	pps[0]	= 0xFF;
	pps[1]	= Flags | (0x0F & nProtocol);
	ppsLen	= 2;

	if(PTS1 == 0x00) {
		// case when PCSC-LITE does not provide PTS1
		// use the default speed as reported in ATR then
		MCU_ATR_RESULT	mcuATRResult;
		MCU_ATR			mcuATR;
		UCHAR			mcuATRTA1;

#ifdef PCSC_DEBUG
		printf("%s: PCSC-LITE does not provide PTS1 -> use default\n",__FILE__);
#endif
		mcuATRResult = MCUAtrInit(&mcuATR, IFDCard.Atr, IFDCard.AtrLength);
		if (mcuATRResult != MCU_ATR_OK)
		{
#ifdef PCSC_DEBUG
			printf("%s: MCUAtrInit failed, 0x%X\n",__FILE__,mcuATRResult);
#endif
			result = STATUS_DATA_ERROR;
			return result;
		}

		mcuATRResult = MCUAtrGetInterfaceByte(	&mcuATR,
												1,
												MCU_ATR_INTERFACE_TA,
												&mcuATRTA1);
		if (mcuATRResult != MCU_ATR_OK)
		{
#ifdef PCSC_DEBUG
			printf("%s: MCUAtrGetInterfaceByte failed, 0x%X\n",__FILE__,mcuATRResult);
#endif
			result = STATUS_DATA_ERROR;
			MCUAtrCleanUp(&mcuATR);
			/* ---- 1.7.1 fix (Default F/D = 0x11 if TA1 is absent) starts here ---- */
#if 0
			return result;
#endif
			mcuATRTA1 = 0x11;
			/* ---- 1.7.1 fix (Default F/D = 0x11 if TA1 is absent) ends here ---- */
		}

		PTS1 = mcuATRTA1;
#ifdef PCSC_DEBUG
		printf("PTS1 = 0x%02X\n",PTS1);
#endif
	}
	/* ---- 1.3 fix (PPS) ends here ---- */

	if ((Flags & 0x10) == 0x10)
	{
		fl = (PTS1 & 0xF0) >> 4;
		dl = (PTS1 & 0x0F);
		if (Adm_SupportPPS(fl, dl) == 0)
		{
#ifdef PCSC_DEBUG
			printf("%s: Requested PPS is not supported\n",__FILE__);
#endif
			result = IFD_COMMUNICATION_ERROR;
			goto ReaderNotSupport;
		}

		pps[ppsLen] = PTS1;
		ppsLen++;
	}

	if ((Flags & 0x20) == 0x20)
	{
		pps[ppsLen] = PTS2;
		ppsLen++;
	}

	if ((Flags & 0x40) == 0x40)
	{
		pps[ppsLen] = PTS3;
		ppsLen++;
	}

	/* ---- 1.3 fix (PPS) starts here ---- */
	// Fixed:	Add the missed PPS_CheckSum
	pps[ppsLen]	= pps[0]^pps[1]^pps[2];	// PPS_CheckSum
	ppsLen++;
	/* ---- 1.3 fix (PPS) ends here ---- */

	/* ---- 1.7 fix (Skip PPS when not needed) starts here ---- */
	{
	ULONG	uSupPrtcl = 0;
	UCHAR	TD = IFDCard.Atr[1];
	UCHAR	idx = 1;
	UCHAR	bShf = 0;

	do {

		for(bShf=4;bShf<8;bShf++) {

			if((TD>>bShf)&0x01)
				idx++;
		}
		
		if((TD&0x80) == 0)
			break;
			
		TD = IFDCard.Atr[idx];
		
		if((TD&0x0F) == 0)
			uSupPrtcl |= SCARD_PROTOCOL_T0;
		else if((TD&0x0F) == 1)
			uSupPrtcl |= SCARD_PROTOCOL_T1;
			
	}while(idx < 36);

#if PCSC_DEBUG
	printf("%s: This card supports %s %s protocol\n",
	__FILE__,
	uSupPrtcl&SCARD_PROTOCOL_T0?"T0":" ",
	uSupPrtcl&SCARD_PROTOCOL_T1?"T1":" ");
#endif

	if((uSupPrtcl == Protocol) &&
	   (uSupPrtcl == SCARD_PROTOCOL_T0 || uSupPrtcl == SCARD_PROTOCOL_T1) &&
	   (fl == 0x01 && dl == 0x01)) {
		
	   	/* Necessary Conditions:	Card supports 1 protocol
	   								Fl/Dl indicates default baudrate
									Selected protocol = supported protocol
	   				  
	   	   Action:	Skip issuing PPS and return success to PCSCLITE
		*/

#if PCSC_DEBUG
		printf("%s: There is no need to issue PPS for this card -> Skipping PPS\n",
		__FILE__);
#endif
		return IFD_SUCCESS;
	}
	}
	/* ---- 1.7 fix (Skip PPS when not needed) ends here ---- */

	rv = Adm_DoPPSExchange(Lun, pps, ppsLen);
	if (rv != STATUS_SUCCESS)
	{
		result = IFD_COMMUNICATION_ERROR;
		goto Adm_DoPPSExchangeFailed;
	}

	return IFD_SUCCESS;

/****** Error Part ******/
Adm_DoPPSExchangeFailed:
ReaderNotSupport:
	return result;
}


RESPONSECODE
IFDHPowerICC(	DWORD	Lun,
				DWORD	Action,
				PUCHAR	Atr,
				PDWORD	AtrLength)
{
	ULONG rv;

	if(Action == IFD_POWER_UP)
	{
		rv = Adm_PowerICC(Lun, Atr, AtrLength);
		if (rv == STATUS_SUCCESS)
		{
			memcpy(IFDCard.Atr, Atr, *AtrLength);
			IFDCard.AtrLength = *AtrLength;
#if defined(DO_PPS)
			(void)Adm_DoPPSExchangeATR(	Lun,
										Atr,
										*AtrLength);
#endif /* defined(DO_PPS) */
			return IFD_SUCCESS;
		}
		else
		{
			return IFD_COMMUNICATION_ERROR;
		}
	}
	else if (Action == IFD_POWER_DOWN)
	{
		rv = Adm_UnPowerICC(Lun);
		if (rv == STATUS_SUCCESS)
		{
			return IFD_SUCCESS;
		}
		else
		{
			return IFD_COMMUNICATION_ERROR;
		}
	}
	else if (Action == IFD_RESET)
	{
		rv = Adm_ResetICC(Lun, Atr, AtrLength);
		if (rv == STATUS_SUCCESS)
		{
			memcpy(IFDCard.Atr, Atr, *AtrLength);
			IFDCard.AtrLength = *AtrLength;
#if defined(DO_PPS)
			(void)Adm_DoPPSExchangeATR(	Lun,
										Atr,
										*AtrLength);
#endif /* defined(DO_PPS) */
			return IFD_SUCCESS;
		}
		else
		{
			return IFD_COMMUNICATION_ERROR;
		}
	}
	else
	{
		return IFD_NOT_SUPPORTED;
	}

	return IFD_SUCCESS;
}

RESPONSECODE IFDHTransmitToICC ( DWORD Lun, SCARD_IO_HEADER SendPci, 
				 PUCHAR TxBuffer, DWORD TxLength, 
				 PUCHAR RxBuffer, PDWORD RxLength, 
				 PSCARD_IO_HEADER RecvPci ) {
  
  ULONG rv;
#ifdef PCSC_DEBUG
  int i;
#endif

#ifdef PCSC_DEBUG
   printf("T=%d -> ", SendPci.Protocol);
   for (i=0; i < TxLength; i++) {
     printf("%x ", TxBuffer[i]);
   } printf("\n");
#endif

  if ( SendPci.Protocol == 0 ) {
#ifdef PCSC_DEBUG
    printf("Doing T0_ExchangeData\n");
#endif
    rv = T0_ExchangeData( Lun, TxBuffer, TxLength, RxBuffer, RxLength );
  } else if ( SendPci.Protocol == 1 ) {
#ifdef PCSC_DEBUG
    printf("Doing T1_ExchangeData\n");
#endif
    rv = T1_ExchangeData( Lun, TxBuffer, TxLength, RxBuffer, RxLength );
  } else {
    return IFD_PROTOCOL_NOT_SUPPORTED;
  }

#ifdef PCSC_DEBUG
   printf("T=%d <- ", SendPci.Protocol);
   for (i=0; i < *RxLength; i++) {
     printf("%x ", RxBuffer[i]);
   } printf("\n");
#endif

  if ( rv == STATUS_SUCCESS ) {
    return IFD_SUCCESS;
  } else {
    return IFD_COMMUNICATION_ERROR;
  }
}

RESPONSECODE
IFDHControl(DWORD	Lun,
			PUCHAR	TxBuffer,
			DWORD	TxLength,
			PUCHAR	RxBuffer,
			PDWORD	RxLength)
{
	/* This function performs a data exchange with the reader
	 * (not the card) specified by Lun. Here XXXX will only be used.
	 * It is responsible for abstracting functionality such as PIN
	 * pads, biometrics, LCD panels, etc.  You should follow the MCT,
	 * CTBCS specifications for a list of accepted commands to
	 * implement.
	 * TxBuffer - Transmit data
	 * TxLength - Length of this buffer.
	 * RxBuffer - Receive data
	 * RxLength - Length of the received data.
	 * This function will be passed the length of the buffer RxBuffer
	 * and it must set this to the length of the received data.
	 *
	 * Notes:
	 * RxLength should be zero on error.
	 */

	ULONG	rv;

	rv = Adm_Control(Lun, TxBuffer, TxLength, RxBuffer, RxLength);
	if (rv == STATUS_SUCCESS)
	{
		return IFD_SUCCESS;
	}
	else
	{
		return IFD_COMMUNICATION_ERROR;
	}
}

RESPONSECODE IFDHICCPresence( DWORD Lun ) {

 ULONG rv;  

   rv = Adm_IsICCPresent(Lun);

   if ( rv == STATUS_SUCCESS ) {
      return IFD_ICC_PRESENT;      
   } else if ( rv == STATUS_UNSUCCESSFUL ) {
      return IFD_ICC_NOT_PRESENT;      
   } else {
      return IFD_COMMUNICATION_ERROR;
   }
}
