#include <adept/statefilter.h>

namespace adept {

StateFilterWidget::StateFilterWidget( QWidget *parent )
    : QWidget( parent )
{
    setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
    QGridLayout *l = new QGridLayout( this );

    l->setSpacing( 2 );

    l->addWidget( new QLabel( i18n( "State: " ), this ), 0, 0,
                  Qt::AlignRight );
    l->addWidget( m_notInstalled = new QLabel( this ), 0, 1 );
    l->addWidget( m_installed = new QLabel( this ), 0, 2 );
    l->addWidget( m_upgradable = new QLabel( this ), 0, 3 );

    l->addWidget( new QLabel( i18n( "Requested: " ), this ), 1, 0,
                  Qt::AlignRight);
    l->addWidget( m_keep = new QLabel( this ), 1, 1 );
    l->addWidget( m_install = new QLabel( this ), 1, 2 );
    l->addWidget( m_remove = new QLabel( this ), 1, 3 );
    l->addWidget( m_upgrade = new QLabel( this ), 1, 4 );

    int size = QFontMetrics( font() ).height() + 2;

    m_installed->setToolTip( i18n( "Show installed packages." ) );
    m_notInstalled->setToolTip(
        i18n( "Show available (not installed) packages." ) );
    m_upgradable->setToolTip( i18n( "Show upgradable packages." ) );

    m_keep->setToolTip(
        i18n( "Show packages without any changes requested." ) );
    m_install->setToolTip(
        i18n( "Show packages marked for installation." ) );
    m_remove->setToolTip( i18n( "Show packages marked for removal." ) );
    m_upgrade->setToolTip( i18n( "Show packages marked for upgrade." ) );

    m_icons[ m_installed ] = scaledIcon( "installed", size );
    m_icons[ m_notInstalled ] = scaledIcon( "notinstalled", size );
    m_icons[ m_upgradable ] = scaledIcon( "upgradable", size );
    m_icons[ m_keep ] = scaledIcon( "notinstalled", size );
    m_icons[ m_install ] = scaledIcon( "install", size );
    m_icons[ m_remove ] = scaledIcon( "remove", size );
    m_icons[ m_upgrade ] = scaledIcon( "upgrade", size );

    for ( QMap< QLabel *, QPixmap >::const_iterator it = m_icons.begin();
          it != m_icons.end(); ++it ) {
        it.key()->setPixmap( *it );
        it.key()->installEventFilter( this );
        m_checked[ it.key() ] = true;
    }
}

void StateFilterWidget::toggle( QLabel *l ) {
    m_checked[ l ] = !m_checked[ l ]; // invert
    if ( m_checked[ l ] ) {
        l->setPixmap( m_icons[ l ] );
    } else {
        QImage img = m_icons[ l ].toImage();
        KIconEffect::toGray( img, 1.0 );
        KIconEffect::toGamma( img, 1.5 );
        l->setPixmap( QPixmap::fromImage( img ) );
    }
}
void StateFilterWidget::setChecked( QLabel *l, bool e )
{
    if ( m_checked[ l ] != e )
        toggle( l );
}

bool StateFilterWidget::eventFilter( QObject *o, QEvent *e )
{
    QLabel *l = dynamic_cast< QLabel * >( o );
    if ( l ) {
        if ( e->type() == QEvent::MouseButtonRelease ) {
            toggle( l );
            changed();
        }
    }
    return false;
}

int StateFilterWidget::stateMask() {
    typedef StateFilter F;
    unsigned mask = 0;
    if ( m_checked[ m_installed ] ) mask |= F::Installed;
    if ( m_checked[ m_notInstalled ] ) mask |= F::NotInstalled;
    if ( m_checked[ m_upgradable ] ) mask |= F::Upgradable;
    if ( m_checked[ m_install ] ) mask |= F::Install;
    if ( m_checked[ m_remove ] ) mask |= F::Remove;
    if ( m_checked[ m_upgrade ] ) mask |= F::Upgrade;
    if ( m_checked[ m_keep ] ) mask |= F::Keep;
    return mask;
}

}
