# Code structure extractor for C language.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/report"
require "adlint/code"

module AdLint #:nodoc:
module C #:nodoc:

  class TypeDeclExtraction < CodeExtraction
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_typedef_declaration += method(:extract_typedef)
      visitor.enter_struct_type_declaration += method(:extract_struct)
      visitor.enter_union_type_declaration += method(:extract_union)
      visitor.enter_enum_type_declaration += method(:extract_enum)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract_typedef(typedef_declaration)
      TYPEDCL(typedef_declaration.identifier.location, "T",
              typedef_declaration.type.name, typedef_declaration.type.image)
    end

    def extract_struct(struct_type_declaration)
      TYPEDCL(struct_type_declaration.identifier.location, "S",
              struct_type_declaration.type.name,
              struct_type_declaration.type.image)
    end

    def extract_union(union_type_declaration)
      TYPEDCL(union_type_declaration.identifier.location, "U",
              union_type_declaration.type.name,
              union_type_declaration.type.image)
    end

    def extract_enum(enum_type_declaration)
      TYPEDCL(enum_type_declaration.identifier.location, "E",
              enum_type_declaration.type.name,
              enum_type_declaration.type.image)
    end
  end

  class GVarDeclExtraction < CodeExtraction
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:extract)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract(variable_declaration, variable)
      if variable.declared_as_extern?
        GVARDCL(variable_declaration.identifier.location,
                variable_declaration.identifier.value,
                variable_declaration.type.brief_image)
      end
    end
  end

  class FuncDeclExtraction < CodeExtraction
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_declared += method(:extract)
      interp.on_block_started += method(:enter_block)
      interp.on_block_ended += method(:leave_block)
      @block_level = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract(function_declaration, function)
      if function.declared_as_extern?
        FUNCDCL(function_declaration.identifier.location, "X",
                @block_level == 0 ? "F" : "B",
                FunctionIdentifier.new(function_declaration.identifier.value,
                                       function_declaration.signature.to_s))
      else
        FUNCDCL(function_declaration.identifier.location, "I",
                @block_level == 0 ? "F" : "B",
                FunctionIdentifier.new(function_declaration.identifier.value,
                                       function_declaration.signature.to_s))
      end
    end

    def enter_block(*)
      @block_level += 1
    end

    def leave_block(*)
      @block_level -= 1
    end
  end

  class VarDefExtraction < CodeExtraction
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:extract_variable)
      interp.on_parameter_defined += method(:extract_parameter)
      interp.on_block_started += method(:enter_block)
      interp.on_block_ended += method(:leave_block)
      @block_level = 0
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract_variable(variable_definition, variable)
      case
      when variable.declared_as_extern?
        VARDEF(variable_definition.identifier.location, "X",
               @block_level == 0 ? "F" : "B",
               storage_class_type(variable_definition.storage_class_specifier),
               variable_definition.identifier.value,
               variable_definition.type.brief_image)
      when variable.declared_as_static?
        VARDEF(variable_definition.identifier.location, "I",
               @block_level == 0 ? "F" : "B",
               storage_class_type(variable_definition.storage_class_specifier),
               variable_definition.identifier.value,
               variable_definition.type.brief_image)
      when variable.declared_as_auto?
        VARDEF(variable_definition.identifier.location, "I",
               @block_level == 0 ? "F" : "B",
               storage_class_type(variable_definition.storage_class_specifier),
               variable_definition.identifier.value,
               variable_definition.type.brief_image)
      when variable.declared_as_register?
        VARDEF(variable_definition.identifier.location, "I",
               @block_level == 0 ? "F" : "B",
               storage_class_type(variable_definition.storage_class_specifier),
               variable_definition.identifier.value,
               variable_definition.type.brief_image)
      end
    end

    def extract_parameter(parameter_definition, variable)
      return unless variable.named?

      VARDEF(parameter_definition.identifier.location, "I", "P",
             storage_class_type(parameter_definition.storage_class_specifier),
             parameter_definition.identifier.value,
             parameter_definition.type.brief_image)
    end

    def enter_block(*)
      @block_level += 1
    end

    def leave_block(*)
      @block_level -= 1
    end

    def storage_class_type(storage_class_specifier)
      return "N" unless storage_class_specifier

      case storage_class_specifier.type
      when :AUTO
        "A"
      when :REGISTER
        "R"
      when :STATIC
        "S"
      when :EXTERN
        "E"
      else
        "N"
      end
    end
  end

  class FuncDefExtraction < CodeExtraction
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_defined += method(:extract_function)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract_function(function_definition, function)
      case
      when function.declared_as_extern?
        FUNCDEF(function_definition.identifier.location, "X", "F",
                FunctionIdentifier.new(function_definition.identifier.value,
                                       function_definition.signature.to_s),
                function_definition.lines)
      when function.declared_as_static?
        FUNCDEF(function_definition.identifier.location, "I", "F",
                FunctionIdentifier.new(function_definition.identifier.value,
                                       function_definition.signature.to_s),
                function_definition.lines)
      end
    end
  end

  class LabelDefExtraction < CodeExtraction
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_generic_labeled_statement += method(:extract)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract(generic_labeled_statement)
      LABELDEF(generic_labeled_statement.label.location,
               generic_labeled_statement.label.value)
    end
  end

  class InitializationExtraction < CodeExtraction
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_definition += method(:extract)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract(variable_definition)
      if variable_definition.initializer
        INITIALIZATION(variable_definition.identifier.location,
                       variable_definition.identifier.value,
                       variable_definition.initializer.to_s)
      end
    end
  end

  class AssignmentExtraction < CodeExtraction
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_simple_assignment_expression += method(:extract)
      visitor.enter_compound_assignment_expression += method(:extract)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract(binary_expression)
      ASSIGNMENT(binary_expression.operator.location,
                 binary_expression.lhs_operand.to_s,
                 stringify(binary_expression))
    end

    def stringify(node)
      "#{node.operator.value} #{node.rhs_operand.to_s}"
    end
  end

  class FuncCallExtraction < CodeExtraction
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_started += method(:update_caller)
      @interp.on_function_call_expr_evaled += method(:extract_function_call)
      @caller = nil
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def update_caller(function_definition, function)
      @caller = function
    end

    def extract_function_call(function_call_expression, function,
                              arg_variables, result_variable)
      if @caller && function.named?
        CALL(function_call_expression.location,
             FunctionIdentifier.new(@caller.name, @caller.signature.to_s),
             FunctionIdentifier.new(function.name, function.signature.to_s))
      end
    end
  end

  class CrossRefExtraction < CodeExtraction
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:update_accessor)
      interp.on_function_ended += method(:clear_accessor)
      interp.on_variable_value_referred += method(:extract_variable_read)
      interp.on_variable_value_updated += method(:extract_variable_write)
      interp.on_function_referred += method(:extract_function_reference)
      @accessor = nil
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def update_accessor(function_definition, function)
      @accessor = function
    end

    def clear_accessor(function_definition, function)
      @accessor = nil
    end

    def extract_variable_read(expression, variable)
      if @accessor && variable.scope.global? && variable.named?
        XREF_VAR(expression.location,
                 FunctionIdentifier.new(@accessor.name,
                                        @accessor.signature.to_s),
                 "R", variable.name)
      end
    end

    def extract_variable_write(expression, variable)
      if @accessor && variable.scope.global? && variable.named?
        XREF_VAR(expression.location,
                 FunctionIdentifier.new(@accessor.name,
                                        @accessor.signature.to_s),
                 "W", variable.name)
      end
    end

    def extract_function_reference(expression, function)
      if @accessor && function.named?
        XREF_FUNC(expression.location,
                  FunctionIdentifier.new(@accessor.name,
                                         @accessor.signature.to_s),
                  "R", FunctionIdentifier.new(function.name,
                                              function.signature.to_s))
      end
    end
  end

  class LiteralExtraction < CodeExtraction
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_constant_specifier += method(:extract_constant)
      visitor.enter_string_literal_specifier += method(:extract_string_literal)
    end

    private
    def do_prepare(context) end
    def do_execute(context) end

    def extract_constant(constant_specifier)
      LIT(constant_specifier.location, type_of(constant_specifier),
          constant_specifier.prefix, constant_specifier.suffix,
          constant_specifier.constant.value)
    end

    def extract_string_literal(string_literal_specifier)
      LIT(string_literal_specifier.location, type_of(string_literal_specifier),
          string_literal_specifier.prefix, nil,
          string_literal_specifier.literal.value)
    end

    def type_of(node)
      case node
      when ConstantSpecifier
        case node.constant.value
        when /\A0x[0-9a-f]+[UL]*\z/i
          "HN"
        when /\A0b[01]+[UL]*\z/i
          "BN"
        when /\A0[0-9]+[UL]*\z/i
          "ON"
        when /\A[0-9]+[UL]*\z/i
          "DN"
        when /\A(?:[0-9]*\.[0-9]*E[+-]?[0-9]+|[0-9]+\.?E[+-]?[0-9]+)[FL]*\z/i,
             /\A(?:[0-9]*\.[0-9]+|[0-9]+\.)[FL]*\z/i
          "FN"
        when /\A'.*'\z/i
          "CN"
        when /\AL'.*'\z/i
          "CW"
        else
          "NA"
        end
      when StringLiteralSpecifier
        case node.literal.value
        when /\A".*"\z/i
          "SN"
        when /\AL".*"\z/i
          "SW"
        else
          "NA"
        end
      else
        raise TypeError
      end
    end
  end

end
end
