# Message detection classes for C preprocessor language.
#
# Author::    Rie Shima <mailto:rkakuuchi@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Rie Shima <mailto:rkakuuchi@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/report"
require "adlint/message"

module AdLint #:nodoc:
module Cpp #:nodoc:

  class W0687 < PassiveMessageDetection
    # NOTE: W0687 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0687

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_macro_undefined += method(:check)
    end

    private
    def check(undef_line, macro)
      if undef_line.identifier.value == "defined"
        W(:W0687, undef_line.identifier.location)
      end
    end
  end

  class W0688 < PassiveMessageDetection
    # NOTE: W0688 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0688

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_line_line += method(:check)
    end

    private
    def check(line_line)
      if line_no_arg = line_no_argument(line_line)
        line_no = Integer(line_no_arg.value)
        unless line_no > 0 && line_no < 32768
          W(:W0688, line_no_arg.location)
        end
      end
    rescue
    end

    def line_no_argument(line_line)
      line_line.pp_tokens ? line_line.pp_tokens.tokens.first : nil
    end
  end

  class W0689 < W0688
    # NOTE: W0689 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0689

    private
    def check(line_line)
      if fname_arg = file_name_argument(line_line)
        unless fname_arg.value =~ /\A".*"\z/
          W(:W0689, fname_arg.location)
        end
      end
    end

    def file_name_argument(line_line)
      line_line.pp_tokens ? line_line.pp_tokens.tokens[1] : nil
    end
  end

  class W0690 < W0688
    # NOTE: W0690 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0690

    private
    def check(line_line)
      if line_no_arg = line_no_argument(line_line)
        Integer(line_no_arg.value)
      end
    rescue
      W(:W0690, line_no_arg.location)
    end
  end

  class W0695 < W0687
    # NOTE: W0695 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0695

    private
    def check(undef_line, macro)
      if undef_line.identifier.value == "assert"
        W(:W0695, undef_line.identifier.location)
      end
    end
  end

  class W0806 < PassiveMessageDetection
    # NOTE: W0806 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0806

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_object_like_macro_defined += method(:check)
    end

    private
    def check(define_line, macro)
      if define_line.identifier.value == "defined"
        W(:W0806, define_line.identifier.location)
      end
    end
  end

  class W0807 < W0687
    # NOTE: W0807 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0807

    private
    def check(undef_line, macro)
      if macro.kind_of?(SpecialMacro)
        identifier = undef_line.identifier
        W(:W0807, identifier.location, identifier.value)
      end
    end
  end

  class W0808 < PassiveMessageDetection
    # NOTE: W0808 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0808

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_object_like_macro_defined += method(:check)
      @macro_table = context[:cpp_macro_table]
    end

    private
    def check(define_line, macro)
      identifier = define_line.identifier

      if @macro_table.lookup(identifier.value)
        W(:W0808, identifier.location, identifier.value)
      end
    end
  end

end
end
