# Analysis phases for C preprocessor language.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/phase"
require "adlint/monitor"
require "adlint/cpp/lexer"
require "adlint/cpp/eval"
require "adlint/cpp/code"
require "adlint/cpp/message"
require "adlint/cpp/message_shima"
require "adlint/cpp/util"

module AdLint #:nodoc:
module Cpp #:nodoc:

  class Prepare1Phase < Phase
    private
    def do_execute(context)
      monitored_region("pr1") do
        context[:c_source] = PreprocessedSource.new(context[:sources].first)

        context[:cpp_macro_table] = MacroTable.new
        context[:cpp_interpreter] = Preprocessor.new
        context[:cpp_visitor] = SyntaxTreeMulticastVisitor.new
      end
    end
  end

  class Prepare2Phase < Phase
    private
    def do_execute(context)
      monitored_region("pr2") do
        context[:cpp_commands] =
          setup_code_extractions(context) + setup_message_detections(context)
      end
    end

    def setup_code_extractions(context)
      [
        ObjectLikeMacroExtraction.new(context),
        FuncLikeMacroExtraction.new(context),
        IncludeDirectiveExtraction.new(context),
        DirectiveExtraction.new(context)
      ]
    end

    def setup_message_detections(context)
      [
        W0001.new(context),
        W0025.new(context),
        W0026.new(context),
        W0053.new(context),
        W0054.new(context),
        W0055.new(context),
        W0056.new(context),
        W0057.new(context),
        W0059.new(context),
        W0060.new(context),
        W0061.new(context),
        W0069.new(context),
        W0072.new(context),
        W0073.new(context),
        W0442.new(context),
        W0443.new(context),
        W0444.new(context),
        W0445.new(context),
        W0477.new(context),
        W0478.new(context),
        W0479.new(context),
        W0480.new(context),
        W0481.new(context),
        W0482.new(context),
        W0483.new(context),
        W0511.new(context),
        W0528.new(context),
        W0541.new(context),
        W0549.new(context),
        W0554.new(context),
        W0574.new(context),
        W0575.new(context),
        W0576.new(context),
        W0577.new(context),
        W0632.new(context),
        W0633.new(context),
        W0634.new(context),
        W0687.new(context),
        W0688.new(context),
        W0689.new(context),
        W0690.new(context),
        W0695.new(context),
        W0696.new(context),
        W0804.new(context),
        W0806.new(context),
        W0807.new(context),
        W0808.new(context),
        W0831.new(context),
        W0832.new(context),
        W9002.new(context)
      ]
    end
  end

  class EvalPhase < Phase
    private
    def do_execute(context)
      monitored_region("cpp") do
        pp_context = PreprocessContext.new(context)
        process_cinit_header(context, pp_context)
        process_pinit_header(context, pp_context)
        process_target_source(context, pp_context)
      end
    end

    def process_cinit_header(context, pp_context)
      if fpath = Traits.instance.of_compiler.initial_header
        init_header = Source.new(fpath)
      else
        init_header = EmptySource.new
      end
      syntax_tree = context[:cpp_interpreter].execute(pp_context, init_header)
      context[:cpp_syntax_tree] = syntax_tree
    end

    def process_pinit_header(context, pp_context)
      if fpath = Traits.instance.of_project.initial_header
        init_header = Source.new(fpath)
      else
        init_header = EmptySource.new
      end
      syntax_tree = context[:cpp_interpreter].execute(pp_context, init_header)
      context[:cpp_syntax_tree].concat(syntax_tree)
    end

    def process_target_source(context, pp_context)
      source = context[:sources].first
      syntax_tree = context[:cpp_interpreter].execute(pp_context, source)
      context[:cpp_syntax_tree].concat(syntax_tree)
    end
  end

  class SubstPhase < Phase
    private
    def do_execute(context)
      monitored_region("sub") do
        context[:c_source].substitute_code_blocks
      end
    end
  end

  class ReviewPhase < Phase
    private
    def do_execute(context)
      monitored_region("rv1") do
        context[:cpp_syntax_tree].accept(context[:cpp_visitor])
      end
    end
  end

  class CommandPhase < Phase
    private
    def do_execute(context)
      monitored_region("cm1") do
        context[:cpp_commands].each { |command| command.execute }
      end
    end
  end

end
end
