/*
Copyright (c) 2003 Bruno T. C. de Oliveira

LICENSE INFORMATION:
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
Copyright (c) 2002 Bruno T. C. de Oliveira

INFORMAES DE LICENA:
Este programa  um software de livre distribuio; voc pode
redistribu-lo e/ou modific-lo sob os termos da GNU General
Public License, conforme publicado pela Free Software Foundation,
pela verso 2 da licena ou qualquer verso posterior.

Este programa  distribudo na esperana de que ele ser til
aos seus usurios, porm, SEM QUAISQUER GARANTIAS; sem sequer
a garantia implcita de COMERCIABILIDADE ou DE ADEQUAO A
QUALQUER FINALIDADE ESPECFICA. Consulte a GNU General Public
License para obter mais detalhes (uma cpia acompanha este
programa, armazenada no arquivo COPYING).
*/

#include "document.h"
#include <stdbool.h>

#include <bores.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#define SUPPRESS_NEVER 0           /* never suppress newlines */
#define SUPPRESS_ONSAMEWIDTH 1     /* only when term width == output width */
#define SUPPRESS_ALWAYS 2          /* always suppress newlines */

bool clear_screen = false;
int suppress_newlines = SUPPRESS_NEVER; 
                           /* when to suppress newlines on output. See
                            * SUPPRESS_ constants */
bool use_color = true;
char *outputfile;
int termwidth;    /* attributed in main() from the COLUMNS env variable */

FILE *f;

Document *doc;    /* the document we are outputting */
int layer_number; /* the layer we will output; -1 means composite */
Layer *lyr;       /* the layer we are printing; NULL if printing composite */
int width, height;

struct decoded_cell_t {  /* represents a decoded document cell. Returned
                          * by decode_cell */

   int fg, bg;      /* the foreground and background color codes (4 bit, 0-15)
                     * for the cell */
   int ch;          /* the character in the cell */

   /* now some of the same information, but in a more broken-down form: */
   bool bold, blink;  /* bold and blink attributes */
   int fg3, bg3;       /* 3-bit color codes (0-7) for the foreground and
                        * background of the cell */
};

struct html_output_state_t {
   bool font_tag_open, blink_tag_open;  /* which html tags are open */
   int color;       /* The color we are currently outputting text in.
                     * This is a full 0-15 color code.*/
};

/* Extracts a cell from the appropriate layer (or build one, 
 * if we are dealing with a composite). Returns it. */
Cell get_cell(int x, int y) {
   if (lyr) return lyr->cells[x][y];
   else     return document_calc_effective_cell(doc, x, y);
}

/* Decodes a cell into components */
struct decoded_cell_t decode_cell(Cell cell) {
   struct decoded_cell_t dec;
   dec.fg     = (cell.attr & 0xf0) >> 4;
   dec.bg     = cell.attr & 0x0f;
   dec.ch     = cell.ch;

   dec.fg3   = (cell.attr & 0x70) >> 4;
   dec.bg3   = cell.attr & 0x07;
   dec.bold  = (cell.attr & 0x80) ? 1 : 0;
   dec.blink = (cell.attr & 0x08) ? 1 : 0;

   return dec;
};

void output_text(void) {
   int x, y;
   struct decoded_cell_t dec;
   int cur_fg = -1;
   int cur_bg = -1; 

   if (clear_screen)
      fputs("\e[2J\e[H", f);

   for (y = 0; y < height; y++) {
      for (x = 0; x < width; x++) {
         dec  = decode_cell(get_cell(x, y));

         /* set up color if necessary */
         if (use_color && (dec.fg != cur_fg || dec.bg != cur_bg)) {
            fputs("\e[", f);
            fputs(dec.bold ? "1;" : "0;", f);
            if (dec.blink) fputs("5;", f);
            fprintf(f, "%d;%dm", 30 + dec.fg3, 40 + dec.bg3);

            cur_fg = dec.fg;
            cur_bg = dec.bg;
         }

         /* output the character */
         if (dec.ch >= 0 && dec.ch <= 32) dec.ch = ' ';
         fputc(dec.ch, f);
      }
      
      if (suppress_newlines == SUPPRESS_NEVER || 
         (suppress_newlines == SUPPRESS_ONSAMEWIDTH && termwidth != width))
            fputc('\n', f);
   }

   fputs("\e[0m", f);
}

#define COL_BLACK 0
#define COL_RED 1
#define COL_GREEN 2
#define COL_YELLOW 3
#define COL_BLUE 4
#define COL_MAGENTA 5
#define COL_CYAN 6
#define COL_WHITE 7

static char *hex_codes[16] = {
     "000000", "990000", "009900", "999900",
     "000099", "990099", "009999", "999999",
     "404040", "ff0000", "00ff00", "ffff00",
     "0000ff", "ff00ff", "00ffff", "ffffff" };

void output_html(void) {
   int x, y;
   int need_closing_font = 0;
   struct html_output_state_t cur_state;
   struct decoded_cell_t dec;

   fputs("<html><body bgcolor=\"#000000\" style='font-family: monospace'>", f);
   
   for (y = 0; y < height; y++) {
      for (x = 0; x < width; x++) {
         dec = decode_cell(get_cell(x, y));

	 /* first set up the blink tag */
	 if (use_color && dec.blink != cur_state.blink_tag_open) {
	    fprintf(f, "<%sblink>", dec.blink ? "" : "/");
	    cur_state.blink_tag_open = dec.blink;
	 }

         /* set up color */
	 if (use_color && dec.fg != cur_state.color) {
	    if (cur_state.font_tag_open) fputs("</font>", f);
	    fprintf(f, "<font color='#%s'>", hex_codes[dec.fg]);
	    cur_state.color = dec.fg;
	    cur_state.font_tag_open = true;
	 }
	 
	 /* now render the character */
         if (dec.ch >= 0 && dec.ch <= 32) fputs("&nbsp;", f);
	 else if (dec.ch == '&')          fputs("&amp;", f);
         else if (dec.ch == '<')          fputs("&lt;", f);
	 else if (dec.ch == '>')          fputs("&gt;", f);
         else                             fputc(dec.ch, f);
      }
      fputs("<br>\n", f);
   }

   /* close tags as needed and call it a day */
   if (cur_state.font_tag_open) fputs("</font>", f);
   if (cur_state.blink_tag_open) fputs("</blink>", f);
   fputs("</body></html>", f);
}

#define MYSYNTAX \
   "Syntax: aecat [-b] [-c] [-{n|N}] [{-p | -l <layer_num>}]\n" \
   "               [-f <format>] [-o <output_file>] inputfile\n" \
   "\n" \
   "   -f : specifies output format - either text or html\n" \
   "   -c : prepend a 'clear screen' escape sequence (only valid when\n" \
   "        outputting text)\n" \
   "   -o : writes output to specified file rather than stdout\n" \
   "   -b : disables output of color (only characters will be\n" \
   "        printed).\n" \
   "   -l : specifies which layer of the document is to be used.\n" \
   "        (must be an index, not a layer name). By default,\n" \
   "        layer 0 will be used.\n" \
   "   -p : exports a composite, that is, overlays all visible layers,\n" \
   "        paying attention to layer transparency, etc. The size of the\n" \
   "        composite will be the size of the first layer\n" \
   "   -n : suppress output of newlines when the width of the terminal\n" \
   "        is equal to the width of the output (this makes it look\n" \
   "        right because the cursor will advance to the next line\n" \
   "        automatically, making the newline superfluous.\n" \
   "   -N : suppress output of newlines no matter what\n" \
   "   -h : prints this help text\n"

void rtfm() {  /* rtfm = Read the FAQ and Manual, of course :-) */
   fprintf(stderr, MYSYNTAX);
   exit(1);
}

#define FMT_TEXT 1
#define FMT_HTML 2

int main(int argc, char **argv) {
   static char optstring[] = "o:chl:bpf:nN";
   char *inputfile;
   char *s;
   int ch;
   
   int out_fmt = 0; /* In what format to output. */

   /* Try to detect terminal width from the COLUMNS environment variable.
    * We could instead use tcgetattr and the like, but we want to use
    * COLUMNS to allow the user to fool the program into using a different
    * value if need be */
   if ( (s = getenv("COLUMNS")) ) termwidth = atoi(s);
   
   while (0 < (ch = getopt(argc, argv, optstring)) ) {
      switch (ch) {
         case 'c': clear_screen = true; break;
         case 'o': outputfile = strdup(optarg); break;
         case 'l': layer_number = atoi(optarg); break;
         case 'p': layer_number = -1; break;
         case 'b': use_color = false; break;
	 case 'n': suppress_newlines = SUPPRESS_ONSAMEWIDTH; break;
	 case 'N': suppress_newlines = SUPPRESS_ALWAYS; break;
         case 'f':
            if(strcmp(optarg, "text") == 0)
               out_fmt = FMT_TEXT;
            else if(strcmp(optarg, "html") == 0)
               out_fmt = FMT_HTML;
            else {
               fprintf(stderr, "Invalid format \"%s\"\n", optarg);
               exit(1);
            }
            break;
         default : rtfm();
      }
   }

   if (!out_fmt)
      out_fmt = FMT_TEXT;
   
   if (out_fmt != FMT_TEXT && clear_screen) 
      fprintf(stderr, "Warning: -c option only valid when outputting text\n");

   if (out_fmt != FMT_TEXT && suppress_newlines)
      fprintf(stderr, "Warning: -n/-N options only valid when "
                      "outputting text\n");
   
   if (optind >= argc) rtfm();
   inputfile = strdup(argv[optind]);

   if (! (doc = document_load_from(inputfile)) ) {
      fprintf(stderr, "Error loading document from %s (bad format?).\n", 
                                                        inputfile);
      fprintf(stderr, "Error description:\n   %s\n", aeff_get_error());
      exit(1);
   }

   if (layer_number >= doc->layer_count) {
      if (layer_number)
         fprintf(stderr, "Document %s does not have the specified layer: %d\n",
                inputfile, layer_number);
      else
         fprintf(stderr, "Document %s has no layers.\n", inputfile);
      exit(1);
   }

   width = doc->layers[0]->width;
   height = doc->layers[0]->height;
   lyr = layer_number >= 0 ? doc->layers[layer_number] : NULL;
   
   f = stdout;  /* notice that we are reusing f, which now identifies
                 * the file we are outputting information to */

   if (outputfile && !(f = fopen(outputfile, "w"))) {
      fprintf(stderr, "Error opening %s for writing.\n", outputfile);
      exit(1);
   }

   switch(out_fmt) {
      case FMT_TEXT: output_text(); break;
      case FMT_HTML: output_html(); break;
   }

   return 0;
}
