/* aewm - An Exiguous Window Manager - vim:sw=4:ts=4:et
 *
 * Copyright 1998-2006 Decklin Foster <decklin@red-bean.com>. This
 * program is free software; please see LICENSE for details. */

#include <X11/Xatom.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#include "aewm.h"
#include "atom.h"

static void handle_button_press(XButtonEvent *);
static void handle_button_release(XButtonEvent *);
static void handle_configure_request(XConfigureRequestEvent *);
static void handle_circulate_request(XCirculateRequestEvent *);
static void handle_map_request(XMapRequestEvent *);
static void handle_unmap_event(XUnmapEvent *);
static void handle_destroy_event(XDestroyWindowEvent *);
static void handle_client_message(XClientMessageEvent *);
static void handle_property_change(XPropertyEvent *);
static void handle_enter_event(XCrossingEvent *);
static void handle_colormap_change(XColormapEvent *);
static void handle_expose_event(XExposeEvent *);
#ifdef SHAPE
static void handle_shape_change(XShapeEvent *);
#endif

static int root_button_pressed = 0;

/* TWM has an interesting and different way of doing this. We might also want
 * to respond to unknown events. */

void event_loop(void)
{
    XEvent ev;

    for (;;) {
        XNextEvent(dpy, &ev);
#ifdef DEBUG
        show_event(ev);
#endif
        switch (ev.type) {
            case ButtonPress:
                handle_button_press(&ev.xbutton);
                break;
            case ButtonRelease:
                handle_button_release(&ev.xbutton);
                break;
            case ConfigureRequest:
                handle_configure_request(&ev.xconfigurerequest);
                break;
            case CirculateRequest:
                handle_circulate_request(&ev.xcirculaterequest);
                break;
            case MapRequest:
                handle_map_request(&ev.xmaprequest);
                break;
            case UnmapNotify:
                handle_unmap_event(&ev.xunmap);
                break;
            case DestroyNotify:
                handle_destroy_event(&ev.xdestroywindow);
                break;
            case ClientMessage:
                handle_client_message(&ev.xclient);
                break;
            case ColormapNotify:
                handle_colormap_change(&ev.xcolormap);
                break;
            case PropertyNotify:
                handle_property_change(&ev.xproperty);
                break;
            case EnterNotify:
                handle_enter_event(&ev.xcrossing);
                break;
            case Expose:
                handle_expose_event(&ev.xexpose);
                break;
#ifdef SHAPE
            default:
                if (shape && ev.type == shape_event)
                    handle_shape_change((XShapeEvent *)&ev);
#endif
        }
    }
}

/* Someone clicked a button. If they clicked on a window, we want the button
 * press, but if they clicked on the root, we're only interested in the button
 * release. Thus, two functions.
 *
 * If it was on the root, we get the click by default. If it's on a window
 * frame, we get it as well. If it's on a client window, it may still fall
 * through to us if the client doesn't select for mouse-click events. The
 * upshot of this is that you should be able to click on the blank part of a
 * GTK window with Button2 to move it. */

static void handle_button_press(XButtonEvent *e)
{
    client_t *c;

    if ((c = find_client(e->window, MATCH_FRAME)))
        do_press(c, e->x, e->y, e->button);
    else if (e->window == root)
        root_button_pressed = 1;
}

static void handle_button_release(XButtonEvent *e)
{
    if (e->window == root && root_button_pressed) {
#ifdef DEBUG
        dump_clients();
#endif
        switch (e->button) {
            case Button1:
                fork_exec(opt_new1);
                break;
            case Button2:
                fork_exec(opt_new2);
                break;
            case Button3:
                fork_exec(opt_new3);
                break;
        }

        root_button_pressed = 0;
    }
}

/* Because we are redirecting the root window, we get ConfigureRequest events
 * from both clients we're handling and ones that we aren't. For clients we
 * manage, we need to adjust the frame and the client window, and for
 * unmanaged windows we have to pass along everything unchanged.
 *
 * Most of the assignments here are going to be garbage, but only the ones
 * that are masked in by e->value_mask will be looked at by the X server. */

static void handle_configure_request(XConfigureRequestEvent *e)
{
    client_t *c;
    geom_t f;
    XWindowChanges wc;

    if ((c = find_client(e->window, MATCH_WINDOW))) {
        if (e->value_mask & CWX)
            c->geom.x = e->x;
        if (e->value_mask & CWY)
            c->geom.y = e->y;
        if (e->value_mask & CWWidth)
            c->geom.w = e->width;
        if (e->value_mask & CWHeight)
            c->geom.h = e->height;
#ifdef DEBUG
        dump_geom(c, "moving to");
#endif
        f = frame_geom(c);
        wc.x = f.x;
        wc.y = f.y;
        wc.width = f.w;
        wc.height = f.h;
        wc.border_width = BW(c);
        wc.sibling = e->above;
        wc.stack_mode = e->detail;
        XConfigureWindow(dpy, c->frame, e->value_mask, &wc);
#ifdef SHAPE
        if (e->value_mask & (CWWidth|CWHeight)) set_shape(c);
#endif
        if (c->zoomed && e->value_mask & (CWX|CWY|CWWidth|CWHeight)) {
            c->zoomed = 0;
            remove_atom(c->win, net_wm_state, XA_ATOM, net_wm_state_mv);
            remove_atom(c->win, net_wm_state, XA_ATOM, net_wm_state_mh);
        }
        send_config(c);
    }

    wc.x = c ? 0 : e->x;
    wc.y = c ? theight(c) : e->y;
    wc.width = e->width;
    wc.height = e->height;
    wc.sibling = e->above;
    wc.stack_mode = e->detail;
    XConfigureWindow(dpy, e->window, e->value_mask, &wc);
}

/* The only window that we will circulate children for is the root (because
 * nothing else would make sense). After a client requests that the root's
 * children be circulated, the server will determine which window needs to be
 * raised or lowered, and so all we have to do is make it so. */

static void handle_circulate_request(XCirculateRequestEvent *e)
{
    if (e->parent == root) {
        if (e->place == PlaceOnBottom)
            XLowerWindow(dpy, e->window);
        else /* e->place == PlaceOnTop */
            XRaiseWindow(dpy, e->window);
    }
}

/* Two possibilities if a client is asking to be mapped. One is that it's a new
 * window, so we handle that if it isn't in our clients list anywhere. The
 * other is that it already exists and wants to de-iconify, which is simple to
 * take care of. Since we iconify all of a window's transients when iconifying
 * that window, de-iconify them here. */

static void handle_map_request(XMapRequestEvent *e)
{
    client_t *c, *p;

    if ((c = find_client(e->window, MATCH_WINDOW))) {
        do_deiconify(c);
        for (p = head; p; p = p->next)
            if (p->trans == c->win) do_deiconify(p);
    } else {
        map_new(e->window);
    }
}

/* We don't get to intercept Unmap events, so this is post mortem. If we
 * caused the unmap ourselves earlier (explictly or by remapping), we will
 * have incremented c->ignore_unmap. If not, time to destroy the client.
 *
 * Because most clients unmap and destroy themselves at once, they're gone
 * before we even get the Unmap event, never mind the Destroy one. Therefore
 * we must be extra careful in remove_client. */

static void handle_unmap_event(XUnmapEvent *e)
{
    client_t *c;

    if ((c = find_client(e->window, MATCH_WINDOW))) {
        if (c->ignore_unmap) c->ignore_unmap--;
        else remove_client(c, REMOVE_WITHDRAW);
    }
}

/* But a window can also go away when it's not mapped, in which case there is
 * no Unmap event. */

static void handle_destroy_event(XDestroyWindowEvent *e)
{
    client_t *c;

    if ((c = find_client(e->window, MATCH_WINDOW)))
        remove_client(c, REMOVE_WITHDRAW);
}

/* If a client wants to manipulate itself or another window it must send a
 * special kind of ClientMessage. As of right now, this only responds to the
 * ICCCM iconify message, but there are more in the EWMH that will be added
 * later. */

static void handle_client_message(XClientMessageEvent *e)
{
    client_t *c;

    if (e->window == root) {
        if (e->message_type == net_cur_desk && e->format == 32)
            goto_desk(e->data.l[0]);
        else if (e->message_type == net_num_desktops && e->format == 32)
            ndesks = e->data.l[0];
    } else if ((c = find_client(e->window, MATCH_WINDOW))) {
        if (e->message_type == wm_change_state && e->format == 32 &&
                e->data.l[0] == IconicState) {
            iconify_win(c);
        } else if (e->message_type == net_active_window && e->format == 32) {
            c->desk = cur_desk;
            map_desk(c);
            do_deiconify(c);
            XRaiseWindow(dpy, c->frame);
        } else if (e->message_type == net_close_window && e->format == 32) {
            send_wm_delete(c);
        }
    }
}

/* If we have something copied to a variable, or displayed on the screen, make
 * sure it is up to date. If redrawing the title is necessary, clear the
 * window because Xft uses alpha rendering. */

static void handle_property_change(XPropertyEvent *e)
{
    client_t *c;
    long supplied;
    unsigned long read, left;
    int i;
    Atom state;

    if ((c = find_client(e->window, MATCH_WINDOW))) {
        if (e->atom == XA_WM_NAME || e->atom == net_wm_name) {
            if (c->name) XFree(c->name);
            c->name = get_wm_name(c->win);
            redraw(c);
        } else if (e->atom == XA_WM_NORMAL_HINTS) {
            XGetWMNormalHints(dpy, c->win, c->size, &supplied);
        } else if (e->atom == net_wm_state) {
            for (i = 0, left = 1; left; i += read) {
                read = get_atoms(c->win, net_wm_state, XA_ATOM, i,
                    &state, 1, &left);
                if (read)
                    handle_net_wm_state_item(c, state);
                else
                    break;
            }
        } else if (e->atom == net_wm_desktop) {
            if (get_atoms(c->win, net_wm_desktop, XA_CARDINAL, 0,
                    &c->desk, 1, &left))
                map_desk(c);
        }
    }
}

/* Lazy focus-follows-mouse and colormap-follows-mouse policy. This does not,
 * however, prevent focus stealing (it's lazy). It is not very efficient
 * either; we can get a lot of enter events at once when flipping through a
 * window stack on startup/desktop change. */

static void handle_enter_event(XCrossingEvent *e)
{
    client_t *c;

    if ((c = find_client(e->window, MATCH_FRAME)))
        set_active(c);
}

/* More colormap policy: when a client installs a new colormap on itself, set
 * the display's colormap to that. We do this even if it's not focused. */

static void handle_colormap_change(XColormapEvent *e)
{
    client_t *c;

    if ((c = find_client(e->window, MATCH_WINDOW)) && e->new) {
        c->cmap = e->colormap;
        XInstallColormap(dpy, c->cmap);
    }
}

/* If we were covered by multiple windows, we will usually get multiple expose
 * events, so ignore them unless e->count (the number of outstanding exposes)
 * is zero. */

static void handle_expose_event(XExposeEvent *e)
{
    client_t *c;

    if ((c = find_client(e->window, MATCH_FRAME)) && e->count == 0)
        redraw(c);
}

#ifdef SHAPE
static void handle_shape_change(XShapeEvent *e)
{
    client_t *c;

    if ((c = find_client(e->window, MATCH_WINDOW)))
        set_shape(c);
}
#endif
