/* aewm - An Exiguous Window Manager - vim:sw=4:ts=4:et
 *
 * Copyright 1998-2006 Decklin Foster <decklin@red-bean.com>. This
 * program is free software; please see LICENSE for details. */

#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <signal.h>
#include <sys/wait.h>
#include <X11/Xatom.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#include "aewm.h"
#include "atom.h"

static void shutdown(void);

void sig_handler(int signum)
{
    switch (signum) {
        case SIGINT:
        case SIGTERM:
        case SIGHUP:
            shutdown();
            break;
        case SIGCHLD:
            wait(NULL);
            break;
    }
}

int handle_xerror(Display *dpy, XErrorEvent *e)
{
    /* XXX: do we want this or not? */
    /* client_t *c = find_client(e->resourceid, MATCH_WINDOW); */

    if (e->error_code == BadAccess && e->resourceid == root) {
        fprintf(stderr, "aewm: root window unavailable\n");
        exit(1);
    } else {
        char msg[255];
        XGetErrorText(dpy, e->error_code, msg, sizeof msg);
        fprintf(stderr, "aewm: X error (%#lx): %s\n", e->resourceid, msg);
#ifdef DEBUG
        if (c) dump_info(c);
#endif
    }

    /* XXX: ditto */
    /* if (c) remove_client(c, REMOVE_WITHDRAW); */
    return 0;
}

/* Ick. Argh. You didn't see this function. */

int ignore_xerror(Display *dpy, XErrorEvent *e)
{
    return 0;
}

void goto_desk(int new_desk)
{
    client_t *c;

    cur_desk = new_desk;
    set_atoms(root, net_cur_desk, XA_CARDINAL, &cur_desk, 1);

    for (c = head; c; c = c->next)
        map_desk(c);
}

/* We use XQueryTree instead of walking the clients list for two reasons:
 * first, we may be in the process of mapping a new window, in which case its
 * client structure has already been inserted into the list. We don't want to
 * consider the strut on the window we're mapping! Second, we may be starting
 * in the middle of an X session, and reparenting a maximized window before a
 * window with a strut set. We check if the window is mapped already; however,
 * we still have to call find_client in case we have reparented the window and
 * our frame is the direct child of root. */

void collect_struts(int desk, strut_t *s)
{
    Window qroot, qparent, *wins, w;
    unsigned int nwins, i;
    unsigned long w_desk = cur_desk;
    XWindowAttributes attr;
    strut_t temp;
    client_t *c;

    s->left = 0;
    s->right = 0;
    s->top = 0;
    s->bottom = 0;

    XQueryTree(dpy, root, &qroot, &qparent, &wins, &nwins);
    for (i = 0; i < nwins; i++) {
        c = find_client(wins[i], MATCH_FRAME);
        w = c ? c->win : wins[i];
        get_atoms(w, net_wm_desktop, XA_CARDINAL, 0, &w_desk, 1, NULL);
        XGetWindowAttributes(dpy, w, &attr);
        if (attr.map_state == IsViewable && IS_ON_DESK(w_desk, desk) &&
                get_strut(w, &temp)) {
            if (temp.left > s->left)
                s->left = temp.left;
            if (temp.right > s->right)
                s->right = temp.right;
            if (temp.top > s->top)
                s->top = temp.top;
            if (temp.bottom > s->bottom)
                s->bottom = temp.bottom;
        }
    }
    XFree(wins);
}

#ifdef DEBUG

/* Bleh, stupid macro names. I'm not feeling creative today. */

#define SHOW_EV(name, memb) \
    case name: ev_type = #name; w = e.memb.window; break;
#define SHOW(name) \
    case name: return #name;

void show_event(XEvent e)
{
    char *ev_type;
    Window w;
    client_t *c;

    switch (e.type) {
        SHOW_EV(ButtonPress, xbutton)
        SHOW_EV(ButtonRelease, xbutton)
        SHOW_EV(ClientMessage, xclient)
        SHOW_EV(ColormapNotify, xcolormap)
        SHOW_EV(ConfigureNotify, xconfigure)
        SHOW_EV(ConfigureRequest, xconfigurerequest)
        SHOW_EV(CirculateRequest, xcirculaterequest)
        SHOW_EV(CreateNotify, xcreatewindow)
        SHOW_EV(DestroyNotify, xdestroywindow)
        SHOW_EV(EnterNotify, xcrossing)
        SHOW_EV(Expose, xexpose)
        SHOW_EV(MapNotify, xmap)
        SHOW_EV(MapRequest, xmaprequest)
        SHOW_EV(MappingNotify, xmapping)
        SHOW_EV(MotionNotify, xmotion)
        SHOW_EV(PropertyNotify, xproperty)
        SHOW_EV(ReparentNotify, xreparent)
        SHOW_EV(ResizeRequest, xresizerequest)
        SHOW_EV(UnmapNotify, xunmap)
        default:
#ifdef SHAPE
            if (shape && e.type == shape_event) {
                ev_type = "ShapeNotify"; w = ((XShapeEvent *)&e)->window;
                break;
            }
#endif
            ev_type = "unknown event";
            w = None;
            break;
    }

    if ((c = find_client(w, MATCH_WINDOW)))
        dump_title(c, ev_type, 'w');
    else if ((c = find_client(w, MATCH_FRAME)))
        dump_title(c, ev_type, 'f');
    else if (w == root)
        dump_win(w, ev_type, 'r');
    else
        /* something we are not managing */
        dump_win(w, ev_type, 'u');
}

static const char *show_state(client_t *c)
{
    switch (get_wm_state(c->win)) {
        SHOW(WithdrawnState)
        SHOW(NormalState)
        SHOW(IconicState)
        default: return "unknown state";
    }
}

static const char *show_grav(client_t *c)
{
    if (!c->size || !(c->size->flags & PWinGravity))
        return "no grav (NW)";

    switch (c->size->win_gravity) {
        SHOW(UnmapGravity)
        SHOW(NorthWestGravity)
        SHOW(NorthGravity)
        SHOW(NorthEastGravity)
        SHOW(WestGravity)
        SHOW(CenterGravity)
        SHOW(EastGravity)
        SHOW(SouthWestGravity)
        SHOW(SouthGravity)
        SHOW(SouthEastGravity)
        SHOW(StaticGravity)
        default: return "unknown grav";
    }
}

void dump_title(client_t *c, const char *label, char flag)
{
    printf("%15.15s: %#010lx [%c] %-47.47s\n", label, c->win, flag, c->name);
}

void dump_win(Window w, const char *label, char flag)
{
    printf("%15.15s: %#010lx [%c] %-47.47s\n", label, w, flag,
        w == root ? "(root window)": "(unknown window)");
}

void dump_info(client_t *c)
{
    printf("%28s[i] frame %#0lx, ignore_unmap %d\n", "",
        c->frame, c->ignore_unmap);
    printf("%28s[i] %ld, %s, %s\n", "",
        c->desk, show_state(c), show_grav(c));
}

void dump_geom(client_t *c, const char *label)
{
    printf("%28s[g] %s %ldx%ld+%ld+%ld\n", "",
        label, c->geom.w, c->geom.h, c->geom.x, c->geom.y);
}

void dump_removal(client_t *c, int mode)
{
    printf("%28s[r] %s, %d pending\n", "",
        mode == REMOVE_WITHDRAW ? "withdraw" : "remap", XPending(dpy));
}

void dump_clients()
{
    client_t *c;

    for (c = head; c; c = c->next) {
        dump_title(c, "dump", 'd');
        dump_geom(c, "current");
        dump_info(c);
    }
}
#endif

/* We use XQueryTree here to preserve the window stacking order, since
 * the order in our linked list is different. */

static void shutdown(void)
{
    unsigned int nwins, i;
    Window qroot, qparent, *wins;
    client_t *c;

    XQueryTree(dpy, root, &qroot, &qparent, &wins, &nwins);
    for (i = 0; i < nwins; i++) {
        c = find_client(wins[i], MATCH_FRAME);
        if (c) remove_client(c, REMOVE_REMAP);
    }
    XFree(wins);

    XFreeFont(dpy, font);
#ifdef X_HAVE_UTF8_STRING
    XFreeFontSet(dpy, font_set);
#endif
#ifdef XFT
    XftFontClose(dpy, xftfont);
#endif
    XFreeCursor(dpy, map_curs);
    XFreeCursor(dpy, move_curs);
    XFreeCursor(dpy, resize_curs);
    XFreeGC(dpy, invert_gc);
    XFreeGC(dpy, border_gc);
    XFreeGC(dpy, string_gc);

    XInstallColormap(dpy, DefaultColormap(dpy, screen));
    XSetInputFocus(dpy, PointerRoot, RevertToPointerRoot, CurrentTime);

    XDeleteProperty(dpy, root, net_supported);
    XDeleteProperty(dpy, root, net_client_list);

    XCloseDisplay(dpy);
    exit(0);
}
