/* 
   Affix - Bluetooth Protocol Stack for Linux
   Copyright (C) 2001 Nokia Corporation
   Original Author: Dmitry Kasatkin <dmitry.kasatkin@nokia.com>

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at your
   option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* 
   $Id: utils.h,v 1.29 2003/04/16 07:56:34 kds Exp $

   General purpose utilites 

   Fixes:
   		Dmitry Kasatkin <dmitry.kasatkin@nokia.com>
*/		

#ifndef _BTUTILS_H
#define _BTUTILS_H

#include <stdint.h>
#include <sys/stat.h>

#include <affix/btcore.h>

__BEGIN_DECLS

#if __BYTE_ORDER == __LITTLE_ENDIAN
static inline void *u8_to_ptr(uint8_t data)
{
	unsigned int	tmp = data;
	return ((void*)tmp);
}
static inline void *u16_to_ptr(uint16_t data)
{
	unsigned int	tmp = data;
	return ((void*)tmp);
}
static inline void *u32_to_ptr(uint32_t data)
{
	unsigned int	tmp = data;
	return ((void*)tmp);
}
#elif __BYTE_ORDER == __BIG_ENDIAN
#warning "BIGENDIAN"
static inline void *u8_to_ptr(uint8_t data)
{
	unsigned int	tmp = data;
	tmp <<= 24;
	return ((void*)tmp);
}
static inline void *u16_to_ptr(uint16_t data)
{
	unsigned int	tmp = data;
	tmp <<= 16;
	return ((void*)tmp);
}
static inline void *u32_to_ptr(uint32_t data)
{
	unsigned int	tmp = data;
	return ((void*)tmp);
}
#else
#error "__BYTE_ORDER is not defined"
#endif

typedef struct _slist_t{
	void		*data;
	struct _slist_t	*next;
} slist_t;

typedef void (slist_func)(void *data, void *param);
typedef int (slist_sort_func)(const void *data1, const void *data2);

static inline slist_t *s_list_next(slist_t *list)
{
	return list->next;
}

static inline void *s_list_data(slist_t *list)
{
	return list->data;
}

static inline unsigned int s_list_uint(slist_t *list)
{
	return (unsigned int)list->data;
}


slist_t *s_list_append(slist_t **list, void *data);
slist_t *s_list_insert(slist_t **list, void *data, int i);
slist_t *s_list_insert_sorted(slist_t **list, void *data, slist_sort_func *func);
int s_list_length(slist_t *list);
void s_list_free(slist_t **list);
void *s_list_nth_data(slist_t *list, int i);
void s_list_foreach(slist_t *list, slist_func *func, void *param);
slist_t *s_list_find_custom(slist_t *list, void *data, slist_sort_func *func);
void s_list_remove(slist_t **list, void *data);
void s_list_remove_custom(slist_t **list, void *data, slist_sort_func *func);
void *s_list_dequeue(slist_t **list);
void s_list_destroy(slist_t **list);

static inline slist_t *s_list_append_uint(slist_t **list, unsigned int value)
{
	return s_list_append(list, (void*)value);
}

static inline unsigned int s_list_nth_uint(slist_t *list, int i)
{
	return (unsigned int)s_list_nth_data(list, i);
}

/* file stuff */
int get_fileinfo(const char *name, char *lastmod);

#define PIDOF_SINGLE	0x01
#define PIDOF_OMIT	0x02
#define PIDOF_POMIT	0x04
#define PIDOF_SCRIPTS	0x08
int affix_pidof(char *name, int flags, pid_t pid);

char *xml_element(char **buf, char **attr);
char *xml_attribute(char **buf, char **value);
int rmkdir(char *new_dir, int mode);
int get_speed(int size, struct timeval *tv_start, struct timeval *tv_end,
		long int *rsec, long int *rusec, double *speed);

//
// Get the filesize
//
static inline int get_filesize(const char *filename)
{
	struct stat stats;
	/*  Need to know the file length */
	stat(filename, &stats);
	return (int) stats.st_size;
}

static inline int get_fdsize(int fd)
{
	struct stat stats;
	/*  Need to know the file length */
	fstat(fd, &stats);
	return (int) stats.st_size;
}


char *val2str(struct affix_tupla *map, int value);
int str2val(struct affix_tupla *map, char *str, unsigned int *val);
int str2mask(struct affix_tupla *map, char *str, unsigned int *mask);
int mask2str(struct affix_tupla *map, char *str, unsigned int mask);
int mask2str_comma(struct affix_tupla *map, char *str, unsigned int mask);
int str2cod(char *str, uint32_t *cod);
int str2cod_svc(char *str, uint32_t *cod);
int str2pkt_type(char *str, unsigned int *pkt_type);
int str2sec_level(char *str, unsigned int *sec_level);


/* device inquiry/known cache */
#define BTDEV_KEY		0x0001

/* Inquiry Cache support */
#define DEVSTATE_RANGE		0x01
#define DEVSTATE_GONE		0x02
#define DEVSTATE_UNKNOWN	0x04
#define DEVSTATE_ALL		0xFF

typedef struct {
	int		flags;
	int		state;	/* to keep inquiry info */
	
	BD_ADDR		bda;
	uint32_t	cod;
	char		name[248];

	__u8		key_type;
	__u8		link_key[16];
	__u8		pin_length;
	__u8		pin[16];
} btdev_struct;

typedef struct {
	slist_t	*head;
	int	count;
	char	*file;
	int	lock;
} btdev_list;

int btdev_cache_lock(btdev_list *list);
void btdev_cache_unlock(btdev_list *list);

btdev_struct *btdev_cache_lookup(btdev_list *list, BD_ADDR *bda);
int btdev_cache_del(btdev_list *list, btdev_struct *entry);
btdev_struct *btdev_cache_add(btdev_list *list, BD_ADDR *bda);
int btdev_cache_reload(btdev_list *list);
int btdev_cache_load(char *cachefile, btdev_list *list);
int btdev_cache_save(btdev_list *list);
void btdev_cache_free(btdev_list *list);

int parse_cod(char *buf, uint32_t  cod);
void argv2str(char *str, char *argv[]);

/* btctl-util.c */
void btdev_cache_retire(btdev_list *list);
void btdev_cache_print(btdev_list *list, int state);
int btdev_cache_resolve(btdev_list *list, BD_ADDR *bda, int id);
btdev_struct *__btdev_cache_add(btdev_list *list, BD_ADDR bda, uint32_t cod, char *name);

int btdev_get_bda(btdev_list *list, BD_ADDR *bda, char *arg);


__END_DECLS

#endif	/* _BTUTILS_H */

