// ---------------------------------------------------------------------------
// - InputCipher.hpp                                                         -
// - afnix cryptography - input cipher class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------


#ifndef  AFNIX_INPUTCIPHER_HPP
#define  AFNIX_INPUTCIPHER_HPP

#ifndef  AFNIX_CIPHER_HPP
#include "Cipher.hpp"
#endif

namespace afnix {

  /// The InputCipher implements an input stream interface that can
  /// stream out an input stream from a cipher. In other word, an input
  /// stream is read and block are encoded as long as the input stream 
  /// read characters. If the cipher is nil, the input cipher simply
  /// read the input stream and is therefore transparent.
  /// @author amaury darsch

  class InputCipher : public Input {
  public:
    /// the ciphering mode
    enum t_mode {
      ECB, // electronic codebook mode
      CBC  // cipher block chaining mode
    };

  private:
    /// the ciphering mode
    t_mode  d_mode;
    /// the block size
    long    d_cbsz;
    /// the output block
    t_byte* p_bo;
    /// the input block
    t_byte* p_bi;
    /// the last block
    t_byte* p_bl;
    /// the cipher reverse flag
    bool    d_rflg;
    /// the cipher object
    Cipher* p_cifr;
    /// the input object
    Input*  p_is;

  public:
    /// create a new input cipher with a cipher only
    /// @param cifr the cipher to bind
    InputCipher (Cipher* cifr);

    /// create a new input cipher with a cipher and mode
    /// @param cifr the cipher to bind
    /// @param mode the input cipher mode
    InputCipher (Cipher* cifr, const t_mode mode);

    /// create a new input cipher stream by cipher and input stream
    /// @param cifr the cipher to bind
    /// @param is   the input stream to bind
    InputCipher (Cipher* cifr, Input* is);

    /// create a new input cipher stream by cipher, input stream and mode
    /// @param cifr the cipher to bind
    /// @param is   the input stream to bind
    /// @param mode the cipher mode
    InputCipher (Cipher* cifr, Input* is, const t_mode mode);

    /// destroyy this input cipher
    ~InputCipher (void);

    /// @return the class name
    String repr (void) const;

    /// duplicate this input cipher with another stream
    /// @param is the input stream to set
    InputCipher* dup (Input* is) const;

    /// set the input stream to this input cipher
    /// @param is the input stream to set
    void setis (Input* is);

    /// @return the new character on the input stream
    char read (void);

    /// @return true if we are at the end of the input string
    bool iseof (void) const;

    /// check if we can read a character
    /// @param tout the timeout value
    bool valid (const long tout) const;
      
    /// get the normal waist from a file size
    /// @param size the size to normalize
    virtual t_long waist (const t_long size) const;

  private:
    // make the copy constructor private
    InputCipher (const InputCipher&);
    // make the assignment operator private
    InputCipher& operator = (const InputCipher&);
    // initialize the input cipher
    void initialize (void);
    //update the input buffer
    void update (void);

  public:
    /// evaluate an object data member
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to evaluate
    static Object* meval (Runnable* robj, Nameset* nset, const long quark);

    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);
    
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
