// ---------------------------------------------------------------------------
// - Input.cpp                                                               -
// - standard object library - input stream class implementation             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Byte.hpp"
#include "Input.hpp"
#include "Ascii.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "Character.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "ccnv.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // return the stream descriptor

  int Input::getsid (void) const {
    return -1;
  }

  // read the next character in byte mode

  t_quad Input::rbyte (void) {
    wrlock ();
    try {
      t_quad result = encode (read ());
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // read the next character in utf8 mode

  t_quad Input::rutf8 (void) {
    wrlock ();
    try {
      t_byte b1 = (t_byte) read ();
      // 1 byte mode
      if (b1 < 0x80) {
	t_quad result = b1;
	unlock ();
	return result;
      }
      // 2 bytes mode
      if (b1 < 0xE0) {
	t_quad result = ((t_quad) (b1 & 0x3F)) << 6;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b2 = (t_byte) read ();
	if ((b2 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b2 & 0x3F);
	if (result < 0x00000080) {
	  throw Exception ("read-error", "invalid long utf-8 sequence");
	}
	unlock ();
	return result;
      }
      // 3 bytes mode
      if (b1 < 0xF0) {
	t_quad result = ((t_quad) (b1 & 0x0f)) << 12;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b2 = (t_byte) read ();
	if ((b2 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= ((t_quad) (b2 & 0x3F)) << 6;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b3 = (t_byte) read ();
	if ((b3 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b3 & 0x3F);
	if (result < 0x00000800) {
	  throw Exception ("read-error", "invalid long utf-8 sequence");
	}
	unlock ();
	return result;
      }
      // 4 bytes mode
      if (b1 < 0xF8) {
	t_quad result = ((t_quad) (b1 & 0x07)) << 18;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b2 = (t_byte) read ();
	if ((b2 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= ((t_quad) (b2 & 0x3F)) << 12;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b3 = (t_byte) read ();
	if ((b3 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b3 & 0x3F) << 6;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b4 = (t_byte) read ();
	if ((b4 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b4 & 0x3F);
	if (result < 0x00010000) {
	  throw Exception ("read-error", "invalid long utf-8 sequence");
	}
	unlock ();
	return result;
      }
      // 5 bytes mode
      if (b1 < 0xFC) {
	t_quad result = ((t_quad) (b1 & 0x03)) << 24;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b2 = (t_byte) read ();
	if ((b2 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= ((t_quad) (b2 & 0x3F)) << 18;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b3 = (t_byte) read ();
	if ((b3 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b3 & 0x3F) << 12;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b4 = (t_byte) read ();
	if ((b4 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b4 & 0x3F) << 6;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b5 = (t_byte) read ();
	if ((b5 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b5 & 0x3F);
	if (result < 0x00200000) {
	  throw Exception ("read-error", "invalid long utf-8 sequence");
	}
	unlock ();
	return result;
      }
      // 6 bytes mode
      if (b1 < 0xFE) {
	t_quad result = ((t_quad) (b1 & 0x01)) << 30;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b2 = (t_byte) read ();
	if ((b2 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= ((t_quad) (b2 & 0x3F)) << 24;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b3 = (t_byte) read ();
	if ((b3 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b3 & 0x3F) << 18;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b4 = (t_byte) read ();
	if ((b4 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b4 & 0x3F) << 12;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b5 = (t_byte) read ();
	if ((b5 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b5 & 0x3F) << 6;
	if (valid (0) == false) {
	  throw Exception ("read-error", 
			   "invalid eos while reading unicode character");
	}
	t_byte b6 = (t_byte) read ();
	if ((b6 & 0x80) != 0x80) {
	  throw Exception ("read-error", 
			   "invalid byte while reading unicode character");
	}
	result |= (t_quad) (b6 & 0x3F);
	if (result < 0x04000000) {
	  throw Exception ("read-error", "invalid long utf-8 sequence");
	}
	unlock ();
	return result;
      }
      throw Exception ("read-error", "invalid unicode character to read");
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // read the next unicode character

  t_quad Input::rduc (void) {
    wrlock ();
    try {
      t_quad result = nilq;
      switch (d_emod) {
      case System::BYTE:
	result = rbyte ();
	break;
      case System::UTF8:
	result = rutf8 ();
	break;
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // read a buffer by size

  Buffer* Input::read (const long size) {
    wrlock ();
    Buffer* result = new Buffer;
    try {
      for (long i = 0; i < size; i++) {
	if (valid (-1) == false) break;
	result->add (read ());
      }
    } catch (...) {
      delete result;
      unlock ();
      throw;
    }
    unlock ();
    return result;
  }

  // read a line in this input stream

  String Input::readln (void) {
    wrlock ();
    // create a buffer to accumulate characters
    Buffer buf;
    bool   flg = false;
    try {
      // read the character in the buffer
      while (valid (-1) == true) {
	t_quad c = rduc ();
	if (c == crlq) {
	  flg = true;
	  continue;
	}
	if (c == eolq) {
	  unlock ();
	  return buf.tostring ();
	}
	if (flg == true) {
	  buf.add (crlq);
	  flg = false;
	}
	buf.add (c);
      }
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
    return buf.tostring ();
  }

  // pushback a character

  void Input::pushback (const char value) {
    wrlock ();
    try {
      d_buffer.pushback (value);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // pushback a c-string on this stream

  void Input::pushback (const char* s) {
    wrlock ();
    try {
      long size = Ascii::strlen (s);
      d_buffer.pushback (s, size);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // pushback a c-string on this stream

  void Input::pushback (const char* s, const long size) {
    wrlock ();
    try {
      d_buffer.pushback (s, size);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // pushback a string on this stream
  
  void Input::pushback (const String& value) {
    wrlock ();
    try {
      // check for byte mode
      if (d_emod == System::BYTE) {
	// get the quad buffer
	t_quad* sbuf = value.toquad ();
	if (sbuf == nilp) {
	  unlock ();
	  return;
	}
	// transcode to the character buffer
	char* cbuf = decode (sbuf);
	delete [] sbuf;
	// write the buffer and clean
	pushback (cbuf);
	delete [] cbuf;
	unlock ();
	return;
      }
      // check for utf-8 mode
      if (d_emod == System::UTF8) {
	char* sbuf = value.encode ();
	if (sbuf == nilp) {
	  unlock ();
	  return;
	}
	// write string content
	pushback (sbuf);
	delete [] sbuf;
	unlock ();
	return;
      }
      throw Exception ("pushback-error", "invalid mode for string pushback");
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the size of the input buffer

  long Input::buflen (void) const {
    rdlock ();
    long result = d_buffer.length ();
    unlock ();
    return result;
  }

  // flush the input stream until a character is reached

  void Input::flush (const char value) {
    wrlock ();
    try {
      while (true) {
	char c = read ();
	if (c == eolc) {
	  unlock ();
	  return;
	}
	if (c == eofc) {
	  pushback (c);
	  unlock ();
	  return;
	}
      }
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // flush the input stream until a character is reached

  void Input::flush (const t_quad value) {
    wrlock ();
    try {
      while (true) {
	t_quad c = rduc ();
	if (c == eolq) {
	  unlock ();
	  return;
	}
	if (c == eofq) {
	  pushback (c);
	  unlock ();
	  return;
	}
      }
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 7;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_EOFP   = zone.intern ("eof-p");
  static const long QUARK_READ   = zone.intern ("read");
  static const long QUARK_RDUC   = zone.intern ("rduc");
  static const long QUARK_PUSHB  = zone.intern ("pushback");
  static const long QUARK_VALIDP = zone.intern ("valid-p");
  static const long QUARK_READLN = zone.intern ("readln");
  static const long QUARK_BUFLEN = zone.intern ("get-buffer-length");

  // return true if the given quark is defined

  bool Input::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Stream::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* Input::apply (Runnable* robj, Nameset* nset, const long quark,
			Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_READ)   return new Byte      (read   ());
      if (quark == QUARK_RDUC)   return new Character (rduc   ());
      if (quark == QUARK_EOFP)   return new Boolean   (iseof  ());
      if (quark == QUARK_VALIDP) return new Boolean   (valid  (-1));
      if (quark == QUARK_READLN) return new String    (readln ());
      if (quark == QUARK_BUFLEN) return new Integer   (buflen ());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_READ) {
	long size = argv->getint (0);
	return read (size);
      }
      if (quark == QUARK_VALIDP) {
	long tout = argv->getint (0);
	return new Boolean (valid (tout));
      }
      if (quark == QUARK_PUSHB) {
	Object* obj = argv->get (0);
	// check for a byte
	Byte* bobj = dynamic_cast <Byte*> (obj);
	if (bobj != nilp) {
	  pushback (bobj->tobyte ());
	  return nilp;
	}
	// check for a character
	Character* cobj = dynamic_cast <Character*> (obj);
	if (cobj != nilp) {
	  pushback (cobj->tochar ());
	  return nilp;
	}
	String* s = dynamic_cast <String*> (obj);
	if (s != nilp) {
	  pushback (*s);
	  return nilp;
	}
	throw Exception ("type-error", "invalid object with pushback method",
			 Object::repr (obj));
      }
    }
    // apply these arguments with the stream object
    return Stream::apply (robj, nset, quark, argv);
  }
}
