// ---------------------------------------------------------------------------
// - InputFile.cpp                                                           -
// - standard object library - input file class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "System.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "InputFile.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "csio.hpp"
#include "cerr.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // this function open a file by name and return a file id
  static int open_file (const String& name) {
    char* fname = name.tochar ();
    int sid     = c_openr (fname);
    if (sid < 0) {
      delete [] fname;
      return -1;
    }
    delete [] fname;
    return sid;
  }

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a new input file by name

  InputFile::InputFile (const String& name) {
    d_name = name;
    d_sid  = open_file (name);
    d_buffer.reset ();
    if (d_sid == -1) throw Exception ("open-error", "cannot open file", name);
  }

  // close and destroy this input file
  InputFile::~InputFile (void) {
    close ();
  }

  // return the class name

  String InputFile::repr (void) const {
    return "InputFile";
  }

  // return the stream descriptor

  int InputFile::getsid (void) const {
    rdlock ();
    int result = d_sid;
    unlock ();
    return result;
  }

  // read one character from the input stream.

  char InputFile::read (void) {
    wrlock ();
    try {
      // check if we can read a character
      if (valid (-1) == false) return eofc;
      // check the pushback buffer first
      if (d_buffer.empty () == false) {
	char result = d_buffer.read ();
	unlock ();
	return result;
      }
      // read by packet the input stream
      char buffer[256];
      long count = 0;
      if ((count = c_read (d_sid, buffer, 256)) < 0) {
	throw Exception ("read-error", c_errmsg (count));
      }
      // check for eof (this should not happen)
      if (count == 0) return eofc;
      // pushback the packet
      d_buffer.pushback (buffer,count);
      char result = read ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return true if the eof flag is set

  bool InputFile::iseof (void) const {
    wrlock ();
    try {
      bool result = !valid (0);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return true if we can read a character

  bool InputFile::valid (const long tout) const {
    wrlock ();
    try {
      if (d_buffer.empty () == false) {
	unlock ();
	return true;
      }
      // check if we can read one character
      bool status = c_rdwait (d_sid, tout);
      if (status == false) {
	unlock ();
	return false;
      }
      // read in the character - might be the eof
      char c = nilc;
      long s = c_read (d_sid, &c, 1);
      if (s < 0)  throw Exception ("iseof-error", c_errmsg (s));
      if (s == 0) {
	unlock ();
	return false;
      }
      d_buffer.pushback (c);
      unlock ();
      return true;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the file name associated with this stream

  String InputFile::getname (void) const {
    rdlock ();
    String result = d_name;
    unlock ();
    return result;
  }

  // close this input file and mark the eof marker
  
  bool InputFile::close (void) {
    wrlock ();
    if ((d_sid < 0) || (Object::uref (this) == false)) {
      unlock ();
      return true;
    }
    if (c_close (d_sid) == false) {
      unlock ();
      return false;
    }
    d_sid = -1;
    unlock ();
    return true;
  }

  // set the input stream at a certain position

  void InputFile::lseek (const t_long pos) {
    wrlock ();
    // move the file cursor
    c_lseek (d_sid, pos);
    // reset everything
    d_buffer.reset ();
    unlock ();
  }

  // return the input file size

  t_long InputFile::length (void) const {
    rdlock ();
    t_long result = c_fsize (d_sid);
    unlock ();
    return result;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 3;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_LSEEK   = zone.intern ("lseek");
  static const long QUARK_CLOSE   = zone.intern ("close");
  static const long QUARK_LENGTH  = zone.intern ("length");

  // create a new object in a generic way

  Object* InputFile::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 1) 
      throw Exception ("argument-error", 
		       "invalid arguments with with input file"); 

    // try to map the file name
    String name = argv->getstring (0);
    return new InputFile (name);
  }

  // return true if the given quark is defined

  bool InputFile::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    // check the nameable class
    bool result = hflg ? Nameable::isquark (quark, hflg) : false;
    // check the input class
    if (result == false) {
      result = hflg ? Input::isquark (quark, hflg) : false;
    }
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* InputFile::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {

    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH)  return new Integer (length ());
      if (quark == QUARK_CLOSE)   return new Boolean (close ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_LSEEK) {
	t_long pos = argv->getint (0);
	lseek (pos);
	return nilp;
      }
    }
    // check the nameable class
    if (Nameable::isquark (quark, true) == true) {
      return Nameable::apply (robj, nset, quark, argv);
    }
    // fallbak with the input method
    return Input::apply (robj, nset, quark, argv);
  }
}
