// ---------------------------------------------------------------------------
// - Item.cpp                                                                -
// - standard object library - item class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Item.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {
  
  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a new enumeration item by id

  Item::Item (const long tid, const long quark) {
    d_type  = STATIC;
    d_tid   = tid;
    d_quark = quark;
  }

  // create a new enumeration item by object

  Item::Item (Object* obj, const long quark) {
    d_type = DYNAMIC;
    Object::iref (p_obj = obj);
    d_quark = quark;
  }

  // copy construct this enumeration item

  Item::Item (const Item& that) {
    that.rdlock ();
    try {
      d_type  = that.d_type;
      d_quark = that.d_quark;
      if (d_type == STATIC) {
	d_tid = that.d_tid;
      } else {
	Object::iref (p_obj = that.p_obj);
      }
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // destroy this enumeration item

  Item::~Item (void) {
    if (d_type == DYNAMIC) Object::dref (p_obj);
  }

  // return the class name

  String Item::repr (void) const {
    return "Item";
  }
 
  // clone this enumeration item

  Object* Item::clone (void) const {
    return new Item (*this);
  }

  // make this enumeration item a shared object
  
  void Item::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if ((d_type == DYNAMIC) && (p_obj != nilp)) p_obj->mksho ();
  }

  // return a literal representation of this item

  String Item::toliteral (void) const {
    return tostring ();
  }

  // return a string representation of this item

  String Item::tostring (void) const {
    return String::qmap (d_quark);
  }

  // compare this item with another one

  bool Item::operator == (const Item& item) const {
    if (d_type != item.d_type) return false;
    if (d_type == DYNAMIC) {
      if (p_obj   != item.p_obj)   return false;      
      if (d_quark != item.d_quark) return false;
    }
    if (d_type == STATIC) {
      if (d_tid   != item.d_tid)   return false;      
      if (d_quark != item.d_quark) return false;
    }
    return true;
  }

  bool Item::operator != (const Item& item) const {
    bool result = (*this == item);
    return result ? false : true;
  }

  // return the item quark

  long Item::getquark (void) const {
    rdlock ();
    long result = d_quark;
    unlock ();
    return result;
  }

  // return true if the item is static with that id
  
  bool Item::scheck (const long tid) const {
    rdlock ();
    bool result = ((d_type == STATIC) && (d_tid == tid));
    unlock ();
    return result;
  }

  // get the item tid

  long Item::gettid (void) const {
    rdlock ();
    if (d_type != STATIC) {
      unlock ();
      throw Exception ("item-error", "trying to access a dynamic item");
    }
    long result = d_tid;
    unlock ();
    return result;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 3;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_EQL     = zone.intern ("==");
  static const long QUARK_NEQ     = zone.intern ("!=");
  static const long QUARK_GETENUM = zone.intern ("get-enum");

  // return true if the given quark is defined

  bool Item::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Literal::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // operate this obejct with another object

  Object* Item::oper (t_oper type, Object* object) {
    Item* iobj = dynamic_cast <Item*> (object);
    switch (type) {
    case Object::EQL:
      if (iobj != nilp) return new Boolean (*this == *iobj);
      break;
    case Object::NEQ:
      if (iobj != nilp) return new Boolean (*this != *iobj);
      break;
    default:
      break;
    }
    throw Exception ("type-error", "invalid operand with item",
		     Object::repr (object));
  }

  // apply this item with a set of arguments and a quark

  Object* Item::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETENUM) {
	rdlock ();
	if (d_type == STATIC) {
	  unlock ();
	  throw Exception ("item-error", "cannot access static enumeration");
	}
	Object* result = p_obj;
	robj->post (result);
	unlock ();
	return result;
      }
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_EQL) {
	Object* result = oper (Object::EQL, argv->get (0));
	return result;
      }
      if (quark == QUARK_NEQ) {
	Object* result = oper (Object::NEQ, argv->get (0));
	return result;
      }
    } 
    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
