// ---------------------------------------------------------------------------
// - Strbuf.hpp                                                              -
// - standard object library - string buffer class definition                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_STRBUF_HPP
#define  AFNIX_STRBUF_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

namespace afnix {
  /// The Strbuf class is a string buffer which can be used to accumulate
  /// unicode characters. The class is similar to the buffer class except
  /// that it operates with quad representation.
  /// @author amaury darsch

  class Strbuf : public virtual Object {
  protected:
    /// the character buffer
    t_quad* p_buffer;
    /// the buffer allocated size
    long d_size;
    /// the buffer length
    long d_length;

  public:
    /// create a new buffer class.
    Strbuf (void);

    /// create a new buffer class with a default size.
    /// @param size the buffer default size
    Strbuf (const long size);
    
    /// create a new buffer and initialize it with a c string
    /// @param value the c string to initialize
    Strbuf (const char* value);

    /// create a new buffer and initialize it with a string
    /// @param value the string to initialize
    Strbuf (const String& value);

    /// destroy this buffer
    ~Strbuf (void);

    /// @return the class name
    String repr (void) const;

    /// reset this buffer
    virtual void reset (void);

    /// @return the length of this buffer
    virtual long length (void) const;

    /// resize this buffer
    /// @param size the new buffer size
    virtual void resize (const long size);

    /// add a character to this buffer
    /// @param value the character to add to this buffer
    virtual void add (const char value);

    /// add a unicode character to this buffer
    /// @param value the character to add to this buffer
    virtual void add (const t_quad value);

    /// add a character string in this buffer
    /// @param value the string to add
    virtual void add (const char* value);

    /// add a character buffer in this buffer
    /// @param buffer the buffer to add
    /// @param size   the buffer size
    virtual void add (const char* buffer, const long size);

    /// add a unicode string in this buffer
    /// @param value the string to add
    virtual void add (const t_quad* value);

    /// add a unicode buffer in this buffer
    /// @param buffer the buffer to add
    /// @param size   the buffer size
    virtual void add (const t_quad* buffer, const long size);

    /// add a string to this buffer
    /// @param value the string to add to this buffer
    virtual void add (const String& value);

    /// @return the buffer contents as a string
    virtual String tostring (void) const;

  private:
    // make the copy constructor private
    Strbuf (const Strbuf&);
    // make the assignment operator private
    Strbuf& operator =  (const Strbuf&);
  };
}

#endif
