// ---------------------------------------------------------------------------
// - Time.cpp                                                                -
// - standard object library - time class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Time.hpp"
#include "Vector.hpp"
#include "Integer.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "cclk.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // return the absolute value
  static inline long abs (const long abs) {
    return (abs < 0) ? -abs : abs;
  }
  static inline t_long abs (const t_long abs) {
    return (abs < 0) ? -abs : abs;
  }

  // -------------------------------------------------------------------------
  // - public section                                                        -
  // -------------------------------------------------------------------------

  // return the time in second since the origin

  t_long Time::gettclk (void) {
    return c_time ();
  }

  // return the timezone in seconds
  
  t_long Time::getzone (void) {
    return c_tzone ();
  }
  
  // return a formated string of the time zone

  String Time::tozone (void) {
    // get the time zone difference
    t_long wclk = c_tzone ();
    // save the sign
    bool negt = (wclk < 0);
    if (negt == true) wclk = -wclk;
    // get the minutes
    long mins = (long) ((wclk / 60LL) % 60LL);
    // compute hours
    long hour = (long) ((wclk / 3600LL) % 24LL);
     // format the result
    String result = negt ? '-' : '+';
    // add the hour
    if (hour < 10) result += '0';
    result += hour;
    // add the minutes
    if (mins < 10) result += '0';
    result += mins;
    // here we are
    return result;
  }

  // -------------------------------------------------------------------------
  // - class section                                                        -
  // -------------------------------------------------------------------------

  // create a current atc clock

  Time::Time (void) {
    d_tclk  = c_time ();
  }

  // create a specific time clock

  Time::Time (const t_long tclk) {
    d_tclk  = tclk;
  }

  // create a specific time clock

  Time::Time (const long hour, const long mins, const long secs) {
    settime (hour, mins, secs);
  }

  // copy construct this time

  Time::Time (const Time& that) {
    that.rdlock ();
    d_tclk = that.d_tclk;
    that.unlock ();
  }

  // return the class name

  String Time::repr (void) const {
    return "Time";
  }

  // return a clone of this object

  Object* Time::clone (void) const {
    return new Time (*this);
  }

  // add a specific time in second to the atc clock

  void Time::add (const t_long tclk) {
    wrlock ();
    d_tclk += tclk;
    unlock ();
  }

  // set the atc clock in seconds

  void Time::settime (const t_long tclk) {
    wrlock ();
    d_tclk = tclk;
    unlock ();
  }

  // set the time by specific elements

  void Time::settime (const long hour, const long mins, const long secs) {
    wrlock ();
    t_long tclk = hour * HSEC + mins * MSEC + secs;
    settime (tclk);
    unlock ();
  }
      
  // return the atc clock

  t_long Time::gettime (void) const {
    rdlock ();
    t_long result = d_tclk;
    unlock ();
    return result;
  }

  // get the number of seconds

  long Time::getsecs (const bool utc) const {
    rdlock ();
    // compute the working clock based on the timezone
    t_long wclk = abs (d_tclk);
    if (utc == false) wclk += (t_long) c_tzone ();
    // compute the seconds
    long result = (long) (wclk % 60LL);
    unlock ();
    return result;
  }

  // get the number of minutes

  long Time::getmins (const bool utc) const {
    rdlock ();
    // compute the working clock based on the timezone
    t_long wclk = abs (d_tclk);
    if (utc == false) wclk += (t_long) c_tzone ();
    // compute the minutes
    long result = (long) ((wclk / 60LL) % 60LL);
    unlock ();
    return result;
  }

  // get the number of hour

  long Time::gethour (const bool utc) const {
    rdlock ();
    // compute the working clock based on the timezone
    t_long wclk = abs (d_tclk);
    if (utc == false) wclk += (t_long) c_tzone ();
    // compute the hour
    long result = (long) ((wclk / 3600LL) % 24LL);
    unlock ();
    return result;
  }

  // format the time

  String Time::format (const bool utc) const {
    rdlock ();
    // compute the working clock based on the timezone
    t_long wclk = abs (d_tclk);
    if (utc == false) wclk += (t_long) c_tzone ();
    // compute the seconds
    long secs = (long) (wclk % 60LL);
    // compute minute
    long mins = (long) ((wclk / 60LL) % 60LL);
    // compute hours
    long hour = (long) ((wclk / 3600LL) % 24LL);
    // initialize the result
    String result;
    if (hour < 10) {
      result = result + '0' + hour + ':';
    } else {
      result = result + hour + ':';
    }
    if (mins < 10) {
      result = result + '0' + mins + ':';
    } else {
      result = result + mins + ':';
    }
    if (secs < 10) {
      result = result + '0' + secs;
    } else {
      result = result + secs;
    }
    unlock ();
    return result;
  }

  // format the time in a form similar to ISO-8601

  String Time::toiso (const bool utc) const {
    rdlock ();
    try {
      // get the time reprsentation (local)
      String result = Time::format (utc);
      // add the suffix if any
      if (utc == true) result += 'Z';
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // format the time in a form similar to RFC-2822

  String Time::torfc (void) const {
    rdlock ();
    try {
      // get the time reprsentation (local)
      String result = Time::format (false);
      // add the timezone difference (including daylight if any)
      result += ' ';
      result += Time::tozone ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 10;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_ADD     = zone.intern ("add");
  static const long QUARK_TOISO   = zone.intern ("to-iso");
  static const long QUARK_TORFC   = zone.intern ("to-rfc");
  static const long QUARK_FORMAT  = zone.intern ("format");
  static const long QUARK_ADDMNS  = zone.intern ("add-minutes");
  static const long QUARK_ADDHRS  = zone.intern ("add-hours");
  static const long QUARK_ADDDAYS = zone.intern ("add-days");
  static const long QUARK_SETTIME = zone.intern ("set-time");
  static const long QUARK_GETTIME = zone.intern ("get-time");
  static const long QUARK_GETSECS = zone.intern ("seconds");
  static const long QUARK_GETMINS = zone.intern ("minutes");
  static const long QUARK_GETHOUR = zone.intern ("hours");

  // create a new object in a generic way
 
  Object* Time::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create a default time object
    if (argc == 0) return new Time;
    if (argc == 1) {
      long tval = argv->getint (0);
      return new Time (tval);
    }
    if (argc == 3) {
      long hour = argv->getint (0);
      long mins = argv->getint (1);
      long secs = argv->getint (2);
      return new Time (hour, mins, secs);
    }
    throw Exception ("argument-error",
                     "too many argument with time constructor");
  }
 
  // return true if the given quark is defined

  bool Time::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* Time::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 arguments
    if (argc == 0) {
      if (quark == QUARK_TOISO)   return new String  (toiso   (false));
      if (quark == QUARK_TORFC)   return new String  (torfc   ());
      if (quark == QUARK_FORMAT)  return new String  (format  (false));
      if (quark == QUARK_GETTIME) return new Integer (gettime ());
      if (quark == QUARK_GETSECS) return new Integer (getsecs (true));
      if (quark == QUARK_GETMINS) return new Integer (getmins (true));
      if (quark == QUARK_GETHOUR) return new Integer (gethour (true));
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADD) {
	t_long tval = argv->getint (0);
	add (tval);
	return nilp;
      }
      if (quark == QUARK_SETTIME) {
	t_long tclk = argv->getint (0);
	settime (tclk);
	return nilp;
      }
      if (quark == QUARK_ADDMNS) {
	t_long num = argv->getint (0);
	t_long sec = num * (t_long) MSEC;
	add (sec);
	return nilp;
      }
      if (quark == QUARK_ADDHRS) {
	t_long num = argv->getint (0);
	t_long sec = num * (t_long) HSEC;
	add (sec);
	return nilp;
      }
      if (quark == QUARK_ADDDAYS) {
	t_long num = argv->getint (0);
	t_long sec = num * (t_long) DSEC;
	add (sec);
	return nilp;
      }
      if (quark == QUARK_FORMAT) {
	bool utc = argv->getbool (0);
	return new String (format (utc));
      }
      if (quark == QUARK_TOISO) {
	bool utc = argv->getbool (0);
	return new String (toiso (utc));
      }
      if (quark == QUARK_GETSECS) {
	bool utc = argv->getbool (0);
	return new Integer (getsecs (utc));
      }
      if (quark == QUARK_GETMINS) {
	bool utc = argv->getbool (0);
	return new Integer (getmins (utc));
      }
      if (quark == QUARK_GETHOUR) {
	bool utc = argv->getbool (0);
	return new Integer (gethour (utc));
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
