// ---------------------------------------------------------------------------
// - Vector.cpp                                                              -
// - standard object library - dynamic vector class implementation           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Byte.hpp"
#include "Cons.hpp"
#include "Real.hpp"
#include "Stdsid.hxx"
#include "Vector.hpp"
#include "Output.hpp"
#include "Method.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create an empty vector

  Vector::Vector (void) {
    d_size   = 0;
    d_length = 0;
    p_vector = nilp;
  }

  // create a vector with an original size
  
  Vector::Vector (const long size) {
    if (size < 0) throw Exception ("size-error","negative vector size");
    d_size   = size;
    d_length = 0;
    p_vector = new Object*[d_size];
    for (long i = 0; i < d_size; i++) p_vector[i] = nilp;
  }
  
  // copy constructor for this vector

  Vector::Vector (const Vector& that) {
    that.rdlock ();
    try {
      d_size   = that.d_size;
      d_length = that.d_length;
      p_vector = new Object*[d_size];
      for (long i = 0; i < d_length; i++)
	p_vector[i] = Object::iref (that.p_vector[i]);
    } catch (...) {
      that.unlock ();
      throw;
    }
    that.unlock ();
  }

  // destroy this vector

  Vector::~Vector (void) {
    for (long i = 0; i < d_length; i++)
      Object::dref (p_vector[i]);
    delete [] p_vector;
  }

  // return the class name

  String Vector::repr (void) const {
    return "Vector";
  }

  // assignment operator for this class

  Vector& Vector::operator = (const Vector& that) {
    // protect against this = that
    if (this == &that) return *this;
    // lock everybody
    wrlock ();
    that.rdlock ();
    // clean vector first
    if (d_length != 0) {
      for (long i = 0; i < d_length; i++)
      Object::dref (p_vector[i]);
      delete [] p_vector;
    }
    // copy old to new
    d_size   = that.d_size;
    d_length = that.d_length;
    p_vector = new Object*[d_size];
    for (long i = 0; i < d_length; i++)
    p_vector[i] = Object::iref (that.p_vector[i]);
    that.unlock ();
    unlock ();
    return *this;
  }

  // make this vector a shared object

  void Vector::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    for (long i = 0; i < d_length; i++) {
      Object* obj = p_vector[i];
      if (obj != nilp) obj->mksho ();
    }
  }

  // return the vector serial code

  t_byte Vector::serialid (void) const {
    return SERIAL_VECT_ID;
  }

  // serialize this vector

  void Vector::wrstream (Output& os) const {
    rdlock ();
    // write the vector length
    Integer vlen (d_length);
    vlen.wrstream (os);
    // write the objects
    for (long i = 0; i < d_length; i++) {
      Object* obj = get (i);
      if (obj == nilp) {
	Serial::wrnilid (os);
      } else {
	Serial* sobj = dynamic_cast <Serial*> (obj);
	if (sobj == nilp) {
	  unlock ();
	  throw Exception ("serial-error", "cannot serialize object", 
			   obj->repr ());
	}
	sobj->serialize (os);
      }
    }
    unlock ();
  }

  // deserialize this vector

  void Vector::rdstream (Input& is) {
    wrlock ();
    reset ();
    // get the vector length
    Integer vlen;
    vlen.rdstream (is);
    long len = vlen.tointeger ();
    // read in each object
    for (long i = 0; i < len; i++) append (Serial::deserialize (is));
    unlock ();
  }

  // reset this vector

  void Vector::reset (void) {
    wrlock ();
    if (p_vector != nilp) {
      for (long i = 0; i < d_length; i++) Object::dref (p_vector[i]);
    }
    d_length = 0;
    unlock ();
  }

  // get the number of element in this vector

  long Vector::length (void) const {
    rdlock ();
    long result = d_length;
    unlock ();
    return result;
  }

  // return true if the vector is empty

  bool Vector::empty (void) const {
    rdlock ();
    bool result = (d_length == 0);
    unlock ();
    return result;
  }

  // add a new element in this vector

  void Vector::append (Object* object) {
    wrlock ();
    try {
      // check if we have to resize the vector
      if (d_length >= d_size) {
	long size = (d_size <= 0) ? 1 : d_size * 2;
	Object** vector = new Object*[size];
	for (long i = 0; i < d_length; i++)
	  vector[i] = p_vector[i];
	delete [] p_vector;
	d_size   = size;
	p_vector = vector;
      }
      // check for shared
      if ((p_shared != nilp) && (object != nilp)) object->mksho ();
      // set the object in this vector
      p_vector[d_length++] = Object::iref (object);
      // unlock the vector
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // set an object at a certain position in this vector. The old object is
  // destroyed.
  
  void Vector::set (const long index, Object* object) {
    wrlock ();
    try {
      // check that we are bounded
      if (index >= d_length) 
	throw Exception ("index-error","index is out of range");
      // check for shared
      if ((p_shared != nilp) && (object != nilp)) object->mksho ();    
      // set the object
      Object::dref (p_vector[index]);
      p_vector[index] = Object::iref (object);
      // unlock the vector
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get an object at a certain position
  
  Object* Vector::get (const long index) const {
    rdlock ();
    try {
      // check that we are bounded
      if ((index < 0) || (index >= d_length))
	throw Exception ("index-error","index is out of range");
      // get the object and unlock
      Object* result = p_vector[index];
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the first object in this vector
    
  Object* Vector::first (void) const {
    rdlock ();
    try {
      Object* result = get (0);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the last object in this vector

  Object* Vector::last (void) const {
    rdlock ();
    try {
      Object* result = get (d_length-1);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // backtrack one position in this vector

  Object* Vector::backt (void) {
    wrlock ();
    try {
      // get the last object
      Object* result = last ();
      // fix length and result object
      d_length--;
      Object::tref (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // pop the first object from this vector

  Object* Vector::pop (void) {
    wrlock ();
    try {
      // get the first object
      Object* result = get (0);
      // shift the remaining arguments
      for (long i = 1; i < d_length; i++) {
	p_vector[i-1] = p_vector[i];
	p_vector[i]   = nilp;
      }
      // fix length and result object
      d_length--;
      Object::tref (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return true if the object exists in this vector

  bool Vector::exists (Object* object) const {
    rdlock ();
    for (long i = 0; i < d_length; i++) {
      if (p_vector[i] == object) {
	unlock ();
	return true;
      }
    }
    unlock ();
    return false;
  }

  // find an object in this vector

  long Vector::find (Object* object) {
    if (object == nilp) return -1;
    rdlock ();
    for (long i = 0; i < d_length; i++) {
      if (p_vector[i] == object) {
	unlock ();
	return i;
      }
    }
    unlock ();
    return -1;
  }

  // remove an object by index and repack the vector

  void Vector::remove (const long index) {
    if ((index < 0) || (index >= d_length))
      throw Exception ("index-error","index is out of range");
    wrlock ();
    try {
      // remove the object
      Object::dref (p_vector[index]);
      // repack the vector
      long mark = d_length - 1;
      for (long i = index; i < mark; i++)
	p_vector[i] = p_vector[i+1];
      d_length = mark;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // remove and object and repack the vector

  void Vector::remove (Object* object) {
    if (object == nilp) return;
    wrlock ();
    try {
      long index = find (object);
      if (index == -1) {
	unlock ();
	return;
      }
      remove (index);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // shift the vector and create a new one

  Vector* Vector::shift (void) const {
    rdlock ();
    Vector* result = new Vector;
    try {
      for (long i = 1; i < d_length; i++) {
	result->append (p_vector[i]);
      }
      unlock ();
      return result;
    } catch (...) {
      delete result;
      unlock ();
      throw;
    }
  }

  // merge a vector into this one

  void Vector::merge (const Vector& v) {
    wrlock ();
    long len = v.length ();
    for (long i = 0; i < len; i++) append (v.get (i));
    unlock ();
  }

  // return a new vector iterator

  Iterator* Vector::makeit (void) {
    return new Vectorit (this);
  }

  // get an integer value from an index

  t_long Vector::getint (const long index) const {
    Object*  obj  = get (index);
    Integer* iobj = dynamic_cast <Integer*> (obj);
    if (iobj == nilp) 
      throw Exception ("type-error", "looking for integer but got",
		       Object::repr (obj));
    return iobj->tointeger ();
  }

  // get a real value from an index

  t_real Vector::getreal (const long index) const {
    Object* obj  = get (index);
    Real*   robj = dynamic_cast <Real*> (obj);
    if (robj == nilp) 
      throw Exception ("type-error", "looking for real but got",
		       Object::repr (obj));
    return robj->toreal ();
  }

  // get a real value from an index (either from an integer)

  t_real Vector::getireal (const long index) const {
    Object*   obj = get (index);
    Integer* iobj = dynamic_cast <Integer*> (obj);
    if (iobj != nilp) return iobj->tointeger ();
    Real* robj = dynamic_cast <Real*> (obj);
    if (robj != nilp) return robj->toreal ();
    // not found
    throw Exception ("type-error", "looking for real but got", 
		     Object::repr (obj));
  }

  // get a boolean value from an index

  bool Vector::getbool (const long index) const {
    Object*   obj = get (index);
    Boolean* bobj = dynamic_cast <Boolean*> (obj);
    if (bobj == nilp) 
      throw Exception ("type-error", "looking for boolean but got",
		       Object::repr (obj));
    return bobj->toboolean ();
  }

  // get a character value from an index

  t_byte Vector::getbyte (const long index) const {
    rdlock ();
    try {
      Object* obj = get (index);
      Byte* bobj = dynamic_cast <Byte*> (obj);
      if (bobj != nilp) {
	t_byte result =  bobj->tobyte ();
	unlock ();
	return result;
      }
      Character* cobj = dynamic_cast <Character*> (obj);
      if (cobj != nilp) {
	t_byte result =  (t_byte) cobj->tochar ();
	unlock ();
	return result;
      }
      throw Exception ("type-error", "looking for character but got",
		       Object::repr (obj));
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get a character value from an index

  t_quad Vector::getchar (const long index) const {
    Object*     obj = get (index);
    Character* cobj = dynamic_cast <Character*> (obj);
    if (cobj == nilp) 
      throw Exception ("type-error", "looking for character but got",
		       Object::repr (obj));
    return cobj->toquad ();
  }

  // get a string value from an index

  String Vector::getstring (const long index) const {
    Object*  obj = get (index);
    String* sobj = dynamic_cast <String*> (obj);
    if (sobj == nilp) 
      throw Exception ("type-error", "looking for string but got",
		       Object::repr (obj));
    return *sobj;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 13;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GET    = zone.intern ("get");
  static const long QUARK_SET    = zone.intern ("set");
  static const long QUARK_POP    = zone.intern ("pop");
  static const long QUARK_FIND   = zone.intern ("find");
  static const long QUARK_LAST   = zone.intern ("last");
  static const long QUARK_FIRST  = zone.intern ("first");
  static const long QUARK_BACKT  = zone.intern ("back-track");
  static const long QUARK_RESET  = zone.intern ("reset");
  static const long QUARK_LENGTH = zone.intern ("length");
  static const long QUARK_APPEND = zone.intern ("append");
  static const long QUARK_EXISTS = zone.intern ("exists-p");
  static const long QUARK_REMOVE = zone.intern ("remove");
  static const long QUARK_EMPTYP = zone.intern ("empty-p");

  // generate a vector of arguments

  Vector* Vector::eval (Runnable* robj, Nameset* nset, Cons* args) {
    long len = 0;
    if ((args == nilp) || ((len = args->length ()) == 0)) return nilp;
    Vector* result = new Vector (len);
  
    // loop in the cons cell and accumulate arguments
    try {
      while (args != nilp) {
	Object* car = args->getcar ();
	if (car == nilp) 
	  result->append ((Object*) nilp);
	else
	  result->append (car->eval (robj,nset));
	args = args->getcdr ();
      }
    } catch (...) {
      delete result;
      throw;
    }
    return result;
  }

  // create a new object in a generic way

  Object* Vector::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Vector;
    return new Vector (*argv);
  }

  // return true if the given quark is defined

  bool Vector::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Iterable::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* Vector::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH) return new Integer (length ());
      if (quark == QUARK_EMPTYP) return new Boolean (empty  ());
      if (quark == QUARK_RESET) {
	reset  ();
	return nilp;
      }
      if (quark == QUARK_FIRST) {
	rdlock ();
	try {
	  Object* result = first ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_LAST) {
	rdlock ();
	try {
	  Object* result = last ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_BACKT) {
	wrlock ();
	try {
	  Object* result = backt ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_POP) {
	wrlock ();
	try {
	  Object* result = pop ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_GET) {
	rdlock ();
	try {
	  Object* result = get (argv->getint (0));
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_APPEND) {
	Object* result = argv->get (0);
	append (result);
	robj->post (result);
	return result;
      }
      if (quark == QUARK_EXISTS) {
	Object* obj = argv->get (0);
	bool result = exists (obj);
	return new Boolean (result);
      }
      if (quark == QUARK_FIND) {
	Object* obj = argv->get (0);
	long index = find (obj);
	if (index == -1) return nilp;
	return new Integer (index);
      }
      if (quark == QUARK_REMOVE) {
	Object* obj = argv->get (0);
	remove (obj);
	return nilp;
      }
    }

    // dispatch 2 arguments
    if (argc == 2) {
      if (quark == QUARK_SET) {
	t_long val = argv->getint (0);
	Object* result = argv->get (1);
	set (val, result);
	robj->post (result);
	return result;
      }
    }

    // call the object method
    return Iterable::apply (robj, nset, quark, argv);
  }

  // -------------------------------------------------------------------------
  // - iterator section                                                      -
  // -------------------------------------------------------------------------

  // create a new vector iterator

  Vectorit::Vectorit (Vector* vec) {
    Object::iref (p_vector = vec);
    if (p_vector != nilp) p_vector->rdlock ();
    begin ();
  }

  // destroy this vector iterator

  Vectorit::~Vectorit (void) {
    if (p_vector != nilp) p_vector->unlock ();
    Object::dref (p_vector);
  }

  // return the class name

  String Vectorit::repr (void) const {
    return "Vectorit";
  }

  // make this vector iterator a shared object

  void Vectorit::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_vector != nilp) p_vector->mksho ();
  }

  // reset the iterator to the begining

  void Vectorit::begin (void) {
    wrlock ();
    d_index = 0;
    unlock ();
  }

  // reset the iterator to the end

  void Vectorit::end (void) {
    wrlock ();
    if (p_vector != nilp) {
      d_index = (p_vector->d_length == 0) ? 0 : p_vector->d_length - 1;
    }
    unlock ();
  }

  // go to the next object

  void Vectorit::next (void) {
    wrlock ();
    if (p_vector != nilp) {
      if (++d_index >= p_vector->d_length) d_index = p_vector->d_length;
    }
    unlock ();
  }

  // go to the previous object

  void Vectorit::prev (void) {
    wrlock ();
    if (--d_index < 0) d_index = 0;
    unlock ();
  }

  // get the object at the current position

  Object* Vectorit::getobj (void) const {
    rdlock ();
    Object* result = nilp;
    if (p_vector != nilp) {
      if (d_index < p_vector->d_length) {
	result = p_vector->get (d_index);
      }
    }
    unlock ();
    return result;
  }

  // return true if the iterator is at the end

  bool Vectorit::isend (void) {
    rdlock ();
    bool result = false;
    if (p_vector != nilp) {
      result = (d_index >= p_vector->d_length);
    }
    unlock ();
    return result;
  }
}
