// ---------------------------------------------------------------------------
// - Uri.hpp                                                                 -
// - afnix:www module - uri class definition                                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_URI_HPP
#define  AFNIX_URI_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

namespace afnix {

  /// The Uri class is a base object used to parse or build a uniform
  /// resource identifer as defined by RFC 3986. The URI can be built by 
  /// specifiying each component or by parsing a string. When a string is
  /// given in the constructor, the class parses the string and extract all
  /// components. The URI components are the scheme, the authority,
  /// the path, the query and the fragment. The class also takes care of the
  /// character escaping.
  /// @author amaury darsch

  class Uri : public Object {
  public:
    /// escape a name by cnverting the special characters
    /// @param name the name to escape
    static String escape (const String& name);

  private:
    /// the uri scheme
    String d_scheme;
    /// the uri authority
    String d_auth;
    /// the uri path
    String d_path;
    /// the uri query
    String d_query;
    /// the uri fragment
    String d_fragment;

  public:
    /// create an empty uri
    Uri (void);

    /// create a uri by string
    /// @param uri the string to parse
    Uri (const String& uri);

    /// copy construct this uri
    /// @param that the uri to copy
    Uri (const Uri& that);

    /// assign a uri to this one
    /// @param that the uri to assign
    Uri& operator = (const Uri& that);

    /// @return the class name
    String repr (void) const;

    /// reset an uri name to its default
    void reset (void);

    /// parse an uri by name
    /// @param uri the uri to parse
    void parse (const String& uri);

    /// @return this uri scheme
    String getscheme (void) const;

    /// @return this uri authority
    String getauth (void) const;

    /// @return this uri path
    String getpath (void) const;

    /// @return this uri query
    String getquery (void) const;

    /// @return this uri fragment
    String getfragment (void) const;

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);
    
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;
    
    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
